/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.dialog;

import java.util.ArrayList;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileGroup;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileManager;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.server.CommandManager;
import jp.sourceforge.tomoyo.core.server.ICommand;
import jp.sourceforge.tomoyo.ui.editor.Activator;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

public class ChangeProfileDialog extends TitleAreaDialog {

	private IProject project;
	private Domain[] domainArray;
	
	public ChangeProfileDialog(Shell parentShell, IProject project, Domain[] domainArray) {
		super(parentShell);
		this.project = project;
		this.domainArray = domainArray;
	}

    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.ChangeProfileDialog_ShellTitle);
    }
    
	protected int getShellStyle() {
//		return SWT.MAX | SWT.RESIZE;
		return SWT.SHELL_TRIM;
	}

	protected Point getInitialSize() {
		return new Point(400, 500);
	}

	@Override
	protected Button createButton(Composite parent, int id, String label, boolean defaultButton) {
		if (id == IDialogConstants.OK_ID) {
			Button okButton = super.createButton(parent, id, label, defaultButton);
			okButton.setEnabled((domainArray != null && domainArray.length > 0));
			return okButton;
		} else {
			return super.createButton(parent, id, label, defaultButton);
		}
	}

	protected Control createDialogArea(Composite parent) {
		setTitle(Messages.ChangeProfileDialog_DialogTitle);
		setTitleImage(Activator.getImage("coloring_wiz.gif")); //$NON-NLS-1$
		
		Composite composite = (Composite)super.createDialogArea(parent);

		Composite container = new Composite(composite, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		layout.marginTop = 5;
		layout.marginLeft = 5;
		layout.marginRight = 5;
		container.setLayout(layout);
		container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		createViewer(container);
		createRadioButtons(container);

		initializeUI();

		/*
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent,
		"jp.sourceforge.tomoyo.doc.open_coloring_dialog");		 //$NON-NLS-1$
		*/
		
	    return composite;
	}

	private TableViewer viewer = null;

	private void createViewer(Composite parent) {
		viewer = new TableViewer(parent, SWT.BORDER);
		Table table = viewer.getTable();
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		table.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		table.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDoubleClick(MouseEvent e) {
				okPressed();
			}
		});
		table.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setErrorMessage(null);
			}
		});
		
		TableColumn column = new TableColumn(table,SWT.CENTER);
		column.setText(Messages.ChangeProfileDialog_ColumnProfile);
		column.setWidth(50);
		
		column = new TableColumn(table,SWT.CENTER);
		column.setText(Messages.ChangeProfileDialog_ColumnComment);
		column.setWidth(100);
		
		column = new TableColumn(table,SWT.CENTER);
		column.setText(Messages.ChangeProfileDialog_ColumnDescription);
		column.setWidth(200);
		
		viewer.setColumnProperties(COLUMN_PROPERTIES); 
		    
		CellEditor[] editors = new CellEditor[] {
		    new TextCellEditor(table),
		    new TextCellEditor(table),
		    new TextCellEditor(table),
		};
		viewer.setCellEditors(editors);
		viewer.setContentProvider(new ArrayContentProvider());
		viewer.setLabelProvider(new ColoringLabelProvider());
		
		ArrayList<ProfileGroup> input = ProfileManager.getInstance().getProjectProfile(project).toList(true);
		viewer.setInput(input);
	}

	private static final String[] COLUMN_PROPERTIES = new String[] {
		"no", //$NON-NLS-1$
		"comment", //$NON-NLS-1$
		"description", //$NON-NLS-1$
    };
	
	private class ColoringLabelProvider extends LabelProvider implements ITableLabelProvider, ITableColorProvider {
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		public String getColumnText(Object element, int columnIndex) {
			ProfileGroup group = (ProfileGroup)element;
			String result = ""; //$NON-NLS-1$
			switch (columnIndex) {
			case 0:
				result = String.valueOf(group.getNo());
				break;
			case 1:
				result = group.getComment();
				break;
			case 2:
				result = group.getDescription();
				break;
			default:
				result = ""; //$NON-NLS-1$
				break;
			}
			return result;
		}
		public Color getBackground(Object element, int columnIndex) {
			ProfileGroup group = (ProfileGroup)element;
			switch (columnIndex) {
			case 0:
			case 1:
			case 2:
			case 3:
				RGB rgb = group.getBackcolor();
				if (rgb != null)
					return Activator.getDefault().getColor(rgb);
			}
			return null;
		}
		public Color getForeground(Object element, int columnIndex) {
			ProfileGroup group = (ProfileGroup)element;
			switch (columnIndex) {
			case 0:
			case 1:
			case 2:
			case 3:
				RGB rgb = group.getForecolor();
				if (rgb != null)
					return Activator.getDefault().getColor(rgb);
			}
			return null;
		}
	}
	
	private Button[] radioButtons = new Button[2];
	
	private void createRadioButtons(Composite composite) {
		radioButtons[0] = createRadioButton(composite, Messages.ChangeProfileDialog_OptionRecursive);
		radioButtons[1] = createRadioButton(composite, Messages.ChangeProfileDialog_OptionNotRecursive);
		radioButtons[0].setSelection(true);
	}
	
    /**
     * Creates a radio button with the given parent and text.
     *
     * @param parent the parent composite
     * @param text the text for the check box
     * @return the radio box button
     */
    protected Button createRadioButton(Composite parent, String text) {
        Button button = new Button(parent, SWT.RADIO);
        button.setText(text);
        button.setFont(parent.getFont());
        return button;
    }
    
	private void initializeUI() {
		viewer.getControl().setFocus();
	}

	protected void okPressed() {
		ProfileGroup group = getGroup();
		if (group == null)
			return;
		if (save(group)) {
			super.okPressed();
		} else {
			setErrorMessage(Messages.ChangeProfileDialog_ChangeNGMessage);
		}
	}

	private boolean save(ProfileGroup selectedGroup) {
		String strDomain = domainArray[0].getText();
		if (selectedGroup == null)
			return false;
		boolean recursive = radioButtons[0].getSelection();
		
		StringBuffer setProfileCommand = new StringBuffer();
		setProfileCommand.append(CommandManager.createCCSToolPath(project, CommandManager.CMD_SETPROFILE));
		setProfileCommand.append(" "); //$NON-NLS-1$
		String option = null;
		if (recursive) {
			option = "-r"; //$NON-NLS-1$
		}
		setProfileCommand.append(selectedGroup.getNo());
		setProfileCommand.append(" "); //$NON-NLS-1$
		setProfileCommand.append("'" + strDomain + "'"); //$NON-NLS-1$ //$NON-NLS-2$
		
		ICommand command = CommandManager.getInstance().exec(
				project,
				CommandManager.createCCSToolPath(project, CommandManager.CMD_SETPROFILE),
				new String[] {option},
				new String[] {String.valueOf(selectedGroup.getNo()), "'" + strDomain + "'"} //$NON-NLS-1$ //$NON-NLS-2$
				);
		if (command.isSuccessful()) {
			LocalResource domainPolicy = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();
			if (ProcResourceManager.getInstance().synchronize(project, domainPolicy)) {
				String message = Messages.ChangeProfileDialog_ChangeOKMessage;
				TomoyoCorePlugin.getDefault().setStatusMessage(message);
				return true;
			} else {
				String message = Messages.ChangeProfileDialog_ChangeNGMessage;
				TomoyoCorePlugin.getDefault().setStatusErrorMessage(message);
				return false;
			}
		}
		return false;
	}
	
	private ProfileGroup getGroup() {
		if (viewer.getSelection() instanceof StructuredSelection) {
			Object firstElement = ((StructuredSelection)viewer.getSelection()).getFirstElement();
			if (firstElement instanceof ProfileGroup) {
				ProfileGroup group = (ProfileGroup)firstElement;
				return group;
			}
		}
		return null;
	}

}
