/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.dialog;

import java.util.ArrayList;
import java.util.Hashtable;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.model.domain.AccessPermission;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.server.CommandManager;
import jp.sourceforge.tomoyo.core.server.WriteCommand;
import jp.sourceforge.tomoyo.ui.editor.Activator;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

public class PatternizeAccessPermissionsDialog extends TitleAreaDialog {

	private IProject project;
	
	private Domain domain;
	private AccessPermission[] accessPermissions;
	
	public PatternizeAccessPermissionsDialog(Shell parentShell, IProject project, Domain domain, AccessPermission[] accessPermissions) {
		super(parentShell);
		this.project = project;
		this.domain = domain;
		this.accessPermissions = accessPermissions;
	}

    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.PatternizeAccessPermissionsDialog_ShellTitle);
    }
    
	protected int getShellStyle() {
		return SWT.MAX | SWT.RESIZE;
	}

	protected Point getInitialSize() {
		return new Point(800, 700);
	}

	@Override
	protected Button createButton(Composite parent, int id, String label, boolean defaultButton) {
		if (id == IDialogConstants.OK_ID) {
			Button okButton = super.createButton(parent, id, label, defaultButton);
			if (domain == null) {
				okButton.setEnabled(false);
			} else {
				okButton.setEnabled((accessPermissions != null && accessPermissions.length > 0));
			}
			return okButton;
		} else {
			return super.createButton(parent, id, label, defaultButton);
		}
	}

	protected Control createDialogArea(Composite parent) {
		setTitle(Messages.PatternizeAccessPermissionsDialog_DialogAreaTitle);
//		setTitleImage(Activator.getImage("")); //$NON-NLS-1$
		
		Composite composite = (Composite)super.createDialogArea(parent);

		Composite container = new Composite(composite, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		layout.marginTop = 5;
		layout.marginLeft = 5;
		layout.marginRight = 5;
		container.setLayout(layout);
		container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		createViewer(container);
		createOptionButtons(container);
		createDeleteOptionButton(container);

		initializeUI();

		/*
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent,
		"jp.sourceforge.tomoyo.doc.open_coloring_dialog");		 //$NON-NLS-1$
		*/
		
	    return composite;
	}

	private TableViewer viewer = null;

	private void createViewer(Composite parent) {
		viewer = new TableViewer(parent, SWT.BORDER);
		Table table = viewer.getTable();
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		table.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		TableColumn column = new TableColumn(table,SWT.LEFT);
		column.setText(Messages.PatternizeAccessPermissionsDialog_ColumnBefore);
		column.setWidth(300);
		
		column = new TableColumn(table,SWT.LEFT);
		column.setText(Messages.PatternizeAccessPermissionsDialog_ColumnAfter);
		column.setWidth(300);
		
		viewer.setColumnProperties(COLUMN_PROPERTIES); 
		    
		CellEditor[] editors = new CellEditor[] {
		    new TextCellEditor(table),
		    new TextCellEditor(table),
		};
		viewer.setCellEditors(editors);
		viewer.setContentProvider(new ArrayContentProvider());
		viewer.setLabelProvider(new ArrayLabelProvider());
		
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				handleTableSelectionChanged();
			}
		});
		
		viewer.setInput(accessPermissions);
	}

	private void handleTableSelectionChanged() {
	}

	private static final String[] COLUMN_PROPERTIES = new String[] {
		"before", //$NON-NLS-1$
		"after", //$NON-NLS-1$
    };

	private class ArrayLabelProvider extends LabelProvider implements ITableLabelProvider, ITableColorProvider {
		
		public void clearCache() {
			foretxTable.clear();
			uniquePatterns.clear();
			statusTable.clear();
			deleteAPs.clear();
		}
		
		private Hashtable<String, String> foretxTable = new Hashtable<String, String>();
		private Hashtable<String, String> uniquePatterns = new Hashtable<String, String>();
		private Hashtable<String, Boolean> statusTable = new Hashtable<String, Boolean>();
		private ArrayList<AccessPermission> deleteAPs = new ArrayList<AccessPermission>();

		private void cacheElement(AccessPermission ap) {
			if (foretxTable.get(ap.getText()) != null)
				return;
			if (isSupported(ap)) {
				String converted = convert(ap);
				if (converted.equals(ap.getText())) {
					foretxTable.put(ap.getText(), "(" + Messages.PatternizeAccessPermissionsDialog_ConvertInvalidMessage + ")"); //$NON-NLS-1$ //$NON-NLS-2$
					statusTable.put(ap.getText(), Boolean.FALSE);
				} else {
					if (uniquePatterns.get(converted) == null) {
						uniquePatterns.put(converted, converted);
						foretxTable.put(ap.getText(), converted);
						statusTable.put(ap.getText(), Boolean.TRUE);
					} else {
						foretxTable.put(ap.getText(), "(" + Messages.PatternizeAccessPermissionsDialog_ConvertDuplicateMessage + ")"); //$NON-NLS-1$ //$NON-NLS-2$
						statusTable.put(ap.getText(), Boolean.FALSE);
					}
					deleteAPs.add(ap);
				}
			} else {
				foretxTable.put(ap.getText(), "(" + Messages.PatternizeAccessPermissionsDialog_ConvertNotSupportedMessage + ")"); //$NON-NLS-1$ //$NON-NLS-2$
				statusTable.put(ap.getText(), Boolean.FALSE);
			}
		}
		
		public String[] getUniquePatternArray() {
			return uniquePatterns.values().toArray(new String[uniquePatterns.size()]);
		}
		
		public AccessPermission[] getPatternizedAccessPermissionArray() {
			return deleteAPs.toArray(new AccessPermission[deleteAPs.size()]);
		}
		
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		
		public String getColumnText(Object element, int columnIndex) {
			AccessPermission ap = (AccessPermission)element;
			String result = ""; //$NON-NLS-1$
			switch (columnIndex) {
			case 0:
				result = ap.getText();
				break;
			case 1:
				cacheElement(ap);
				result = foretxTable.get(ap.getText());
				break;
			default:
				result = ""; //$NON-NLS-1$
				break;
			}
			return result;
		}
		
		public static final int PATTERNIZE_METHOD_FILENAME_WHOLE = 0;
		public static final int PATTERNIZE_METHOD_FILENAME_ONLY = 1;
		public static final int PATTERNIZE_METHOD_FILENAME_EXTENSION = 2;
			
		private int patternizeMethod = PATTERNIZE_METHOD_FILENAME_WHOLE;
		
		public void setPatternizeMethod(int id) {
			patternizeMethod = id;
		}
		
		private String convert(AccessPermission ap) {
			if (isSupported(ap)) {
				String result = null;
				String directive = ap.getDirective();
				String content = ap.getContents();
				String[] paths = toPaths(content);
				switch (patternizeMethod) {
				case PATTERNIZE_METHOD_FILENAME_WHOLE:
					paths[paths.length - 1] = "\\*"; //$NON-NLS-1$
					result = directive + " " + toPath(paths); //$NON-NLS-1$
					break;
				case PATTERNIZE_METHOD_FILENAME_ONLY:
					paths[paths.length - 1] = patternizeFilename(paths[paths.length - 1]);
					result = directive + " " + toPath(paths); //$NON-NLS-1$
					break;
				case PATTERNIZE_METHOD_FILENAME_EXTENSION:
					paths[paths.length - 1] = patternizeExtension(paths[paths.length - 1]);
					result = directive + " " + toPath(paths); //$NON-NLS-1$
					break;
				default:
					
				}
				return result;
			} else {
				return ap.getText();
			}
		}

		private String[] toPaths(String content) {
			if (content == null)
				return new String[0];
			else {
				String[] paths = content.split("/"); //$NON-NLS-1$
				return paths;
			}
		}

		private String patternizeFilename(String filename) {
			int index = filename.lastIndexOf("."); //$NON-NLS-1$
			if (index > 0) {
				StringBuffer sb = new StringBuffer();
				sb.append("\\*"); //$NON-NLS-1$
				sb.append("."); //$NON-NLS-1$
				sb.append(filename.substring(index + 1));
				return sb.toString();
			} else {
				if (filename.length() > 0) {
					return "\\*"; //$NON-NLS-1$
				}
				return filename;
			}
		}
		
		private String patternizeExtension(String filename) {
			int index = filename.lastIndexOf("."); //$NON-NLS-1$
			if (index > 0) {
				StringBuffer sb = new StringBuffer();
				sb.append(filename.substring(0, index));
				sb.append("."); //$NON-NLS-1$
				sb.append("\\*"); //$NON-NLS-1$
				return sb.toString();
			}
			return filename;
		}
		
		private String toPath(String[] paths) {
			StringBuffer sb = new StringBuffer();
			for (int cnt = 0; cnt < paths.length; cnt++) {
				sb.append(paths[cnt]);
				if (cnt < paths.length - 1)
					sb.append("/"); //$NON-NLS-1$
			}
			return sb.toString();
		}

		private boolean isSupported(AccessPermission ap) {
			String directive = ap.getDirective();
			if (directive.equals("1") || directive.equals("2") || directive.equals("3") || //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					directive.equals("4") || directive.equals("5") || directive.equals("6") || //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					directive.equals("7") || //$NON-NLS-1$
					directive.equals("allow_create") || //$NON-NLS-1$
					directive.equals("allow_rewrite") || //$NON-NLS-1$
					directive.equals("allow_truncate") || //$NON-NLS-1$
					directive.equals("allow_rewrite") || //$NON-NLS-1$
					directive.equals("allow_unlink") //$NON-NLS-1$
					) {
				return true;
			} else {
				return false;
			}
		}
		@Override
		public Color getBackground(Object element, int columnIndex) {
			return null;
		}
		@Override
		public Color getForeground(Object element, int columnIndex) {
			AccessPermission ap = (AccessPermission)element;
			if (columnIndex == 1) {
				cacheElement(ap);
				if (!statusTable.get(ap.getText()))
					return Activator.getStandardDisplay().getSystemColor(SWT.COLOR_GRAY);
			}
			return null;
		}
	}

	private Button[] radioButtons = new Button[3];
	
	private void createOptionButtons(Composite composite) {
		Group group = new Group(composite, SWT.NULL);
		group.setText(Messages.PatternizeAccessPermissionsDialog_FilePatternizingMethodGroupTitle);
		group.setLayout(new GridLayout());
		group.setLayoutData(new GridData());

		radioButtons[0] = createRadioButton(group, Messages.PatternizeAccessPermissionsDialog_OptionIncludeExtension);
		radioButtons[1] = createRadioButton(group, Messages.PatternizeAccessPermissionsDialog_OptionWithoutExtension);
		radioButtons[2] = createRadioButton(group, Messages.PatternizeAccessPermissionsDialog_OptionWithoutFilename);
	}
	
	private Button deleteOptionButton;
	
	private void createDeleteOptionButton(Composite composite) {
		deleteOptionButton = new Button(composite, SWT.CHECK);
		deleteOptionButton.setText(Messages.PatternizeAccessPermissionsDialog_DeleteOptionButton);
		deleteOptionButton.setSelection(true);
	}

    /**
     * Creates a radio button with the given parent and text.
     *
     * @param parent the parent composite
     * @param text the text for the check box
     * @return the radio box button
     */
	protected Button createRadioButton(Composite parent, String text) {
		Button button = new Button(parent, SWT.RADIO);
		button.setText(text);
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleButtonSelected((Button)e.getSource());
			}
		});
		return button;
    }

	private void handleButtonSelected(Button source) {
		ArrayLabelProvider labelProvider = (ArrayLabelProvider)viewer.getLabelProvider();
		if (source == radioButtons[0]) {
			labelProvider.setPatternizeMethod(ArrayLabelProvider.PATTERNIZE_METHOD_FILENAME_WHOLE);
		}
		if (source == radioButtons[1]) {
			labelProvider.setPatternizeMethod(ArrayLabelProvider.PATTERNIZE_METHOD_FILENAME_ONLY);
		}
		if (source == radioButtons[2]) {
			labelProvider.setPatternizeMethod(ArrayLabelProvider.PATTERNIZE_METHOD_FILENAME_EXTENSION);
		}

		labelProvider.clearCache();
		viewer.refresh();
	}

	private void initializeUI() {
		viewer.getControl().setFocus();
		
		radioButtons[1].setSelection(true);
		handleButtonSelected(radioButtons[1]);
	}

	protected void okPressed() {
		ArrayLabelProvider labelProvider = (ArrayLabelProvider)viewer.getLabelProvider();
		String[] saveLines = labelProvider.getUniquePatternArray();
		AccessPermission[] deleteAPs = labelProvider.getPatternizedAccessPermissionArray();
		if (save(createSaveline(saveLines))) {
			TomoyoCorePlugin.getDefault().setStatusMessage(Messages.PatternizeAccessPermissionsDialog_StatusMessagePattenizedOK, true);
			if (deleteOptionButton.getSelection()) {
				if (delete(deleteAPs)) {
					TomoyoCorePlugin.getDefault().setStatusMessage(Messages.PatternizeAccessPermissionsDialog_StatusMessageDeletePattenizedOK, true);
					synchronize();
					super.okPressed();
				} else {
					synchronize();
					setErrorMessage(Messages.PatternizeAccessPermissionsDialog_DeleteNGMessage);
					getButton(IDialogConstants.OK_ID).setEnabled(false);
				}
			}
		} else {
			setErrorMessage(Messages.PatternizeAccessPermissionsDialog_SaveNGMessage);
			getButton(IDialogConstants.OK_ID).setEnabled(false);
		}
	}

	private void synchronize() {
		LocalResource domainPolicy = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();
		ProcResourceManager.getInstance().synchronize(project, domainPolicy);
	}

	public String createSaveline(String[] saveLines) {
		StringBuffer input = new StringBuffer();
		input.append(domain.getText());
		input.append("\n"); //$NON-NLS-1$
		for (int cnt = 0; cnt < saveLines.length; cnt++) {
			input.append(saveLines[cnt]);
			input.append("\n"); //$NON-NLS-1$
		}
		return input.toString();
	}

	private boolean save(String saveLine) {
		LocalResource domainPolicy = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();
		WriteCommand command = CommandManager.getInstance().write(project, domainPolicy, saveLine);
		if (command.isSuccessful()) {
			return true;
		} else {
			return false;
		}
	}

	private boolean delete(AccessPermission[] deleteAPs) {
		LocalResource domainPolicy = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();
		
		WriteCommand command = CommandManager.getInstance().write(
				project, domainPolicy, createDeleteline(deleteAPs));

		if (command.isSuccessful()) {
			return true;
		} else {
			String message = Messages.PatternizeAccessPermissionsDialog_DeleteNGMessage;
			TomoyoCorePlugin.getDefault().setStatusMessage(message);
			return false;
		}
	}

	public String createDeleteline(AccessPermission[] deleteAPs) {
		StringBuffer input = new StringBuffer();
		input.append(domain.getText());
		input.append("\n"); //$NON-NLS-1$
		for (int cnt = 0; cnt < deleteAPs.length; cnt++) {
			AccessPermission access = deleteAPs[cnt];
			input.append("delete"); //$NON-NLS-1$
			input.append(" "); //$NON-NLS-1$
			input.append(access.getText());
			input.append("\n"); //$NON-NLS-1$
		}
		return input.toString();
	}
	
}
