/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form;


import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.model.PolicyElementDefinition;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.ViewerExpandCollapseKeyListener;
import jp.sourceforge.tomoyo.ui.editor.CoreEditorOutlineMouseListener;
import jp.sourceforge.tomoyo.ui.editor.form.ResourceFormPage;
import jp.sourceforge.tomoyo.ui.editor.text.outline.ExceptPolicyOutlineContentProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.ExceptPolicyOutlineLabelProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.PolicyOutlineMenuListener;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.widgets.Section;

public abstract class DefinitionPolicyPage extends ResourceFormPage {

	public DefinitionPolicyPage(FormEditor editor, String pageID, String title) {
		super(editor, pageID, title);
	}

    //-----------------------------------------------------------------------------------------
	// Overwrides (body)
	//-----------------------------------------------------------------------------------------

	protected void createClient(Composite body) {

		Composite composite = new Composite(body, SWT.NULL);
		GridLayout layout = new GridLayout();
		composite.setBackground(body.getBackground());
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));

		createMainSection(composite);
	}

	protected Section createMainSection(Composite parent) {

		Section section = getManagedForm().getToolkit().createSection(parent, Section.TITLE_BAR | Section.DESCRIPTION);
		
		section.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		String title = Messages.DefinitionPolicyPage_MainSectionTitle;
		if (title != null)
			section.setText(title);
		String description = Messages.DefinitionPolicyPage_MainSectionDescription;
		if (description != null)
			section.setDescription(description);
		
		Composite container = getManagedForm().getToolkit().createComposite(section);
		container.setLayout(setBorder0(new GridLayout()));
		container.setLayoutData(new GridData(GridData.FILL_BOTH));

		createContent(container);
		
		section.setTextClient(createToolbar(section));

		getManagedForm().getToolkit().paintBordersFor(container);
		section.setClient(container);

		return section;
	}
	
	protected void createContent(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		container.setLayout(setBorder0(new GridLayout(2, false)));
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		container.setLayoutData(data);
	
		SashForm sash = new SashForm(container, SWT.HORIZONTAL);
		sash.setBackground(container.getBackground());
		GridLayout layout = new GridLayout();
		sash.setLayout(layout);
		sash.setLayoutData(new GridData(GridData.FILL_BOTH));

		Composite left = getManagedForm().getToolkit().createComposite(sash, SWT.NULL);
		left.setBackground(parent.getBackground());
		left.setLayout(setBorder0(new GridLayout()));
		left.setLayoutData(new GridData(GridData.FILL_BOTH));
	
		createElementComposite(left);

		Composite right = getManagedForm().getToolkit().createComposite(sash, SWT.NULL);
		right.setBackground(parent.getBackground());
		right.setLayout(setBorder0(new GridLayout()));
		right.setLayoutData(new GridData(GridData.FILL_BOTH));
		createContentComposite(right);

		sash.setWeights(new int[] {1, 2});
	}
	
	protected void restoreDefault() {
		super.restoreDefault();
		handleModifyText(null);
		setDefaultSelection();
	}

	private void setDefaultSelection() {
		Object[] visibles = elementViewer.getVisibleExpandedElements();
		if (visibles == null || visibles.length == 0)
			return;
		elementViewer.setSelection(new StructuredSelection(visibles[0]));
	}
	
    //-----------------------------------------------------------------------------------------
	// Overwrides (TODO section)
	//-----------------------------------------------------------------------------------------

	protected void handleTodoLinkEvent(String href) {
		if (href.equals("href.search")) { //$NON-NLS-1$
		}
	}

    //-----------------------------------------------------------------------------------------
	// Overwrides (Event)
	//-----------------------------------------------------------------------------------------

	protected void handleHistoryElementSelectionChanged(ISelection selection) {
		if (selection instanceof StructuredSelection) {
			StructuredSelection structured = (StructuredSelection)selection;
			if (structured.size() == 1) {
			}
		}
	}

    //-----------------------------------------------------------------------------------------
	// Create element viewer
	//-----------------------------------------------------------------------------------------
	
	private TreeViewer elementViewer;
	
	protected Control createElementComposite(Composite parent) {		
		elementViewer = new TreeViewer(parent, SWT.BORDER | SWT.MULTI);
		elementViewer.setContentProvider(new ExceptPolicyOutlineContentProvider(getProject()));
		elementViewer.setLabelProvider(new ExceptPolicyOutlineLabelProvider(getProject()));
		elementViewer.addSelectionChangedListener(new ElementViewerSelectionChangeLister());
		elementViewer.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));
		elementViewer.getControl().addKeyListener(new ViewerExpandCollapseKeyListener(elementViewer));
		elementViewer.getControl().addKeyListener(new DefinitionPolicyViewerKeyListener(getProject(), elementViewer, getLocalResource()));
		elementViewer.getControl().addMouseListener(new CoreEditorOutlineMouseListener(elementViewer));
		elementViewer.getControl().addFocusListener(new TreeFocusAdapter(elementViewer));

		elementViewer.setInput(createInputModel());
		elementViewer.expandAll();

    	setupMenuListener(parent, elementViewer);

		return elementViewer.getControl();
	}
	
	protected Object createInputModel() {
		return getLocalResource().getModel();
	}

	private class ElementViewerSelectionChangeLister implements ISelectionChangedListener {
		public void selectionChanged(SelectionChangedEvent event) {
			updateLabelProvider();
			
			StructuredSelection selection = (StructuredSelection)elementViewer.getSelection();
			Iterator<?> it = selection.iterator();
//			List<PolicyElement> elementList = new ArrayList<PolicyElement>();
			List<PolicyElementDefinition> definitionList = new ArrayList<PolicyElementDefinition>();
			while (it.hasNext()) {
				Object next = it.next();
				if (next instanceof PolicyElementDefinition) {
					PolicyElementDefinition definition = (PolicyElementDefinition)next;
					definitionList.add(definition);
				}
			}
			
			updateContentViewer(definitionList);
		}
		private void updateLabelProvider() {
			StructuredSelection selection = (StructuredSelection)elementViewer.getSelection();
			if (selection.size() > 1) {
				verboseButton.setSelection(true);
				setDirectiveVerbose(true);
			}
		}
	}
	
	//-----------------------------------------------------------------------------------------
	// Create content viewer
	//-----------------------------------------------------------------------------------------

	public TreeViewer contentViewer;

	protected Composite createContentComposite(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(2, false);
		container.setLayout(setBorder0(layout));
		GridData data = new GridData(GridData.FILL_BOTH);
		container.setLayoutData(data);

		contentViewer = new TreeViewer(container, SWT.BORDER | SWT.MULTI);
		contentViewer.setContentProvider(new ExceptPolicyOutlineContentProvider(getProject()));
		contentViewer.setLabelProvider(new ExceptPolicyOutlineLabelProvider(getProject()));
		contentViewer.addSelectionChangedListener(new ContentViewerSelectionChangeLister());
		contentViewer.getTree().addKeyListener(new ViewerExpandCollapseKeyListener(contentViewer));
		contentViewer.getTree().addKeyListener(new DefinitionPolicyViewerKeyListener(getProject(), contentViewer, getLocalResource()));
		contentViewer.getTree().addMouseListener(new CoreEditorOutlineMouseListener(contentViewer));
		contentViewer.setSorter(new ContentViewerSorter());
		GridData layoutData = new GridData(GridData.FILL_BOTH);
		contentViewer.getTree().setLayoutData(layoutData);
		contentViewer.getControl().addFocusListener(new TreeFocusAdapter(contentViewer));

    	setupMenuListener(container, contentViewer);

		createContentViewerContoler(container);

		return container;
	}
	
	private void updateContentViewer(List<PolicyElementDefinition> definitionList) {
		contentViewer.setInput(definitionList.toArray(new PolicyElementDefinition[definitionList.size()]));
		restoreViewerSelectioned(contentViewer);
		contentViewer.refresh();
	}

	private void updateContentViewer() {
		contentViewer.refresh();
	}
	
	private class ContentViewerSelectionChangeLister implements ISelectionChangedListener {
		public void selectionChanged(SelectionChangedEvent event) {
			StructuredSelection selection = (StructuredSelection)event.getSelection();
			storeViewerSelectioned((Viewer)event.getSource(), selection);
		}
	}
	
	private class ContentViewerSorter extends ViewerSorter implements Comparator<Object> {
		private boolean ascend = true;
		public ContentViewerSorter() {
		}
		public void setOrientation(boolean b) {
			ascend = b;
		}
		public int compare(Viewer viewer, Object e1, Object e2) {
			return compare(e1, e2);
		}
		public int compare(Object e1, Object e2) {
			String text1;
			String text2;
			if (e1 instanceof PolicyElement) {
				PolicyElement element1 = (PolicyElement)e1;
				PolicyElement element2 = (PolicyElement)e2;
				text1 = element1.getText();
				text2 = element2.getText();
			} else {
				return 0;
			}
			if (ascend)
				return text1.compareTo(text2);
			else
				return text2.compareTo(text1);
		}
	}
	
	private void setupMenuListener(Composite parent, TreeViewer viewer) {
    	MenuManager menuManager = new MenuManager("#PopupMenu"); //$NON-NLS-1$
    	menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new DefinitionOutlineMenuListener(getProject(), viewer));
		Menu menu = menuManager.createContextMenu(parent);
		viewer.getTree().setMenu(menu);
    }

	private Button sortButton;
	private Button verboseButton;
	
	private void createContentViewerContoler(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(1, false);
		container.setLayout(setBorder0(layout));
		GridData data = new GridData(SWT.RIGHT, SWT.FILL, false, true, 1, 1);
		data.horizontalIndent = 0;
		container.setLayoutData(data);
		
		sortButton = createButton(container, Activator.getImage("sort_asc.gif"), SWT.TOGGLE); //$NON-NLS-1$
		sortButton.setToolTipText(Messages.DefinitionPolicyPage_SortButtonTooltip);
		verboseButton = createButton(container, Activator.getImage("verbose.gif"), SWT.TOGGLE); //$NON-NLS-1$
		verboseButton.setToolTipText(Messages.DefinitionPolicyPage_VerboseButtonTooltip);
	}
	
	protected void handleButtonSelected(Button button) {
		if (button == sortButton) {
			ContentViewerSorter sorter = (ContentViewerSorter)contentViewer.getSorter();
			sorter.setOrientation(button.getSelection());
			contentViewer.refresh();
		}
		if (button == verboseButton) {
			setDirectiveVerbose(button.getSelection());
		}
	}
	
	private void setDirectiveVerbose(boolean verbose) {
		ExceptPolicyOutlineLabelProvider labelProvider = (ExceptPolicyOutlineLabelProvider)contentViewer.getLabelProvider();
		labelProvider.setDirectiveVerbose(verbose);
		contentViewer.refresh();
	}
	
	private class DefinitionOutlineMenuListener extends PolicyOutlineMenuListener {

		public DefinitionOutlineMenuListener(IProject project, StructuredViewer viewer) {
			super(project, viewer);
		}
		
		public void menuAboutToShow(IMenuManager manager) {
			String directive = null;
			String context = null;
			if (getSelection(contentViewer).isEmpty()) {
				IStructuredSelection selection = getSelection(elementViewer);
				if (selection.getFirstElement() instanceof PolicyElementDefinition) {
					PolicyElementDefinition definition = (PolicyElementDefinition)selection.getFirstElement();
					directive = definition.getDirectiveName();
				}
			} else {
				IStructuredSelection selection = getSelection(contentViewer);
				if (selection.getFirstElement() instanceof PolicyElement) {
					PolicyElement policyElement = (PolicyElement)selection.getFirstElement();
					directive = policyElement.getDirective();
				}
			}
			manager.add(createCreateDefinitionAction(getLocalResource(), directive, context));
			manager.add(new Separator());
			super.menuAboutToShow(manager);
		}

	}
	
	//-----------------------------------------------------------------------------------------
	// Notifications
	//-----------------------------------------------------------------------------------------

	protected void notifyPolicyUpdated(IProject project, LocalResource localResource) {
		super.notifyPolicyUpdated(project, localResource);
		if (project == getProject() && localResource.equals(getLocalResource())) {
			elementViewer.refresh();
			updateContentViewer();
		}
	}

}
