/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import jp.sourceforge.tomoyo.core.ProjectProperty;
import jp.sourceforge.tomoyo.core.ProjectPropertyManager;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileManager;
import jp.sourceforge.tomoyo.core.local.parser.PolicyParserManager;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.server.ConnectionManager;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.ICoreEditorOutlinePageProvider;
import jp.sourceforge.tomoyo.ui.editor.form.outline.ProjectOutlinePage;
import jp.sourceforge.tomoyo.ui.editor.form.overview.ProjectOverviewPage;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Widget;

import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.contentoutline.ContentOutlinePage;

public abstract class PlainFormPage extends FormPage implements ICoreEditorOutlinePageProvider {
	
	public PlainFormPage(FormEditor editor, String id, String title) {
		super(editor, id, title);
	}

	protected void createFormContent(IManagedForm managedForm) {
		final ScrolledForm form = managedForm.getForm();

		String formTitle = getFormTitle();
		form.setData("title", formTitle); //$NON-NLS-1$
		form.setText(formTitle);
		
		createToolBarActions(managedForm);
		
		crateBody(setupBody(form));
/*
		getSite().getPage().addSelectionListener(new ISelectionListener() {
			public void selectionChanged(IWorkbenchPart part, ISelection selection) {
				if (part instanceof GrantLogView) {
					handleAuditLogSelectionChanged(selection);
				}
				if (part instanceof HistoryView) {
					handleHistoryElementSelectionChanged(selection);
				}
			}
		});
*/
		restoreDefault();
		
		managedForm.refresh();

		setupObservers();
		
		markInitialized();
	}

	protected void setupObservers() {
		PolicyParserManager.getInstance().addObserver(new PolicyUpdateObserver());
		ProfileManager.getInstance().addObserver(new ProfileUpdateObserver());
		ConnectionManager.getInstance().addObserver(new ConnectionStatusObserver());
	}

	private Composite setupBody(ScrolledForm form) {
		Composite body = form.getBody();
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		body.setLayout(layout);
		return body;
	}

	protected abstract String getFormTitle();

	protected void handleAuditLogSelectionChanged(ISelection selection) {
	}
	
	protected void handleHistoryElementSelectionChanged(ISelection selection) {
	}

	private SashForm todoSash;

	protected void crateBody(Composite body) {
		todoSash = new SashForm(body, SWT.VERTICAL);
		GridLayout layout1 = new GridLayout();
		todoSash.setBackground(body.getBackground());
		todoSash.setLayout(layout1);
		todoSash.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		createTodoSection(todoSash);
		
		createClient(todoSash);
	}

	protected abstract void createClient(Composite client);
		
	protected void restoreDefault() {
		showTODOAction.setChecked(false);
		showTODOAction.run();
		verticalOrientAction.setChecked(true);
		verticalOrientAction.run();
		
		updateConnectToolbarAction(ConnectionManager.getInstance().isConnected(getProject()));
	}
	
	//-----------------------------------------------------------------------------------------
	// Form toolbar actions
	//-----------------------------------------------------------------------------------------

	protected void createToolBarActions(IManagedForm managedForm) {
		List<Action> actionList = getToolBarActions();
		if (actionList == null)
			return;
		final ScrolledForm form = managedForm.getForm();
		for (int cnt = 0; cnt < actionList.size(); cnt++) {
			Action action = (Action)actionList.get(cnt);
			form.getToolBarManager().add(action);
		}
		form.updateToolBar();
	}

	protected List<Action> getToolBarActions() {
		List<Action> actionList = new ArrayList<Action>();
//		actionList.add(showTODOAction);
//		actionList.add(verticalOrientAction);
//		actionList.add(horizontalOrientAction);
		actionList.add(back2homeAction);
		actionList.add(connectAction);
//		actionList.add(showHelpAction);
		return actionList;
	}


	private ShowTODOAction showTODOAction = new ShowTODOAction();
	private SwitchSashOrientationAction verticalOrientAction = new SwitchSashOrientationAction(SWT.VERTICAL);
	private SwitchSashOrientationAction horizontalOrientAction = new SwitchSashOrientationAction(SWT.HORIZONTAL);
	protected Action back2homeAction = createBack2HomeAction();
	protected Action connectAction = createConnectAction();
//	protected Action showHelpAction = createShowHelpAction();

	private class ShowTODOAction extends Action {
		public ShowTODOAction() {
			super("", Action.AS_CHECK_BOX); //$NON-NLS-1$
			setImageDescriptor(Activator.getImageDescriptor("todo.gif")); //$NON-NLS-1$
		}
		public void run() {
			showTODO(isChecked());
			setSashEnabled(isChecked());
		}
		private void showTODO(boolean b) {
			if (b)
				todoSash.setMaximizedControl(null);
			else
				todoSash.setMaximizedControl(todoSash.getChildren()[1]);
		}
		private void setSashEnabled(boolean b) {
			verticalOrientAction.setEnabled(b);
			horizontalOrientAction.setEnabled(b);
		}
	}
	
	private class SwitchSashOrientationAction extends Action {
		private int orientation;
		public SwitchSashOrientationAction(int orientation) {
			super("", Action.AS_RADIO_BUTTON); //$NON-NLS-1$
			this.orientation = orientation;
			if (orientation == SWT.VERTICAL) {
				setImageDescriptor(Activator.getImageDescriptor("vertical.gif")); //$NON-NLS-1$
			} else {
				setImageDescriptor(Activator.getImageDescriptor("horizontal.gif")); //$NON-NLS-1$
			}
		}
		public void run() {
			if (orientation == SWT.VERTICAL) {
				setHorizontal();
			} else {
				setVertical();
			}
		}
		private void setVertical() {
			todoSash.setOrientation(SWT.VERTICAL);
			todoSash.setWeights(new int[] {1 , 4});
		}
		private void setHorizontal() {
			todoSash.setOrientation(SWT.HORIZONTAL);
			todoSash.setWeights(new int[] {1 , 4});
		}
	}
	
	protected Action createBack2HomeAction() {
		Action action = new Action("", Action.AS_PUSH_BUTTON) { //$NON-NLS-1$
			public void run() {
				FormEditor formEditor = (FormEditor)getEditor();
				formEditor.setActivePage(ProjectOverviewPage.PAGE_ID);
			}
		};
		action.setToolTipText(Messages.PlainFormPage_Back2HomeToolTipText);
		action.setImageDescriptor(Activator.getImageDescriptor("home.gif")); //$NON-NLS-1$
		return action;
	}

	protected Action createConnectAction() {
		Action action = new Action(Messages.PlainFormPage_Connect2ServerActionTitle, Action.AS_CHECK_BOX) {
			public void run() {
				if (isChecked())
					ConnectionManager.getInstance().connect(getProject());
				else
					ConnectionManager.getInstance().disConnect(getProject());
			}
		};
		action.setToolTipText(action.getText());
		action.setImageDescriptor(Activator.getImageDescriptor("disconnected.gif")); //$NON-NLS-1$
		return action;
	}

	protected Action createShowHelpAction() {
		Action action = new Action(Messages.PlainFormPage_ShowHelpResourceActionTitle, Action.AS_PUSH_BUTTON) {
			public void run() {
				final ScrolledForm form = getManagedForm().getForm();
				BusyIndicator.showWhile(form.getDisplay(), new Runnable() {
					public void run() {
						PlatformUI.getWorkbench().getHelpSystem().displayHelpResource(getHelpPath());
					}
				});
			}
		};
		action.setToolTipText(action.getText());
		action.setImageDescriptor(Activator.getImageDescriptor("help.gif")); //$NON-NLS-1$
		return action;
	}

	protected abstract String getHelpPath();

	/*
			PlatformUI.getWorkbench().getHelpSystem().setHelp(page.getControl(),
				"jp.sourceforge.tomoyo.doc.create_a_new_project");
	 */
	
	private FormText createFormText(Composite section, String content) {
        FormText text = getManagedForm().getToolkit().createFormText(section, true);
        try {
        	text.setText(content, true, false);
        } catch (SWTException e) {
        	text.setText(e.getMessage(), false, false);
        }
        section.setLayoutData(new TableWrapData(TableWrapData.FILL, TableWrapData.FILL_GRAB));
        return text;
    }
	
	//-------------------------------------------------------------------------------------------
	// Sections
	//-------------------------------------------------------------------------------------------
/*	
	protected Composite createToolbar(FormToolkit toolkit, Section section) {
		Composite toolbar = new Composite(section, SWT.NULL);
		
		toolbar.setBackground(section.getTitleBarGradientBackground());
//		toolkit.adapt(toolbar, true, true);
		
		RowLayout layout = new RowLayout(SWT.HORIZONTAL);
		layout.marginTop = 0;
		layout.marginBottom = 0;
		toolbar.setLayout(layout);
		toolbar.setLayoutData(new GridData(GridData.FILL_BOTH));

		addHelpResourceLink(toolkit, section, toolbar);
		
		return toolbar;
	}
	
	protected void addHelpResourceLink(FormToolkit toolkit, Section section, Composite toolbar) {
		ImageHyperlink helpLink = new ImageHyperlink(toolbar, SWT.NULL);
		toolkit.adapt(helpLink, true, true);
		helpLink.setImage(Activator.getImage("_help.gif")); //$NON-NLS-1$
		helpLink.setBackground(section.getTitleBarGradientBackground());
//		helpLink.addHyperlinkListener(new ViewHelpResourceHyperlinkEvent(helppath));
		helpLink.setVisible(false);
		helpLink.setEnabled(false);
	}
*/	

	protected Section createSection(Composite parent, String gridID) {

		Section section = getManagedForm().getToolkit().createSection(parent, Section.TITLE_BAR | getStyle(gridID));
		section.setLayoutData(new GridData(GridData.FILL_BOTH, GridData.FILL_BOTH));
		
		String title = getSectionTitle(gridID);
		if (title != null)
			section.setText(title);
		String description = getSectionDescription(gridID);
		if (description != null)
			section.setDescription(description);
		section.setData("gridID", gridID); //$NON-NLS-1$
		
		Composite container = getManagedForm().getToolkit().createComposite(section);
		container.setLayout(new GridLayout());
		container.setLayoutData(new GridData(GridData.FILL_BOTH, GridData.FILL_BOTH));

		createContent(container, gridID);
		
		section.setTextClient(createToolbar(section));

		getManagedForm().getToolkit().paintBordersFor(container);
		section.setClient(container);

		return section;
	}

	protected Composite createToolbar(Section section) {
		return null;
	}

	protected int getStyle(String gridID) {
		return SWT.NULL;
	}
	protected Object getSectionLayoutData(String gridID) {
		return null;
	}
	protected String getSectionTitle(String gridID) {
		return null;
	}
	protected String getSectionDescription(String gridID) {
		return null;
	}
	protected void createContent(Composite parent, String gridID) {
	}

	//-------------------------------------------------------------------------------------------
	// Section grids
	//-------------------------------------------------------------------------------------------

	protected Section createSection(Composite parent, FormToolkit toolkit, String title, String description, String helppath, int style) {
		Section section = toolkit.createSection(parent, style);
		section.setText(title);
		if (description != null)
			section.setDescription(description);
		section.setLayout(new GridLayout());
		section.setLayoutData(new GridData(GridData.FILL_BOTH, GridData.FILL_BOTH));

		if (helppath != null) {
//			setHelpResource("/jp.sourceforge.tomoyo.core/" + helppath); //$NON-NLS-1$
		}
//		section.setTextClient(createToolbar(toolkit, section));
		return section;
	}
	
	//---------------------------------------------------------------------------------------------------------------
	// Custom composites
	//---------------------------------------------------------------------------------------------------------------
/*	
	protected Composite createFillBothComposite(Composite parent) {
		Composite composite = getManagedForm().getToolkit().createComposite(parent, SWT.NULL);
		TableWrapLayout layout = new TableWrapLayout();
		composite.setLayout(layout);
		composite.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB, TableWrapData.FILL));
		return composite;
	}
*/	


	
	//-----------------------------------------------------------------------------------------
	// Status bar
	//-----------------------------------------------------------------------------------------

	protected static final String STATUS_MSG_CONNECTING = Messages.PlainFormPage_StatusMessageConnecting;
	protected static final String STATUS_MSG_SAVING = Messages.PlainFormPage_StatusMessageSaving;
	protected static final String STATUS_MSG_LOADING = Messages.PlainFormPage_StatusMessageLoading;

	protected static final String STATUS_MSG_CONNECTED = Messages.PlainFormPage_StatusMessageConnected;
	protected static final String STATUS_MSG_DISCONNECTED = Messages.PlainFormPage_StatusMessageDisconnected;
	protected static final String STATUS_MSG_CANT_CONNECTED = Messages.PlainFormPage_StatusMessageCannotConnect2Server;
	protected static final String STATUS_MSG_SAVED = Messages.PlainFormPage_StatusMessageSaved;
	protected static final String STATUS_MSG_LOADED = Messages.PlainFormPage_StatusMessageLoaded;
	
	protected static final String STATUS_MSG_ERROR = Messages.PlainFormPage_StatusMessageErrorOccured;
	
	//-----------------------------------------------------------------------------------------
	// Others
	//-----------------------------------------------------------------------------------------

	protected IProject getProject() {
		return ((FileEditorInput)getEditor().getEditorInput()).getFile().getProject();
	}

	protected ProjectProperty getProjectProperty() {
		return ProjectPropertyManager.getInstance().getProperty(getProject());
	}

	protected FormToolkit getToolKit() {
		return getManagedForm().getToolkit();
	}

	protected Shell getActiveShell() {
		return Activator.getStandardDisplay().getActiveShell();
	}
	
	//-----------------------------------------------------------------------------------------
	// Viewer forcus adapter
	//-----------------------------------------------------------------------------------------

    protected class ViewerFocusAdapter extends FocusAdapter {
    	private Viewer viewer;
    	public ViewerFocusAdapter(Viewer viewer) {
    		this.viewer = viewer;
    	}
		public void focusGained(FocusEvent e) {
			handleViewerFocusGaind(viewer);
		}
    }
    
    private Class focusedSectionClass;
    
	private void handleViewerFocusGaind(Viewer viewer) {
		focusedSectionClass = (Class)viewer.getData("filter"); //$NON-NLS-1$
	}
	
	protected Class getFocusedSectionClass() {
		return focusedSectionClass;
	}
	
	//-----------------------------------------------------------------------------------------
	// Notifications
	//-----------------------------------------------------------------------------------------

    private class PolicyUpdateObserver implements Observer {
		public void update(Observable o, Object arg) {
			Display.getDefault().asyncExec(new PolicyUpdateThread(o, arg));
		}
    }

    private class PolicyUpdateThread implements Runnable {
    	private Observable o;
    	private Object arg;
    	public PolicyUpdateThread(Observable o, Object arg) {
    		this.o = o;
    		this.arg = arg;
    	}
		public void run() {
			if (o instanceof PolicyParserManager) {
				LocalResource localResource = (LocalResource)arg;
				IProject project = localResource.getProject();
				if (getPartControl().isDisposed())
					return;
				notifyPolicyUpdated(project, localResource);
			}
		}
    }

    private class ProfileUpdateObserver implements Observer {
		public void update(Observable o, Object arg) {
			Display.getDefault().asyncExec(new ProfileUpdateThread(o, arg));
		}
    }

    private class ProfileUpdateThread implements Runnable {
    	private Observable o;
    	private Object arg;
    	public ProfileUpdateThread(Observable o, Object arg) {
    		this.o = o;
    		this.arg = arg;
    	}
		public void run() {
			if (o instanceof ProfileManager) {
				Hashtable table = (Hashtable)arg;
				IProject project = (IProject)table.get(IProject.class);
				notifyProfileUpdated(project);
			}
		}
    }

    private class ConnectionStatusObserver implements Observer {
    	public void update(Observable o, Object arg) {
			Display.getDefault().asyncExec(new ConnectionStatusUpdateThread(o, arg));
		}
    }

    private class ConnectionStatusUpdateThread implements Runnable {
//    	private Observable o;
//    	private Object arg;
    	public ConnectionStatusUpdateThread(Observable o, Object arg) {
//   		this.o = o;
//    		this.arg = arg;
    	}
		public void run() {
			boolean isConnected = ConnectionManager.getInstance().isConnected(getProject());
			notifyConnectionUpdated(getProject(), isConnected);
		}
    }

	public void notifyPageChanged(PlainFormPage editorPage) {
		if (editorPage == this) {
		}
	}

	protected void notifyPolicyUpdated(IProject project, LocalResource localResource) {
	}

	protected void notifyProfileUpdated(IProject project) {
	}
	
	protected void notifyConnectionUpdated(IProject project, boolean isConnected) {
		updateConnectToolbarAction(isConnected);
	}
	
	private void updateConnectToolbarAction(boolean isConnected) {
		connectAction.setChecked(isConnected);
		if (isConnected)
			connectAction.setImageDescriptor(Activator.getImageDescriptor("connected.gif")); //$NON-NLS-1$
		else
			connectAction.setImageDescriptor(Activator.getImageDescriptor("disconnected.gif")); //$NON-NLS-1$
	}
	
	//------------------------------------------------------------------------
	// check widget
	//------------------------------------------------------------------------
	
	private ArrayList<Widget> widgedList = new ArrayList<Widget>();
	
	protected Widget addWidget(Widget widget) {
		widgedList.add(widget);
		return widget;
	}
	
	protected boolean hasDisposedWidgets() {
		Iterator<Widget> it = widgedList.iterator();
		while (it.hasNext()) {
			Object next = it.next();
			if (next instanceof Widget) {
				Widget widget = (Widget)next;
				if (widget.isDisposed())
					return false;
			}
		}
		return true;
	}
	
	private boolean initialzed = false;

	private void markInitialized() {
		initialzed = true;
	}

	public boolean isInitizlized() {
		return initialzed;
	}

	protected boolean checkWidgets() {
		return !hasDisposedWidgets() && isInitizlized();
	}
	
    //-----------------------------------------------------------------------------------------
	// TODO section
	//-----------------------------------------------------------------------------------------

	protected void createTodoSection(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.horizontalSpacing = 0;
		
		composite.setBackground(parent.getBackground());
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));

		Section section = createSection(composite, getManagedForm().getToolkit(),
				getTodoSectionTitle(), getTodoSectionDescription(),
				"helppath", //$NON-NLS-1$
				Section.TITLE_BAR);
		
		section.setLayout(new TableWrapLayout());
		section.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		Composite container = getManagedForm().getToolkit().createComposite(section, SWT.NULL);
		TableWrapLayout containerLayout = new TableWrapLayout();
		containerLayout.numColumns = 1;
		containerLayout.makeColumnsEqualWidth = false;
		containerLayout.horizontalSpacing = 0;
		
		container.setLayout(containerLayout);

		StringBuffer formHTML = new StringBuffer();
		formHTML.append("<form>"); //$NON-NLS-1$
		for (int cnt = 0; cnt < getTodoCount(); cnt++) {
			formHTML.append("<li style=\"image\" value=\"img" + cnt + "\" bindent=\"5\">"); //$NON-NLS-1$ //$NON-NLS-2$
			formHTML.append("<a href=\"" + getHref(cnt) + "\">"); //$NON-NLS-1$ //$NON-NLS-2$
			formHTML.append(getLinkText(cnt));
			formHTML.append("</a>"); //$NON-NLS-1$
			formHTML.append(" : "); //$NON-NLS-1$
			formHTML.append(getLinkDescription(cnt));
			formHTML.append("</li>"); //$NON-NLS-1$
		}
		formHTML.append("</form>"); //$NON-NLS-1$
		FormText compileFormText = createFormText(container, formHTML.toString());
		for (int cnt = 0; cnt < getTodoCount(); cnt++) {
			compileFormText.setImage("img" + cnt, getLinkImage(cnt)); //$NON-NLS-1$ //$NON-NLS-2$
		}
		compileFormText.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				String href = e.getHref().toString();
				handleTodoLinkEvent(href);
			}
		});
		
		section.setClient(container);
	}

	protected abstract void handleTodoLinkEvent(String href);

	protected String getTodoSectionTitle() {
		return "TODO"; //$NON-NLS-1$
	}
	
	protected String getTodoSectionDescription() {
		return null;
	}

	protected class Todo {
		private Image image;
		private String linkText;
		private String description;
		private String href;
		public Todo(Image image, String linkText, String description, String href) {
			super();
			this.image = image;
			this.linkText = linkText;
			this.description = description;
			this.href = href;
		}
		public String getDescription() {
			return description;
		}
		public String getHref() {
			return href;
		}
		public Image getImage() {
			return image;
		}
		public String getLinkText() {
			return linkText;
		}
	}

	protected int getTodoCount() {
		Todo[] todoList = getTodoList();
		if (todoList == null)
			return 0;
		return todoList.length;
	}
	
	protected abstract Todo[] getTodoList();

	protected Image getLinkImage(int index) {
		Todo[] todoList = getTodoList();
		if (todoList == null)
			return null;
		return todoList[index].getImage();
	}

    protected String getLinkText(int index) {
		Todo[] todoList = getTodoList();
		if (todoList == null)
			return null;
		return todoList[index].getLinkText();
    }

    protected String getLinkDescription(int index) {
		Todo[] todoList = getTodoList();
		if (todoList == null)
			return null;
		return todoList[index].getDescription();
    }

    protected String getHref(int index) {
		Todo[] todoList = getTodoList();
		if (todoList == null)
			return null;
		return todoList[index].getHref();
    }
    
	protected GridLayout setBorder0(GridLayout layout) {
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		return layout;
	}
	
	protected GridLayout setMargin(GridLayout layout, int width) {
		layout.horizontalSpacing = 10;
		return layout;
	}
/*
	protected GridLayout setMargin0(GridLayout layout) {
		layout.marginLeft = 0;
		layout.marginRight = 0;
		return layout;
	}
*/	
	/*
	protected GridLayout setBorder0(GridLayout layout) {
		layout.marginLeft = 0;
		layout.marginRight = 0;
		layout.marginTop = 0;
		layout.marginBottom = 0;
		return layout;
	}
	 */

	//-----------------------------------------------------------------------------------------
	// implements ICoreEditorOutlilePage
	//-----------------------------------------------------------------------------------------

	private ProjectOutlinePage outlinePage = null;
	
	public ContentOutlinePage getContentOutlinePage() {
		if (outlinePage == null) {
			outlinePage = new ProjectOutlinePage(getEditor());	
//			outlinePage.addSelectionChangedListener(new ProjectOutlineSelectionChangedListener());
		}
		return outlinePage;
	}

}
