/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form.domain;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;

import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.domain.AccessPermission;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.MimicCUIEditorKeyListener;
import jp.sourceforge.tomoyo.ui.editor.dialog.NewAccessPermissionDialog;

public class DomainPolicyViewerKeyListener extends MimicCUIEditorKeyListener {

	private TreeViewer domainViewer;
	private TreeViewer accessViewer;
	private DomainSearchDialog dialog;
	private DomainSearchDialogObserver observer;
	
	public DomainPolicyViewerKeyListener(IProject project, TreeViewer domainViewer, TreeViewer accessViewer, DomainSearchDialogObserver observer) {
		super(project);
		this.domainViewer = domainViewer;
		this.accessViewer = accessViewer;
		this.observer = observer;
	}
	
	public void keyPressed(KeyEvent e) {
		super.keyPressed(e);
		if (e.keyCode == SWT.F2 || ((e.stateMask & SWT.CTRL) != 0 && e.keyCode == 'f')) {
			if (observer != null) {
	    		dialog = DomainSearchDialog.getInstance(Activator.getStandardDisplay().getActiveShell());
	    		observer.setDialog(dialog);
	    		observer.setViewer(domainViewer);
	        	dialog.open();
			}
		}
		if (e.keyCode == SWT.F3) {
			if (observer != null)
				observer.notifySearchConditionChanged(true);
		}
		if (e.keyCode == SWT.F4) {
			if (observer != null)
				observer.notifySearchConditionChanged(false);
        }
		if (e.keyCode == SWT.DEL) {
		}
		if (e.keyCode == SWT.CR) {
			handleEnter();
		}
	}
	
	private boolean isDomainViewer() {
		return domainViewer != null && accessViewer == null;
	}
	
	private boolean isAccessViewer() {
		return domainViewer != null && accessViewer != null && domainViewer != accessViewer;
	}
	
	private boolean isCustomViewer() {
		return domainViewer != null && accessViewer != null && domainViewer == accessViewer;
	}
	
	protected void handleAdd() {
		if (isDomainViewer()) {
			if (isDomainOnly(getSelection(domainViewer))) {
				_addAccessPermission();
//				_addDomain(true, true);
			}
		}
		if (isAccessViewer()) {
			if (isAccessPermissionOnly(getSelection(accessViewer))) {
				AccessPermission accessPermission = (AccessPermission)getSelection(accessViewer).getFirstElement();
				if (accessPermission == null)
					return;
				Domain domain = (Domain)accessPermission.getParent();
				if (domain == null)
					return;
				NewAccessPermissionDialog dialog = new NewAccessPermissionDialog(
						Activator.getStandardDisplay().getActiveShell(), project, domain, accessPermission.getDirective() + " ");
		    	dialog.open();
			}
			if (getSelection(accessViewer).isEmpty()) {
				_addAccessPermission();
			}
		}
		if (isCustomViewer()) {
			if (isDomainOnly(getSelection(accessViewer))) {
				Domain domain = (Domain)getSelection(accessViewer).getFirstElement();
				NewAccessPermissionDialog dialog = new NewAccessPermissionDialog(
						Activator.getStandardDisplay().getActiveShell(), project, domain, null);
		    	dialog.open();
			}
			if (isAccessPermissionOnly(getSelection(accessViewer))) {
				AccessPermission accessPermission = (AccessPermission)getSelection(accessViewer).getFirstElement();
				if (accessPermission == null)
					return;
				Domain domain = (Domain)accessPermission.getParent();
				if (domain == null)
					return;
				NewAccessPermissionDialog dialog = new NewAccessPermissionDialog(
						Activator.getStandardDisplay().getActiveShell(), project, domain, accessPermission.getDirective() + " ");
		    	dialog.open();
			}
		}
	}
	
	private void _addAccessPermission() {
		IStructuredSelection selection = getSelection(domainViewer);
		if (selection.size() == 1) {
			if (selection.getFirstElement() instanceof Domain) {
				Domain domain = (Domain)selection.getFirstElement();
				createCreateAccessPermissionAction(domain, null).run();
			}
		}
	}

	protected void handleDelete() {
		if (isDomainViewer()) {
			_deleteDomain();
		}
		if (isAccessViewer()) {
			_deleteAccess();
		}
		if (isCustomViewer()) {
			_deleteDomain();
			_deleteAccess();
		}
	}
	
	private void _deleteDomain() {
		IStructuredSelection selection = getSelection(domainViewer);
		if (areDomainsOnly(selection)) {
			createDeleteDomainAction(toDomainArray(selection)).run();
		}
	}
	
	private void _deleteAccess() {
		if (areAccessPermissionsOnly(getSelection(accessViewer), true)) {
			AccessPermission[] apArray = toAccessPermissionArray(getSelection(accessViewer));
			boolean b = MessageDialog.openQuestion(Activator.getStandardDisplay().getActiveShell(), Messages.DomainPolicyPage_DeleteAPConfirmTitle,
					Messages.DomainPolicyPage_DeleteAPConfirmMessage);
			if (b) {
				new DeleteAccessPermissionAction(apArray).run();
			}
		}
	}
	
	protected void handleInsert() {
		if (isDomainViewer()) {
			if (isDomainOnly(getSelection(domainViewer))) {
				_addDomain(false, true);
			}
		}
		if (isAccessViewer()) {
			if (isAccessPermissionOnly(getSelection(accessViewer))) {
				handleAdd();
			}
			if (getSelection(accessViewer).isEmpty()) {
				_addAccessPermission();
			}
		}
		if (isCustomViewer()) {
			if (isDomainOnly(getSelection(accessViewer))) {
				_addDomain(false, true);
			}
			if (isAccessPermissionOnly(getSelection(accessViewer))) {
				handleAdd();
			}
		}
	}
	
	protected void handleCopy() {
		if (isDomainViewer()) {
			if (isDomainOnly(getSelection(domainViewer))) {
				_addDomain(true, false);
			}
		}
		if (isAccessViewer()) {
			_copy();
		}
		if (isCustomViewer()) {
			if (isDomainOnly(getSelection(accessViewer))) {
				_addDomain(true, false);
			}
			if (isAccessPermissionOnly(getSelection(accessViewer))) {
				_copy();
			}
		}
	}

	private void _copy() {
		if (isAccessPermissionOnly(getSelection(accessViewer))) {
			AccessPermission accessPermission = (AccessPermission)getSelection(accessViewer).getFirstElement();
			if (accessPermission == null)
				return;
			Domain domain = (Domain)accessPermission.getParent();
			if (domain == null)
				return;
			NewAccessPermissionDialog dialog = new NewAccessPermissionDialog(
					Activator.getStandardDisplay().getActiveShell(), project, domain, accessPermission.getText());
	    	dialog.open();
		}
	}

	private void _addDomain(boolean isSelectedBased, boolean withSuffix) {
		String argDomainText = null;
		IStructuredSelection selection = getSelection(domainViewer);
		if (selection.getFirstElement() instanceof Domain) {
			Domain domain = (Domain)selection.getFirstElement();
			if (isSelectedBased)
				argDomainText = domain.getText();
			else
				argDomainText = domain.getText2(domain.getDepth() - 1);
		}
		createCreateDomainAction(argDomainText, withSuffix).run();
	}
	
	protected void handleSetProfile() {
		IStructuredSelection selection = getSelection(domainViewer);
		if (areDomainsOnly(selection)) {
			Domain[] domainArray = toDomainArray(selection);
			createChangeProfileAction(domainArray).run();
		}
	}
	
	protected void handleRefresh() {
		LocalResource localResource = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();
		boolean syncOK = ProcResourceManager.getInstance().synchronize(project, localResource);
		if (syncOK)
			;
		else
			;
	}

	/*
	private PolicyAdditionObserver addtionObserver = new PolicyAdditionObserver();
	
	private class PolicyAdditionObserver implements Observer {
		public void update(Observable o, Object arg) {
			if (arg instanceof PolicyElement) {
				domainViewer.setExpandedState((PolicyElement)arg, true);
			}
		}
	}
	*/
	
	@Override
	protected LocalResource _getRefreshingLocalResource() {
		LocalResource localResource = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();
		return localResource;
	}

	@Override
	protected void handleFindNext() {
		if (observer != null)
			observer.notifySearchConditionChanged(true);
	}

	@Override
	protected void handleFindPrevious() {
		if (observer != null)
			observer.notifySearchConditionChanged(false);
	}
	
	private void handleEnter() {
		if (isInitializingOnly(getSelection(domainViewer))) {
			AccessPermission permission = (AccessPermission)getSelection(domainViewer).getFirstElement();
        	if (permission.isInitializedSource()) {
        		String processName = permission.getContents();
				Domain foundDomain = (Domain)PolicyCacheManager.getInstance().getInitializerTarget(project, processName);
				if (foundDomain == null)
					return;
				domainViewer.setSelection(new StructuredSelection(foundDomain));
        	}
		}
	}

	protected boolean isInitializingOnly(IStructuredSelection selection) {
		if (selection.size() == 1) {
			if (selection.getFirstElement() instanceof AccessPermission) {
				AccessPermission permission = (AccessPermission)selection.getFirstElement();
				return (permission.isInitializedSource());
			}
		}
		return false;
	}

}
