/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form.profile;		

import java.util.Iterator;
import java.util.List;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.model.status.Profile;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileGroup;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileManager;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileMetaData;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileTypeChoise;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileTypeRange;
import jp.sourceforge.tomoyo.core.local.model.status.ProjectProfile;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.server.ConnectionManager;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.form.ResourceFormPage;
import jp.sourceforge.tomoyo.ui.editor.form.profile.wizard.NewProfileWizard;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColorCellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellEditorListener;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.widgets.Section;

public class ProjectProfilePage extends ResourceFormPage {

	public static final String PAGE_ID = "Profile"; //$NON-NLS-1$

	public ProjectProfilePage(FormEditor editor) {
		super(editor, PAGE_ID, Messages.ProjectProfilePage_PageTitle);
		setContentDescription(Messages.ProjectProfilePage_PageDescription);
	}

	protected List<Action> getToolBarActions() {
		List<Action> actionList = super.getToolBarActions();
		openEditorAction.setEnabled(false);
		return actionList;
	}

	//-----------------------------------------------------------------------------------------
	// Overwrides 
	//-----------------------------------------------------------------------------------------
	
	protected LocalResource getLocalResource() {
		return ProcResourceManager.getInstance().getProcResourceSet(getProject()).getStatus();
	}

    //-----------------------------------------------------------------------------------------
	// Overwrides (form)
	//-----------------------------------------------------------------------------------------


	protected String getFormTitle() {
		return Messages.ProjectProfilePage_FormTitle;
	}
	
	protected void createClient(Composite body) {
		
		Composite composite = new Composite(body, SWT.NULL);
		GridLayout layout = new GridLayout();
		composite.setBackground(body.getBackground());
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));

		createMainSection(composite);
	}
	
	protected Section createMainSection(Composite parent) {

		Section section = getManagedForm().getToolkit().createSection(parent, Section.TITLE_BAR | Section.DESCRIPTION);
		
		section.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		String title = Messages.ProjectProfilePage_ProfileTableSectionTitle;
		if (title != null)
			section.setText(title);
		String description = Messages.ProjectProfilePage_PageDescription;
		if (description != null)
			section.setDescription(description);

		Composite container = getManagedForm().getToolkit().createComposite(section);
		container.setBackground(parent.getBackground());
		container.setLayout(new GridLayout(2, false));
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		container.setLayoutData(data);

		treeViewer = createProfileTableTree(container);
		createButtonContoler(container);

		section.setTextClient(createToolbar(section));

		getManagedForm().getToolkit().paintBordersFor(container);
		section.setClient(container);

		return section;
	}
	
	protected void restoreDefault() {
		super.restoreDefault();
		
		expandButton.setEnabled(true);
		collapseButton.setEnabled(true);
		boolean isConnected = ConnectionManager.getInstance().isConnected(getProject());
		saveButton.setEnabled(isConnected);
		rollbackButton.setEnabled(false);

		treeViewer.setInput(createModel());
		
		packColumn();
	}

	private void packColumn() {
		if (true)
			return;
		Tree tree = treeViewer.getTree();
		for (int cnt = 0; cnt < tree.getColumnCount(); cnt++) {
			tree.getColumn(cnt).pack();
		}
	}

    //-----------------------------------------------------------------------------------------
	// TODO section
	//-----------------------------------------------------------------------------------------

	protected Todo[] getTodoList() {
		Todo[] todoList = new Todo[] {
		};
		return todoList;
	}

	protected void handleTodoLinkEvent(String href) {
	}

	//-------------------------------------------------------------------------------------------
	// Expand items
	//-------------------------------------------------------------------------------------------

	private Button expandButton;
	private Button collapseButton;
	private Button saveButton;
	private Button rollbackButton;
	private Button newButton;
	
	private void createButtonContoler(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(1, false);
		layout.horizontalSpacing = 0;
		layout.marginLeft = 0;
		layout.horizontalSpacing = 0;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.RIGHT, SWT.FILL, false, true, 1, 1);
		data.horizontalIndent = 0;
		container.setLayoutData(data);
		
		expandButton = (Button)addWidget(createButton(container, Activator.getImage("expandall.gif"), SWT.NULL)); //$NON-NLS-1$
		expandButton.setToolTipText(Messages.ProjectProfilePage_ExpandAllButtonToolTip);
		collapseButton = (Button)addWidget(createButton(container, Activator.getImage("collapseall.gif"), SWT.NULL)); //$NON-NLS-1$
		collapseButton.setToolTipText(Messages.ProjectProfilePage_CollapseAllButtonToolTip);
		saveButton = (Button)addWidget(createButton(container, Activator.getImage("save.gif"), SWT.NULL)); //$NON-NLS-1$
		saveButton.setToolTipText(Messages.ProjectProfilePage_SaveChangesButtonToolTip);
		rollbackButton = (Button)addWidget(createButton(container, Activator.getImage("rollback.gif"), SWT.NULL)); //$NON-NLS-1$
		rollbackButton.setToolTipText(Messages.ProjectProfilePage_ResetChangesButtonToolTip);
		newButton = (Button)addWidget(createButton(container, Activator.getImage("new.gif"), SWT.NULL)); //$NON-NLS-1$
		newButton.setToolTipText(Messages.ProjectProfilePage_CreateNewProfileButtonToolTip);
	}

	ISelection selectionCache = null;
	
	protected void handleButtonSelected(Button button) {
		if (button == expandButton) {
			if (!treeViewer.getSelection().isEmpty())
				selectionCache = treeViewer.getSelection();
			treeViewer.getTree().setRedraw(false);
			treeViewer.expandAll();
			treeViewer.getTree().setRedraw(true);
			if (selectionCache == null)
				treeViewer.setSelection(new StructuredSelection(treeViewer.getTree().getItem(0)));
			else
				treeViewer.setSelection(selectionCache);
		}
		if (button == collapseButton) {
			if (!treeViewer.getSelection().isEmpty())
				selectionCache = treeViewer.getSelection();
			treeViewer.getTree().setRedraw(false);
			treeViewer.collapseAll();
			treeViewer.getTree().setRedraw(true);
		}
		if (button == saveButton) {
			saveProfile();
		}
		if (button == rollbackButton) {
			rollback();
		}
		if (button == newButton) {
			startCreateDialog();
		}
	}

	private void saveProfile() {
		ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(getProject());
		if (pp.hasDirty()) {
			boolean updateYes = MessageDialog.openQuestion(
					Activator.getStandardDisplay().getActiveShell(),
					Messages.ProjectProfilePage_SaveConfirmationTitle,
					Messages.ProjectProfilePage_SaveConfirmationMessage);
			if (!updateYes)
				return;
			boolean isSaved = pp.save();
			if (isSaved) {
				treeViewer.refresh();
				updateButtonEnabled();
				TomoyoCorePlugin.getDefault().setStatusMessage(STATUS_MSG_SAVED);
			} else {
				TomoyoCorePlugin.getDefault().setStatusErrorMessage(STATUS_MSG_ERROR);
			}
		}
		updateButtonEnabled();
	}

	private void rollback() {
		ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(getProject());
		pp.rollback();
		treeViewer.refresh();
		updateButtonEnabled();
	}
	
	private void startCreateDialog() {
		final ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(getProject());
		if (pp.hasDirty()) {
			boolean updateYes = MessageDialog.openQuestion(
					Activator.getStandardDisplay().getActiveShell(),
					Messages.ProjectProfilePage_StartWizardConfirmationTitle,
					Messages.ProjectProfilePage_StartWizardConfirmationMessage);
			if (!updateYes)
				return;
		}

		NewProfileWizard wizard = new NewProfileWizard(getProject());
		wizard.setDefaultGroupNo(pp.getNewGroupNo());
		WizardDialog dialog = new WizardDialog(getActiveShell(), wizard) {
		    protected void configureShell(Shell shell) {
		    	super.configureShell(shell);
//		    	shell.setText(Messages.ProjectProfilePage_CreateNewProfileWizardTitle);
//		    	shell.setImage(Activator.getImage("new.gif")); //$NON-NLS-1$
		    }
		};	

		if (dialog.open() == Window.OK) {
			int profileNo = wizard.getGroupNo();
			ProfileGroup group = pp.createGroup(profileNo);
			group.createProfile("COMMENT").setUpdateValue(wizard.getComment()); //$NON-NLS-1$
			group.createProfile("MAC_FOR_FILE").setUpdateValue(String.valueOf(wizard.getMacForFile())); //$NON-NLS-1$
			group.createProfile("MAC_FOR_NETWORK").setUpdateValue(String.valueOf(wizard.getMacForNetwork())); //$NON-NLS-1$
			group.createProfile("TOMOYO_VERBOSE").setUpdateValue(String.valueOf(wizard.getVerbose())); //$NON-NLS-1$
			boolean isSaved = pp.save();
			if (isSaved) {
				addColumn(treeViewer, profileNo);
				updateColumn();
				restoreViewerSelectioned(treeViewer);
				treeViewer.refresh();
				TomoyoCorePlugin.getDefault().setStatusMessage(STATUS_MSG_SAVED);
			} else {
				TomoyoCorePlugin.getDefault().setStatusErrorMessage(STATUS_MSG_ERROR);
			}
		}
	}
	
	//-----------------------------------------------------------------------------------------
	// Table viewer
	//-----------------------------------------------------------------------------------------
	
	private TreeViewer treeViewer;
	
	private TreeViewer createProfileTableTree(Composite parent) {
		TreeViewer viewer = new TreeViewer(parent, SWT.BORDER);
		GridData layoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
    	viewer.getControl().setLayoutData(layoutData);
    	setupColumn(viewer);
    	viewer.setContentProvider(new ProfileViewerContentProvider(getProject()));
		viewer.setLabelProvider(new ProfileViewerLabelProvider(getProject(), viewer));
		
		viewer.getTree().addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent event) {
				StructuredSelection selection = (StructuredSelection)treeViewer.getSelection();
				if (selection.size() == 1) {
					if (event.button == 3)	// ignore right double click
						return;
					/*
					Object object = selection.getFirstElement();
					if (object instanceof ProfileMetaData) {
						ProfileMetaData meta = (ProfileMetaData)object;
						TomoyoCorePlugin.getDefault().setStatusMessage(meta.getDescription()); //$NON-NLS-1$
					}
					*/
				}
			}
			public void mouseDoubleClick(MouseEvent event) {
				StructuredSelection selection = (StructuredSelection)treeViewer.getSelection();
				if (selection.size() == 1) {
					if (event.button == 3)	// ignore right double click
						return;
					Object object = selection.getFirstElement();
					if (treeViewer.getExpandedState(object))
						treeViewer.collapseToLevel(object, 1);
					else
						treeViewer.expandToLevel(object, 1);
				}
			}
		});
		viewer.getTree().addSelectionListener(new SelectionListener() {
			public void widgetDefaultSelected(SelectionEvent event) {
			}
			public void widgetSelected(SelectionEvent event) {
				handleSelection(event);
			}
		});
		viewer.getTree().addKeyListener(new KeyListener() {
			public void keyPressed(KeyEvent e) {
				if (e.keyCode == SWT.CR | e.keyCode == ' ') {
					ISelection selection = treeViewer.getSelection();
					Iterator<?> it = ((IStructuredSelection)selection).iterator();
					while (it.hasNext()) {
						Object object = it.next();
						if (treeViewer.getExpandedState(object))
							treeViewer.collapseToLevel(object, 1);
						else
							treeViewer.expandToLevel(object, 1);
					}
				}
			}
			public void keyReleased(KeyEvent arg0) {
			}
		});
    	viewer.getControl().addFocusListener(new TreeFocusAdapter(viewer));

//    	setupMenuListener(parent, viewer);
	    
		return viewer;
	}

	private void handleSelection(SelectionEvent event) {
		StructuredSelection selection = (StructuredSelection)treeViewer.getSelection();

		storeViewerSelectioned(treeViewer, selection);

		switchCellEditor();
	}

	private void switchCellEditor() {
		if (treeViewer == null)
			return;
		StructuredSelection selection = (StructuredSelection)treeViewer.getSelection();
		if (selection == null)
			return;
		Object object = selection.getFirstElement();
		if (object == null)
			return;
		
		CellEditor[] editors = treeViewer.getCellEditors();
		
		for (int cnt = 1; cnt < editors.length; cnt++) {
			if (object instanceof ProfileMetaData) {
				ProfileMetaData meta = (ProfileMetaData)object;
				if (meta.getType() instanceof ProfileTypeChoise) {
					ProfileTypeChoise choise = (ProfileTypeChoise)meta.getType();
					editors[cnt] = new ComboBoxCellEditor(treeViewer.getTree(), choise.choiseTexts, SWT.CENTER);
				} else {
					editors[cnt] = new TextCellEditor(treeViewer.getTree(), SWT.CENTER);
					editors[cnt].addListener(new ICellEditorListener() {
						public void applyEditorValue() {
						}
						public void cancelEditor() {
						}
						public void editorValueChanged(boolean oldValidState, boolean newValidState) {
						}
					});
				}
			}
			if (object instanceof Class) {
				Class cls = (Class)object;
				if (cls.equals(String.class)) {
					editors[cnt] = new TextCellEditor(treeViewer.getTree(), SWT.CENTER);
				}
				if (cls.equals(ForeColor.class)) {
					editors[cnt] = new ColorCellEditor(treeViewer.getTree());
				}
				if (cls.equals(BackColor.class)) {
					editors[cnt] = new ColorCellEditor(treeViewer.getTree());
				}
			}
		}
	}
	
	private static final String TITLE_COLUMN_PROPERTY = ""; //$NON-NLS-1$

	private void setupColumn(TreeViewer viewer) {
		Tree tree = viewer.getTree();
//		TableLayout layout = new TableLayout();
//		table.setLayout(layout);
		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);
		
		createBlankColumn(tree);

		ProfileGroup[] groups = ProfileManager.getInstance().getProjectProfile(getProject()).toArray(true);
		
		String[] properties = new String[groups.length + 1];
		CellEditor[] editors = new CellEditor[groups.length + 1];

		properties[0] = TITLE_COLUMN_PROPERTY; //$NON-NLS-1$
		editors[0] = null;
		
		for (int cnt = 0; cnt < groups.length; cnt++) {
			ProfileGroup group = groups[cnt];
			TreeColumn column = new TreeColumn(tree, SWT.NONE);
			column.setText(group.getLongName()); //$NON-NLS-1$
			column.setAlignment(SWT.CENTER);
			column.setWidth(100);
			column.setData(Integer.valueOf(group.getNo()));
			properties[cnt + 1] = String.valueOf(group.getNo());
			editors[cnt + 1] = null;
		}		    
		
		viewer.setColumnProperties(properties);
		viewer.setCellEditors(editors);
		viewer.setCellModifier(createCellModifier(viewer));
	    
	}
	
	private void addColumn(TreeViewer viewer, int groupNo) {
		Tree tree = viewer.getTree();
		TreeColumn column = new TreeColumn(tree, SWT.NONE);
		column.setAlignment(SWT.CENTER);
		column.setWidth(100);
		column.setData(Integer.valueOf(groupNo));

		String[] oldProperties = (String[])treeViewer.getColumnProperties();
		CellEditor[] oldEditors = treeViewer.getCellEditors();
		
		String[] newProperties = new String[oldProperties.length + 1];
		CellEditor[] newEditors = new CellEditor[oldEditors.length + 1];

		System.arraycopy(oldProperties, 0, newProperties, 0, oldProperties.length);
		System.arraycopy(oldEditors, 0, newEditors, 0, oldEditors.length);
        
		newProperties[newProperties.length - 1] = String.valueOf(groupNo);
		newEditors[newEditors.length - 1] = null;
        
		viewer.setColumnProperties(newProperties);
		viewer.setCellEditors(newEditors);
	}
	
	private void updateColumn() {
		TreeColumn[] columns = treeViewer.getTree().getColumns();
		for (int cnt = 0; cnt < columns.length; cnt++) {
			if (cnt == 0)
				continue;
			TreeColumn column = columns[cnt];
			Integer groupNo = (Integer)column.getData();
			ProfileGroup group = ProfileManager.getInstance().getProjectProfile(getProject()).getGroup(groupNo.intValue());
			columns[cnt].setText(group.getLongName()); //$NON-NLS-1$
		}
	}
	
	private void createBlankColumn(Tree tree) {
		TreeColumn column = new TreeColumn(tree, SWT.NONE);
		column.setWidth(200);
	}

	//-----------------------------------------------------------------------------------------
	// Models
	//-----------------------------------------------------------------------------------------

	protected Object createModel() {
		return ProfileManager.getInstance().getProjectProfile(getProject());
	}

	//-----------------------------------------------------------------------------------------
	// Cell modifier
	//-----------------------------------------------------------------------------------------
	
	private CellModifier cellModifier;
	
	private ICellModifier createCellModifier(TreeViewer viewer) {
		if (cellModifier == null)
			cellModifier = new CellModifier(viewer);
		return cellModifier;
	}

	public class CellModifier implements ICellModifier {
		  
		private TreeViewer viewer;

		public CellModifier(TreeViewer viewer) {
			this.viewer = viewer;
		}

		private Object lastElement = null;
		
		public boolean canModify(Object element, String property) {
			boolean modifiable = false;
			try {
				if (element instanceof ProfileMetaData) {
					modifiable = (lastElement == element);
				} else if (element instanceof Class) {
					modifiable = (lastElement == element);
				} else {
				}
				return modifiable;
			} finally {
				lastElement = element;
			}
		}

		public Object getValue(Object element, String property) {
			if (property.equals(TITLE_COLUMN_PROPERTY))
				return null;

			CellEditor[] editors = treeViewer.getCellEditors();
			CellEditor editor = editors[1];
			
			if (element instanceof ProfileMetaData) {
				ProfileMetaData meta = (ProfileMetaData)element;
				ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(getProject());
				ProfileGroup group = pp.getGroup(Integer.parseInt(property));
				Profile profile = group.getProfile(meta);
				if (editor instanceof TextCellEditor) {
					if (profile == null) {		
						return ""; //$NON-NLS-1$
					}
					return profile.getValue();
				}
				if (editor instanceof ComboBoxCellEditor) {
					if (profile == null) {		
						return new Integer(0);
					}
					if (meta.getType() instanceof ProfileTypeChoise) {
						return new Integer(profile.getValue());
					}
				}
			}
			
			if (element instanceof Class) {
				Class cls = (Class)element;
				ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(getProject());
				ProfileGroup group = pp.getGroup(Integer.parseInt(property));
				if (editor instanceof TextCellEditor) {
					String description = group.getDescription();
					if (description == null)
						return ""; //$NON-NLS-1$
					else
						return description;
				}
				if (editor instanceof ColorCellEditor) {
					if (cls.equals(ForeColor.class)) {
						RGB color = group.getForecolor();
						if (color == null)
							return new RGB(0, 0, 0);
						else
							return color;
					}
					if (cls.equals(BackColor.class)) {
						RGB color = group.getBackcolor();
						if (color == null)
							return new RGB(0, 0, 0);
						else
							return color;
					}
				}
				if (editor instanceof ComboBoxCellEditor) {
					return new Integer(1);
				}
			}
			
			return null;
		}

		public void modify(Object element, String property, Object value) {
			if (property.equals(TITLE_COLUMN_PROPERTY))
				return;

			TreeItem treeItem = (TreeItem)element;
			
			if (treeItem.getData() instanceof ProfileMetaData) {
				ProfileMetaData meta = (ProfileMetaData)treeItem.getData();
				ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(getProject());
				ProfileGroup group = pp.getGroup(Integer.parseInt(property));
				Profile profile = group.getProfile(meta);
				if (profile == null) {
					profile = group.createProfile(meta);
				}	
				if (validate(meta, value.toString())) {
					profile.setUpdateValue(translate(value.toString()));
				}
			}
			
			if (treeItem.getData() instanceof Class) {
				Class cls = (Class)treeItem.getData();
				ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(getProject());
				ProfileGroup group = pp.getGroup(Integer.parseInt(property));
				if (cls.equals(ForeColor.class)) {
					RGB rgb = (RGB)value;
					group.saveForecolor(rgb);
				}
				if (cls.equals(BackColor.class)) {
					RGB rgb = (RGB)value;
					group.saveBackcolor(rgb);
				}
				if (cls.equals(String.class)) {
					String username = (String)value;
					if (username == null || username.trim().length() == 0)
						group.saveDescription(null);
					else
						group.saveDescription(username);
					pp.setUpdate();
				}
			}

			viewer.refresh(true);
//			viewer.update(element, null);	// This doesn't make changes visible.
			
			updateButtonEnabled();
		}

		private String translate(String string) {
			if (string != null && string.equals("-1")) //$NON-NLS-1$
				return null;
			return string;
		}

		private boolean validate(ProfileMetaData meta, String string) {
			if (meta.getType() instanceof ProfileTypeRange) {
				ProfileTypeRange type = (ProfileTypeRange)meta.getType();
				try {
					int value = Integer.parseInt(string);
					if (value < type.min || type.max < value) {
						MessageDialog.openWarning(getActiveShell(),
								Messages.ProjectProfilePage_InputErrorDialogTitle,
								Messages.ProjectProfilePage_MustBetweenErrorMessage + type.min + Messages.ProjectProfilePage_MustBetweenErrorMessageAnd + type.max);
						return false;
					}
				} catch (NumberFormatException e1) {
					MessageDialog.openWarning(getActiveShell(),
							Messages.ProjectProfilePage_InputErrorDialogTitle,
							Messages.ProjectProfilePage_MustBeNumberErrorMessage);
					return false;
				}
			}
			return true;
		}

	}
	
	private void updateButtonEnabled() {
		boolean dirty = ProfileManager.getInstance().getProjectProfile(getProject()).hasDirty();
		saveButton.setEnabled(dirty);
		rollbackButton.setEnabled(dirty);
	}

	protected void notifyConnectionUpdated(IProject project, boolean isConnected) {
		super.notifyConnectionUpdated(project, isConnected);
		if (!checkWidgets())
			return;
		if (project == getProject()) {
			saveButton.setEnabled(isConnected);
		}
	}
	
	protected void notifyProfileUpdated(IProject project) {
		super.notifyProfileUpdated(project);
		
		updateColumn();	// for update column text when comment and description are updated.
	}

	protected void notifyLocalResourceUpdated(LocalResource localResource) {
		super.notifyLocalResourceUpdated(localResource);
		treeViewer.refresh();
		updateButtonEnabled();
	}

	protected String getHelpPath() {
		return "/jp.sourceforge.tomoyo.doc/" + "html/" + "page-profile.html"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

}
