/*******************************************************************************
  TPI - flexible but useless plug-in framework.
  Copyright (C) 2002-2009 Silky

  This library is free software; you can redistribute it and/or modify it under
  the terms of the GNU Lesser General Public License as published by the Free
  Software Foundation; either version 2.1 of the License, or (at your option)
  any later version.

  This library is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License along
  with this library; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

  $Id: frm_main.cpp 482 2011-02-09 13:29:52Z sirakaba $
*******************************************************************************/

#include "lychee.h"

#include "frm_main.h"
#include "cls_filedroptarget.h"
#include "dlg_make.h"
#include "dlg_process.h"
#include "functions.h"

#include <wx/arrimpl.cpp>

WX_DEFINE_OBJARRAY(ArrayTPI_FILEINFO);

#define SetMenuToolState(id, state) this->toolbar->EnableTool(XRCID(id), state); this->menubar->Enable(XRCID(id), state)

//******************************************************************************
//    グローバル変数
//******************************************************************************

wxImageList g_hIconT(16, 16), g_hIconLL(32, 32), g_hIconLS(16, 16);
int g_nSortColumn;
bool g_fSortAscend;

//******************************************************************************
// MainFrame
//******************************************************************************

MainFrame::MainFrame(): wxFrame()
{
}

MainFrame::~MainFrame()
{
	wxCommandEvent e;
	this->OnArcClose(e);

	// 設定を記録。
	if (! this->IsIconized() && ! this->IsMaximized())
	{
		int a, b;
		this->GetSize(& a, & b);
		this->conf.WriteId(CONF_WINDOW_WIDTH,  a);
		this->conf.WriteId(CONF_WINDOW_HEIGHT, b);
		this->GetPosition(& a, & b);
		this->conf.WriteId(CONF_WINDOW_X, a);
		this->conf.WriteId(CONF_WINDOW_Y, b);
	}
	this->conf.WriteId(CONF_WINDOW_SPLITTER_POS, this->window_splitter->GetSashPosition());

	// ツールバー/ステータスバー関連。
	this->conf.WriteId(CONF_WINDOW_STATUSBAR, this->statusbar->IsShown());
	this->conf.WriteId(CONF_WINDOW_TOOLBAR,   this->toolbar->IsShown());

	// ListView関連。
	this->conf.WriteId(CONF_LISTVIEW_SHOWMODE, this->menubar->IsChecked(XRCID("Exe_View_Icons")) ? 1 : this->menubar->IsChecked(XRCID("Exe_View_List")) ? 2 : 0);
	if (this->menubar->IsChecked(XRCID("Exe_View_Details")))
	{
		this->conf.WriteId(CONF_LISTVIEW_C_FILENAME, this->list_ctrl->GetColumnWidth(0));
		this->conf.WriteId(CONF_LISTVIEW_C_UNPACKED, this->list_ctrl->GetColumnWidth(1));
		this->conf.WriteId(CONF_LISTVIEW_C_PACKED,   this->list_ctrl->GetColumnWidth(2));
		this->conf.WriteId(CONF_LISTVIEW_C_RATIO,    this->list_ctrl->GetColumnWidth(3));
		this->conf.WriteId(CONF_LISTVIEW_C_METHOD,   this->list_ctrl->GetColumnWidth(4));
		this->conf.WriteId(CONF_LISTVIEW_C_ATTR,	 this->list_ctrl->GetColumnWidth(5));
		this->conf.WriteId(CONF_LISTVIEW_C_LASTMOD,  this->list_ctrl->GetColumnWidth(6));
		this->conf.WriteId(CONF_LISTVIEW_C_PATH,     this->list_ctrl->GetColumnWidth(7));
		this->conf.WriteId(CONF_LISTVIEW_C_TYPE,     this->list_ctrl->GetColumnWidth(8));
		this->conf.WriteId(CONF_LISTVIEW_C_NO,       this->list_ctrl->GetColumnWidth(9));
		this->conf.WriteId(CONF_LISTVIEW_C_COMMENT,  this->list_ctrl->GetColumnWidth(10));
		this->conf.WriteId(CONF_LISTVIEW_S_COLUMN,   g_nSortColumn);
		this->conf.WriteId(CONF_LISTVIEW_S_ASCEND,   g_fSortAscend);
	}

	this->Close(true);
}

//******************************************************************************
// Event Table.
//******************************************************************************

BEGIN_EVENT_TABLE(MainFrame, wxFrame)
	EVT_INIT_DIALOG(      MainFrame::OnInit)
	// Menu
	EVT_MENU(XRCID("Arc_Create"),  MainFrame::OnArcCreate)
	EVT_MENU(XRCID("Arc_Open"),    MainFrame::OnArcOpen)
	EVT_MENU(XRCID("Arc_Close"),   MainFrame::OnArcClose)
	EVT_MENU(XRCID("Arc_Clone"),   MainFrame::OnArcClone)
	EVT_MENU(XRCID("Arc_Add"),     MainFrame::OnArcAdd)
	EVT_MENU(XRCID("Arc_SFX"),     MainFrame::OnArcConvert)
	EVT_MENU(XRCID("Arc_UnSFX"),   MainFrame::OnArcConvert)
	EVT_MENU(XRCID("Exe_Exit"),    MainFrame::OnExit)
	EVT_MENU(XRCID("Arc_Extract"), MainFrame::OnArcExtract)
	EVT_MENU(XRCID("Arc_Delete"),  MainFrame::OnArcDelete)
	EVT_MENU(XRCID("Arc_Test"),    MainFrame::OnArcTest)
	EVT_MENU(XRCID("Arc_Repair"),  MainFrame::OnArcRepair)
	EVT_MENU(XRCID("Exe_Copy"),    MainFrame::OnExeCopy)
	EVT_MENU(XRCID("Exe_View_Icons"),  MainFrame::OnViewMode)
	EVT_MENU(XRCID("Exe_View_Details"),MainFrame::OnViewMode)
	EVT_MENU(XRCID("Exe_View_List"),   MainFrame::OnViewMode)
	EVT_MENU(XRCID("Exe_View_ToolBar"),MainFrame::OnShowToolBar)
	EVT_MENU(XRCID("Exe_View_StatusBar"),MainFrame::OnShowStatusBar)
	EVT_MENU(XRCID("Exe_View_SelectAll"),MainFrame::OnSelectAll)
	// TreeView
	EVT_TREE_SEL_CHANGED(XRCID("TreeView"), MainFrame::OnTreeChanged)
	EVT_TREE_BEGIN_DRAG( XRCID("TreeView"), MainFrame::OnTreeBeginDrag)
	EVT_COMMAND_CONTEXT_MENU(XRCID("TreeView"), MainFrame::OnContextMenu)
	// ListView
	EVT_LIST_ITEM_SELECTED(  XRCID("ListView"), MainFrame::OnListItemSelect)
	EVT_LIST_ITEM_DESELECTED(XRCID("ListView"), MainFrame::OnListItemSelect)
	EVT_LIST_ITEM_ACTIVATED( XRCID("ListView"), MainFrame::OnListItemDClick)
	EVT_LIST_BEGIN_DRAG(     XRCID("ListView"), MainFrame::OnListBeginDrag)
	EVT_COMMAND_CONTEXT_MENU(XRCID("ListView"), MainFrame::OnContextMenu)
	// Filter
	EVT_TEXT(XRCID("tcFilter"), MainFrame::OnFilter)
END_EVENT_TABLE()

//******************************************************************************
// Event handler.
//******************************************************************************

void MainFrame::OnInit(wxInitDialogEvent&)
{
	// XRCと結びつけ。
	this->menubar    = this->GetMenuBar();
	this->toolbar    = this->GetToolBar();
	this->statusbar = XRCCTRL(* this, "statusbar", wxStatusBar);
	this->tree_ctrl = XRCCTRL(* this, "TreeView", wxTreeCtrl);
	this->list_ctrl = XRCCTRL(* this, "ListView", myListCtrl);
	this->window_splitter = XRCCTRL(* this, "window_splitter", wxSplitterWindow);
	this->tcFilter  = XRCCTRL(* this->toolbar, "tcFilter", wxTextCtrl);

	// 設定を読み込み。
	this->SetSize(this->conf.ReadId(CONF_WINDOW_X, 0l), this->conf.ReadId(CONF_WINDOW_Y, 0l), this->conf.ReadId(CONF_WINDOW_WIDTH, 800l), this->conf.ReadId(CONF_WINDOW_HEIGHT, 400l), wxSIZE_ALLOW_MINUS_ONE);
	wxTheMimeTypesManager->Initialize(wxMAILCAP_ALL);

	// 初期値設定。
	{
		wxIcon icon;
		icon.CopyFromBitmap(wxBitmap(L_DIR_S_ICO wxT("app.png"), wxBITMAP_TYPE_ANY));
		this->SetIcon(icon);
	}
	wxCommandEvent e;
	this->OnArcClose(e);
	this->SetDropTarget(new myFileDropTarget(this));

	// スプリッター設定。
	this->window_splitter->SetSashPosition(this->conf.ReadId(CONF_WINDOW_SPLITTER_POS, 200l));

	// リストビュー設定。
	int nIconMode = this->conf.ReadId(CONF_LISTVIEW_SHOWMODE, 0l);
	e.SetId(nIconMode == 1 ? XRCID("Exe_View_Icons") : (nIconMode == 2 ? XRCID("Exe_View_List") : XRCID("Exe_View_Details")));
	this->OnViewMode(e);
	// wxGTKでは直接wxLC_VIRTUALを指定しないと反映されない。
	this->list_ctrl->SetSingleStyle(wxLC_VIRTUAL);
	this->list_ctrl->InsertColumn(0, _("Filename"),      wxLIST_FORMAT_LEFT,   this->conf.ReadId(CONF_LISTVIEW_C_FILENAME, 140l));
	this->list_ctrl->InsertColumn(1, _("Unpacked"),      wxLIST_FORMAT_RIGHT,  this->conf.ReadId(CONF_LISTVIEW_C_UNPACKED,  80l));
	this->list_ctrl->InsertColumn(2, _("Packed"),        wxLIST_FORMAT_RIGHT,  this->conf.ReadId(CONF_LISTVIEW_C_PACKED,    80l));
	this->list_ctrl->InsertColumn(3, _("Ratio"),         wxLIST_FORMAT_RIGHT,  this->conf.ReadId(CONF_LISTVIEW_C_RATIO,     50l));
	this->list_ctrl->InsertColumn(4, _("Method"),        wxLIST_FORMAT_LEFT,   this->conf.ReadId(CONF_LISTVIEW_C_METHOD,    60l));
	this->list_ctrl->InsertColumn(5, _("Attr"),          wxLIST_FORMAT_LEFT,   this->conf.ReadId(CONF_LISTVIEW_C_ATTR,      50l));
	this->list_ctrl->InsertColumn(6, _("Last modified"), wxLIST_FORMAT_RIGHT,  this->conf.ReadId(CONF_LISTVIEW_C_LASTMOD,  150l));
	this->list_ctrl->InsertColumn(7, _("Path"),          wxLIST_FORMAT_LEFT,   this->conf.ReadId(CONF_LISTVIEW_C_PATH,     100l));
	this->list_ctrl->InsertColumn(8, _("Type"),          wxLIST_FORMAT_LEFT,   this->conf.ReadId(CONF_LISTVIEW_C_TYPE,     100l));
	this->list_ctrl->InsertColumn(9, _("No."),           wxLIST_FORMAT_RIGHT,  this->conf.ReadId(CONF_LISTVIEW_C_NO,        35l));
	this->list_ctrl->InsertColumn(10,_("Comment"),       wxLIST_FORMAT_LEFT,   this->conf.ReadId(CONF_LISTVIEW_C_COMMENT,   35l));
	g_nSortColumn = this->conf.ReadId(CONF_LISTVIEW_S_COLUMN, 9l);
	g_fSortAscend = this->conf.ReadId(CONF_LISTVIEW_S_ASCEND, true);

	// ツールバー/ステータスバー設定。
	int nStatusBarParts[] = {70, 70, 180, 50, -1};
	this->statusbar->SetFieldsCount(5, nStatusBarParts);
	this->SetStatusBarPane(-1);
	bool fShow = this->conf.ReadId(CONF_WINDOW_STATUSBAR, true);
	this->menubar->Check(XRCID("Exe_View_StatusBar"), fShow);
	this->statusbar->Show(fShow);
	fShow = this->conf.ReadId(CONF_WINDOW_TOOLBAR, true);
	this->menubar->Check(XRCID("Exe_View_ToolBar"), fShow);
	this->toolbar->Show(fShow);
}

// MenuBar

void MainFrame::OnExit(wxCommandEvent&)
{
	this->Close(true);
}

void MainFrame::OnArcCreate(wxCommandEvent& e)
{
	TPI_SWITCHES swInfo;
	swInfo.pCustomSwitches = NULL;

	// 作成ダイアログを設定。
	MakeDialog mkDlg;
	mkDlg.SetParent(this);
	if (e.GetClientData() == NULL)
	{
		// 処理対象のファイルを選択。
		wxFileDialog fd(this, _("Choose files to compress"), this->conf.ReadHistory(CONF_HISTORY_PATH, 0), wxEmptyString, wxFileSelectorDefaultWildcardStr, wxFD_OPEN | wxFD_FILE_MUST_EXIST | wxFD_MULTIPLE);
		if (fd.ShowModal() == wxID_CANCEL)
		{
			return;
		}
		swInfo.fnDestinationDirectory = wxFileName::DirName(fd.GetDirectory());
		this->conf.WriteHistory(CONF_HISTORY_PATH, fd.GetDirectory());
		fd.GetFilenames(mkDlg.files);
	}
	else
	{
		mkDlg.files = * (wxArrayString *) e.GetClientData();
		swInfo.fnDestinationDirectory = wxFileName::DirName(wxFileName(mkDlg.files[0]).GetPath());
		// 相対パスに変換。
		for (size_t n = 0; n < mkDlg.files.GetCount(); n++)
		{
			wxFileName fn(mkDlg.files[n]);
			fn.MakeRelativeTo(swInfo.fnDestinationDirectory.GetPath());
			mkDlg.files[n] = fn.GetFullPath();
		}
	}

	this->OnArcClose(e);
	// 書庫名はファイル名の拡張子より前、もしくはディレクトリ名。
	this->fnArchive = wxFileName(mkDlg.files[0]);
	this->fnArchive.SetName(mkDlg.files.GetCount() == 1 ? (this->fnArchive.GetName().IsEmpty() ? this->fnArchive.GetDirs().Last() : this->fnArchive.GetName()) : swInfo.fnDestinationDirectory.GetDirs().Last());
	this->fnArchive.SetEmptyExt();
	this->fnArchive.SetPath(swInfo.fnDestinationDirectory.GetPath());

	// ダイアログを表示。
	if (mkDlg.ShowModal() == wxID_CANCEL)
	{
		return;
	}

	// 各種設定。
	swInfo.fStoreDirectoryPathes= ! mkDlg.cbIgnorePath->IsChecked();
	swInfo.fMakeSFX			    = mkDlg.cbMakeSFX->IsChecked();
	swInfo.fSolid               = mkDlg.cbSolid->IsChecked();
	swInfo.fMMOptimize          = mkDlg.cbMMOptimize->IsChecked();
	swInfo.fEncryptHeader       = mkDlg.cbEncryptHeader->IsChecked();
	swInfo.fCompressHeader      = mkDlg.cbCompressHeader->IsChecked();
	swInfo.nCompressLevel       = mkDlg.scLevel->GetValue();
	swInfo.nRecoveryRecord      = mkDlg.scRR->GetValue();
	swInfo.szPassword           = mkDlg.tcPassword->GetValue();
	swInfo.szKeyFile            = mkDlg.tcKeyfile->GetValue();
	swInfo.szComment            = mkDlg.tcComment->GetValue();
	mkDlg.cbSplitSize->GetValue().ToULongLong(& swInfo.nSplitSize);

	// TPIを読み込み。
	int nSelected = mkDlg.chType->GetSelection();
	this->fnArchive = wxFileName(mkDlg.cbDir->GetValue(), mkDlg.cbFileName->GetValue());
	if (! tpi.InitLibrary(mkDlg.afInfo[nSelected].szTPIName, this->fnArchive.GetFullPath(), mkDlg.afInfo[nSelected].nTypeId))
	{
		this->ErrorCheck(tpi.nErrorCode, wxT("InitLibrary"));
		return;
	}

	// コールバック関数を設定。
	tpi.SetCallbackProc(TPICallbackProc);
	this->ErrorCheck(tpi.nErrorCode, wxT("SetCallbackProc"));

	// 処理を行う。
	{
		ProcessDialog procDlg;
		procDlg.fnArchive  = & this->fnArchive;
		procDlg.nFileCount = mkDlg.files.GetCount();
		procDlg.Show(true);

		tpi.Command(TPI_COMMAND_CREATE, & swInfo, this->fnArchive.GetFullPath(), mkDlg.files);
		this->ErrorCheck(tpi.nErrorCode);
		procDlg.Show(false);
	}
	tpi.FreeLibrary();

	if (mkDlg.cbOpenAfter->IsChecked())
	{
		// 作成先を開く。
		::wxExecute(DIR_APP + QuoteString(swInfo.fnDestinationDirectory.GetFullPath()));
	}

	if (mkDlg.cbExitAfter->IsChecked())
	{
		// 終了。
		this->Close(true);
	}

	// 終了しない場合は書庫を開く。
	this->OnArcOpen(e);
}

void MainFrame::OnArcOpen(wxCommandEvent& e)
{
	// モード取得。通常は0, それ以外で開く場合は1, ファイルDnDなら2。
	int nMode = e.GetInt() == 2 ? 2 : e.GetId() == XRCID("Arc_Open") ? 0 : 1;

	// 書庫を選択。
	if (nMode == 0)
	{
		wxFileDialog fd(this, _("Choose an archive"), this->conf.ReadHistory(CONF_HISTORY_PATH, 0), wxEmptyString, wxFileSelectorDefaultWildcardStr, wxFD_OPEN | wxFD_FILE_MUST_EXIST);
		if (fd.ShowModal() == wxID_CANCEL)
		{
			return;
		}
		this->conf.WriteHistory(CONF_HISTORY_PATH, fd.GetDirectory());
		this->fnArchive = wxFileName(fd.GetPath());
	}

	// 進捗ダイアログ表示。
	ProcessDialog procDlg;
	procDlg.fnArchive  = & this->fnArchive;
	procDlg.Show(true);

	// DnD以外で書庫を開く場合、TPIを読み込み。
	TPI_PROCESSINFO piInfo;
	if (! this->LoadTPI(this->fnArchive.GetFullPath(), & piInfo.fiInfo.nUnpackedSize))
	{
		procDlg.Show(false);
		if (nMode == 2)
		{
			// DnDの場合は書庫を作成する。
			this->OnArcCreate(e);
		}
		else if (this->IsShown())
		{
			wxBell();
			this->statusbar->SetStatusText(_("No plug-in supporting this archive was found!"), 4);
		}
		else
		{
			wxLogError(_("No plug-in supporting this archive was found!"));
			this->Close(true);
		}
		return;
	}

	piInfo.eMessage = TPI_MESSAGE_STATUS;
	piInfo.eStatus = 0x1001;
	if (this->ErrorCheck(procDlg.CallbackProc(TPI_NOTIFY_COMMON, & piInfo), wxT("Callback")) == TPI_CALLBACK_CANCEL)
	{
		procDlg.Show(false);
		tpi.CloseArchive();
		wxCommandEvent e;
		this->OnArcClose(e);
		return;
	}

	// 配列のサイズを確保。
	this->fileinfo.Alloc(piInfo.fiInfo.nUnpackedSize);

	// 履歴に追加。
	this->conf.WriteHistory(CONF_HISTORY_PATH, this->fnArchive.GetPath());
	this->conf.WriteHistory(CONF_HISTORY_NAME, this->fnArchive.GetFullName());
	this->conf.WriteHistory(CONF_HISTORY_FULL, this->fnArchive.GetFullPath());

	// 書庫のアイコンを取得し、書庫ルートを作成。
	g_hIconT.Add(wxBitmap(L_DIR_S_ICO wxT("folder_closed.png"), wxBITMAP_TYPE_ANY));
	g_hIconT.Add(wxBitmap(L_DIR_S_ICO wxT("folder_open.png"), wxBITMAP_TYPE_ANY));
	this->tree_ctrl->SetImageList(& g_hIconT);
	wxTreeItemId
		idRoot = this->tree_ctrl->AddRoot(wxEmptyString),
#ifdef __WINDOWS__
		idArchive = this->tree_ctrl->AppendItem(idRoot, this->fnArchive.GetFullName(), g_hIconT.Add(GetFileTypeIcon(piInfo.fiInfo.fnFileName))),
#else
		idArchive = this->tree_ctrl->AppendItem(idRoot, this->fnArchive.GetFullName(), g_hIconT.Add(GetFileTypeIcon(piInfo.fiInfo.fnFileName).ConvertToImage().Rescale(16, 16))),
#endif
		idArcRoot = this->tree_ctrl->AppendItem(idRoot, wxT("-----"), 0, 1);

	// ファイル情報をロード。
	if (tpi.GetFileInformation(& piInfo.fiInfo, true))
	{
		piInfo.eStatus = 0x1002;
		piInfo.nProcessedSize = 0;
		do
		{
			piInfo.nProcessedSize++;
			if (this->ErrorCheck(procDlg.CallbackProc(TPI_NOTIFY_COMMON, & piInfo), wxT("Callback")) == TPI_CALLBACK_CANCEL)
			{
				procDlg.Show(false);
				tpi.CloseArchive();
				wxCommandEvent e;
				this->OnArcClose(e);
				return;
			}

			// 拡張子のみ設定されている場合。
			if (piInfo.fiInfo.szStoredName.IsEmpty())
			{
				piInfo.fiInfo.szStoredName = this->fnArchive.GetName();
				if (piInfo.fiInfo.fnFileName.HasExt())
				{
					piInfo.fiInfo.szStoredName += wxT('.') + piInfo.fiInfo.fnFileName.GetExt();
				}
				piInfo.fiInfo.fnFileName = wxFileName(piInfo.fiInfo.szStoredName);
			}

			// セキュリティチェック。
			// ルート記号を削除。
			wxString szPath = piInfo.fiInfo.fnFileName.GetPathWithSep(wxPATH_UNIX);
			if (szPath.StartsWith(wxT("/")) || szPath.StartsWith(wxT("./")))
			{
				piInfo.fiInfo.fnFileName = wxFileName(szPath.AfterFirst(wxT('/')), piInfo.fiInfo.fnFileName.GetFullName(), wxPATH_DOS);
			}
			// ルートメンバの情報は無視する。
			if (piInfo.fiInfo.fnFileName.GetFullPath().IsEmpty() || piInfo.fiInfo.fnFileName.GetFullPath() == wxT("."))
			{
				continue;
			}

			// 改行文字/タブ文字などを削除。
			if (piInfo.fiInfo.szStoredName.Find(wxT('\r')) != wxNOT_FOUND
			||  piInfo.fiInfo.szStoredName.Find(wxT('\n')) != wxNOT_FOUND
			||  piInfo.fiInfo.szStoredName.Find(wxT('\t')) != wxNOT_FOUND)
			{
				wxString sz = piInfo.fiInfo.fnFileName.GetFullPath();
				sz.Replace(wxT("\r"), wxT(" "));
				sz.Replace(wxT("\n"), wxT(" "));
				sz.Replace(wxT("\t"), wxT(" "));
				piInfo.fiInfo.eDanger = TRUE;
				piInfo.fiInfo.fnFileName = wxFileName(sz);
				wxLogWarning(_("This archive may contain files whose name contains some special characters like CR(\\r), LF(\\n), Tab(\\t) and some problem would be happen if you extract these files. Don\'t extract these files carelessly.\nDanger file is:\n%s"), piInfo.fiInfo.fnFileName.GetFullPath().c_str());
			}

			// DTV検査。
			if (piInfo.fiInfo.fnFileName.GetPathWithSep(wxPATH_UNIX).Find(wxT("../")) != wxNOT_FOUND)
			{
				piInfo.fiInfo.eDanger = TRUE;
				wxLogWarning(_("This archive may have Directory Traversal Vulnerability(DTV) problem, and some danger files may be extracted to the unexpected system directory! You should use the \"Ignore file pathes\" option when extracting this archive.\nDanger file is:\n%s"), piInfo.fiInfo.fnFileName.GetFullPath().c_str());
			}
			// 空白の連続による拡張子偽装を検査。
			if (piInfo.fiInfo.fnFileName.GetFullName().Find(wxT("        ")) != wxNOT_FOUND)
			{
				piInfo.fiInfo.eDanger = TRUE;
				wxLogWarning(_("This archive may contain extension-disguised files whose real extension is hidden by using many blank charactor and you may mistake that it is a \"safe\" file. Don\'t execute these files carelessly.\nDanger file is:\n%s"), piInfo.fiInfo.fnFileName.GetFullPath().c_str());
			}
			// Unicode制御文字を検査。
			for (wxChar c = 0x200c; c <= 0x206f; c++)
			{
				if (piInfo.fiInfo.fnFileName.GetFullName().Find(c) != wxNOT_FOUND)
				{
					piInfo.fiInfo.eDanger = TRUE;
					wxLogWarning(_("This archive may contain extension-disguised files whose real extension is hidden by using Unicode control character and you may mistake that it is a \"safe\" file. Don\'t execute these files carelessly.\nDanger file is:\n%s"), piInfo.fiInfo.fnFileName.GetFullPath().c_str());
				}
				switch (c)
				{
				case 0x200f: c = 0x2027; break;
				case 0x202e: c = 0x2060; break;
				}
			}

			// ツリービューに反映。
			bool fDir = piInfo.fiInfo.dwAttribute & TPI_ATTRIBUTE_DIRECTORY ? true : false;
			TreeView_CheckNewerItem(this->tree_ctrl, idArcRoot, fDir ? piInfo.fiInfo.fnFileName.GetFullPath() : piInfo.fiInfo.fnFileName.GetPath(), true);

			// ディレクトリ属性を含むものについては情報を保存しない。
			if (fDir)
			{
				continue;
			}

			// 情報を保存してカウントアップ。
			this->fileinfo.Add(piInfo.fiInfo);
		}
		while (tpi.GetFileInformation(& piInfo.fiInfo));
	}

	// GetFileInformationがエラー終了した場合。
	this->ErrorCheck(tpi.nErrorCode, wxT("GetFileInformation"));

	// 書庫の情報を取得。
	tpi.GetArchiveInformation(& this->aiArchive);
	this->ErrorCheck(tpi.nErrorCode, wxT("GetArchiveInformation"));

	// 書庫を閉じる。
	tpi.CloseArchive();
	this->ErrorCheck(tpi.nErrorCode, wxT("CloseArchive"));

	// 以下、UI処理。
	this->fileinfo.Shrink();
	this->tree_ctrl->ExpandAll();
	this->tree_ctrl->ScrollTo(idArchive);
	this->tree_ctrl->SelectItem(idArchive);
	this->tree_ctrl->SetScrollPos(wxHORIZONTAL, 0);
	this->list_ctrl->atDangerItem.SetTextColour(* wxRED);
	this->list_ctrl->atEncryptedItem.SetTextColour(wxColour(wxT("forest green")));

	// ステータスバー設定。
	this->statusbar->SetStatusText(this->aiArchive.fiInfo.szTypeName, 0);
	this->statusbar->SetStatusText(wxString::Format(_("%u file(s)"), this->fileinfo.GetCount()), 1);
	this->statusbar->SetStatusText(wxString::Format(wxString("%" wxLongLongFmtSpec "uB -> %" wxLongLongFmtSpec "uB"), this->aiArchive.nUnpackedSize, this->aiArchive.nPackedSize), 2);
	this->statusbar->SetStatusText(wxString::Format(wxT("%3.1f%%"), this->aiArchive.wCompressRatio / 10.0), 3);
	this->statusbar->SetStatusText(this->fnArchive.GetFullPath(), 4);
	this->statusbar->SetToolTip(
		wxString::Format(
			_("%s(%s)\nModify: %s\nCreate: %s\nTPI: %s(%s)"),
			this->fnArchive.GetFullName().c_str(), this->aiArchive.fiInfo.szTypeName.c_str(),
			this->aiArchive.tmModify.Format(_("%Y/%m/%d %H:%M:%S")).c_str(),
			this->aiArchive.tmCreate.Format(_("%Y/%m/%d %H:%M:%S")).c_str(),
			this->aiArchive.fiInfo.szTPIName.c_str(), this->aiArchive.fiInfo.szEngineName.c_str()
		)
	);

	// ツールバー・メニューバー設定。ファイル選択時しか動作しない削除などは別に設定。
	SetMenuToolState("Arc_Close",   true);
	SetMenuToolState("Arc_Add",     (this->aiArchive.fiInfo.eSupportedCommand & TPI_COMMAND_ADD)   == TPI_COMMAND_ADD   && this->aiArchive.fiInfo.fArchive);
	SetMenuToolState("Arc_SFX",     (this->aiArchive.fiInfo.eSupportedCommand & TPI_COMMAND_SFX)    == TPI_COMMAND_SFX   && ! this->aiArchive.fSFX);
	SetMenuToolState("Arc_UnSFX",   (this->aiArchive.fiInfo.eSupportedCommand & TPI_COMMAND_UNSFX)  == TPI_COMMAND_UNSFX && this->aiArchive.fSFX);
	SetMenuToolState("Arc_Extract", (this->aiArchive.fiInfo.eSupportedCommand & TPI_COMMAND_EXTRACT)== TPI_COMMAND_EXTRACT);
	SetMenuToolState("Arc_Test",    (this->aiArchive.fiInfo.eSupportedCommand & TPI_COMMAND_TEST)   == TPI_COMMAND_TEST);
	SetMenuToolState("Arc_Repair",  (this->aiArchive.fiInfo.eSupportedCommand & TPI_COMMAND_REPAIR) == TPI_COMMAND_REPAIR);
	this->menubar->Enable(XRCID("Arc_Clone"), true);

	procDlg.Show(false);
	this->Raise();
}

void MainFrame::OnArcClose(wxCommandEvent& e)
{
	// ツリービュー・リストビュー設定。
	this->tree_ctrl->DeleteAllItems();
	this->list_ctrl->DeleteAllItems();
	this->list_ctrl->apShowFile.Clear();

	// ツールバー・メニューバー設定。
	SetMenuToolState("Arc_Close",   false);
	SetMenuToolState("Arc_Add",     false);
	SetMenuToolState("Arc_SFX",     false);
	SetMenuToolState("Arc_UnSFX",   false);
	SetMenuToolState("Arc_Extract", false);
	SetMenuToolState("Arc_Delete",  false);
	SetMenuToolState("Arc_Test",    false);
	SetMenuToolState("Arc_Repair",  false);
	this->menubar->Enable(XRCID("Arc_Clone"), false);

	for (int i = 0; i < this->statusbar->GetFieldsCount(); i++)
	{
		this->statusbar->SetStatusText(wxEmptyString, i);
	}
	this->statusbar->SetToolTip(wxEmptyString);
	this->fileinfo.Clear();
	this->aiArchive.szComment.Empty();
	this->aiArchive.fnArchive.Clear();
	this->aiArchive.fiInfo.szTypeName.Empty();
	this->aiArchive.fiInfo.szSuffix.Empty();
	this->aiArchive.fiInfo.szEngineName.Empty();
	this->aiArchive.fiInfo.szTPIName.Empty();

	g_hIconT.RemoveAll();
	g_hIconLL.RemoveAll();
	g_hIconLS.RemoveAll();

	// DnDで書庫を開くときは既に読み込まれているTPIを用いるので、解放してはいけない。
	if (e.GetExtraLong() == 0)
	{
		tpi.FreeLibrary();
	}
}

void MainFrame::OnArcClone(wxCommandEvent&)
{
	// 保存先を尋ねる。
	wxFileDialog fd(this, _("Clone archive"), this->fnArchive.GetPath(), this->fnArchive.GetFullName(), wxFileSelectorDefaultWildcardStr, wxFD_SAVE | wxFD_OVERWRITE_PROMPT);
	if (fd.ShowModal() == wxID_CANCEL)
	{
		return;
	}
	this->conf.WriteHistory(CONF_HISTORY_PATH, fd.GetDirectory());

	// コピー。
	::wxCopyFile(this->fnArchive.GetFullPath(), fd.GetPath());
	wxFileName fn(fd.GetPath());
	wxDateTime dtAccess, dtModify, dtCreate;
	this->fnArchive.GetTimes(& dtAccess, & dtModify, & dtCreate);
	fn.SetTimes(& dtAccess, & dtModify, & dtCreate);
}

void MainFrame::OnArcAdd(wxCommandEvent& e)
{
	// 作成ダイアログを設定。
	MakeDialog mkDlg;
	mkDlg.SetParent(this);
	mkDlg.uCommand = TPI_COMMAND_ADD;

	// 処理対象のファイルを選択。
	wxFileDialog fd(this, _("Choose files to add"), this->conf.ReadHistory(CONF_HISTORY_PATH, 0), wxEmptyString, wxFileSelectorDefaultWildcardStr, wxFD_OPEN | wxFD_FILE_MUST_EXIST | wxFD_MULTIPLE);
	if (fd.ShowModal() == wxID_CANCEL)
	{
		return;
	}
	fd.GetFilenames(mkDlg.files);
	this->conf.WriteHistory(CONF_HISTORY_PATH, fd.GetDirectory());

	// ダイアログを表示。
	if (mkDlg.ShowModal() == wxID_CANCEL)
	{
		return;
	}

	// 各種設定。
	TPI_SWITCHES swInfo;
	swInfo.pCustomSwitches       = NULL;
	swInfo.fMakeSFX              = false;
	swInfo.fnDestinationDirectory= wxFileName::DirName(fd.GetDirectory());
	swInfo.fStoreDirectoryPathes = ! mkDlg.cbIgnorePath->IsChecked();
	swInfo.fSolid               = mkDlg.cbSolid->IsChecked();
	swInfo.fMMOptimize          = mkDlg.cbMMOptimize->IsChecked();
	swInfo.fEncryptHeader       = mkDlg.cbEncryptHeader->IsChecked();
	swInfo.nCompressLevel       = mkDlg.scLevel->GetValue();
	swInfo.nRecoveryRecord      = mkDlg.scRR->GetValue();
	swInfo.szPassword           = mkDlg.tcPassword->GetValue();
	swInfo.szKeyFile            = mkDlg.tcKeyfile->GetValue();
	swInfo.szComment            = mkDlg.tcComment->GetValue();

	// 処理を行う。
	{
		ProcessDialog procDlg;
		procDlg.fnArchive  = & this->fnArchive;
		procDlg.nFileCount = mkDlg.files.GetCount();
		procDlg.Show(true);

		tpi.Command(TPI_COMMAND_ADD, & swInfo, this->fnArchive.GetFullPath(), mkDlg.files);
		this->ErrorCheck(tpi.nErrorCode);
		procDlg.Show(false);
	}

	if (mkDlg.cbOpenAfter->IsChecked())
	{
		// 作成先を開く。
		::wxExecute(DIR_APP + QuoteString(swInfo.fnDestinationDirectory.GetFullPath()));
	}

	if (mkDlg.cbExitAfter->IsChecked())
	{
		// 終了。
		this->Close(true);
	}

	// 終了しない場合は書庫を再読み込み。
	this->OnArcOpen(e);
}

void MainFrame::OnArcConvert(wxCommandEvent& e)
{
	TPI_SWITCHES swInfo;
	swInfo.fMakeSFX = e.GetId() == XRCID("Arc_SFX");

	// 保存先を尋ねる。
	wxFileDialog fd(this, swInfo.fMakeSFX ? _("Save as SFX") : _("Save as normal archive"), this->fnArchive.GetPath(), this->fnArchive.GetName() + (swInfo.fMakeSFX ? EXE_EXT : (wxString) wxEmptyString), wxFileSelectorDefaultWildcardStr, wxFD_SAVE | wxFD_OVERWRITE_PROMPT);
	if (fd.ShowModal() == wxID_CANCEL)
	{
		return;
	}
	swInfo.fnDestinationDirectory = wxFileName::DirName(fd.GetDirectory());
	this->conf.WriteHistory(CONF_HISTORY_PATH, fd.GetDirectory());

	wxArrayString files;
	files.Add(fd.GetPath());

	ProcessDialog procDlg;
	procDlg.Show(true);
	tpi.Command(swInfo.fMakeSFX ? TPI_COMMAND_SFX : TPI_COMMAND_UNSFX, & swInfo, this->fnArchive.GetFullPath(), files);
	this->ErrorCheck(tpi.nErrorCode);
	procDlg.Show(false);
}

void MainFrame::OnArcExtract(wxCommandEvent& e)
{
	TPI_SWITCHES swInfo;
	swInfo.pCustomSwitches = NULL;

	// モード取得。通常は0, 実行なら1, ファイルDnDなら2、ディレクトリDnDなら3、クリップボードなら4、コンテキストメニューからなら8。
	int nMode = e.GetInt();
	if (e.GetInt() == 8)
	{
		nMode = 0;
	}
	// 実行時のみ使用。
	wxFileType * ftFile = NULL;

	// 展開ダイアログを作成。DnDまたは実行時は表示しない。
	MakeDialog mkDlg;
	mkDlg.SetParent(this);
	mkDlg.uCommand = TPI_COMMAND_EXTRACT;
	mkDlg.files    = MakeTargetFileList(this, nMode == 1);

	if (nMode != 0)
	{
		// 作業ディレクトリ作成。
		swInfo.fStoreDirectoryPathes = false;
		wxString szDestDirBase = nMode == 3 ? this->tree_ctrl->GetItemText(this->tree_ctrl->GetSelection()) : wxT("tpi_tmp");
		wxStandardPaths p;
		if (szDestDirBase == wxT("-----"))
		{
			// 書庫ルートのときは書庫名にしておく。
			szDestDirBase = this->fnArchive.GetName();
		}
		swInfo.fnDestinationDirectory = MakeDirPath(wxFileName::DirName(p.GetTempDir()), szDestDirBase, true);
		if (! swInfo.fnDestinationDirectory.IsOk())
		{
			wxLogError(_("Unable to make the temporary directory!"));
			return;
		}
	}
	else
	{
		if (mkDlg.ShowModal() == wxID_CANCEL)
		{
			return;
		}

		// 各種設定。
		swInfo.fStoreDirectoryPathes = ! mkDlg.cbIgnorePath->IsChecked();
		swInfo.fnDestinationDirectory = wxFileName::DirName(mkDlg.cbDir->GetValue());
		swInfo.szPassword = mkDlg.tcPassword->GetValue();
		swInfo.szKeyFile  = mkDlg.tcKeyfile->GetValue();

		// 必要なら書庫名でディレクトリを作成する。
		if (WillMakeDirByArcName(this, & mkDlg))
		{
			swInfo.fnDestinationDirectory = MakeDirPath(swInfo.fnDestinationDirectory, this->fnArchive.GetName(), true);
			if (! swInfo.fnDestinationDirectory.IsOk())
			{
				wxLogError(_("Unable to make the destination directory!"));
				return;
			}
		}
	}

	// 処理を行う。
	{
		ProcessDialog procDlg;
		procDlg.fnArchive  = & this->fnArchive;
		procDlg.nFileCount = mkDlg.files.GetCount();
		procDlg.Show(true);

		tpi.Command(TPI_COMMAND_EXTRACT, & swInfo, this->fnArchive.GetFullPath(), mkDlg.files);
		this->ErrorCheck(tpi.nErrorCode);
		procDlg.Show(false);
	}

	if (nMode == 0)
	{
		if (mkDlg.cbOpenAfter->IsChecked())
		{
			// 展開先を開く。
			::wxExecute(DIR_APP + QuoteString(swInfo.fnDestinationDirectory.GetFullPath()));
		}

		if (mkDlg.cbExitAfter->IsChecked())
		{
			// 終了。
			this->Close(true);
		}
	}
	else
	{
		if (nMode == 1)
		{
			// コマンドを取得。
			ftFile = wxTheMimeTypesManager->GetFileTypeFromExtension(wxFileName(mkDlg.files[0], wxPATH_DOS).GetExt());
			if (! ftFile || ftFile->GetOpenCommand(wxEmptyString).IsEmpty())
			{
				// 種類が取得できないときはテキストとみなす。
				ftFile = wxTheMimeTypesManager->GetFileTypeFromExtension(this->conf.ReadId(CONF_DEFAULT_EXT, (wxString) wxT("txt")));
			}

			// コマンドを実行。
			wxString szTempFile = swInfo.fnDestinationDirectory.GetPathWithSep() + wxFileName(mkDlg.files[0], wxPATH_DOS).GetFullName();
			bool fSuccess = tpi.nErrorCode == TPI_ERROR_SUCCESS && ftFile != NULL;
			myProcess * pCallback = new myProcess(szTempFile, swInfo.fnDestinationDirectory.GetPath());
			if (fSuccess)
			{
#ifdef __LINUX__
				// Linuxでは引用符で囲む必要がある。
				fSuccess = ::wxExecute(ftFile->GetOpenCommand(QuoteString(szTempFile)), wxEXEC_ASYNC, pCallback) > 0;
#else
				fSuccess = ::wxExecute(ftFile->GetOpenCommand(szTempFile), wxEXEC_ASYNC, pCallback) > 0;
#endif
			}
			if (! fSuccess)
			{
				pCallback->OnTerminate(0, 0);
			}
		}
		else
		{
			// 展開対象を決定。
			wxArrayString asFiles;
			myFileDataObject * objFile = new myFileDataObject();
			objFile->szTempDir = nMode == 3 ? swInfo.fnDestinationDirectory.GetPath() : swInfo.fnDestinationDirectory.GetFullPath();
			for (size_t i = 0; i < mkDlg.files.GetCount(); i++)
			{
				wxString szFileName = swInfo.fnDestinationDirectory.GetPathWithSep() + wxFileName(mkDlg.files[i], wxPATH_DOS).GetFullName();
				if (nMode == 3)
				{
					asFiles.Add(szFileName);
				}
				else
				{
					// リストに追加。
					objFile->AddFile(szFileName);
				}
			}
			if (nMode == 3)
			{
				objFile->AddFile(objFile->szTempDir);
			}

			if (nMode == 4)
			{
				wxTheClipboard->SetData(objFile);
			}
			else
			{
				// 自身にドロップされると煩雑なので、一時的にドロップを受け付けないようにしておく。
				this->SetDropTarget(NULL);

				// DnD開始。
				wxDropSource dropSource(* objFile, this);
				wxDragResult drResult = dropSource.DoDragDrop();
				if (drResult != wxDragCancel && drResult != wxDragNone && drResult != wxDragMove)
				{
#ifdef __LINUX__
					// Linuxではまだ処理が終わっていない(コンテキストメニューが表示されている)ので、とりあえず3秒だけ待つ。
					sleep(3);
#endif
				}
				this->SetDropTarget(new myFileDropTarget(this));

				// ディレクトリDnDのときは、先にディレクトリの中のファイルを消しておく。
				if (nMode == 3)
				{
					for (size_t i = 0; i < asFiles.GetCount(); i++)
					{
						chmod(asFiles[i].ToUTF8(), 0600);
						::wxRemoveFile(asFiles[i]);
					}
				}

				delete objFile;
			}
		}
	}
}

void MainFrame::OnArcDelete(wxCommandEvent& e)
{
	// 全ファイル削除は危険ではないかと。
	if (this->list_ctrl->GetSelectedItemCount() == 0)
	{
		// wxのバグで自動では無効化できないので、実行しようとしたときに無効化する。
		SetMenuToolState("Arc_Delete", false);
		return;
	}

	if (::AskDlg(_("Are you sure to delete selected files?"), this) == wxNO)
	{
		return;
	}

	// 処理を行う。
	{
		wxArrayString asFiles = MakeTargetFileList(this, false);
		ProcessDialog procDlg;
		procDlg.fnArchive  = & this->fnArchive;
		procDlg.nFileCount = asFiles.GetCount();
		procDlg.Show(true);

		TPI_SWITCHES swInfo;
		tpi.Command(TPI_COMMAND_DELETE, & swInfo, this->fnArchive.GetFullPath(), asFiles);
		this->ErrorCheck(tpi.nErrorCode);
		procDlg.Show(false);
	}

	// 書庫を再読み込みする。
	this->OnArcOpen(e);
}

void MainFrame::OnArcTest(wxCommandEvent&)
{
	// 処理を行う。
	wxArrayString asFiles = MakeTargetFileList(this, false);
	ProcessDialog procDlg;
	procDlg.fnArchive  = & this->fnArchive;
	procDlg.nFileCount = asFiles.GetCount();
	procDlg.Show(true);

	TPI_SWITCHES swInfo;
	bool bIsCorrect = tpi.Command(TPI_COMMAND_TEST, & swInfo, this->fnArchive.GetFullPath(), asFiles);
	procDlg.Show(false);

	if (bIsCorrect)
	{
		wxLogMessage(_("This is a correct archive."));
	}
	else
	{
		this->ErrorCheck(tpi.nErrorCode);
	}
}

void MainFrame::OnArcRepair(wxCommandEvent&)
{
	// 処理を行う。
	wxArrayString asFiles = MakeTargetFileList(this, false);
	ProcessDialog procDlg;
	procDlg.fnArchive  = & this->fnArchive;
	procDlg.nFileCount = asFiles.GetCount();
	procDlg.Show(true);

	TPI_SWITCHES swInfo;
	tpi.Command(TPI_COMMAND_REPAIR, & swInfo, this->fnArchive.GetFullPath(), asFiles);
	this->ErrorCheck(tpi.nErrorCode);
	procDlg.Show(false);	
}

void MainFrame::OnExeCopy(wxCommandEvent & e)
{
	if (this->list_ctrl->GetSelectedItemCount() == 0)
	{
		// wxのバグで自動では無効化できないので、実行しようとしたときに無効化する。
		this->menubar->Enable(XRCID("Exe_Copy"), false);
		return;
	}

	if (! wxTheClipboard->Open())
	{
		return;
	}

	e.SetInt(4);
	this->OnArcExtract(e);
	wxTheClipboard->Close();
}

void MainFrame::OnViewMode(wxCommandEvent & e)
{
	int n = e.GetId();
	this->menubar->Check(n, true);
	this->list_ctrl->SetSingleStyle(wxLC_REPORT, n == XRCID("Exe_View_Details"));
	this->list_ctrl->SetSingleStyle(wxLC_ICON,   n == XRCID("Exe_View_Icons"));
	this->list_ctrl->SetSingleStyle(wxLC_LIST,   n == XRCID("Exe_View_List"));
}

void MainFrame::OnShowToolBar(wxCommandEvent & e)
{
	this->toolbar->Show(e.IsChecked());
}

void MainFrame::OnShowStatusBar(wxCommandEvent & e)
{
	this->statusbar->Show(e.IsChecked());
}

void MainFrame::OnSelectAll(wxCommandEvent &)
{
	for (int i = 0; i < this->list_ctrl->GetItemCount(); i++)
	{
		this->list_ctrl->Select(i, true);
	}
}

// TreeView

void MainFrame::OnTreeChanged(wxTreeEvent& e)
{
	// ツリービューからパスを取得。
	wxString szNodePath = TreeView_GetItemPath(this->tree_ctrl, e.GetItem());
	// リストビューを初期化。
	this->list_ctrl->apShowFile.Clear();
	this->list_ctrl->DeleteAllItems();
	g_hIconLL.RemoveAll();
	g_hIconLS.RemoveAll();

	// 現在のアイテムの選択状態を無効にしておく。
	long nSelected = this->list_ctrl->GetFirstSelected();
	while (nSelected != -1)
	{
		this->list_ctrl->Select(nSelected, false);
		nSelected = this->list_ctrl->GetNextSelected(nSelected);
	}

	// アイコン設定。
	this->list_ctrl->SetImageList(& g_hIconLL, wxIMAGE_LIST_NORMAL);
	this->list_ctrl->SetImageList(& g_hIconLS, wxIMAGE_LIST_SMALL);

	// 配列と比較し、パスが一致しなければ消す。
	for (size_t i = 0; i < this->fileinfo.GetCount(); i++)
	{
		// パスを比較。
		if (szNodePath == wxT("*") || szNodePath == this->fileinfo[i].fnFileName.GetPath())
		{
			// 項目がフォルダであるなら無視。
			if (this->fileinfo[i].fnFileName.IsDir() || ! TreeView_CheckNewerItem(this->tree_ctrl, this->tree_ctrl->GetLastChild(this->tree_ctrl->GetRootItem()), this->fileinfo[i].fnFileName.GetFullPath(), false))
			{
				continue;
			}

			// フィルタにかからなければ無視。
			if (this->fileinfo[i].fnFileName.GetFullName().MakeLower().Find(this->tcFilter->GetValue().MakeLower()) == wxNOT_FOUND)
			{
				continue;
			}

			this->list_ctrl->apShowFile.Add(& this->fileinfo[i]);
		}
	}

	// ソートして表示。
	this->list_ctrl->apShowFile.Sort(& ListCtrlCompareProc);
	this->list_ctrl->SetItemCount(this->list_ctrl->apShowFile.GetCount());
}

void MainFrame::OnTreeBeginDrag(wxTreeEvent& e)
{
	// TODO : アイテムが子階層を持っていても展開できない。
	this->tree_ctrl->SelectItem(e.GetItem());

	wxCommandEvent e1;
	e1.SetInt(3);
	this->OnArcExtract(e1);
}

void MainFrame::OnContextMenu(wxContextMenuEvent& e)
{
	wxPoint p = e.GetPosition();
	this->list_ctrl->PopupMenu(this->menubar->GetMenu(1), p == wxDefaultPosition ? this->list_ctrl->GetPosition() : this->list_ctrl->ScreenToClient(p));
}

// ListView

void MainFrame::OnListItemSelect(wxListEvent&)
{
	// ファイルに対する動作の設定。但し、選択解除時はwxのバグで呼び出されない。
	bool fEnable = this->list_ctrl->GetSelectedItemCount() > 0;
	SetMenuToolState("Arc_Delete", fEnable && (this->aiArchive.fiInfo.eSupportedCommand & TPI_COMMAND_DELETE) == TPI_COMMAND_DELETE && this->aiArchive.fiInfo.fArchive);
	this->menubar->Enable(XRCID("Exe_Copy"), fEnable);
}

void MainFrame::OnListItemDClick(wxListEvent&)
{
	wxCommandEvent e;
	e.SetInt(1);
	this->OnArcExtract(e);
}

void MainFrame::OnListBeginDrag(wxListEvent&)
{
	if (this->list_ctrl->GetSelectedItemCount() == 0)
	{
		// アイテムを選択せずドラッグしようとした場合。
		return;
	}

	wxCommandEvent e;
	e.SetInt(2);
	this->OnArcExtract(e);
}

// Filter

void MainFrame::OnFilter(wxCommandEvent&)
{
	wxTreeEvent e;
	e.SetItem(this->tree_ctrl->GetSelection());
	this->OnTreeChanged(e);
}

// イベントハンドラ以外。

bool MainFrame::LoadTPI(const wxString & szFileName, wxULongLong_t * llFileCount)
{
	// 書庫を開いていれば閉じておく。
	wxCommandEvent e;
	this->OnArcClose(e);

	// TPIを読み込み。
	wxDir fs(L_DIR_B_LIB);
	wxString szTPIName;
	if (fs.GetFirst(& szTPIName, wxT("*" TPI_EXT)))
	{
		do
		{
			// 初期化。
			if (! tpi.InitLibrary(L_DIR_B_LIB + szTPIName, szFileName))
			{
				tpi.FreeLibrary();
				continue;
			}

			// コールバック関数を設定。
			tpi.SetCallbackProc(TPICallbackProc);

			// 対応確認。
			if (! tpi.OpenArchive(szFileName, llFileCount))
			{
				tpi.FreeLibrary();
				* llFileCount = 0;
				continue;
			}
			return true;
		}
		while (fs.GetNext(& szTPIName));
	}
	return false;
}

int MainFrame::ErrorCheck(int nErrorCode, const wxString & szAPIName)
{
	switch (nErrorCode)
	{
	case TPI_ERROR_SUCCESS:
	case TPI_ERROR_S_ENDOFDATA:
	case TPI_CALLBACK_CONTINUE:
		break;
	case TPI_ERROR_D_UNSUPPORTED:
		wxLogError(_("Sorry, this function is not supported yet."));
		break;
	case TPI_ERROR_D_SKIPPED:
	case TPI_CALLBACK_CANCEL:
		wxLogError(_("This operation is canceled by the user."));
		break;
	default:
		wxLogError(_("Error: %s()!\nError code: %d"), szAPIName.c_str(), nErrorCode);
	}
	return nErrorCode;
}
