/*
 * FilePathPatternFilterTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.ObjectInspector;
import ts.tester.function.print.*;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.List;
import java.util.LinkedList;

/**
 * {@link ts.util.file.FilePathPatternFilter FilePathPatternFilter}NX̋@\NXB
 *
 * @author  V.
 * @version $Revision: 1.2 $, $Date: 2007/05/27 16:13:11 $
 */
public class FilePathPatternFilterTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);

      run(FilePathPatternFilterTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  protected void preTesting()
  {
    MSG("FilePathPatternFilter NX̋@\sB");
  }

  /* -- test case -- */

  public void constructor()
  {
    MSG("RXgN^̊mFB");

    FileFilter filter = new FilePathPatternFilter(new File("."), "*");

    List<List<List<String>>> lst = new LinkedList<List<List<String>>>();
    List<List<String>> lst2 = new LinkedList<List<String>>();
    lst.add(lst2);
    List<String> lst3 = new LinkedList<String>();
    lst2.add(lst3);
    lst3.add("");
    lst3.add("");

    try {
      ObjectInspector oi = new ObjectInspector(this);
      oi.expect("basePath_", new File(".").getCanonicalPath() + File.separator);
      oi.expect("pathPatternLLL_", lst);
      oi.inspect(filter);
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void constructor_1()
  {
    MSG("k̏ꍇB");

    try {
      FileFilter filter = new FilePathPatternFilter(new File("."), null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileFilter filter = new FilePathPatternFilter(null, "*");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void constructor_2()
  {
    MSG("x[XEfBNg̎w肪sȏꍇB");

    try {
      FileFilter filter =
        new FilePathPatternFilter(new File("./build.xml"), "*");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileFilter filter = new FilePathPatternFilter(new File("./aaa"), "*");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void constructor_3()
  {
    MSG("pXEp^[̎w肪sȏꍇB");

    try {
      FileFilter filter = new FilePathPatternFilter(new File("/"), "/aaa/*");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void splitWithAsterisk()
  {
    MSG("pXEp^[AX^XNŕB");
    MSG("- pXEp^[B");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> patternL = new LinkedList<String>();
    FilePathPatternFilter filter =
      new FilePathPatternFilter(new File("."), "*");

    List patternLLL = null;

    try {
      patternLLL = List.class.cast(oi.invokeMethod(
        filter, "splitWithAsterisk", List.class, patternL));
    } catch (Exception e) {
      NG(e);
    }
    EQUAL(patternLLL.size(), 1);
    EQUAL(List.class.cast(patternLLL.get(0)).size(), 0);
  }

  public void splitWithAsterisk_1()
  {
    MSG("pXEp^[AX^XNŕB");
    MSG("- AX^XN܂܂Ȃp^[B");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> patternL = new LinkedList<String>();
    FilePathPatternFilter filter =
      new FilePathPatternFilter(new File("."), "*");
    List patternLLL = null;

    patternL.add("aaa");
    patternL.add("bbb");
    patternL.add("ccc");

    try {
      patternLLL = List.class.cast(oi.invokeMethod(
        filter, "splitWithAsterisk", List.class, patternL));
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(patternLLL.size(), 1);

    List patternLL = List.class.cast(patternLLL.get(0));
    EQUAL(patternLL.size(), 3);

    List patternLst = List.class.cast(patternLL.get(0));
    EQUAL(patternLst.size(), 1);
    EQUAL(patternLst.get(0), "aaa");

    patternLst = List.class.cast(patternLL.get(1));
    EQUAL(patternLst.size(), 1);
    EQUAL(patternLst.get(0), "bbb");

    patternLst = List.class.cast(patternLL.get(2));
    EQUAL(patternLst.size(), 1);
    EQUAL(patternLst.get(0), "ccc");
  }

  public void splitWithAsterisk_2()
  {
    MSG("pXEp^[AX^XNŕB");
    MSG("- pXvf̒ɃAX^XN܂ޏꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> patternL = new LinkedList<String>();
    FilePathPatternFilter filter =
      new FilePathPatternFilter(new File("."), "*");

    patternL.add("abc*def");
    patternL.add("ghi*jkl*");
    patternL.add("*mno.pqr");

    List patLLL = null;
    try {
      patLLL = List.class.cast(oi.invokeMethod(
        filter, "splitWithAsterisk", List.class, patternL));
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(patLLL.size(), 1);

    List patLL = null;
    patLL = List.class.cast(patLLL.get(0));
    EQUAL(patLL.size(), 3);

    List patL = null;
    patL = List.class.cast(patLL.get(0));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "abc");
    EQUAL(patL.get(1), "def");
    patL = List.class.cast(patLL.get(1));
    EQUAL(patL.size(), 3);
    EQUAL(patL.get(0), "ghi");
    EQUAL(patL.get(1), "jkl");
    EQUAL(patL.get(2), "");
    patL = List.class.cast(patLL.get(2));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "");
    EQUAL(patL.get(1), "mno.pqr");
  }

  public void splitWithAsterisk_3()
  {
    MSG("pXEp^[AX^XNŕB");
    MSG("- pXvf̊ԂɓdAX^XNꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FilePathPatternFilter filter =
      new FilePathPatternFilter(new File("."), "*");
    List<String> patternL = new LinkedList<String>();

    patternL.add("**");
    patternL.add("abc*def");
    patternL.add("**");
    patternL.add("ghi*jkl*");
    patternL.add("*");
    patternL.add("*mno.pqr");

    List patLLL = null, patLL = null, patL = null;
    try {
      patLLL = List.class.cast(oi.invokeMethod(
        filter, "splitWithAsterisk", List.class, patternL));
    } catch (Exception e) {
      NG(e);
    }
    EQUAL(patLLL.size(), 3);

    patLL = List.class.cast(patLLL.get(0)); 
    EQUAL(patLL.size(), 0);

    patLL = List.class.cast(patLLL.get(1)); 
    EQUAL(patLL.size(), 1);

    patL = List.class.cast(patLL.get(0));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "abc");
    EQUAL(patL.get(1), "def");

    patLL = List.class.cast(patLLL.get(2)); 
    EQUAL(patLL.size(), 3);

    patL = List.class.cast(patLL.get(0));
    EQUAL(patL.size(), 3);
    EQUAL(patL.get(0), "ghi");
    EQUAL(patL.get(1), "jkl");
    patL = List.class.cast(patLL.get(1));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "");
    EQUAL(patL.get(1), "");
    patL = List.class.cast(patLL.get(2));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "");
    EQUAL(patL.get(1), "mno.pqr");

    patternL.clear();

    patternL.add("**");
    patternL.add("abc*def");
    patternL.add("**");
    patternL.add("**");
    patternL.add("ghi*jkl*");
    patternL.add("*");
    patternL.add("*mno.pqr");

    try {
      patLLL = List.class.cast(oi.invokeMethod(
        filter, "splitWithAsterisk", List.class, patternL));
    } catch (Exception e) {
      NG(e);
    }
    EQUAL(patLLL.size(), 4);

    patLL = List.class.cast(patLLL.get(0));
    EQUAL(patLL.size(), 0);

    patLL = List.class.cast(patLLL.get(1));
    EQUAL(patLL.size(), 1);
    patL = List.class.cast(patLL.get(0));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "abc");
    EQUAL(patL.get(1), "def");

    patLL = List.class.cast(patLLL.get(2));
    EQUAL(patLL.size(), 0);

    patLL = List.class.cast(patLLL.get(3));
    EQUAL(patLL.size(), 3);
    patL = List.class.cast(patLL.get(0));
    EQUAL(patL.size(), 3);
    EQUAL(patL.get(0), "ghi");
    EQUAL(patL.get(1), "jkl");
    EQUAL(patL.get(2), "");
    patL = List.class.cast(patLL.get(1));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "");
    EQUAL(patL.get(1), "");
    patL = List.class.cast(patLL.get(2));
    EQUAL(patL.size(), 2);
    EQUAL(patL.get(0), "");
    EQUAL(patL.get(1), "mno.pqr");
  }

  public void splitWithAsterisk_4()
  {
    MSG("k̏ꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FilePathPatternFilter filter =
      new FilePathPatternFilter(new File("."), "*");

    try {
      oi.invokeMethod(filter, "splitWithAsterisk", List.class, null);
      NG();
    } catch (NullPointerException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void splitWithAsterisk_5()
  {
    MSG("dAX^XNt@Cɑ΂郏ChEJ[hƂ" +
        "gpĂꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FilePathPatternFilter filter =
      new FilePathPatternFilter(new File("."), "*");
    List<String> patternL = new LinkedList<String>();
    List<List<List<String>>> patternLLL = null;

    patternL.add("**");
    patternL.add("abc**def");
    patternL.add("**");
    patternL.add("ghi*jkl*");
    patternL.add("*");
    patternL.add("*mno.pqr");

    try {
      oi.invokeMethod(filter, "splitWithAsterisk", List.class, patternL);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    patternL.clear();
    patternL.add("**abcf");
    patternL.add("def");
    patternL.add("ghi*jkl*");
    patternL.add("*");
    patternL.add("*mno.pqr");

    try {
      oi.invokeMethod(filter, "splitWithAsterisk", List.class, patternL);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void parsePathPattern()
  {
    MSG("pXEp^[̕B");

    ObjectInspector oi = new ObjectInspector(this);
    FilePathPatternFilter f = new FilePathPatternFilter(new File("."), "*");

    List patternLLL = null;
    try {
      patternLLL = List.class.cast(
        oi.invokeMethod(f, "parsePathPattern", String.class, ""));
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      patternLLL = List.class.cast(
        oi.invokeMethod(f, "parsePathPattern", String.class, "a"));
      EQUAL(patternLLL.size(), 1);

      List patternLL = List.class.cast(patternLLL.get(0));
      EQUAL(patternLL.size(), 1);

      List patternL = List.class.cast(patternLL.get(0));
      EQUAL(patternL.size(), 1);
      EQUAL(patternL.get(0), "a");
    } catch (Exception e) {
      NG(e);
    }

    try {
      patternLLL = List.class.cast(
        oi.invokeMethod(f, "parsePathPattern", String.class, "*"));
      EQUAL(patternLLL.size(), 1);

      List patternLL = List.class.cast(patternLLL.get(0));
      EQUAL(patternLL.size(), 1);

      List patternL = List.class.cast(patternLL.get(0));
      EQUAL(patternL.size(), 2);
      EQUAL(patternL.get(0), "");
      EQUAL(patternL.get(1), "");
    } catch (Exception e) {
      NG(e);
    }

    try {
      patternLLL = List.class.cast(
        oi.invokeMethod(f, "parsePathPattern", String.class, "*/aaa"));
      EQUAL(patternLLL.size(), 1);

      List patternLL = List.class.cast(patternLLL.get(0));
      EQUAL(patternLL.size(), 2);

      List patternL = List.class.cast(patternLL.get(0));
      EQUAL(patternL.size(), 2);
      EQUAL(patternL.get(0), "");
      EQUAL(patternL.get(1), "");

      patternL = List.class.cast(patternLL.get(1));
      EQUAL(patternL.size(), 1);
      EQUAL(patternL.get(0), "aaa");
    } catch (Exception e) {
      NG(e);
    }

    try {
      patternLLL = List.class.cast(
        oi.invokeMethod(f, "parsePathPattern", String.class, "*/*/*"));
      EQUAL(patternLLL.size(), 1);

      List patternLL = List.class.cast(patternLLL.get(0));
      EQUAL(patternLL.size(), 3);

      List patternL = List.class.cast(patternLL.get(0));
      EQUAL(patternL.size(), 2);
      EQUAL(patternL.get(0), "");
      EQUAL(patternL.get(1), "");
      patternL = List.class.cast(patternLL.get(1));
      EQUAL(patternL.size(), 2);
      EQUAL(patternL.get(0), "");
      EQUAL(patternL.get(1), "");
      patternL = List.class.cast(patternLL.get(2));
      EQUAL(patternL.size(), 2);
      EQUAL(patternL.get(0), "");
      EQUAL(patternL.get(1), "");
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.invokeMethod(f, "parsePathPattern", String.class, "*//");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void matchPatternElement()
  {
    MSG("pXEXgƃp^[EXg̗vf̔rB");
    MSG("- p^[ɃAX^XN܂܂ȂꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> pathL = new LinkedList<String>();
    List<List<String>> patternLL = new LinkedList<List<String>>();

    FilePathPatternFilter f = new FilePathPatternFilter(new File("."), "*");
    Class[] t = { List.class, Integer.TYPE, List.class };
    Object[] v = { pathL, 0, patternLL };

    pathL.add("abc");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abc");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abc");
    pathL.add("def");
    pathL.add("ghi");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abc");
    patternLL.get(1).add("def");
    patternLL.get(2).add("ghi");

    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abc");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abd");

    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abc");
    pathL.add("def");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abc");
    patternLL.get(1).add("ddd");

    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void matchPatternElement_1()
  {
    MSG("pXEXgƃp^[EXg̗vf̔rB");
    MSG("- p^[ɃAX^XN܂ޏꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> pathL = new LinkedList<String>();
    List<List<String>> patternLL = new LinkedList<List<String>>();

    FilePathPatternFilter f = new FilePathPatternFilter(new File("."), "*");
    Class[] t = { List.class, Integer.TYPE, List.class };
    Object[] v = { pathL, 0, patternLL };

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("ghi");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("defg");
    patternLL.get(0).add("");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("a");
    patternLL.get(0).add("ef");
    patternLL.get(0).add("i");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("a");
    patternLL.get(0).add("c");
    patternLL.get(0).add("e");
    patternLL.get(0).add("g");
    patternLL.get(0).add("i");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("mno");
    patternLL.get(1).add("");
    patternLL.get(2).add("");
    patternLL.get(2).add("uvw");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");  // <==
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("mno");
    patternLL.get(1).add("");
    patternLL.get(2).add("");
    patternLL.get(2).add("uvw");
    patternLL.get(2).add("");  // <==
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void matchPatternElement_1a()
  {
    MSG("pXEXgƃp^[EXg̗vf̔rB");
    MSG("- p^[ɃAX^XN܂ޏꍇivȂP[XjB");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> pathL = new LinkedList<String>();
    List<List<String>> patternLL = new LinkedList<List<String>>();

    FilePathPatternFilter f = new FilePathPatternFilter(new File("."), "*");
    Class[] t = { List.class, Integer.TYPE, List.class };
    Object[] v = { pathL, 0, patternLL };

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("gh");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abxd");
    patternLL.get(0).add("");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("dddd");
    patternLL.get(0).add("");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("a");
    patternLL.get(0).add("ee");
    patternLL.get(0).add("i");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("a");
    patternLL.get(0).add("c");
    patternLL.get(0).add("e");
    patternLL.get(0).add("g");
    patternLL.get(0).add("j");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("mno");
    patternLL.get(1).add("");
    patternLL.get(2).add("");
    patternLL.get(2).add("uvx");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("a");  // <==
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("mno");
    patternLL.get(1).add("");
    patternLL.get(2).add("");
    patternLL.get(2).add("uvw");
    patternLL.get(2).add("");  // <==
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void matchPatternElement_2()
  {
    MSG("CfbNXOȊȌꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> pathL = new LinkedList<String>();
    List<List<String>> patternLL = new LinkedList<List<String>>();

    FilePathPatternFilter f = new FilePathPatternFilter(new File("."), "*");
    Class[] t = { List.class, Integer.TYPE, List.class };
    Object[] v = { pathL, 1, patternLL };

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("ghi");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("defg");
    patternLL.get(0).add("");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("a");
    patternLL.get(0).add("ef");
    patternLL.get(0).add("i");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("a");
    patternLL.get(0).add("c");
    patternLL.get(0).add("e");
    patternLL.get(0).add("g");
    patternLL.get(0).add("i");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("mno");
    patternLL.get(1).add("");
    patternLL.get(2).add("");
    patternLL.get(2).add("uvw");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("mno");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("uvw");
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");  // <==
    patternLL.get(0).add("abcd");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("mno");
    patternLL.get(1).add("");
    patternLL.get(2).add("");
    patternLL.get(2).add("uvw");
    patternLL.get(2).add("");  // <==
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      FALSE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.clear();
    patternLL.clear();

    pathL.add("abcdefghi");
    pathL.add("jklmnop");
    pathL.add("qrstuvw");
    patternLL.add(new LinkedList<String>());
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("mno");
    patternLL.get(0).add("");
    patternLL.get(1).add("");
    patternLL.get(1).add("uvw");
    patternLL.get(1).add("");  // <==
    try {
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void matchPatternElement_3()
  {
    MSG("k̏ꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> pathL = new LinkedList<String>();
    List<List<String>> patternLL = new LinkedList<List<String>>();

    FilePathPatternFilter f = new FilePathPatternFilter(new File("."), "*");
    Class[] t = { List.class, Integer.TYPE, List.class };

    pathL.add("abcdefghi");
    patternLL.add(new LinkedList<String>());
    patternLL.get(0).add("");
    patternLL.get(0).add("gh");
    try {
      Object[] v = { null, 0, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      NG();
    } catch (NullPointerException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      Object[] v = { pathL, 0, null };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      NG();
    } catch (NullPointerException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void matchPatternElement_4()
  {
    MSG("p^[EXg̏ꍇsȏꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    List<String> pathL = new LinkedList<String>();
    List<List<String>> patternLL = new LinkedList<List<String>>();

    FilePathPatternFilter f = new FilePathPatternFilter(new File("."), "*");
    Class[] t = { List.class, Integer.TYPE, List.class };

    try {
      Object[] v = { pathL, 0, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.add("abc");

    try {
      Object[] v = { pathL, 0, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.add("def");

    try {
      Object[] v = { pathL, 0, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    try {
      Object[] v = { pathL, 1, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    pathL.add("ghi");

    try {
      Object[] v = { pathL, 0, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    try {
      Object[] v = { pathL, 1, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }

    try {
      Object[] v = { pathL, 2, patternLL };
      Boolean bool = (Boolean) oi.invokeMethod(f, "matchPatternElement", t, v);
      TRUE(bool.booleanValue());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void accept()
  {
    MSG("t@CEpXƃp^[̔rB");
    File baseDir = new File(".");
    FilePathPatternFilter filter;

    filter = new FilePathPatternFilter(baseDir, "*");
    TRUE(filter.accept(new File(baseDir, "src")));
    TRUE(filter.accept(new File(baseDir, "src/")));
    FALSE(filter.accept(new File(baseDir, "src/javadoc.css")));

    filter = new FilePathPatternFilter(baseDir, "src");
    TRUE(filter.accept(new File(baseDir, "src")));
    FALSE(filter.accept(new File(baseDir, "src/javadoc.css")));

    filter = new FilePathPatternFilter(baseDir, "src/javadoc.css");
    FALSE(filter.accept(new File(baseDir, "src")));
    TRUE(filter.accept(new File(baseDir, "src/javadoc.css")));

    filter = new FilePathPatternFilter(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java");
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/ts/util/file/*Filter.java");
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/ts/util/file/FilePathPattern*.java");
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/**/FilePathPatternFilter.java");
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/*/FilePathPatternFilter.java");
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));

    filter = new FilePathPatternFilter(baseDir,
      "**/FilePathPatternFilter.java");
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/ts/**");
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "s*c/*ts*/**/File*Test.java");
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/ts/util/file/*");
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/ts/util/file/*/*.java");
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));
    FALSE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilterTest.java")));

    filter = new FilePathPatternFilter(baseDir,
      "src/ts/util/file/**/*.java");
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilter.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FilePathPatternFilterTest.java")));
    TRUE(filter.accept(new File(baseDir,
      "src/ts/util/file/FileSizeFilterTest.java")));
  }

  public void accept_1()
  {
    MSG("k̏ꍇB");

    File baseDir = new File(".");
    FilePathPatternFilter filter;
    filter = new FilePathPatternFilter(baseDir, "*");
    try {
      filter.accept(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void accept_2()
  {
    MSG("x[XEfBNgvȂꍇB");

    FilePathPatternFilter filter;

    File baseDir = new File(".");
    File baseDir2 = new File(baseDir, "classes");
    File baseDir3 = new File(baseDir, "src");

    filter = new FilePathPatternFilter(baseDir2, "*");
    FALSE(filter.accept(new File(baseDir3, "javadoc.css")));
  }
}

