/*
 * FileOperation08Move3Test class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import ts.tester.function.*;
import ts.tester.function.print.*;
import java.io.*;
import java.util.*;
import ts.util.*;
import ts.util.text.*;

/**
 * {@link ts.util.file.FileOperation FileOperation}NX̋@\NXB
 *
 * @author  V.
 * @version $Revision: 1.2 $, $Date: 2007/10/09 17:04:50 $
 */
public class FileOperation08Move3Test extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      run(FileOperation08Move3Test.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  final File WORKDIR = new File("test/data/ts/util/file/FileOperation");
  File[] DIRS = new File[6];
  File[] FILES = new File[10];

  protected void preInvocation(String method)
  {
    WORKDIR.mkdirs();
    DIRS[0] = new File(WORKDIR, "DIRS0");
    DIRS[1] = new File(DIRS[0], "DIRS1");
    DIRS[2] = new File(DIRS[0], "DIRS2");
    DIRS[3] = new File(DIRS[1], "DIRS3");
    DIRS[4] = new File(DIRS[2], "DIRS4");
    DIRS[5] = new File(DIRS[2], "DIRS5");
    FILES[0] = new File(DIRS[0], "FILES0"); 
    FILES[1] = new File(DIRS[1], "FILES1"); 
    FILES[2] = new File(DIRS[2], "FILES2"); 
    FILES[3] = new File(DIRS[3], "FILES3"); 
    FILES[4] = new File(DIRS[4], "FILES4"); 
    FILES[5] = new File(DIRS[0], "FILES5"); 
    FILES[6] = new File(DIRS[1], "FILES6"); 
    FILES[7] = new File(DIRS[2], "FILES7"); 
    FILES[8] = new File(DIRS[3], "FILES8"); 
    FILES[9] = new File(DIRS[4], "FILES9"); 
    try {
      for (File d : DIRS) d.mkdirs();
      for (File f : FILES) f.createNewFile();
    } catch (Exception e) {}

    /*
       DIRS[0]
        +- DIRS[1]
          +- DIRS[3]
            +- FILES[3]
            +- FILES[8]
          +- FILES[1]
          +- FILES[6]
        +- DIRS[2]
          +- DIRS[4]
            +- FILES[4]
            +- FILES[9]
          +- DIRS[5]
          +- FILES[2]
          +- FILES[7]
        +- FILES[0]
        +- FILES[5]
    */
  }
  
  protected void postInvocation(String method)
  {
    try {
      FileOperation.deleteRecursive(WORKDIR);
    } catch (Exception e) {}
  }

  private boolean isMoveToTemporary(File f) {
    if (f == null) return false;
    while (true) {
      File p = f.getParentFile();
      if (p == null) break;
      if (p.getName().startsWith("._")) {
        return true;
      }
      f = p;
    }
    return false;
  }

  /* -- test case -- */

  public void moveInner_renameTo_failed()
  {
    MSG("UNIXŕʃt@CEVXerenameTosƎŝ͋[B");
    MSG("ʏ̃t@C̈ړB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();

    try {
      File dstFile = FileOperation.getDestinationFile(FILES[4], DIRS[4], 
        WORKDIR);
      OK(dstFile.getPath());

      TRUE(FILES[3].exists());
      TRUE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());

      @SuppressWarnings("serial")
      File fff = new File(FILES[3].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = fff;
      values[1] = dstFile;
      oi.invokeMethod(fop, "moveInner", types, values);

      FALSE(FILES[3].exists());
      TRUE(dstFile.getParentFile().exists());
      TRUE(dstFile.exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_00()
  {
    MSG("fBNg̈ړB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();

    try {
      File dstDir = new File(WORKDIR, "dest");
      File dstFile = FileOperation.getDestinationFile(DIRS[4], DIRS[0], dstDir);

      TRUE(DIRS[4].exists());
      TRUE(FILES[4].exists());
      TRUE(FILES[9].exists());
      FALSE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());

      @SuppressWarnings("serial")
      File fff = new File(DIRS[4].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = fff;
      values[1] = dstFile;

      oi.invokeMethod(fop, "moveInner", types, values);

      FALSE(DIRS[4].exists());
      TRUE(dstFile.exists());
      TRUE(FileOperation.getDestinationFile(FILES[4],DIRS[0],dstDir).exists());
      TRUE(FileOperation.getDestinationFile(FILES[9],DIRS[0],dstDir).exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_01()
  {
    MSG("ړ̃t@C݂ȂꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();

    @SuppressWarnings("serial")
    File srcFile = new File(WORKDIR, "fff") {
      @Override public boolean renameTo(File f) {
        if (isMoveToTemporary(f)) {
          return super.renameTo(f);
        }
        return false;
      }
    };
    File dstFile = new File(WORKDIR, "ggg");

    FALSE(srcFile.exists());
    FALSE(dstFile.exists());
    TRUE(dstFile.getParentFile().exists());

    try {
      values[0] = srcFile;
      values[1] = dstFile;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_02()
  {
    MSG("ړ̃fBNgłȂt@Cɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    TRUE(FILES[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());

    try {
      @SuppressWarnings("serial")
      File fff = new File(FILES[3].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = fff;
      values[1] = FILES[4];
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());

    try {
      values[0] = DIRS[3];
      values[1] = FILES[4];
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());
  }

  public void moveInner_renameTo_failed_03()
  {
    MSG("ړ̃fBNgɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    try {
      TRUE(DIRS[3].exists());
      TRUE(FILES[3].exists());
      TRUE(FILES[8].exists());
      TRUE(DIRS[4].exists());
      FALSE(fop.getDestinationFile(FILES[3], DIRS[3], DIRS[4]).exists());
      FALSE(fop.getDestinationFile(FILES[8], DIRS[3], DIRS[4]).exists());
  
      try {
        @SuppressWarnings("serial")
        File fff = new File(DIRS[3].getPath()) {
          @Override public boolean renameTo(File f) {
            if (isMoveToTemporary(f)) {
              return super.renameTo(f);
            }
            return false;
          }
        };
        values[0] = fff;
        values[1] = DIRS[4];
        oi.invokeMethod(fop, "moveInner", types, values);
      } catch (Exception e) {
        NG(e);
      }
  
      FALSE(DIRS[3].exists());
      TRUE(DIRS[4].exists());
      TRUE(fop.getDestinationFile(FILES[3], DIRS[3], DIRS[4]).exists());
      TRUE(fop.getDestinationFile(FILES[8], DIRS[3], DIRS[4]).exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_04()
  {
    MSG("ړɃ^Cv̈قȂt@Cɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    File ddd = new File(WORKDIR, "aaa");
    File fff = new File(ddd, "DIRS1/DIRS3/FILES3");
    fff.mkdirs();

    try {
      @SuppressWarnings("serial")
      File f = new File(DIRS[0].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = f;
      values[1] = ddd;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    for (File d : DIRS) TRUE(d.exists());
    for (File f : DIRS) TRUE(f.exists());
    try {
      List<File> lst = FileOperation.listRecursive(ddd);
      EQUAL(lst.size(), 3);
      EQUAL(lst.get(0), new File(ddd, "DIRS1"));
      EQUAL(lst.get(1), new File(ddd, "DIRS1/DIRS3"));
      EQUAL(lst.get(2), new File(ddd, "DIRS1/DIRS3/FILES3"));
      TRUE(new File(ddd, "DIRS1/DIRS3/FILES3").isDirectory());
    } catch (Exception e) {
      NG(e);
    }

    File dd = new File(WORKDIR, "bbb");
    File ff = new File(dd, "DIRS1/DIRS3");
    try {
      FileOperation.createNewFile(ff);
    } catch (Exception e) {
      NG(e);
    }

    try {
      @SuppressWarnings("serial")
      File f = new File(DIRS[0].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = f;
      values[1] = dd;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    for (File d : DIRS) TRUE(d.exists());
    for (File f : DIRS) TRUE(f.exists());
    try {
      List<File> lst = FileOperation.listRecursive(dd);
      EQUAL(lst.size(), 2);
      EQUAL(lst.get(0), new File(dd, "DIRS1"));
      EQUAL(lst.get(1), new File(dd, "DIRS1/DIRS3"));
      FALSE(new File(dd, "DIRS1/DIRS3").isDirectory());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_05()
  {
    MSG("ړɎsƂ̃[obN̊mFB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();
    File dstDir = new File(WORKDIR, "dest");

    File errFile = new File(dstDir, "DIRS2/DIRS4/FILES4");
    try {
      FileOperation.createNewFile(errFile);
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> beforeLst = FileOperation.listRecursive(WORKDIR);

      try {
        @SuppressWarnings("serial")
        File fff = new File(DIRS[0].getPath()) {
          @Override public boolean renameTo(File f) {
            if (isMoveToTemporary(f)) {
              return super.renameTo(f);
            }
            return false;
          }
        };
        values[0] = fff;
        values[1] = dstDir;
        oi.invokeMethod(fop, "moveInner", types, values);
        NG();
      } catch (FileAlreadyExistsException e) {
        OK(e);
      } catch (Exception e) {
        NG(e);
      }
  
      List<File> afterLst = FileOperation.listRecursive(WORKDIR);
      EQUAL(beforeLst.size(), afterLst.size());
      for (int i=0; i<beforeLst.size(); i++) {
        EQUAL(beforeLst.get(i), afterLst.get(i));
      }
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_06()
  {
    MSG("ړ̐efBNg݂ȂꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FileOperation fop = new FileOperation();
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    @SuppressWarnings("serial")
    File src = new File(DIRS[3].getPath()) {
      @Override public boolean renameTo(File f) {
        if (isMoveToTemporary(f)) {
          return super.renameTo(f);
        }
        return false;
      }
    };
    File dst =new File(WORKDIR, "aaa");
    TRUE(DIRS[3].exists());
    FALSE(dst.exists());

    try {
      values[0] = src;
      values[1] = dst;
      oi.invokeMethod(fop, "moveInner", types, values);
    } catch (Exception e) {
      NG(e);
      e.printStackTrace();
    }

    FALSE(DIRS[3].exists());
    TRUE(dst.exists());
    TRUE(new File(WORKDIR, "aaa/FILES3").exists());
    TRUE(new File(WORKDIR, "aaa/FILES8").exists());
  }

  public void moveInner_renameTo_failed_07()
  {
    MSG("ړ̐efBNg̓t@C̏ꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FileOperation fop = new FileOperation();
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    @SuppressWarnings("serial")
    File src = new File(DIRS[3].getPath()) {
      @Override public boolean renameTo(File f) {
        if (isMoveToTemporary(f)) {
          return super.renameTo(f);
        }
        return false;
      }
    };
    File dst =new File(WORKDIR, "aaa/bbb/ccc");
    TRUE(DIRS[3].exists());
    FALSE(dst.exists());

    try {
      FileOperation.createNewFile(new File(WORKDIR, "aaa"));
    } catch (Exception e) {
      NG(e);
    }

    try {
      values[0] = src;
      values[1] = dst;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    FALSE(dst.exists());
  }

  public void moveInner_withFilter_renameTo_failed()
  {
    MSG("UNIXŕʃt@CEVXerenameTosƎŝ͋[B");
    MSG("ʏ̃t@C̈ړB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      File dstFile = FileOperation.getDestinationFile(FILES[4], DIRS[4], 
        WORKDIR);
      OK(dstFile.getPath());

      TRUE(FILES[3].exists());
      TRUE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());

      @SuppressWarnings("serial")
      File fff = new File(FILES[3].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = fff;
      values[1] = dstFile;
      values[2] = filter;
      oi.invokeMethod(fop, "moveInner", types, values);

      FALSE(FILES[3].exists());
      TRUE(dstFile.getParentFile().exists());
      TRUE(dstFile.exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_withFilter_renameTo_failed00()
  {
    MSG("fBNg̈ړB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      File dstDir = new File(WORKDIR, "dest");
      File dstFile = FileOperation.getDestinationFile(DIRS[4], DIRS[0], dstDir);

      TRUE(DIRS[4].exists());
      TRUE(FILES[4].exists());
      TRUE(FILES[9].exists());
      FALSE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());

      @SuppressWarnings("serial")
      File fff = new File(DIRS[4].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = fff;
      values[1] = dstFile;
      values[2] = filter;
      oi.invokeMethod(fop, "moveInner", types, values);

      TRUE(DIRS[4].exists());
      FALSE(FILES[4].exists());
      FALSE(FILES[9].exists());
      TRUE(dstFile.exists());
      TRUE(FileOperation.getDestinationFile(FILES[4],DIRS[0],dstDir).exists());
      TRUE(FileOperation.getDestinationFile(FILES[9],DIRS[0],dstDir).exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_withFilter_renameTo_failed01()
  {
    MSG("ړ̃t@C݂ȂꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    @SuppressWarnings("serial")
    File srcFile = new File(WORKDIR, "fff") {
      @Override public boolean renameTo(File f) {
        if (isMoveToTemporary(f)) {
          return super.renameTo(f);
        }
        return false;
      }
    };
    File dstFile = new File(WORKDIR, "ggg");

    FALSE(srcFile.exists());
    FALSE(dstFile.exists());
    TRUE(dstFile.getParentFile().exists());

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      values[0] = srcFile;
      values[1] = dstFile;
      values[2] = filter;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_withFilter_renameTo_failed02()
  {
    MSG("ړ̃fBNgłȂt@Cɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    TRUE(DIRS[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      @SuppressWarnings("serial")
      File fff = new File(FILES[3].getPath()) {
        @Override public boolean renameTo(File f) {
          if (isMoveToTemporary(f)) {
            return super.renameTo(f);
          }
          return false;
        }
      };
      values[0] = fff;
      values[1] = FILES[4];
      values[2] = filter;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());

    try {
      values[0] = DIRS[3];
      values[1] = FILES[4];
      values[2] = filter;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());
  }

  public void moveInner_withFilter_renameTo_failed04()
  {
    MSG("ړ̃fBNgɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      TRUE(DIRS[3].exists());
      TRUE(FILES[3].exists());
      TRUE(FILES[8].exists());
      TRUE(DIRS[4].exists());
      FALSE(fop.getDestinationFile(FILES[3], DIRS[3], DIRS[4]).exists());
      FALSE(fop.getDestinationFile(FILES[8], DIRS[3], DIRS[4]).exists());
  
      try {
        @SuppressWarnings("serial")
        File f = new File(DIRS[3].getPath()) {
          @Override public boolean renameTo(File f) {
            if (isMoveToTemporary(f)) {
              return super.renameTo(f);
            }
            return false;
          }
        };
        values[0] = f;
        values[1] = DIRS[4];
        values[2] = filter;
        oi.invokeMethod(fop, "moveInner", types, values);
      } catch (Exception e) {
        NG(e);
      }
  
      TRUE(DIRS[3].exists());
      FALSE(FILES[3].exists());
      FALSE(FILES[8].exists());
      TRUE(DIRS[4].exists());
      TRUE(fop.getDestinationFile(FILES[3], DIRS[3], DIRS[4]).exists());
      TRUE(fop.getDestinationFile(FILES[8], DIRS[3], DIRS[4]).exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_withFilter_renameTo_failed05()
  {
    MSG("ړɎsƂ̃[obN̊mFB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    File dstDir = new File(WORKDIR, "dest");

    File errFile = new File(dstDir, "DIRS2/DIRS4/FILES4");
    try {
      FileOperation.createNewFile(errFile);
    } catch (Exception e) {
      NG(e);
    }

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      List<File> beforeLst = FileOperation.listRecursive(WORKDIR);

      try {
        @SuppressWarnings("serial")
        File f = new File(DIRS[0].getPath()) {
          @Override public boolean renameTo(File f) {
            if (isMoveToTemporary(f)) {
              return super.renameTo(f);
            }
            return false;
          }
        };
        values[0] = f;
        values[1] = dstDir;
        values[2] = filter;
        oi.invokeMethod(fop, "moveInner", types, values);
        NG();
      } catch (FileAlreadyExistsException e) {
        OK(e);
      } catch (Exception e) {
        NG(e);
      }
  
      List<File> afterLst = FileOperation.listRecursive(WORKDIR);
      EQUAL(beforeLst.size(), afterLst.size());
      for (int i=0; i<beforeLst.size(); i++) {
        EQUAL(beforeLst.get(i), afterLst.get(i));
      }
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_withFilter_renameTo_failed06()
  {
    MSG("ړ̐efBNg݂ȂꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FileOperation fop = new FileOperation();
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    @SuppressWarnings("serial")
    File src = new File(DIRS[3].getPath()) {
      @Override public boolean renameTo(File f) {
        if (isMoveToTemporary(f)) {
          return super.renameTo(f);
        }
        return false;
      }
    };
    File dst =new File(WORKDIR, "aaa");
    TRUE(DIRS[3].exists());
    FALSE(dst.exists());

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      values[0] = src;
      values[1] = dst;
      values[2] = filter;
      oi.invokeMethod(fop, "moveInner", types, values);
    } catch (Exception e) {
      NG(e);
      e.printStackTrace();
    }

    TRUE(DIRS[3].exists());
    FALSE(FILES[3].exists());
    FALSE(FILES[8].exists());
    TRUE(dst.exists());
    TRUE(new File(WORKDIR, "aaa/FILES3").exists());
    TRUE(new File(WORKDIR, "aaa/FILES8").exists());
  }

  public void moveInner_withFilter_renameTo_failed07()
  {
    MSG("ړ̐efBNg̓t@C̏ꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FileOperation fop = new FileOperation();
    Class[] types = { File.class, File.class, FileFilter.class };
    Object[] values = new Object[3];

    @SuppressWarnings("serial")
    File src = new File(DIRS[3].getPath()) {
      @Override public boolean renameTo(File f) {
        if (isMoveToTemporary(f)) {
          return super.renameTo(f);
        }
        return false;
      }
    };
    File dst = new File(WORKDIR, "aaa/bbb/ccc");
    TRUE(DIRS[3].exists());
    FALSE(dst.exists());

    try {
      FileOperation.createNewFile(new File(WORKDIR, "aaa"));
    } catch (Exception e) {
      NG(e);
    }

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      values[0] = src;
      values[1] = dst;
      values[2] = filter;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    FALSE(dst.exists());
  }

  public void moveInner_renameTo_failed_rollback()
  {
    MSG("UNIXŕʃt@CEVXerenameTosƎŝ͋[B");
    MSG("ʏ̃t@C̈ړB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();

    File dstFile = null;

    try {
      dstFile = FileOperation.getDestinationFile(FILES[4], DIRS[4], WORKDIR);
      OK(dstFile.getPath());

      TRUE(FILES[3].exists());
      TRUE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());

      @SuppressWarnings("serial")
      File fff = new File(FILES[3].getPath()) {
        @Override public boolean renameTo(File f) {
          return false;
        }
        @Override public boolean delete() {
          return false;
        }
      };
      values[0] = fff;
      values[1] = dstFile;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (Exception e) {
      OK(e);
    }

    try {
      TRUE(FILES[3].exists());
      TRUE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_rollback_00()
  {
    MSG("fBNg̈ړB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();

    File dstFile = null;
    try {
      File dstDir = new File(WORKDIR, "dest");
      dstFile = FileOperation.getDestinationFile(DIRS[4], DIRS[0], dstDir);

      TRUE(DIRS[4].exists());
      TRUE(FILES[4].exists());
      TRUE(FILES[9].exists());
      FALSE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());

      @SuppressWarnings("serial")
      File fff = new File(DIRS[4].getPath()) {
        @Override public boolean renameTo(File f) {
          return false;
        }
      };
      values[0] = fff;
      values[1] = dstFile;
      oi.invokeMethod(fop, "moveInner", types, values);
    } catch (Exception e) {
      OK(e);
    }

    try {
      TRUE(DIRS[4].exists());
      TRUE(FILES[4].exists());
      TRUE(FILES[9].exists());
      FALSE(dstFile.getParentFile().exists());
      FALSE(dstFile.exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_rollback_01()
  {
    MSG("ړ̃t@C݂ȂꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();

    @SuppressWarnings("serial")
    File srcFile = new File(WORKDIR, "fff") {
      @Override public boolean renameTo(File f) {
        return false;
      }
      @Override public boolean delete() {
        return false;
      }
    };
    File dstFile = new File(WORKDIR, "ggg");

    FALSE(srcFile.exists());
    FALSE(dstFile.exists());
    TRUE(dstFile.getParentFile().exists());

    try {
      values[0] = srcFile;
      values[1] = dstFile;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    FALSE(srcFile.exists());
    FALSE(dstFile.exists());
    TRUE(dstFile.getParentFile().exists());
  }

  public void moveInner_renameTo_failed_rollback_02()
  {
    MSG("ړ̃fBNgłȂt@Cɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    TRUE(FILES[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());

    try {
      @SuppressWarnings("serial")
      File fff = new File(FILES[3].getPath()) {
        @Override public boolean renameTo(File f) {
          return false;
        }
        @Override public boolean delete() {
          return false;
        }
      };
      values[0] = fff;
      values[1] = FILES[4];
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());

    try {
      @SuppressWarnings("serial")
      File f = new File(DIRS[3].getPath()) {
        @Override public boolean renameTo(File f) {
          return false;
        }
        @Override public boolean delete() {
          return false;
        }
      };
      values[0] = f;
      values[1] = FILES[4];
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    TRUE(FILES[3].exists());
    TRUE(FILES[8].exists());
    TRUE(FILES[4].exists());
  }

  public void moveInner_renameTo_failed_rollback_03()
  {
    MSG("ړ̃fBNgɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    try {
      TRUE(DIRS[3].exists());
      TRUE(FILES[3].exists());
      TRUE(FILES[8].exists());
      TRUE(DIRS[4].exists());
      FALSE(fop.getDestinationFile(FILES[3], DIRS[3], DIRS[4]).exists());
      FALSE(fop.getDestinationFile(FILES[8], DIRS[3], DIRS[4]).exists());
  
      try {
        @SuppressWarnings("serial")
        File fff = new File(DIRS[3].getPath()) {
          @Override public boolean renameTo(File f) {
            return false;
          }
        };
        values[0] = fff;
        values[1] = DIRS[4];
        oi.invokeMethod(fop, "moveInner", types, values);
        NG();
      } catch (Exception e) {
        OK(e);
      }
  
      TRUE(DIRS[3].exists());
      TRUE(FILES[3].exists());
      TRUE(FILES[8].exists());
      TRUE(DIRS[4].exists());
      FALSE(fop.getDestinationFile(FILES[3], DIRS[3], DIRS[4]).exists());
      FALSE(fop.getDestinationFile(FILES[8], DIRS[3], DIRS[4]).exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_rollback_04()
  {
    MSG("ړɃ^Cv̈قȂt@Cɑ݂ꍇB");

    FileOperation fop = new FileOperation();
    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    File ddd = new File(WORKDIR, "aaa");
    File fff = new File(ddd, "DIRS1/DIRS3/FILES3");
    fff.mkdirs();

    try {
      @SuppressWarnings("serial")
      File f = new File(DIRS[0].getPath()) {
        @Override public boolean renameTo(File f) {
          return false;
        }
      };
      values[0] = f;
      values[1] = ddd;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (Exception e) {
      OK(e);
    }

    for (File d : DIRS) TRUE(d.exists());
    for (File f : DIRS) TRUE(f.exists());
    try {
      List<File> lst = FileOperation.listRecursive(ddd);
      EQUAL(lst.size(), 3);
      EQUAL(lst.get(0), new File(ddd, "DIRS1"));
      EQUAL(lst.get(1), new File(ddd, "DIRS1/DIRS3"));
      EQUAL(lst.get(2), new File(ddd, "DIRS1/DIRS3/FILES3"));
      TRUE(new File(ddd, "DIRS1/DIRS3/FILES3").isDirectory());
    } catch (Exception e) {
      NG(e);
    }

    File dd = new File(WORKDIR, "bbb");
    File ff = new File(dd, "DIRS1/DIRS3");
    try {
      FileOperation.createNewFile(ff);
    } catch (Exception e) {
      NG(e);
    }

    try {
      @SuppressWarnings("serial")
      File f = new File(DIRS[0].getPath()) {
        @Override public boolean renameTo(File f) {
          return false;
        }
      };
      values[0] = f;
      values[1] = dd;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (Exception e) {
      OK(e);
    }

    for (File d : DIRS) TRUE(d.exists());
    for (File f : DIRS) TRUE(f.exists());
    try {
      List<File> lst = FileOperation.listRecursive(dd);
      EQUAL(lst.size(), 2);
      EQUAL(lst.get(0), new File(dd, "DIRS1"));
      EQUAL(lst.get(1), new File(dd, "DIRS1/DIRS3"));
      FALSE(new File(dd, "DIRS1/DIRS3").isDirectory());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_rollback_05()
  {
    MSG("ړɎsƂ̃[obN̊mFB");

    ObjectInspector oi = new ObjectInspector(this);
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    FileOperation fop = new FileOperation();
    File dstDir = new File(WORKDIR, "dest");

    File errFile = new File(dstDir, "DIRS2/DIRS4/FILES4");
    try {
      FileOperation.createNewFile(errFile);
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> beforeLst = FileOperation.listRecursive(WORKDIR);

      try {
        @SuppressWarnings("serial")
        File fff = new File(DIRS[0].getPath()) {
          @Override public boolean renameTo(File f) {
            return false;
          }
        };
        values[0] = fff;
        values[1] = dstDir;
        oi.invokeMethod(fop, "moveInner", types, values);
        NG();
      } catch (Exception e) {
        OK(e);
      }
  
      List<File> afterLst = FileOperation.listRecursive(WORKDIR);
      EQUAL(beforeLst.size(), afterLst.size());
      for (int i=0; i<beforeLst.size(); i++) {
        EQUAL(beforeLst.get(i), afterLst.get(i));
      }
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveInner_renameTo_failed_rollback_06()
  {
    MSG("ړ̐efBNg݂ȂꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FileOperation fop = new FileOperation();
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    @SuppressWarnings("serial")
    File src = new File(DIRS[3].getPath()) {
      @Override public boolean renameTo(File f) {
        return false;
      }
    };
    File dst =new File(WORKDIR, "aaa");
    TRUE(DIRS[3].exists());
    FALSE(dst.exists());

    try {
      values[0] = src;
      values[1] = dst;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (Exception e) {
      OK(e);
    }

    TRUE(DIRS[3].exists());
    FALSE(dst.exists());
  }

  public void moveInner_renameTo_failed_rollback_07()
  {
    MSG("ړ̐efBNg̓t@C̏ꍇB");

    ObjectInspector oi = new ObjectInspector(this);
    FileOperation fop = new FileOperation();
    Class[] types = { File.class, File.class };
    Object[] values = new Object[2];

    @SuppressWarnings("serial")
    File src = new File(DIRS[3].getPath()) {
      @Override public boolean renameTo(File f) {
        return false;
      }
    };
    File dst =new File(WORKDIR, "aaa/bbb/ccc");
    TRUE(DIRS[3].exists());
    FALSE(dst.exists());

    try {
      FileOperation.createNewFile(new File(WORKDIR, "aaa"));
    } catch (Exception e) {
      NG(e);
    }

    try {
      values[0] = src;
      values[1] = dst;
      oi.invokeMethod(fop, "moveInner", types, values);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(DIRS[3].exists());
    FALSE(dst.exists());
  }
}
