/*
 * StringConcatenatorTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.text;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.print.*;
import ts.util.*;
import java.text.*;
import java.util.*;

/**
 * <code>StringConcatenator</code>NX̋@\NXB
 *
 * @author  V. 
 * @version $Revision: 1.1.1.1 $, $Date: 2010-10-16 00:03:52 $
 */
public class StringConcatenatorTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);

      run(StringConcatenatorTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /* -- test case -- */

  public void constructor()
  {
    MSG("ʏ̑");

    NOTNULL(new StringConcatenator());
  }

  public void concatenate_strings_delimiter()
  {
    MSG("ʏ̑");

    String[] arr = {
      "","","","Ă"
    };

    EQUAL(StringConcatenator.concatenate(arr, ""),
      "Ă");
    EQUAL(StringConcatenator.concatenate(arr, ","),
      ",,,Ă");
    EQUAL(StringConcatenator.concatenate(arr, "A"),
      "AAAĂ");
  }

  public void concatenate_strings_delimiter_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((String[])null, "");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.concatenate(new String[0], null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_strings_delimiter_2()
  {
    MSG("vfɋ󕶎ƃk܂܂Ăꍇ");

    String[] arr = {
      "","","","",null,"Ă"
    };
    EQUAL(StringConcatenator.concatenate(arr, ""),
      "nullĂ");
    EQUAL(StringConcatenator.concatenate(arr, ","),
      ",,,,null,Ă");
    EQUAL(StringConcatenator.concatenate(arr, "A"),
      "AAAAnullAĂ");
  }

  public void concatenate_stirngs_delimiter_3()
  {
    MSG("vf[̏ꍇ");

    EQUAL(StringConcatenator.concatenate(new String[0], ""), "");
    EQUAL(StringConcatenator.concatenate(new String[0], ","), "");
  }

  public void concatenate_enm_delimiter()
  {
    MSG("ʏ̑");

    List<String> lst = new ArrayList<String>(4);
    lst.add("");
    lst.add("");
    lst.add("");
    lst.add("Ă");

    Enumeration<String> enm = Collections.enumeration(lst);
    EQUAL(StringConcatenator.concatenate(enm, ""),
      "Ă");

    enm = Collections.enumeration(lst);
    EQUAL(StringConcatenator.concatenate(enm, ","),
      ",,,Ă");

    enm = Collections.enumeration(lst);
    EQUAL(StringConcatenator.concatenate(enm, "A"),
      "AAAĂ");
  }

  public void concatenate_enm_delimiter_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((Enumeration<String>)null, "");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.concatenate(new ArrayList<String>(), null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_enm_delimiter_2()
  {
    MSG("vfɋ󕶎ƃk܂܂Ăꍇ");

    List<String> lst = new ArrayList<String>();
    lst.add("");
    lst.add("");
    lst.add("");
    lst.add("");
    lst.add(null);
    lst.add("Ă");

    Enumeration<String> enm = Collections.enumeration(lst);
    EQUAL(StringConcatenator.concatenate(enm, ""),
      "nullĂ");

    enm = Collections.enumeration(lst);
    EQUAL(StringConcatenator.concatenate(enm, ","),
      ",,,,null,Ă");

    enm = Collections.enumeration(lst);
    EQUAL(StringConcatenator.concatenate(enm, "A"),
      "AAAAnullAĂ");
  }

  public void concatenate_enm_delimiter_3()
  {
    MSG("vf[̏ꍇ");

    List<String> lst = new ArrayList<String>();
    Enumeration<String> enm = Collections.enumeration(lst);

    EQUAL(StringConcatenator.concatenate(enm, ""), "");
    EQUAL(StringConcatenator.concatenate(enm, ","), "");
  }

  public void concatenate_coll_delimiter()
  {
    MSG("ʏ̑");

    List<String> lst = new ArrayList<String>(4);
    lst.add("");
    lst.add("");
    lst.add("");
    lst.add("Ă");

    EQUAL(StringConcatenator.concatenate(lst, ""),
      "Ă");
    EQUAL(StringConcatenator.concatenate(lst, ","),
      ",,,Ă");
    EQUAL(StringConcatenator.concatenate(lst, "A"),
      "AAAĂ");
  }

  public void concatenate_coll_delimiter_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((List<String>)null, "");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.concatenate(new ArrayList<String>(), null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_coll_delimiter_2()
  {
    MSG("vfɋ󕶎ƃk܂܂Ăꍇ");

    List<String> lst = new ArrayList<String>(4);
    lst.add("");
    lst.add("");
    lst.add("");
    lst.add("");
    lst.add(null);
    lst.add("Ă");

    EQUAL(StringConcatenator.concatenate(lst, ""),
      "nullĂ");
    EQUAL(StringConcatenator.concatenate(lst, ","),
      ",,,,null,Ă");
    EQUAL(StringConcatenator.concatenate(lst, "A"),
      "AAAAnullAĂ");
  }

  public void concatenate_coll_delimiter_3()
  {
    MSG("vf[̏ꍇ");

    EQUAL(StringConcatenator.concatenate(new ArrayList<String>(0), ""), "");
    EQUAL(StringConcatenator.concatenate(new ArrayList<String>(0), ","), "");
  }

  @Deprecated
  public void repeat_repeated_count()
  {
    MSG("ʏ̏ꍇ");
    EQUAL(StringConcatenator.repeat('a', 0), "");
    EQUAL(StringConcatenator.repeat('a', 1), "a");
    EQUAL(StringConcatenator.repeat('a', 2), "aa");
    EQUAL(StringConcatenator.repeat('a', 3), "aaa");
    EQUAL(StringConcatenator.repeat('a', 4), "aaaa");
  }

  @Deprecated
  public void repeat_repeated_count_1()
  {
    MSG("Jԉ񐔂̏ꍇ");
    try {
      StringConcatenator.repeat('a', -1);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void repeat_repeatedStr_count()
  {
    MSG("ʏ̏ꍇ");
    EQUAL(StringConcatenator.repeat("a", 0), "");
    EQUAL(StringConcatenator.repeat("a", 1), "a");
    EQUAL(StringConcatenator.repeat("a", 2), "aa");
    EQUAL(StringConcatenator.repeat("a", 3), "aaa");
    EQUAL(StringConcatenator.repeat("a", 4), "aaaa");
  }

  public void repeat_repeatedStr_count_1()
  {
    MSG("Jԉ񐔂̏ꍇ");
    try {
      StringConcatenator.repeat("a", -1);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void repeat_repeated_count_delimiter()
  {
    MSG("ʏ̏ꍇ");

    EQUAL(StringConcatenator.repeat("", 3, ""), "");
    EQUAL(StringConcatenator.repeat("", 3, ","), ",,");
    EQUAL(StringConcatenator.repeat("A", 3, ","), "A,A,A");
    EQUAL(StringConcatenator.repeat("ABC", 3, ",,"), "ABC,,ABC,,ABC");

    EQUAL(StringConcatenator.repeat("ABC", 0, ","), "");
    EQUAL(StringConcatenator.repeat("ABC", 1, ","), "ABC");
    EQUAL(StringConcatenator.repeat("ABC", 2, ","), "ABC,ABC");
  }

  public void repeat_repeated_count_delimiter_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.repeat(null, 1, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.repeat("ABC", 1, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void repeat_repeated_count_delimiter_2()
  {
    MSG("Jԉ񐔂̏ꍇ");

    try {
      StringConcatenator.repeat("ABC", -1, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_objs_fmt_delim()
  {
    MSG("ʏ̏ꍇ - DateFormat");

    DateFormat df = new SimpleDateFormat("yyyyNMMdd");
    Date[] dates = new Date[5];
    dates[0] = (new DateTime(2003,11,20)).getDate();
    dates[1] = (new DateTime(2003,11,30)).getDate();
    dates[2] = (new DateTime(2003,12,10)).getDate();
    dates[3] = (new DateTime(2003,12,20)).getDate();
    dates[4] = (new DateTime(2003,12,30)).getDate();

    EQUAL(StringConcatenator.concatenate(dates, df, ""),
      "2003N11202003N11302003N12102003N12202003N1230");
    EQUAL(StringConcatenator.concatenate(dates, df, ","),
      "2003N1120,2003N1130," +
      "2003N1210,2003N1220,2003N1230");
  }

  public void concatenate_objs_fmt_delim_0()
  {
    MSG("ʏ̏ꍇ - MessageFormat");

    MessageFormat mf = new MessageFormat("{0}łB");

    Integer[] arr = { 11, 22 };
    EQUAL(StringConcatenator.concatenate(arr, mf, ""),
      "11łB22łB");
    EQUAL(StringConcatenator.concatenate(arr, mf, "|"),
      "11łB|22łB");
  }

  public void concatenate_objs_fmt_delim_1()
  {
    MSG("k̏ꍇ");

    MessageFormat mf = new MessageFormat("{0}łB");

    try {
      StringConcatenator.concatenate((Object[])null, mf, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.concatenate(new Integer[0], (Format)null, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.concatenate(new Integer[0], mf, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_objs_fmt_delim_2()
  {
    MSG("ɂϊɎsꍇ");

    DateFormat df = new SimpleDateFormat("yyyyNMMdd");
    Object[] objs = new Object[1];
    objs[0] = "abc";

    try {
      NG(StringConcatenator.concatenate(objs, df, ","));
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    MessageFormat mf = new MessageFormat("{0,number}");

    try {
      NG(StringConcatenator.concatenate(objs, mf, ","));
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_objs_fmt_delim_3()
  {
    MSG("vf[̏ꍇ");

    DateFormat df = new SimpleDateFormat("yyyyNMMdd");
    Date[] dates = new Date[0];
    EQUAL(StringConcatenator.concatenate(dates, df, ","), "");

    MessageFormat mf = new MessageFormat("{0,number}");
    Integer[] ints = new Integer[0];
    EQUAL(StringConcatenator.concatenate(ints, mf, ","), "");
  }

  public void concatenate_enm_fmt_delim()
  {
    MSG("ʏ̏ꍇ - DateFormat");

    DateFormat df = new SimpleDateFormat("yyyyNMMdd");

    List<Object> dateL = new ArrayList<Object>();
    dateL.add(new DateTime(2003,11,20).getDate());
    dateL.add(new DateTime(2003,11,30).getDate());
    dateL.add(new DateTime(2003,12,10).getDate());
    dateL.add(new DateTime(2003,12,20).getDate());
    dateL.add(new DateTime(2003,12,30).getDate());

    Enumeration<Object> dateE = Collections.enumeration(dateL);
    EQUAL(StringConcatenator.concatenate(dateE, df, ""),
      "2003N11202003N11302003N12102003N12202003N1230");

    dateE = Collections.enumeration(dateL);
    EQUAL(StringConcatenator.concatenate(dateE, df, ","),
      "2003N1120,2003N1130," +
      "2003N1210,2003N1220,2003N1230");
  }

  public void concatenate_enm_fmt_delim_0()
  {
    MSG("ʏ̏ꍇ - MessageFormat");

    MessageFormat mf = new MessageFormat("{0}łB");

    List<Object> intL = new ArrayList<Object>();
    intL.add(11);
    intL.add(22);

    Enumeration<Object> intE = Collections.enumeration(intL);
    EQUAL(StringConcatenator.concatenate(intE, mf, ""),
      "11łB22łB");

    intE = Collections.enumeration(intL);
    EQUAL(StringConcatenator.concatenate(intE, mf, "|"),
      "11łB|22łB");
  }

  public void concatenate_enm_fmt_delim_1()
  {
    MSG("k̏ꍇ");

    MessageFormat mf = new MessageFormat("{0}łB");

    try {
      StringConcatenator.concatenate((Enumeration<Object>)null, mf, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    List<Object> intL = new ArrayList<Object>();
    intL.add(1);
    Enumeration<Object> intE = Collections.enumeration(intL);

    try {
      StringConcatenator.concatenate(intE, (Format)null, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.concatenate(intE, mf, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_enm_fmt_delim_2()
  {
    MSG("ɂϊɎsꍇ");

    DateFormat df = new SimpleDateFormat("yyyyNMMdd");
    List<Object> sL = new ArrayList<Object>(1);
    sL.add("");

    Enumeration<Object> sE = Collections.enumeration(sL);

    try {
      NG(StringConcatenator.concatenate(sE, df, ","));
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    MessageFormat mf = new MessageFormat("{0,number}");

    sE = Collections.enumeration(sL);
    try {
      NG(StringConcatenator.concatenate(sE, mf, ","));
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_enm_fmt_delim_3()
  {
    MSG("vf[̏ꍇ");

    DateFormat df = new SimpleDateFormat("yyyy/MM/dd");
    List<Object> dateL = new ArrayList<Object>();
    Enumeration<Object> dateE = Collections.enumeration(dateL);
    EQUAL(StringConcatenator.concatenate(dateE, df, ","), "");

    MessageFormat mf = new MessageFormat("{0,number}");
    List<Object> intL = new ArrayList<Object>();
    Enumeration<Object> intE = Collections.enumeration(intL);
    EQUAL(StringConcatenator.concatenate(intE, mf, ","), "");
  }

  public void concatenate_coll_fmt_delim()
  {
    MSG("ʏ̏ꍇ - DateFormat");

    DateFormat df = new SimpleDateFormat("yyyyNMMdd");

    List<Date> dateL = new ArrayList<Date>();
    dateL.add(new DateTime(2003,11,20).getDate());
    dateL.add(new DateTime(2003,11,30).getDate());
    dateL.add(new DateTime(2003,12,10).getDate());
    dateL.add(new DateTime(2003,12,20).getDate());
    dateL.add(new DateTime(2003,12,30).getDate());

    EQUAL(StringConcatenator.concatenate(dateL, df, ""),
      "2003N11202003N11302003N12102003N12202003N1230");

    EQUAL(StringConcatenator.concatenate(dateL, df, ","),
      "2003N1120,2003N1130," +
      "2003N1210,2003N1220,2003N1230");
  }

  public void concatenate_coll_fmt_delim_0()
  {
    MSG("ʏ̏ꍇ - MessageFormat");

    MessageFormat mf = new MessageFormat("{0}łB");

    List<Integer> intL = new ArrayList<Integer>();
    intL.add(11);
    intL.add(22);

    EQUAL(StringConcatenator.concatenate(intL, mf, ""),
      "11łB22łB");

    EQUAL(StringConcatenator.concatenate(intL, mf, "|"),
      "11łB|22łB");
  }

  public void concatenate_coll_fmt_delim_1()
  {
    MSG("k̏ꍇ");

    MessageFormat mf = new MessageFormat("{0}łB");

    try {
      StringConcatenator.concatenate((Collection)null, mf, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    List<Integer> intL = new ArrayList<Integer>();
    intL.add(1);

    try {
      StringConcatenator.concatenate(intL, (Format)null, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringConcatenator.concatenate(intL, mf, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_coll_fmt_delim_2()
  {
    MSG("ɂϊɎsꍇ");

    DateFormat df = new SimpleDateFormat("yyyyNMMdd");
    List<String> sL = new ArrayList<String>(1);
    sL.add("");

    try {
      NG(StringConcatenator.concatenate(sL, df, ","));
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    MessageFormat mf = new MessageFormat("{0,number}");

    try {
      NG(StringConcatenator.concatenate(sL, mf, ","));
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_coll_fmt_delim_3()
  {
    MSG("vf[̏ꍇ");

    DateFormat df = new SimpleDateFormat("yyyy/MM/dd");
    List<Date> dateL = new ArrayList<Date>();
    EQUAL(StringConcatenator.concatenate(dateL, df, ","), "");

    MessageFormat mf = new MessageFormat("{0,number}");
    List<Integer> intL = new ArrayList<Integer>();
    EQUAL(StringConcatenator.concatenate(intL, mf, ","), "");
  }

  @Deprecated
  public void concatenate_strings_delim_escape()
  {
    MSG("ʏ̏ꍇ");

    String[] a = { "!#$%&'()", "=-~^|\\" };
    EQUAL(StringConcatenator.concatenate(a, ',', '@'), "!#$%&'(),=-~^|\\");
    EQUAL(StringConcatenator.concatenate(a, '|', '@'), "!#$%&'()|=-~^@|\\");
    EQUAL(StringConcatenator.concatenate(a, ',', '$'), "!#$$%&'(),=-~^|\\");
  }

  @Deprecated
  public void concatenate_strings_delim_escape_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((String[])null, ',', '\\');
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  @Deprecated
  public void concatenate_strings_delim_escape_2()
  {
    MSG("z̃TCY[̏ꍇ");

    String[] a = new String[0];
    EQUAL(StringConcatenator.concatenate(a, ',', '$'), "");
  }

  public void concatenate_strings_delimStr_escapeStr()
  {
    MSG("ʏ̏ꍇ");

    String[] a = { "!#$%&'()", "=-~^|\\" };
    EQUAL(StringConcatenator.concatenate(a, ",", "@"), "!#$%&'(),=-~^|\\");
    EQUAL(StringConcatenator.concatenate(a, "|", "@"), "!#$%&'()|=-~^@|\\");
    EQUAL(StringConcatenator.concatenate(a, ",", "$"), "!#$$%&'(),=-~^|\\");
  }

  public void concatenate_strings_delimStr_escapeStr_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((String[])null, ",", "\\");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_strings_delimStr_escapeStr_2()
  {
    MSG("z̃TCY[̏ꍇ");

    String[] a = new String[0];
    EQUAL(StringConcatenator.concatenate(a, ",", "$"), "");
  }

  @Deprecated
  public void concatenate_enm_delim_escape()
  {
    MSG("ʏ̏ꍇ");

    List<String> sL = new ArrayList<String>();
    sL.add("!#$%&'()");
    sL.add("=-~^|\\");
    Enumeration<String> e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, ',', '@'), "!#$%&'(),=-~^|\\");
    e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, '|', '@'), "!#$%&'()|=-~^@|\\");
    e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, ',', '$'), "!#$$%&'(),=-~^|\\");
  }

  @Deprecated
  public void concatenate_enm_delim_escape_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate(
        (Enumeration<String>)null, ',', '\\');
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  @Deprecated
  public void concatenate_enm_delim_escape_2()
  {
    MSG("vf[̏ꍇ");

    List<String> sL = new ArrayList<String>();
    Enumeration<String> e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, ',', '@'), "");
  }

  public void concatenate_enm_delimStr_escapeStr()
  {
    MSG("ʏ̏ꍇ");

    List<String> sL = new ArrayList<String>();
    sL.add("!#$%&'()");
    sL.add("=-~^|\\");
    Enumeration<String> e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, ",", "@"), "!#$%&'(),=-~^|\\");
    e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, "|", "@"), "!#$%&'()|=-~^@|\\");
    e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, ",", "$"), "!#$$%&'(),=-~^|\\");
  }

  public void concatenate_enm_delimStr_escapeStr_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate(
        (Enumeration<String>)null, ",", "\\");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_enm_delimStr_escapeStr_2()
  {
    MSG("vf[̏ꍇ");

    List<String> sL = new ArrayList<String>();
    Enumeration<String> e = Collections.enumeration(sL);
    EQUAL(StringConcatenator.concatenate(e, ",", "@"), "");
  }

  @Deprecated
  public void concatenate_coll_delim_escape()
  {
    MSG("ʏ̏ꍇ");

    List<String> c = new ArrayList<String>();
    c.add("!#$%&'()");
    c.add("=-~^|\\");
    EQUAL(StringConcatenator.concatenate(c, ',', '@'), "!#$%&'(),=-~^|\\");
    EQUAL(StringConcatenator.concatenate(c, '|', '@'), "!#$%&'()|=-~^@|\\");
    EQUAL(StringConcatenator.concatenate(c, ',', '$'), "!#$$%&'(),=-~^|\\");
  }

  @Deprecated
  public void concatenate_coll_delim_escape_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((Collection<String>) null, ',', '\\');
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  @Deprecated
  public void concatenate_coll_delim_escape_2()
  {
    MSG("vf[̏ꍇ");

    List<String> c = new ArrayList<String>();
    EQUAL(StringConcatenator.concatenate(c, ',', '@'), "");
  }

  public void concatenate_coll_delimStr_escapeStr()
  {
    MSG("ʏ̏ꍇ");

    List<String> c = new ArrayList<String>();
    c.add("!#$%&'()");
    c.add("=-~^|\\");
    EQUAL(StringConcatenator.concatenate(c, ",", "@"), "!#$%&'(),=-~^|\\");
    EQUAL(StringConcatenator.concatenate(c, "|", "@"), "!#$%&'()|=-~^@|\\");
    EQUAL(StringConcatenator.concatenate(c, ",", "$"), "!#$$%&'(),=-~^|\\");
  }

  public void concatenate_coll_delimStr_escapeStr_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((Collection<String>) null, ",", "\\");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_coll_delimStr_escapeStr_2()
  {
    MSG("vf[̏ꍇ");

    List<String> c = new ArrayList<String>();
    EQUAL(StringConcatenator.concatenate(c, ",", "@"), "");
  }

  @Deprecated
  public void concatenate_strings_delim_openPar_closePar_escape()
  {
    MSG("ʏ̏ꍇ");

    String[] a = { "!#$%&'()", "=-~^|\\" };
    EQUAL(StringConcatenator.concatenate(a, ',', '[', ']', '@'),
      "[!#$%&'()],[=-~^|\\]");
    EQUAL(StringConcatenator.concatenate(a, '|', '(',')', '@'),
      "(!#$%&'@(@))|(=-~^@|\\)");
    EQUAL(StringConcatenator.concatenate(a, ',', '/', '/', '$'),
      "/!#$$%&'()/,/=-~^|\\/");
  }

  @Deprecated
  public void concatenate_strings_delim_openPar_closePar_escape_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((String[])null, ',', '(', ')', '\\');
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  @Deprecated
  public void concatenate_strings_delim_openPar_closePar_escape_2()
  {
    MSG("vf[̏ꍇ");

    String[] a = new String[0];
    EQUAL(StringConcatenator.concatenate(a, ',', '(', ')', '$'), "");
  }

  public void concatenate_strings_delimStr_openPar_closePar_escapeStr()
  {
    MSG("ʏ̏ꍇ");

    String[] a = { "!#$%&'()", "=-~^|\\" };
    EQUAL(StringConcatenator.concatenate(a, ",", "[", "]", "@"),
      "[!#$%&'()],[=-~^|\\]");
    EQUAL(StringConcatenator.concatenate(a, "|", "(",")", "@"),
      "(!#$%&'@(@))|(=-~^@|\\)");
    EQUAL(StringConcatenator.concatenate(a, ",", "/", "/", "$"),
      "/!#$$%&'()/,/=-~^|\\/");
  }

  public void concatenate_strings_delimStr_openPar_closePar_escapeStr_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((String[])null, ",", "(", ")", "\\");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_strings_delimStr_openPar_closePar_escapeStr_2()
  {
    MSG("vf[̏ꍇ");

    String[] a = new String[0];
    EQUAL(StringConcatenator.concatenate(a, ",", "(", ")", "$"), "");
  }

  @Deprecated
  public void concatenate_enm_delim_openPar_closePar_escape()
  {
    MSG("ʏ̑");

    List<String> c = new ArrayList<String>();
    c.add("!#$%&'()");
    c.add("=-~^|\\");

    Enumeration<String> e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, ',', '[', ']', '@'),
      "[!#$%&'()],[=-~^|\\]");

    e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, '|', '(',')', '@'),
      "(!#$%&'@(@))|(=-~^@|\\)");

    e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, ',', '/', '/', '$'),
      "/!#$$%&'()/,/=-~^|\\/");
  }

  @Deprecated
  public void concatenate_enm_delim_openPar_closePar_escape_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate(
        (Enumeration<String>) null, ',','(',')','@');
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  @Deprecated
  public void concatenate_enm_delim_openPar_closePar_escape_2()
  {
    MSG("vf[̏ꍇ");

    List<String> c = new ArrayList<String>();
    Enumeration<String> e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, ',', '[', ']', '@'), "");
  }


  public void concatenate_enm_delimStr_openPar_closePar_escapeStr()
  {
    MSG("ʏ̑");

    List<String> c = new ArrayList<String>();
    c.add("!#$%&'()");
    c.add("=-~^|\\");

    Enumeration<String> e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, ",", "[", "]", "@"),
      "[!#$%&'()],[=-~^|\\]");

    e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, "|", "(",")", "@"),
      "(!#$%&'@(@))|(=-~^@|\\)");

    e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, ",", "/", "/", "$"),
      "/!#$$%&'()/,/=-~^|\\/");
  }

  public void concatenate_enm_delimStr_openPar_closePar_escapeStr_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate(
        (Enumeration<String>) null, ",","(",")","@");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_enm_delimStr_openPar_closePar_escapeStr_2()
  {
    MSG("vf[̏ꍇ");

    List<String> c = new ArrayList<String>();
    Enumeration<String> e = Collections.enumeration(c);
    EQUAL(StringConcatenator.concatenate(e, ",", "[", "]", "@"), "");
  }

  @Deprecated
  public void concatenate_coll_delim_openPar_closePar_escape()
  {
    MSG("ʏ̏ꍇ");

    List<String> c = new ArrayList<String>();
    c.add("!#$%&'()");
    c.add("=-~^|\\");

    EQUAL(StringConcatenator.concatenate(c, ',', '[', ']', '@'),
      "[!#$%&'()],[=-~^|\\]");

    EQUAL(StringConcatenator.concatenate(c, '|', '(',')', '@'),
      "(!#$%&'@(@))|(=-~^@|\\)");

    EQUAL(StringConcatenator.concatenate(c, ',', '/', '/', '$'),
      "/!#$$%&'()/,/=-~^|\\/");
  }

  @Deprecated
  public void concatenate_coll_delim_openPar_closePar_escape_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((Collection<String>)
        null, ',','(',')','@');
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  @Deprecated
  public void concatenate_coll_delim_openPar_closePar_escape_2()
  {
    MSG("vf[̏ꍇ");

    List<String> c = new ArrayList<String>();
    EQUAL(StringConcatenator.concatenate(c, ',', '[', ']', '@'), "");
  }

  public void concatenate_coll_delimStr_openPar_closePar_escapeStr()
  {
    MSG("ʏ̏ꍇ");

    List<String> c = new ArrayList<String>();
    c.add("!#$%&'()");
    c.add("=-~^|\\");

    EQUAL(StringConcatenator.concatenate(c, ",", "[", "]", "@"),
      "[!#$%&'()],[=-~^|\\]");

    EQUAL(StringConcatenator.concatenate(c, "|", "(",")", "@"),
      "(!#$%&'@(@))|(=-~^@|\\)");

    EQUAL(StringConcatenator.concatenate(c, ",", "/", "/", "$"),
      "/!#$$%&'()/,/=-~^|\\/");
  }

  public void concatenate_coll_delimStr_openPar_closePar_escapeStr_1()
  {
    MSG("k̏ꍇ");

    try {
      StringConcatenator.concatenate((Collection<String>)
        null, ",","(",")","@");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void concatenate_coll_delimStr_openPar_closePar_escapeStr_2()
  {
    MSG("vf[̏ꍇ");

    List<String> c = new ArrayList<String>();
    EQUAL(StringConcatenator.concatenate(c, ",", "[", "]", "@"), "");
  }
}
