/*
 * QueryConfig class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import ts.util.ReasonedRuntimeException;
import ts.util.resource.Resource;
import ts.util.resource.PropertyResource;
import java.io.InputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;

/**
 * クエリの初期設定を保持するためのクラス。
 * <br>
 * <tt>ts.query.QueryConfig.properties</tt>から初期設定パラメータをロードして
 * 保持する。
 * <br>
 * このクラスのインスタンスはシングルトン・オブジェクトであり、同一プロセス内
 * にインスタンスが一つだけ生成されて、それが共用される。
 * <br>
 * また、このクラスは{@link ts.util.TypedGetter}インターフェイスをインプリメント
 * しており、初期設定の値を基本データ型や文字列、{@link ts.util.DateTime}
 * オブジェクトに変換して取得することが可能である。
 *
 * @author 佐藤隆之
 * @version $Id: QueryConfig.java,v 1.6 2011-08-09 14:24:18 tayu Exp $
 */
public class QueryConfig extends AbstractTypedGetter<String,String>
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 7870866298172485389L;

  /** このクラスのシングルトン・オブジェクト。 */
  private static final QueryConfig INSTANCE = createInstance();

  /** 初期設定を保持するリソース・オブジェクト。 */
  private final Resource resource;
  
  /**
   * このクラスで発生しうるエラーを定義する列挙型。
   */
  public enum Error {
    FileNotFound,
    FailToLoad,
  }

  /**
   * このクラスのシングルトン・インスタンスを取得する。
   */
  public static QueryConfig getInstance()
  {
    return INSTANCE;
  }

  /**
   * このクラスのシングルトン・インスタンスを作成する。
   *
   * @return このクラスのシングルトン・インスタンス。
   * @throws ReasonedRuntimeException 初期設定ファイルの読込に失敗した場合。
   */
  protected static QueryConfig createInstance() throws ReasonedRuntimeException
  {
    return new QueryConfig();
  }

  /**
   * デフォルト・コンストラクタ。
   *
   * @throws ReasonedRuntimeException 初期設定ファイルの読込に失敗した場合。
   */
  protected QueryConfig() throws ReasonedRuntimeException
  {
    String path = getClass().getName().replace('.','/') + ".properties";

    try {
      this.resource = new PropertyResource(path) {
        static final long serialVersionUID = 7870866298172485390L;
        protected InputStream getInputStream(String path) throws IOException {
          InputStream iostream = ClassLoader.getSystemResourceAsStream(path);
          if (iostream == null) {
            if (! ClassLoader.getSystemResources(path).hasMoreElements())
              throw new FileNotFoundException();
          }
          return iostream;
        }
      };
    }
    catch (FileNotFoundException e) {
      throw new ReasonedRuntimeException(Error.FileNotFound, path);
    }
    catch (IOException e) {
      throw new ReasonedRuntimeException(Error.FailToLoad, path);
    }
  }

  /**
   * 指定されたキーに結びつけられた値オブジェクトを取得する。
   *
   * @param key キー。
   * @return 値オブジェクト。
   * @throws NullPointerException 引数がヌルの場合。
   */
  public String get(String key)
  {
    return this.resource.getFirstValue(key.toString());
  }

  /**
   * 指定されたキーに結びつけられた値オブジェクトをリストに変換して取得する。
   *
   * @param key キー。
   * @return 複数の値オブジェクトを格納したリスト・オブジェクト。
   * @throws NullPointerException 引数がヌルの場合。
   */
  public List<String> getList(String key)
  {
    return this.resource.getValues(key.toString());
  }
}
