package ts.query;

import ts.tester.UnitTest;
import ts.util.resource.*;
import ts.util.table.*;
import ts.util.*;
import java.io.*;
import java.util.*;

public class DefaultQueryTransactionTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(DefaultQueryTransactionTest.class, args);
  }

  static StringWriter SW = null;
  static PrintWriter PW = null;

  static String OUTPUT_LOG() {
    return SW.toString();
  }
  static void OPEN_LOG() {
    SW = new StringWriter();
    PW = new PrintWriter(SW);
  }
  static void CLOSE_LOG() {
    PW.close();
  }
  static void CLEAR_LOG() {
    CLOSE_LOG();
    OPEN_LOG();
  }

  @Override
  protected void preInvocation(String method)
  {
    OPEN_LOG();
  }
  @Override
  protected void postInvocation(String method)
  {
    CLOSE_LOG();
  }

  static class MyConnection implements QueryConnection {
    private long limitTm = -1L;
    private boolean isClosed = true;
    private final String connId;
    private final QueryResultList resultLst; 
    public MyConnection(String connId, QueryTransaction tran) {
      this.connId = connId;
      this.resultLst = tran.getQueryResultList();
      this.limitTm = tran.getLimitTimeMillis();
    }
    @Override
    public String getId() {
      return this.connId;
    }
    @Override
    public void open() {
      this.isClosed = false;
      PW.print("[" + connId + " is opened!]");
    }
    @Override
    public void commit() {
      PW.print("[" + connId + " is committed!]");
    }
    @Override
    public void rollback() {
      PW.print("[" + connId + " is rollbacked!]");
    }
    @Override
    public void close() {
      this.isClosed = true;
      PW.print("[" + connId + " is closed!]");
    }
    @Override
    public boolean isClosed() {
      return this.isClosed;
    }
    @Override
    public long getLimitTimeMillis() {
      return this.limitTm;
    }
    @Override
    public Query getQuery(QueryResource res) {
      return null;
    }
    @Override
    public int executeContent(QueryContent cont) throws ReasonedException {
      return 0;
    }
    @Override
    public int executeContent(QueryContent cont, Table<String,Object> table)
      throws ReasonedException
    {
      return 0;
    }
    @Override
    public QueryResultList getTransactionQueryResultList() {
      return this.resultLst;
    }
  }

  static class MyConnectionFactory implements QueryConnectionFactory {
    @Override
    public QueryConnection create(String connId, QueryTransaction tran)
      throws ReasonedException
    {
      if ("FAIL".equals(connId)) {
        throw new ReasonedException(
          QueryConnectionFactory.Error.FailToCreate, connId);
      }
      return new MyConnection(connId, tran);
    }
  }


  public void constructor()
  {
    MSG("デフォルト・コンストラクタの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    TRUE(tr instanceof QueryTransaction);
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);
  }

  public void setTimeoutMillis()
  {
    MSG("タイムアウト時間を設定するメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();

    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.setTimeoutMillis(123L);
    EQUAL(tr.getTimeoutMillis(), 123L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.begin();
    EQUAL(tr.getTimeoutMillis(), 123L);
    POSITIVE(tr.getBeginTimeMillis());
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 123L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
      EQUAL(e.getMessage(), QueryTransaction.State.Begined.name());
    }
    EQUAL(tr.getTimeoutMillis(), 123L);
    POSITIVE(tr.getBeginTimeMillis());
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 123L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.commit();
    EQUAL(tr.getTimeoutMillis(), 123L);
    POSITIVE(tr.getBeginTimeMillis());
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 123L);
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
      EQUAL(e.getMessage(), QueryTransaction.State.Committed.name());
    }
    EQUAL(tr.getTimeoutMillis(), 123L);
    POSITIVE(tr.getBeginTimeMillis());
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 123L);
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getTimeoutMillis(), 123L);
    POSITIVE(tr.getBeginTimeMillis());
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 123L);
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
      EQUAL(e.getMessage(), QueryTransaction.State.Ended.name());
    }
    EQUAL(tr.getTimeoutMillis(), 123L);
    POSITIVE(tr.getBeginTimeMillis());
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 123L);
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getQueryResultList().countResults(), 0);
  }

  public void normal_flow()
  {
    MSG("このクラスの通常の処理の流れを確認する。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    long tm0 = System.currentTimeMillis();
    tr.begin();
    long tm1 = System.currentTimeMillis();
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getTimeoutMillis(), -1L);
    long tm = tr.getBeginTimeMillis();
    TRUE(tm0 <= tm && tm <= tm1);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), tm);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), tm);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);
  }

  public void begin_timeout()
  {
    MSG("トランザクションを開始するメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getTimeoutMillis(), -1L);
    long bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.begin(1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.begin(1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.begin(1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void begin_timeout_Negative()
  {
    MSG("引数が負の場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.begin(-1000L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getTimeoutMillis(), -1L);
    long bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.begin(-1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.begin(-1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.begin(-1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void commit()
  {
    MSG("トランザクションの結果を確定するメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.commit();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getTimeoutMillis(), -1L);
    long bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.rollback();
    EQUAL(tr.getState(), QueryTransaction.State.Rollbacked);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.commit();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void rollback()
  {
    MSG("トランザクションの結果を取り消すメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.rollback();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getTimeoutMillis(), -1L);
    long bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.rollback();
    EQUAL(tr.getState(), QueryTransaction.State.Rollbacked);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.rollback();
    EQUAL(tr.getState(), QueryTransaction.State.Rollbacked);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.rollback();
    EQUAL(tr.getState(), QueryTransaction.State.Rollbacked);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    try {
      tr.rollback();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void end()
  {
    MSG("トランザクションを終了するメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getTimeoutMillis(), -1L);
    long bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);


    tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
    EQUAL(tr.getQueryResultList().countResults(), 0);


    tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
    EQUAL(tr.getQueryResultList().countResults(), 0);
  }

  public void getConnection_connId()
  {
    MSG("コネクションを取得するメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(OUTPUT_LOG(), "");

    tr.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(OUTPUT_LOG(), "");

    tr.begin();
    EQUAL(OUTPUT_LOG(), "");

    tr.commit();
    EQUAL(OUTPUT_LOG(), "");

    tr.end();
    EQUAL(OUTPUT_LOG(), "");

    CLEAR_LOG();


    tr = new DefaultQueryTransaction();
    EQUAL(OUTPUT_LOG(), "");

    tr.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(OUTPUT_LOG(), "");

    tr.begin();
    EQUAL(OUTPUT_LOG(), "");

    QueryConnection conn0 = null, conn1 = null, conn2 = null;
    try {
      conn0 = tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c0 is opened!]");
      FALSE(conn0.isClosed());

      conn1 = tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c0 is opened!]");
      FALSE(conn0.isClosed());

      conn2 = tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c0 is opened!]");
      FALSE(conn0.isClosed());

      tr.commit();
      EQUAL(OUTPUT_LOG(), "[c0 is opened!][c0 is committed!]");
      FALSE(conn0.isClosed());
    }
    catch (Exception e) {
      tr.rollback();
      EQUAL(OUTPUT_LOG(), "[c0 is opened!][c0 is rollbacked!]");
      FALSE(conn0.isClosed());
    }
    finally {
      tr.end();
      EQUAL(OUTPUT_LOG(), "[c0 is opened!][c0 is committed!][c0 is closed!]");
      TRUE(conn0.isClosed());
    }

    CLEAR_LOG();

    tr = new DefaultQueryTransaction();
    EQUAL(OUTPUT_LOG(), "");

    tr.addConnectionFactory("c0", new MyConnectionFactory());
    tr.addConnectionFactory("c1", new MyConnectionFactory());
    tr.addConnectionFactory("c2", new MyConnectionFactory());
    EQUAL(OUTPUT_LOG(), "");

    tr.begin();
    EQUAL(OUTPUT_LOG(), "");

    try {
      conn0 = tr.getConnection("c1");
      EQUAL(OUTPUT_LOG(), "[c1 is opened!]");
      FALSE(conn0.isClosed());

      conn1 = tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!]");
      FALSE(conn0.isClosed());
      FALSE(conn1.isClosed());

      conn2 = tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!]");
      FALSE(conn0.isClosed());
      FALSE(conn1.isClosed());

      throw new Exception();
    }
    catch (Exception e) {
      tr.rollback();
      EQUAL(OUTPUT_LOG(),
        "[c1 is opened!][c0 is opened!]" +
        "[c1 is rollbacked!][c0 is rollbacked!]"
      );
      FALSE(conn0.isClosed());
      FALSE(conn1.isClosed());
    }
    finally {
      tr.end();
      EQUAL(OUTPUT_LOG(),
        "[c1 is opened!][c0 is opened!]" +
        "[c1 is rollbacked!][c0 is rollbacked!]" +
        "[c1 is closed!][c0 is closed!]"
      );
      TRUE(conn0.isClosed());
      TRUE(conn1.isClosed());
    }
  }

  public void getConnectoin_connId_NoConnection()
  {
    MSG("引数のIDに対するコネクションが存在しない場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.begin();
    try {
      tr.getConnection("c0");
      NG();
    }
    catch (ReasonedException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.ConnectionNotFound);
      EQUAL(e.getMessage(), "c0");
    }
    tr.end();


    tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    tr.begin();
    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    }
    catch (ReasonedException e) {
      NG();
    }
    try {
     tr.getConnection("c1");
     NG();
    }
    catch (ReasonedException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.ConnectionNotFound);
      EQUAL(e.getMessage(), "c1");
    }
    tr.end();
  }

  public void getConnection_connId_BadState()
  {
    MSG("トランザクションの状態が不正な場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    try {
     tr.getConnection("c0");
     NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    } catch (ReasonedException e) {
      NG(e);
    }

    tr.begin();

    try {
     EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    tr.rollback();

    try {
     tr.getConnection("c0");
     NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    } catch (ReasonedException e) {
      NG(e);
    }

    tr.end();

    try {
     tr.getConnection("c0");
     NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    } catch (ReasonedException e) {
      NG(e);
    }
  }

  public void getConnection_connId_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    tr.begin();

    try {
      tr.getConnection(null);
      NG();
    } catch (ReasonedException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.ConnectionNotFound);
      EQUAL(e.getMessage(), null);
    }

    tr.end();
  }

  public void getConnection_connId_FailToCreate()
  {
    MSG("コネクションの作成に失敗した場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("FAIL", new MyConnectionFactory());
    tr.begin();

    try {
      tr.getConnection("FAIL");
      NG();
    } catch (ReasonedException e) {
      EQUAL(e.getReason(), QueryConnectionFactory.Error.FailToCreate);
      EQUAL(e.getMessage(), "FAIL");
    }

    tr.end();
  }

  public void addConnectionFactory_connId_factory_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();

    try {
      tr.addConnectionFactory(null, new MyConnectionFactory());
      NG();
    } catch (AssertionError e) {
      OK(e);
    }

    try {
      tr.addConnectionFactory("c0", null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void addConnectionFactory_connId_factory_BadState()
  {
    MSG("トランザクションの状態が不正な場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    tr.begin();
    try {
      tr.addConnectionFactory("c1", new MyConnectionFactory());
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.rollback();
    try {
      tr.addConnectionFactory("c2", new MyConnectionFactory());
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    try {
      tr.addConnectionFactory("c3", new MyConnectionFactory());
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }
}
