/*
 * QueryConfig class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import ts.util.ReasonedRuntimeException;
import ts.util.resource.Resource;
import ts.util.resource.PropertyResource;
import java.io.InputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;

/**
 * クエリの初期設定を保持するためのクラス。
 * <br>
 * リソース・パス <tt>ts/query/QueryConfig.properties</tt> にある初期設定
 * ファイルからプロパティをロードして保持する。
 * <br>
 * このクラスのインスタンスはシングルトン・インスタンスであり、同一プロセス内
 * にインスタンスが一つだけ生成されて、それが共用される。
 * <br>
 * また、このクラスは{@link ts.util.TypedGetter}インターフェイスをインプリメント
 * しており、初期設定プロパティ値を基本データ型や文字列、{@link
 * ts.util.DateTime}オブジェクトに変換して取得することが可能である。
 *
 * @author 佐藤隆之
 * @version $Id: QueryConfig.java,v 1.7 2011-08-14 16:27:16 tayu Exp $
 */
public final class QueryConfig extends AbstractTypedGetter<String,String>
{
  /** シリアル・バージョン番号。 */
  private static final long serialVersionUID = 7870866298172485389L;

  /** このクラスのシングルトン・インスタンス。 */
  private static final QueryConfig INSTANCE = new QueryConfig();

  /** 初期設定を保持するリソース・オブジェクト。 */
  private final Resource resource;

  /** このクラスで発生しうるエラーを定義する列挙型。 */
  public enum Error {
    /** 初期設定ファイルが存在しない場合。 */
    FileNotFound,

    /** 初期設定ファイルのロードに失敗した場合。 */
    FailToLoad,
  }

  /**
   * このクラスのシングル・インスタンスを取得する。
   */
  public static QueryConfig getInstance()
  {
    return INSTANCE;
  }

  /**
   * デフォルト・コンストラクタ。
   *
   * @throws ReasonedRuntimeException 初期設定ファイルの読み込みに失敗した場合。
   */
  private QueryConfig() throws ReasonedRuntimeException
  {
    this.resource = loadResource();
  }

  /**
   * 初期設定ファイルを読み込んだリソース・オブジェクトを取得する。
   *
   * @return 初期設定ファイルを読み込んだリソース・オブジェクト。
   * @throws ReasonedRuntimeException 初期設定ファイルの読み込みに失敗した場合。
   */
  static final Resource loadResource() throws ReasonedRuntimeException
  {
    String path = QueryConfig.class.getName().replace('.','/') + ".properties";
    try {
      return new PropertyResource(path) {
        private static final long serialVersionUID = 7870866298172485390L;
        protected InputStream getInputStream(String path) throws IOException {
          InputStream istream = ClassLoader.getSystemResourceAsStream(path);
          if (istream == null) {
            if (! ClassLoader.getSystemResources(path).hasMoreElements())
              throw new FileNotFoundException(path);
          }
          return istream;
        }
      };
    }
    catch (FileNotFoundException e) {
      throw new ReasonedRuntimeException(Error.FileNotFound, path);
    }
    catch (IOException e) {
      throw new ReasonedRuntimeException(Error.FailToLoad, path);
    }
  }

  /**
   * 指定されたキーに結びつけられた値オブジェクトを取得する。
   *
   * @param key キー。
   * @return 値オブジェクト。
   * @throws IllegalArgumentException 引数がヌルの場合。
   */
  public String get(String key)
  {
    if (key == null)
      throw new IllegalArgumentException("@param:key is null.");

    return this.resource.getFirstValue(key);
  }

  /**
   * 指定されたキーに結びつけられた値オブジェクトをリストに変換して取得する。
   *
   * @param key キー。
   * @return 複数の値オブジェクトを格納したリスト・オブジェクト。
   * @throws IllegalArgumentException 引数がヌルの場合。
   */
  public List<String> getList(String key)
  {
    if (key == null)
      throw new IllegalArgumentException("@param:key is null.");

    return this.resource.getValues(key);
  }
}
