package ts.query;

import ts.tester.UnitTest;
import ts.util.*;
import ts.util.file.FileOperation;
import java.io.*;
import java.math.*;
import java.nio.channels.*;
import java.util.*;

public class QueryConfigTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryConfigTest.class, args);
  }

  @Override
  protected void preTesting()
  {
    QueryConfig.getInstance();
  }

  File DATA_DIR = new File("test/data");
  File FILE0 = new File(DATA_DIR, "ts/query/QueryConfig.dat");
  File FILE1 = new File(DATA_DIR, "ts/query/QueryConfig.properties");

  public void getInstance()
  {
    MSG("シングルトン・オブジェクトを取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();
    EQUAL(config.get("a1"), "xxx1");
  }

  public void createInstance_FileNotFound()
  {
    MSG("プロパティ・ファイルが見つからない場合。");

    try {
      if (FILE1.exists()) FileOperation.delete(FILE1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConfig.loadResource();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryConfig.Error.FileNotFound);
      EQUAL(e.getMessage(), "ts/query/QueryConfig.properties"); 
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      try {
        FileOperation.copy(FILE0, FILE1);
      }
      catch (Exception e) {
        NG(e);
      }
    }
  }

  public void createInstance_FailToLoad()
  {
    MSG("プロパティ・ファイルがロードできない場合。");
    MSG("（ファイルにロックをかけてロードできないようにしている）");

    RandomAccessFile raf = null;
    FileLock lock = null;

    try {
      raf = new RandomAccessFile(FILE1, "rw");
      FileChannel channel = raf.getChannel();
      lock = channel.lock(0, Long.MAX_VALUE, false);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConfig.loadResource();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryConfig.Error.FailToLoad);
      EQUAL(e.getMessage(), "ts/query/QueryConfig.properties"); 
    }
    catch (Throwable e) {
      NG(e);
    }
    finally {
      try {
        if (lock != null) try { lock.release(); } catch (Exception e) {}
        if (raf != null) try { raf.close(); } catch (Exception e) {}
      }
      catch (Exception e) {
        NG(e);
      }
    }
  }

  public void get_key()
  {
    MSG("キーに対する値を取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.get(""), "");
    EQUAL(config.get("a1"), "xxx1");
    EQUAL(config.get("a1.b1"), "xxx1.yyy1");
    EQUAL(config.get("a1.b1.c1"), "xxx1.yyy1.zzz1");
    EQUAL(config.get("a2.b2.bool1"), "true");
    EQUAL(config.get("a2.b2.bool2"), "false");
    EQUAL(config.get("a2.b2.bool3"), "");
    EQUAL(config.get("a2.b2.byte1"), "123");
    EQUAL(config.get("a2.b2.byte2"), "-123");
    EQUAL(config.get("a2.b2.byte3"), "");
    EQUAL(config.get("a2.b2.char1"), "A");
    EQUAL(config.get("a2.b2.char2"), "a");
    EQUAL(config.get("a2.b2.char3"), "");
    EQUAL(config.get("a2.b2.short1"), "2345");
    EQUAL(config.get("a2.b2.short2"), "-2345");
    EQUAL(config.get("a2.b2.short3"), "");
    EQUAL(config.get("a2.b2.int1"), "123456789");
    EQUAL(config.get("a2.b2.int2"), "-123456789");
    EQUAL(config.get("a2.b2.int3"), "");
    EQUAL(config.get("a2.b2.float1"), "1.23456");
    EQUAL(config.get("a2.b2.float2"), "-1.23456");
    EQUAL(config.get("a2.b2.float3"), "");
    EQUAL(config.get("a2.b2.double1"), "987.654321");
    EQUAL(config.get("a2.b2.double2"), "-987.654321");
    EQUAL(config.get("a2.b2.double3"), "");
    EQUAL(config.get("a2.b2.string1"), "Abcdefg");
    EQUAL(config.get("a2.b2.string2"), "あいうえお");
    EQUAL(config.get("a2.b2.string3"), "");
    EQUAL(config.get("a2.b2.datetime1"), "2011-01-01");
    EQUAL(config.get("a2.b2.datetime2"), "2011-12-31");
    EQUAL(config.get("a2.b2.datetime3"), "");
    EQUAL(config.get("a3.b3"), "AB3-1");
  }

  public void get_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.get(null);
      NG();
    }
    catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void get_key_NotExist()
  {
    MSG("引数に存在しないキーが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.get("aaa"), "");
    EQUAL(config.get("a1.b2"), "");
  }

  public void getList_key()
  {
    MSG("キーに対する値をリストとして取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    List list0 = config.getList("a3.b3");
    EQUAL(list0.size(), 5);
    EQUAL(list0.get(0), "AB3-1");
    EQUAL(list0.get(1), "AB3-2");
    EQUAL(list0.get(2), "AB3-3");
    EQUAL(list0.get(3), "AB3-4");
    EQUAL(list0.get(4), "AB3-5");

    List list1 = config.getList("a1.b1");
    EQUAL(list1.size(), 1);
    EQUAL(list1.get(0), "xxx1.yyy1");

    List list2 = config.getList("");
    EQUAL(list2.size(), 1);
    EQUAL(list2.get(0), "");
  }

  public void getList_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getList(null);
      NG();
    }
    catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getList_key_NotExist()
  {
    MSG("引数に存在しないキーが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    List list0 = config.getList("aaa.bbbb");
    EQUAL(list0.size(), 0);

    List list1 = config.getList("a1.b2");
    EQUAL(list1.size(), 0);
  }

  public void getBoolean_key()
  {
    MSG("キーに対する値をboolean値で取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    config.removeTrueObject("xxx1.yyy1");
    config.removeTrueObject("a");
    config.removeTrueObject("2345");
    config.removeTrueObject("ABC3-1");

    FALSE(config.getBoolean(""));
    FALSE(config.getBoolean("a1"));
    FALSE(config.getBoolean("a1.b1"));
    FALSE(config.getBoolean("a1.b1.c1"));
    TRUE (config.getBoolean("a2.b2.bool1"));
    FALSE(config.getBoolean("a2.b2.bool2"));
    FALSE(config.getBoolean("a2.b2.bool3"));
    FALSE(config.getBoolean("a2.b2.byte1"));
    FALSE(config.getBoolean("a2.b2.byte2"));
    FALSE(config.getBoolean("a2.b2.byte3"));
    FALSE(config.getBoolean("a2.b2.char1"));
    FALSE(config.getBoolean("a2.b2.char2"));
    FALSE(config.getBoolean("a2.b2.char3"));
    FALSE(config.getBoolean("a2.b2.short1"));
    FALSE(config.getBoolean("a2.b2.short2"));
    FALSE(config.getBoolean("a2.b2.short3"));
    FALSE(config.getBoolean("a2.b2.int1"));
    FALSE(config.getBoolean("a2.b2.int2"));
    FALSE(config.getBoolean("a2.b2.int3"));
    FALSE(config.getBoolean("a2.b2.float1"));
    FALSE(config.getBoolean("a2.b2.float2"));
    FALSE(config.getBoolean("a2.b2.float3"));
    FALSE(config.getBoolean("a2.b2.double1"));
    FALSE(config.getBoolean("a2.b2.double2"));
    FALSE(config.getBoolean("a2.b2.double3"));
    FALSE(config.getBoolean("a2.b2.string1"));
    FALSE(config.getBoolean("a2.b2.string2"));
    FALSE(config.getBoolean("a2.b2.string3"));
    FALSE(config.getBoolean("a2.b2.datetime1"));
    FALSE(config.getBoolean("a2.b2.datetime2"));
    FALSE(config.getBoolean("a2.b2.datetime3"));
    FALSE(config.getBoolean("a3.b3"));
  }

  public void getBoolean_key_And_addTrueObject()
  {
    MSG("trueと判定する値オブジェクトを指定した場合。");

    QueryConfig config = QueryConfig.getInstance();

    config.addTrueObject("xxx1.yyy1");
    config.addTrueObject("a");
    config.addTrueObject("2345");
    config.addTrueObject("ABC3-1");

    FALSE(config.getBoolean(""));
    FALSE(config.getBoolean("a1"));
    TRUE (config.getBoolean("a1.b1"));
    FALSE(config.getBoolean("a1.b1.c1"));
    TRUE (config.getBoolean("a2.b2.bool1"));
    FALSE(config.getBoolean("a2.b2.bool2"));
    FALSE(config.getBoolean("a2.b2.bool3"));
    FALSE(config.getBoolean("a2.b2.byte1"));
    FALSE(config.getBoolean("a2.b2.byte2"));
    FALSE(config.getBoolean("a2.b2.byte3"));
    FALSE(config.getBoolean("a2.b2.char1"));
    TRUE (config.getBoolean("a2.b2.char2"));
    FALSE(config.getBoolean("a2.b2.char3"));
    TRUE (config.getBoolean("a2.b2.short1"));
    FALSE(config.getBoolean("a2.b2.short2"));
    FALSE(config.getBoolean("a2.b2.short3"));
    FALSE(config.getBoolean("a2.b2.int1"));
    FALSE(config.getBoolean("a2.b2.int2"));
    FALSE(config.getBoolean("a2.b2.int3"));
    FALSE(config.getBoolean("a2.b2.float1"));
    FALSE(config.getBoolean("a2.b2.float2"));
    FALSE(config.getBoolean("a2.b2.float3"));
    FALSE(config.getBoolean("a2.b2.double1"));
    FALSE(config.getBoolean("a2.b2.double2"));
    FALSE(config.getBoolean("a2.b2.double3"));
    FALSE(config.getBoolean("a2.b2.string1"));
    FALSE(config.getBoolean("a2.b2.string2"));
    FALSE(config.getBoolean("a2.b2.string3"));
    FALSE(config.getBoolean("a2.b2.datetime1"));
    FALSE(config.getBoolean("a2.b2.datetime2"));
    FALSE(config.getBoolean("a2.b2.datetime3"));
    FALSE(config.getBoolean("a3.b3"));
  }

  public void getBoolean_key_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getBoolean(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getInteger_key()
  {
    MSG("キーに対する値をint値で取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.getInteger(""), 0);
    EQUAL(config.getInteger("a1"), 0);
    EQUAL(config.getInteger("a1.b1"), 0);
    EQUAL(config.getInteger("a1.b1.c1"), 0);
    EQUAL(config.getInteger("a2.b2.bool1"), 0);
    EQUAL(config.getInteger("a2.b2.bool2"), 0);
    EQUAL(config.getInteger("a2.b2.bool3"), 0);
    EQUAL(config.getInteger("a2.b2.byte1"), 123);
    EQUAL(config.getInteger("a2.b2.byte2"), -123);
    EQUAL(config.getInteger("a2.b2.byte3"), 0);
    EQUAL(config.getInteger("a2.b2.char1"), 0);
    EQUAL(config.getInteger("a2.b2.char2"), 0);
    EQUAL(config.getInteger("a2.b2.char3"), 0);
    EQUAL(config.getInteger("a2.b2.short1"), 2345);
    EQUAL(config.getInteger("a2.b2.short2"), -2345);
    EQUAL(config.getInteger("a2.b2.short3"), 0);
    EQUAL(config.getInteger("a2.b2.int1"), 123456789);
    EQUAL(config.getInteger("a2.b2.int2"), -123456789);
    EQUAL(config.getInteger("a2.b2.int3"), 0);
    EQUAL(config.getInteger("a2.b2.float1"), 1);
    EQUAL(config.getInteger("a2.b2.float2"), -1);
    EQUAL(config.getInteger("a2.b2.float3"), 0);
    EQUAL(config.getInteger("a2.b2.double1"), 987);
    EQUAL(config.getInteger("a2.b2.double2"), -987);
    EQUAL(config.getInteger("a2.b2.double3"), 0);
    EQUAL(config.getInteger("a2.b2.string1"), 0);
    EQUAL(config.getInteger("a2.b2.string2"), 0);
    EQUAL(config.getInteger("a2.b2.string3"), 0);
    EQUAL(config.getInteger("a2.b2.datetime1"), 0);
    EQUAL(config.getInteger("a2.b2.datetime2"), 0);
    EQUAL(config.getInteger("a2.b2.datetime3"), 0);
    EQUAL(config.getInteger("a3.b3"), 0);
  }

  public void getInteger_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getInteger(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getLong_key()
  {
    MSG("キーに対する値をlong値で取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.getLong(""), 0L);
    EQUAL(config.getLong("a1"), 0L);
    EQUAL(config.getLong("a1.b1"), 0L);
    EQUAL(config.getLong("a1.b1.c1"), 0L);
    EQUAL(config.getLong("a2.b2.bool1"), 0L);
    EQUAL(config.getLong("a2.b2.bool2"), 0L);
    EQUAL(config.getLong("a2.b2.bool3"), 0L);
    EQUAL(config.getLong("a2.b2.byte1"), 123L);
    EQUAL(config.getLong("a2.b2.byte2"), -123L);
    EQUAL(config.getLong("a2.b2.byte3"), 0L);
    EQUAL(config.getLong("a2.b2.char1"), 0L);
    EQUAL(config.getLong("a2.b2.char2"), 0L);
    EQUAL(config.getLong("a2.b2.char3"), 0L);
    EQUAL(config.getLong("a2.b2.short1"), 2345L);
    EQUAL(config.getLong("a2.b2.short2"), -2345L);
    EQUAL(config.getLong("a2.b2.short3"), 0L);
    EQUAL(config.getLong("a2.b2.int1"), 123456789L);
    EQUAL(config.getLong("a2.b2.int2"), -123456789L);
    EQUAL(config.getLong("a2.b2.int3"), 0L);
    EQUAL(config.getLong("a2.b2.float1"), 1L);
    EQUAL(config.getLong("a2.b2.float2"), -1L);
    EQUAL(config.getLong("a2.b2.float3"), 0L);
    EQUAL(config.getLong("a2.b2.double1"), 987L);
    EQUAL(config.getLong("a2.b2.double2"), -987L);
    EQUAL(config.getLong("a2.b2.double3"), 0L);
    EQUAL(config.getLong("a2.b2.string1"), 0L);
    EQUAL(config.getLong("a2.b2.string2"), 0L);
    EQUAL(config.getLong("a2.b2.string3"), 0L);
    EQUAL(config.getLong("a2.b2.datetime1"), 0L);
    EQUAL(config.getLong("a2.b2.datetime2"), 0L);
    EQUAL(config.getLong("a2.b2.datetime3"), 0L);
    EQUAL(config.getLong("a3.b3"), 0L);
  }

  public void getLong_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getLong(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getFloat_key()
  {
    MSG("キーに対する値をfloat値で取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.getFloat(""), 0.0f);
    EQUAL(config.getFloat("a1"), 0.0f);
    EQUAL(config.getFloat("a1.b1"), 0.0f);
    EQUAL(config.getFloat("a1.b1.c1"), 0.0f);
    EQUAL(config.getFloat("a2.b2.bool1"), 0.0f);
    EQUAL(config.getFloat("a2.b2.bool2"), 0.0f);
    EQUAL(config.getFloat("a2.b2.bool3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.byte1"), 123.0f);
    EQUAL(config.getFloat("a2.b2.byte2"), -123.0f);
    EQUAL(config.getFloat("a2.b2.byte3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.char1"), 0.0f);
    EQUAL(config.getFloat("a2.b2.char2"), 0.0f);
    EQUAL(config.getFloat("a2.b2.char3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.short1"), 2345.0f);
    EQUAL(config.getFloat("a2.b2.short2"), -2345.0f);
    EQUAL(config.getFloat("a2.b2.short3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.int1"), 123456789.0f);
    EQUAL(config.getFloat("a2.b2.int2"), -123456789.0f);
    EQUAL(config.getFloat("a2.b2.int3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.float1"), 1.23456f);
    EQUAL(config.getFloat("a2.b2.float2"), -1.23456f);
    EQUAL(config.getFloat("a2.b2.float3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.double1"), 987.654321f);
    EQUAL(config.getFloat("a2.b2.double2"), -987.654321f);
    EQUAL(config.getFloat("a2.b2.double3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.string1"), 0.0f);
    EQUAL(config.getFloat("a2.b2.string2"), 0.0f);
    EQUAL(config.getFloat("a2.b2.string3"), 0.0f);
    EQUAL(config.getFloat("a2.b2.datetime1"), 0.0f);
    EQUAL(config.getFloat("a2.b2.datetime2"), 0.0f);
    EQUAL(config.getFloat("a2.b2.datetime3"), 0.0f);
    EQUAL(config.getFloat("a3.b3"), 0.0f);
  }

  public void getFloat_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getFloat(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getDouble_key()
  {
    MSG("キーに対する値をdouble値で取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.getDouble(""), 0.0);
    EQUAL(config.getDouble("a1"), 0.0);
    EQUAL(config.getDouble("a1.b1"), 0.0);
    EQUAL(config.getDouble("a1.b1.c1"), 0.0);
    EQUAL(config.getDouble("a2.b2.bool1"), 0.0);
    EQUAL(config.getDouble("a2.b2.bool2"), 0.0);
    EQUAL(config.getDouble("a2.b2.bool3"), 0.0);
    EQUAL(config.getDouble("a2.b2.byte1"), 123.0);
    EQUAL(config.getDouble("a2.b2.byte2"), -123.0);
    EQUAL(config.getDouble("a2.b2.byte3"), 0.0);
    EQUAL(config.getDouble("a2.b2.char1"), 0.0);
    EQUAL(config.getDouble("a2.b2.char2"), 0.0);
    EQUAL(config.getDouble("a2.b2.char3"), 0.0);
    EQUAL(config.getDouble("a2.b2.short1"), 2345.0);
    EQUAL(config.getDouble("a2.b2.short2"), -2345.0);
    EQUAL(config.getDouble("a2.b2.short3"), 0.0);
    EQUAL(config.getDouble("a2.b2.int1"), 123456789.0);
    EQUAL(config.getDouble("a2.b2.int2"), -123456789.0);
    EQUAL(config.getDouble("a2.b2.int3"), 0.0);
    EQUAL(config.getDouble("a2.b2.float1"), 1.23456);
    EQUAL(config.getDouble("a2.b2.float2"), -1.23456);
    EQUAL(config.getDouble("a2.b2.float3"), 0.0);
    EQUAL(config.getDouble("a2.b2.double1"), 987.654321);
    EQUAL(config.getDouble("a2.b2.double2"), -987.654321);
    EQUAL(config.getDouble("a2.b2.double3"), 0.0);
    EQUAL(config.getDouble("a2.b2.string1"), 0.0);
    EQUAL(config.getDouble("a2.b2.string2"), 0.0);
    EQUAL(config.getDouble("a2.b2.string3"), 0.0);
    EQUAL(config.getDouble("a2.b2.datetime1"), 0.0);
    EQUAL(config.getDouble("a2.b2.datetime2"), 0.0);
    EQUAL(config.getDouble("a2.b2.datetime3"), 0.0);
    EQUAL(config.getDouble("a3.b3"), 0.0);
  }

  public void getDouble_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getDouble(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getBigDecimal_key()
  {
    MSG("キーに対する値をBigDecimalオブジェクトとして取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.getBigDecimal(""), new BigDecimal(0.0));
    EQUAL(config.getBigDecimal("a1"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a1.b1"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a1.b1.c1"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.bool1"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.bool2"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.bool3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.byte1"), new BigDecimal(123.0));
    EQUAL(config.getBigDecimal("a2.b2.byte2"), new BigDecimal(-123.0));
    EQUAL(config.getBigDecimal("a2.b2.byte3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.char1"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.char2"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.char3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.short1"), new BigDecimal(2345.0));
    EQUAL(config.getBigDecimal("a2.b2.short2"), new BigDecimal(-2345.0));
    EQUAL(config.getBigDecimal("a2.b2.short3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.int1"), new BigDecimal(123456789.0));
    EQUAL(config.getBigDecimal("a2.b2.int2"), new BigDecimal(-123456789.0));
    EQUAL(config.getBigDecimal("a2.b2.int3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.float1"), new BigDecimal("1.23456"));
    EQUAL(config.getBigDecimal("a2.b2.float2"), new BigDecimal("-1.23456"));
    EQUAL(config.getBigDecimal("a2.b2.float3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.double1"), new BigDecimal("987.654321"));
    EQUAL(config.getBigDecimal("a2.b2.double2"), new BigDecimal("-987.654321"));
    EQUAL(config.getBigDecimal("a2.b2.double3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.string1"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.string2"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.string3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.datetime1"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.datetime2"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a2.b2.datetime3"), new BigDecimal(0L));
    EQUAL(config.getBigDecimal("a3.b3"), new BigDecimal(0L));
  }

  public void getBigDecimal_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getBigDecimal(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getString_key()
  {
    MSG("キーに対する値を文字列として取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.getString(""), "");
    EQUAL(config.getString("a1"), "xxx1");
    EQUAL(config.getString("a1.b1"), "xxx1.yyy1");
    EQUAL(config.getString("a1.b1.c1"), "xxx1.yyy1.zzz1");
    EQUAL(config.getString("a2.b2.bool1"), "true");
    EQUAL(config.getString("a2.b2.bool2"), "false");
    EQUAL(config.getString("a2.b2.bool3"), "");
    EQUAL(config.getString("a2.b2.byte1"), "123");
    EQUAL(config.getString("a2.b2.byte2"), "-123");
    EQUAL(config.getString("a2.b2.byte3"), "");
    EQUAL(config.getString("a2.b2.char1"), "A");
    EQUAL(config.getString("a2.b2.char2"), "a");
    EQUAL(config.getString("a2.b2.char3"), "");
    EQUAL(config.getString("a2.b2.short1"), "2345");
    EQUAL(config.getString("a2.b2.short2"), "-2345");
    EQUAL(config.getString("a2.b2.short3"), "");
    EQUAL(config.getString("a2.b2.int1"), "123456789");
    EQUAL(config.getString("a2.b2.int2"), "-123456789");
    EQUAL(config.getString("a2.b2.int3"), "");
    EQUAL(config.getString("a2.b2.float1"), "1.23456");
    EQUAL(config.getString("a2.b2.float2"), "-1.23456");
    EQUAL(config.getString("a2.b2.float3"), "");
    EQUAL(config.getString("a2.b2.double1"), "987.654321");
    EQUAL(config.getString("a2.b2.double2"), "-987.654321");
    EQUAL(config.getString("a2.b2.double3"), "");
    EQUAL(config.getString("a2.b2.string1"), "Abcdefg");
    EQUAL(config.getString("a2.b2.string2"), "あいうえお");
    EQUAL(config.getString("a2.b2.string3"), "");
    EQUAL(config.getString("a2.b2.datetime1"), "2011-01-01");
    EQUAL(config.getString("a2.b2.datetime2"), "2011-12-31");
    EQUAL(config.getString("a2.b2.datetime3"), "");
    EQUAL(config.getString("a3.b3"), "AB3-1");
  }

  public void getString_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getString(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }

  public void getNumberString_key()
  {
    MSG("キーに対する値を数値文字列として取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();

    EQUAL(config.getNumberString("", "#,###"), "");
    EQUAL(config.getNumberString("a1", "#,###"), "");
    EQUAL(config.getNumberString("a1.b1", "#,###"), "");
    EQUAL(config.getNumberString("a1.b1.c1", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.bool1", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.bool2", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.bool3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.byte1", "#,###"), "123");
    EQUAL(config.getNumberString("a2.b2.byte2", "#,###"), "-123");
    EQUAL(config.getNumberString("a2.b2.byte3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.char1", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.char2", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.char3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.short1", "#,###"), "2,345");
    EQUAL(config.getNumberString("a2.b2.short2", "#,###"), "-2,345");
    EQUAL(config.getNumberString("a2.b2.short3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.int1", "#,###"), "123,456,789");
    EQUAL(config.getNumberString("a2.b2.int2", "#,###"), "-123,456,789");
    EQUAL(config.getNumberString("a2.b2.int3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.float1", "#,###"), "1");
    EQUAL(config.getNumberString("a2.b2.float2", "#,###"), "-1");
    EQUAL(config.getNumberString("a2.b2.float3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.double1", "#,###"), "988");
    EQUAL(config.getNumberString("a2.b2.double2", "#,###"), "-988");
    EQUAL(config.getNumberString("a2.b2.double3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.string1", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.string2", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.string3", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.datetime1", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.datetime2", "#,###"), "");
    EQUAL(config.getNumberString("a2.b2.datetime3", "#,###"), "");
    EQUAL(config.getNumberString("a3.b3", "#,###"), "");
  }

  public void getNumberString_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getNumberString(null, "#,###");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }

    try {
      config.getNumberString("key", null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void getDateTimeString_key()
  {
    MSG("キーに対する値を日時文字列として取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();
    config.setDateTimeFormat("yyyy-MM-dd");

    EQUAL(config.getDateTimeString("", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a1.b1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a1.b1.c1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.bool1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.bool2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.bool3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.byte1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.byte2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.byte3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.char1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.char2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.char3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.short1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.short2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.short3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.int1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.int2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.int3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.float1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.float2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.float3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.double1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.double2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.double3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.string1", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.string2", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.string3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a2.b2.datetime1", "yyyy/MM/dd"), "2011/01/01");
    EQUAL(config.getDateTimeString("a2.b2.datetime2", "yyyy/MM/dd"), "2011/12/31");
    EQUAL(config.getDateTimeString("a2.b2.datetime3", "yyyy/MM/dd"), "");
    EQUAL(config.getDateTimeString("a3.b3", "yyyy/MM/dd"), "");
  }

  public void getDateTimeString_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getDateTimeString(null, "yyyy-MM-dd");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }

    try {
      config.getDateTimeString("key", null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void getDateTime_key()
  {
    MSG("キーに対する値を日時オブジェクトとして取得するメソッドの確認。");

    QueryConfig config = QueryConfig.getInstance();
    config.setDateTimeFormat("yyyy-MM-dd");

    EQUAL(config.getDateTime(""), new DateTime(0L));
    EQUAL(config.getDateTime("a1"), new DateTime(0L));
    EQUAL(config.getDateTime("a1.b1"), new DateTime(0L));
    EQUAL(config.getDateTime("a1.b1.c1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.bool1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.bool2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.bool3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.byte1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.byte2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.byte3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.char1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.char2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.char3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.short1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.short2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.short3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.int1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.int2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.int3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.float1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.float2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.float3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.double1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.double2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.double3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.string1"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.string2"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.string3"), new DateTime(0L));
    EQUAL(config.getDateTime("a2.b2.datetime1"),new DateTime(2011,1,1));
    EQUAL(config.getDateTime("a2.b2.datetime2"),new DateTime(2011,12,31));
    EQUAL(config.getDateTime("a2.b2.datetime3"), new DateTime(0L));
    EQUAL(config.getDateTime("a3.b3"), new DateTime(0L));
  }

  public void getDateTime_key_Null()
  {
    MSG("引数にヌルが指定された場合。");

    QueryConfig config = QueryConfig.getInstance();

    try {
      config.getDateTime(null);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    }
  }
}
