/*
 * QueryConnection class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.table.Table;

/**
 * クエリを実行する対象への接続を表すコネクション・クラスのインターフェイス。
 * <br>
 * 接続を操作するためのメソッド{@link #open()}、{@link #commit()}、{@link
 * #rollback()}、{@link #close()}を宣言している。
 * これらのメソッドは{@link QueryTransaction}オブジェクトから呼び出される。
 * <br>
 * またこのクラスは、クエリを実行する{@link Query}オブジェクトを取得するための
 * メソッド{@link #getQuery(QueryResource)}、そのメソッドの中で{@link
 * QueryContent}オブジェクトに格納された実行内容を実行する{@link
 * #executeContent(QueryContent)}メソッド、実行したクエリの結果を格納する{@link
 * QueryResultList}オブジェクトを取得するためのメソッド{@link
 * #getTransactionQueryResultList()}を宣言している。
 *
 * @author 佐藤隆之
 * @version $Id: QueryConnection.java,v 1.8 2011-09-18 16:15:16 tayu Exp $
 */
public interface QueryConnection
{
  /**
   * 接続先を識別するコネクションIDを取得する。
   */
  String getId();

  /**
   * 接続を開始する。
   */
  void open();

  /**
   * 接続先への更新を確定する。
   */
  void commit();

  /**
   * 接続先への更新を取り消す。
   */
  void rollback();

  /**
   * 接続を終了する。
   */
  void close();

  /**
   * このオブジェクトが示す接続先へのコネクションが閉じられたかどうかを確認
   * する。
   *
   * @return コネクションが閉じられている場合は<tt>true</tt>を返す。
   */
  boolean isClosed();

  /**
   * クエリ実行のタイムアウトのリミット時刻をミリ秒単位で取得する。
   * <br>
   * 戻り値が負値の場合は、リミット時刻が指定されていないものとする。
   *
   * @return タイムアウトのリミット時刻[msec]。
   */
  long getLimitTimeMillis();

  /**
   * クエリ・オブジェクトを作成する。
   * <br>
   * 引数に指定されたリソースと、内部に保持するコネクション・オブジェクトを
   * 使ってクエリを実行する{@link Query}オブジェクトを作成して返す。
   *
   * @param resource クエリの実行内容の元となる情報を格納する{@link
   *          QueryResource}オブジェクト。
   * @return {@link Query}オブジェクト。
   */
  Query getQuery(QueryResource resource);

  /**
   * 引数の{@link QueryContent}オブジェクトに格納されたクエリの実行内容を、
   * このオブジェクトが示す接続先に対して実行する。
   *
   * @param cont {@link QueryContent}オブジェクト。
   * @param table クエリを実行した結果として取得されたデータを格納する{@link
   *          Table}オブジェクト。
   * @return クエリを実行して処理されたデータの件数。
   * @throws ReasonedException クエリの実行に失敗した場合。
   */
  int executeContent(QueryContent cont, Table<String,Object> table)
    throws ReasonedException;

  /**
   * 引数の{@link QueryContent}オブジェクトに格納されたクエリの実行内容を、
   * このオブジェクトが示す接続先に対して実行する。
   *
   * @param cont {@link QueryContent}オブジェクト。
   * @return クエリを実行して処理されたデータの件数。
   * @throws ReasonedException クエリの実行に失敗した場合。
   */
  int executeContent(QueryContent cont) throws ReasonedException;

  /**
   * このコネクションが属するトランザクションのクエリ実行結果リストを取得する。
   *
   * @return {@link QueryResultList}オブジェクト。
   */
  QueryResultList getTransactionQueryResultList();
}
