/*
 * QueryResource class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * クエリの実行内容の元となる情報を格納するリソース・クラス。
 * <br>
 * 引数として渡された入力パラメータと以前に実行されたクエリの結果リストを使って
 * クエリの実行内容を格納した{@link QueryContent}オブジェクトを作成するメソッド
 * を宣言している。
 * <br>
 * また、このオブジェクトから作成可能な全ての実行内容のクエリIDを取得するための
 * メソッドも宣言している。
 *
 * @author 佐藤隆之
 * @version 1.0, 2011/09/12.
 */
public interface QueryResource
{
  /** このクラスで発生しうるエラーを示す列挙型。 */
  enum Error {
    /** 指定されたクエリIDが見つからなかった場合。 */
    QueryIdNotFound,

    /** クエリの実行内容の作成に失敗した場合。 */
    FailToCreateContent,
  }

  /**
   * 指定されたクエリIDに対応づけられたクエリの実行内容を格納した{@link
   * QueryContent}オブジェクトを作成する。
   *
   * @param queryId クエリID。
   * @param inputMap 入力パラメータ。
   * @param rsltLst クエリの実行結果を格納するリスト。
   * @return {@link QueryContent}オブジェクト。
   * @throws ReasonedException 指定されたクエリIDが存在しない場合、又はクエリの
   *           実行内容の作成に失敗した場合。
   */
  QueryContent createContent(
    String queryId, Map<String,Object> inputMap, QueryResultList rsltLst
  ) throws ReasonedException;

  /**
   * このオブジェクトが作成可能なクエリ実行内容に対応づけられた全てのクエリID
   * を定義順に取得する。
   *
   * @return クエリIDを格納したリスト。
   */
  List<String> listAllQueryIds();
}
