/*
 * QueryResult class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.table.Table;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.io.Serializable;

/**
 * クエリの実行結果を表すクラス。
 * <br>
 * クエリを実行した結果状況（実行成否、処理件数、処理時間、実行に失敗した場合
 * の例外オブジェクト）と、結果データを保持する。
 * さらに、実行したクエリの内容を保持する{@link QueryContent}オブジェクトを
 * 属性に持つ。
 *
 * @author 佐藤隆之
 * @version $Id: QueryResult.java,v 1.6 2011-09-13 14:28:24 tayu Exp $
 */
public class QueryResult implements Serializable
{
  /** シリアル・バージョン番号。 */
  private static final long serialVersionUID = 8309940887092598304L;

  /** クエリID。 */
  private final String queryId;

  /** コネクションID。 */
  private final String connId;

  /** クエリの実行成否フラグ。 */
  private boolean isSuccess;

  /** クエリの処理件数。 */
  private int resultCount = 0;

  /** クエリの処理時間。 */
  private long spentTimeMillis = 0L;

  /** クエリの実行が失敗した時に発生した例外オブジェクト。 */
  private ReasonedException exception = null;

  /** クエリの結果を格納するテーブル・オブジェクト。 */
  private Table<String,Object> resultTable = null;

  /** クエリの内容を保持するオブジェクト。 */
  private QueryContent content = null;

  /**
   * コネクションIDとクエリIDを引数にとるコンストラクタ。
   */
  public QueryResult(String connId, String queryId)
  {
    assert (connId != null  && queryId != null) : 
      (connId  == null) ? "@param:connId is null." :
      (queryId == null) ? "@param:connId is null." : "";

    this.queryId = queryId;
    this.connId = connId;
  }

  /**
   * クエリIDを取得する。
   *
   * @return クエリID。
   */
  public String getQueryId()
  {
    return this.queryId;
  }

  /**
   * コネクションIDを取得する。
   *
   * @return コネクションIDを取得する。
   */
  public String getConnectionId()
  {
    return this.connId;
  }

  /**
   * クエリの内容を保持するオブジェクトを設定する。
   *
   * @param cont クエリの内容を保持するオブジェクト。
   */
  public void setContent(QueryContent cont)
  {
    assert (cont != null) : "@param:cont is null.";
    this.content = cont;
  }

  /**
   * クエリの内容を保持するオブジェクトを取得する。
   *
   * @return クエリの内容を保持するオブジェクト。
   */
  public QueryContent getContent()
  {
    return this.content;
  }

  /**
   * クエリ実行の成否を設定する。
   *
   * @param b クエリ実行が成功した場合は<tt>true</tt>を設定する。
   */
  protected void setSuccess(boolean b)
  {
    this.isSuccess = b;
  }

  /**
   * クエリの実行の成否を取得する。
   *
   * @return クエリの実行が成功した場合は<tt>true</tt>を返す。
   */
  public boolean isSuccess()
  {
    return this.isSuccess;
  }

  /**
   * クエリの処理件数を設定する。
   *
   * @param resultCount クエリの処理件数。
   */
  protected void setResultCount(int resultCount)
  {
    assert (resultCount >= 0) : "@param:resultCount is negative.";
    this.resultCount = resultCount;
  }

  /**
   * クエリの処理件数を取得する。
   *
   * @return クエリの処理件数。
   */
  public int getResultCount()
  {
    return this.resultCount;
  }

  /**
   * クエリの処理時間をミリ秒単位で設定する。
   *
   * @param spentTime クエリの処理時間[msec]。
   */
  protected void setSpentTimeMillis(long spentTime)
  {
    assert (spentTime >= 0L) : "@param:spentTime is negative.";
    this.spentTimeMillis = spentTime;
  }

  /**
   * クエリの処理時間をミリ秒単位で取得する。
   *
   * @return クエリの処理時間[msec]。
   */
  public long getSpentTimeMillis()
  {
    return this.spentTimeMillis;
  }

  /**
   * クエリの実行が失敗した時の例外オブジェクトを設定する。
   * <br>
   * クエリの実行が成功した場合はヌルを設定する。
   *
   * @param exc クエリの実行が失敗した時の例外オブジェクト。
   */
  protected void setException(Exception exc)
  {
    if (exc == null) {
      this.exception = null;
    }
    else if (exc instanceof ReasonedException) {
      this.exception = ReasonedException.class.cast(exc);
    }
    else if (exc instanceof ReasonedRuntimeException) {
      ReasonedRuntimeException rre = ReasonedRuntimeException.class.cast(exc);
      this.exception = new ReasonedException(rre.getReason(), rre);
      this.exception.setStackTrace(exc.getStackTrace());
    }
    else {
      this.exception = new ReasonedException(Query.Error.FailToExecute, exc);
      this.exception.setStackTrace(exc.getStackTrace());
    }
  }

  /**
   * クエリの実行が失敗した時の例外オブジェクトを取得する。
   * <br>
   * クエリの実行が成功した場合はヌルを返す。
   *
   * @return クエリの実行が失敗した時の例外オブジェクト。
   */
  public ReasonedException getReasonedException()
  {
    return this.exception;
  }
  
  /**
   * 結果テーブルを設定する。
   *
   * @param table 結果テーブル。
   */
  protected void setResultTable(Table<String,Object> table)
  {
    this.resultTable = table;
  }

  /**
   * 結果テーブルを取得する。
   *
   * @return 結果テーブル。
   */
  public Table<String,Object> getResultTable()
  {
    return this.resultTable;
  }
}
