/*
 * QueryResultList class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import java.io.Serializable;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;

/**
 * 複数のクエリ結果オブジェクトを格納するためのクラス。
 * <br>
 * 複数のクエリを実行する{@link Query#executeSet(String[], Map)}
 * などのメソッドの戻り値として、またトランザクションの中で実行された複数の
 * クエリの実行履歴として使用される。
 * <br>
 * {@link #isSuccess()}メソッドは、格納されている結果オブジェクトの中に1つでも
 * 失敗したものがあれば<tt>false</tt>を返し、全てが成功の場合に限り
 * <tt>true</tt>を返す。
 * <br>
 * {@link #getSpentTimeMillis()}メソッドは、デフォルトでは格納されている
 * 結果オブジェクトの処理時間の合計を返すが、{@link #setSpentTimeMillis(long)}
 * メソッドを使って処理時間を設定した場合はその値を返す。
 * <br>
 * また、格納されている結果オブジェクトを様々な方法で取り出すメソッドを用意
 * している：
 * 最初に格納されている結果オブジェクトを取得するメソッド、最後に格納されている
 * 結果オブジェクトを取得するメソッド、クエリIDをキーに取得するメソッド、
 * 例外オブジェクトをキーに取得するメソッド等がある。
 *
 * @author 佐藤隆之。
 * @version $Id: QueryResultList.java,v 1.6 2011-09-18 16:15:17 tayu Exp $
 */
public class QueryResultList implements Serializable
{
  /** シリアル・バージョン番号。 */
  private static final long serialVersionUID = 7570497809387060417L;

  /** 結果オブジェクトを順番に格納するリスト。 */
  private final List<QueryResult> resultLst = new LinkedList<QueryResult>();

  /** クエリIDと結果オブジェクトの対応付けを保持するマップ。 */
  private final Map<String,List<QueryResult>> qid2RsltMap =
    new HashMap<String,List<QueryResult>>();

  /** コネクションIDと結果オブジェクトの対応付けを保持するマップ。 */
  private final Map<String,List<QueryResult>> cid2RsltMap =
    new HashMap<String,List<QueryResult>>();

  /** 例外オブジェクトと結果オブジェクトの対応付けを保持するマップ。 */
  private final Map<ReasonedException,QueryResult> exc2RsltMap =
    new HashMap<ReasonedException,QueryResult>();

  /** 所要時間[msec]。 */
  private long spentTimeMillis = -1L;

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryResultList()
  {}

  /**
   * クエリの結果オブジェクトを追加する。
   *
   * @param result 結果オブジェクト。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addResult(QueryResult result)
  {
    assert (result != null) : "@param:result is null.";

    this.resultLst.add(result);

    entryMappedList(this.qid2RsltMap, result.getQueryId(), result);
    entryMappedList(this.cid2RsltMap, result.getConnectionId(), result);

    if (result.getReasonedException() != null)
      exc2RsltMap.put(result.getReasonedException(), result);
  }

  /**
   * 指定されたキーに対応づけられたリストに結果オブジェクトを登録する。
   *
   * @param map 登録先のマップ。
   * @param key キー。
   * @param result 登録される結果オブジェクト。 
   */
  protected void entryMappedList(
    Map<String,List<QueryResult>> map, String key, QueryResult result)
  {
    List<QueryResult> lst = map.get(key);
    if (lst == null) {
      lst = new LinkedList<QueryResult>();
      map.put(key, lst);
    }
    lst.add(result);
  }

  /**
   * 複数のクエリが全て成功したかどうかを取得する。
   * <br>
   * このオブジェクトに格納されている複数の結果オブジェクトが全て成功かどうかを
   * 調べる。
   * このオブジェクトに一つでも失敗の結果オブジェクトが含まれている場合は
   * <tt>false</tt>を返す。
   * <br>
   * クエリ結果が一つも格納されていない場合は<tt>true</tt>を返す。
   *
   * @return このオブジェクトに格納されている結果が全て成功の場合は
   *         <tt>true</tt>を返す。
   */
  public boolean isSuccess()
  {
    for (QueryResult rslt : this.resultLst) {
      if (! rslt.isSuccess())
        return false;
    }
    return true;
  }

  /**
   * 複数のクエリを実行するのに要した時間を取得する。
   * <br>
   * このオブジェクトに格納されている複数の結果オブジェクトの実行時間の合計を
   * 算出して返す。
   * <br>
   * 但し、{@link #setSpentTimeMillis(long)}メソッドを使って処理時間を設定した
   * 場合は、その値を返す。
   * <br>
   * 時間の単位はミリ秒である。
   *
   * @return 複数のクエリを実行するのに要した時間。
   */
  public long getSpentTimeMillis()
  {
    if (this.spentTimeMillis >= 0L) {
      return this.spentTimeMillis;
    }
    else {
      long tm = 0L;
      for (QueryResult rslt : this.resultLst) {
        tm += rslt.getSpentTimeMillis();
      }
      return tm;
    }
  }

  /**
   * 複数のクエリを全て実行するのに要した時間を設定する。
   * <br>
   * もし引数に負の値を設定した場合は、例外をスローする。
   *
   * @param millis 複数のクエリを実行するのに要した時間[msec]。
   * @throws IllegalArgumentException 引数が負値の場合。
   */
  public void setSpentTimeMillis(long millis)
  {
    if (millis < 0L)
      throw new IllegalArgumentException("@param:millis is negative.");

    this.spentTimeMillis = millis;
  }

  /**
   * 複数のクエリを全て実行するのに要した時間をリセットする。
   * <br>
   * このメソッドを実行することにより、過去に{@link #setSpentTimeMillis(long)}
   * メソッドを使って設定した所要時間をリセットし、{@link #getSpentTimeMillis()}
   * メソッドがこのオブジェクトに格納されて結果オブジェクトの所要時間の合計を
   * 返すようにする。
   */
  public void resetSpentTimeMillis()
  {
    this.spentTimeMillis = -1L;
  }

  /**
   * このオブジェクトに格納されている結果オブジェクトの数を取得する。
   *
   * @return このオブジェクトに格納されている結果オブジェクトの数。
   */
  public int countResults()
  {
    return this.resultLst.size();
  }

  /**
   * このオブジェクトに格納されている全ての結果オブジェクトのリストを取得する。
   *
   * @return 結果オブジェクトを格納したリスト。
   */
  public List<QueryResult> getAllResults()
  {
    return Collections.unmodifiableList(this.resultLst);
  }

  /**
   * このオブジェクトに格納されている最初の結果オブジェクトを取得する。
   * <br>
   * 格納されている結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @return このオブジェクトに格納されている最初の結果オブジェクト。
   */
  public QueryResult getFirstResult()
  {
    if (this.resultLst.isEmpty())
      return null;

    return this.resultLst.get(0);
  }

  /**
   * このオブジェクトに格納されている最後の結果オブジェクトを取得する。
   * <br>
   * 格納されている結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @return このオブジェクトに格納されている最後の結果オブジェクト。
   */
  public QueryResult getLastResult()
  {
    if (this.resultLst.isEmpty())
      return null;

    return this.resultLst.get(this.resultLst.size() -1);
  }

  /**
   * 指定されたコネクションIDに対する結果オブジェクトのリストを取得する。
   * <br>
   * 該当する結果オブジェクトが存在しない場合は空のリストを返す。
   *
   * @param connId コネクションID。
   * @return 指定されたコネクションIDに対する結果オブジェクトのリスト。
   */
  public List<QueryResult> getResultsByConnectionId(String connId)
  {
    List<QueryResult> lst = this.cid2RsltMap.get(connId);
    if (lst == null) {
      lst = Collections.emptyList();
      return lst;
    }

    return Collections.unmodifiableList(lst);
  }

  /**
   * 指定されたコネクションIDに対する最初の結果オブジェクトを取得する。
   * <br>
   * 該当する結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @param connId コネクションID。
   * @return 指定されたコネクションIDに対する結果オブジェクト。
   */
  public QueryResult getFirstResultByConnectionId(String connId)
  {
    List<QueryResult> lst = this.cid2RsltMap.get(connId);
    if (lst == null || lst.isEmpty())
      return null;

    return lst.get(0);
  }

  /**
   * 指定されたコネクションIDに対する結果オブジェクトのリストを取得する。
   * <br>
   * 該当する結果オブジェクトが存在しない場合は空のリストを返す。
   *
   * @param connId コネクションID。
   * @return 指定されたコネクションIDに対する結果オブジェクト。
   */
  public QueryResult getLastResultByConnectionId(String connId)
  {
    List<QueryResult> lst = this.cid2RsltMap.get(connId);
    if (lst == null || lst.isEmpty())
      return null;

    return lst.get(lst.size() -1);
  }

  /**
   * 指定されたクエリIDに対する結果オブジェクトのリストを取得する。
   * <br>
   * 該当する結果オブジェクトが存在しない場合は空のリストを返す。
   *
   * @param queryId クエリID。
   * @return 指定されたクエリIDに対する結果オブジェクトのリスト。
   */
  public List<QueryResult> getResultsByQueryId(String queryId)
  {
    List<QueryResult> lst = this.qid2RsltMap.get(queryId);
    if (lst == null) {
      lst = Collections.emptyList();
      return lst;
    }

    return Collections.unmodifiableList(lst);
  }

  /**
   * 指定されたクエリIDに対する最初の結果オブジェクトを取得する。
   * <br>
   * 該当する結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @param queryId クエリID。
   * @return 指定されたクエリIDに対する結果オブジェクト。
   */
  public QueryResult getFirstResultByQueryId(String queryId)
  {
    List<QueryResult> lst = this.qid2RsltMap.get(queryId);
    if (lst == null || lst.isEmpty())
      return null;

    return lst.get(0);
  }

  /**
   * 指定されたクエリIDに対する結果オブジェクトのリストを取得する。
   * <br>
   * 該当する結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @param queryId クエリID。
   * @return 指定されたクエリIDに対する結果オブジェクト。
   */
  public QueryResult getLastResultByQueryId(String queryId)
  {
    List<QueryResult> lst = this.qid2RsltMap.get(queryId);
    if (lst == null || lst.isEmpty())
      return null;

    return lst.get(lst.size() -1);
  }

  /**
   * 指定された例外オブジェクトを持った結果オブジェクトのリストを取得する。
   * <br>
   * 該当する結果オブジェクトが存在しない場合はヌルを返す。
   * また、引数の例外オブジェクトがヌルの場合もヌルを返す。
   *
   * @param exc 理由付き例外オブジェクト。
   * @return 引数の例外オブジェクトを持つ結果オブジェクト。
   */
  public QueryResult getResultByException(ReasonedException exc)
  {
    return this.exc2RsltMap.get(exc);
  }

  /**
   * このオブジェクトをスレッド・セーフ化した{@link QueryResultList}
   * オブジェクトを取得する。
   *
   * @return このオブジェクトをスレッド・セーフ化した{@link QueryResultList}
   *         オブジェクト。
   */
  QueryResultList toThreadSafe()
  {
    return new ThreadSafeQueryResultList(this);
  }


  /**
   * コンストラクタで指定された{@link QueryResultList}オブジェクトをスレッド・
   * セーフ化した結果リスト・オブジェクト。
   */
  protected static class ThreadSafeQueryResultList extends QueryResultList
  {
    /** シリアル・バージョン番号。 */
    private static final long serialVersionUID = -1759409178810650333L;

    /** スレッド・セーフ化される{@link QueryResultList}オブジェクト。 */
    private final QueryResultList inner;

    /**
     * スレッド・セーフ化する{@link QueryResultList}オブジェクトを引数にとる
     * コンストラクタ。
     *
     * @param rsltLst スレッド・セーフ化される{@link QueryResultList}
     *          オブジェクト。
     */
    public ThreadSafeQueryResultList(QueryResultList rsltLst)
    {
      this.inner = rsltLst;
    }

    /**
     * スレッド・セーフ化される{@link QueryResultList}オブジェクトを取得する。
     *
     * @return スレッド・セーフ化される{@link QueryResultList}オブジェクト。
     */
    protected QueryResultList getInner()
    {
      return this.inner;
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void addResult(QueryResult rslt)
    {
      synchronized (this.inner) {
        getInner().addResult(rslt);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized boolean isSuccess()
    {
      synchronized (this.inner) {
        return getInner().isSuccess();
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized long getSpentTimeMillis()
    {
      synchronized (this.inner) {
        return getInner().getSpentTimeMillis();
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void setSpentTimeMillis(long millis)
    {
      synchronized (this.inner) {
        getInner().setSpentTimeMillis(millis);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void resetSpentTimeMillis()
    {
      synchronized (this.inner) {
        getInner().resetSpentTimeMillis();
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized int countResults()
    {
      synchronized (this.inner) {
        return getInner().countResults();
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized List<QueryResult> getAllResults()
    {
      synchronized (this.inner) {
        return getInner().getAllResults();
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized QueryResult getFirstResult()
    {
      synchronized (this.inner) {
        return getInner().getFirstResult();
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized QueryResult getLastResult()
    {
      synchronized (this.inner) {
        return getInner().getLastResult();
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized List<QueryResult> getResultsByConnectionId(
      String connId)
    {
      synchronized (this.inner) {
        return getInner().getResultsByConnectionId(connId);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized QueryResult getFirstResultByConnectionId(String connId)
    {
      synchronized (this.inner) {
        return getInner().getFirstResultByConnectionId(connId);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized QueryResult getLastResultByConnectionId(String connId)
    {
      synchronized (this.inner) {
        return getInner().getLastResultByConnectionId(connId);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized List<QueryResult> getResultsByQueryId(String queryId)
    {
      synchronized (this.inner) {
        return getInner().getResultsByQueryId(queryId);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized QueryResult getFirstResultByQueryId(String queryId)
    {
      synchronized (this.inner) {
        return getInner().getFirstResultByQueryId(queryId);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized QueryResult getLastResultByQueryId(String queryId)
    {
      synchronized (this.inner) {
        return getInner().getLastResultByQueryId(queryId);
      }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized QueryResult getResultByException(ReasonedException exc)
    {
      synchronized (this.inner) {
       return getInner().getResultByException(exc);
      }
    }

    /**
     * {@inheritDoc}
     */
    QueryResultList toThreadSafe()
    {
      return this;
    }
  }
}
