/*
 * QueryTransaction class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;

/**
 * ひとまとまりの処理を表すトランザクション・クラスのインターフェイス。
 * <br>
 * {@link #begin()}メソッドの実行から{@link #end()}メソッドの実行までを一つの 
 * トランザクションとして、その間に実行されたクエリによる各接続先への更新の
 * 確定または取消をまとめて実行する。
 *
 * @author 佐藤隆之
 * @version $Id: QueryTransaction.java,v 1.5 2011-09-18 15:59:39 tayu Exp $
 */
public interface QueryTransaction
{
  /** トランザクションの状態を示す列挙型。 */
  enum State {
    /** トランザクションが作成されたという状態を示す列挙型の値。 */
    Created,

    /** トランザクションが開始処理中という状態を示す列挙型の値。 */
    Begining,

    /** トランザクションが開始されたという状態を示す列挙型の値。 */
    Begined,

    /** トランザクションが確定処理中という状態を示す列挙型の値。 */
    Committing,

    /** トランザクションが確定されたという状態を示す列挙型の値。 */
    Committed,

    /** トランザクションが取消処理中という状態を示す列挙型の値。 */
    Rollbacking,

    /** トランザクションが取消されたという状態を示す列挙型の値。 */
    Rollbacked,

    /** トランザクションが終了処理中という状態を示す列挙型の値。 */
    Ending,

    /** トランザクションが終了されたという状態を示す列挙型の値。 */
    Ended,
  }

  /**
   * このインターフェイスをインプリメントしたクラスで発生しうるエラーの列挙型。
   */
  enum Error {
    /**
     * ある操作を実行しようとしたときに、トランザクションがあるべき状態で
     * なかったことを示す列挙型の値。
     */
    IllegalState,

    /**
     * 指定されたコネクションIDに対応づけられたコネクション・オブジェクトが
     * 見つからなかったことを示す列挙型の値。
     */
    ConnectionNotFound,
  }

  /**
   * トランザクションの開始処理を実行する。
   * <br>
   * トランザクションのタイムアウト時間は、このオブジェクトに設定されている
   * デフォルトのタイムアウト時間を使用する。
   * デフォルトのタイムアウト時間が負の値をとる場合は、タイムアウトの制限は
   * 行われない。
   */
  void begin();

  /**
   * タイムアウト時間を指定して、トランザクションの開始処理を実行する。
   * <br>
   * このオブジェクトが保持しているデフォルトのタイムアウト時間は変更しない。
   * 引数に負値を指定した場合は、タイムアウト時間の指定がないものとする。
   *
   * @param timeout タイムアウト時間[sec]。
   */
  void begin(long timeout);

  /**
   * トランザクション内のクエリの実行結果の確定処理を実行する。
   */
  void commit();

  /**
   * トランザクション内のクエリの実行結果の取消処理を実行する。
   */
  void rollback();

  /**
   * トランザクションを終了する。
   */
  void end();

  /**
   * トランザクションの状態を取得する。
   *
   * @return トランザクションの状態。
   */
  State getState();

  /**
   * トランザクションの開始時刻を取得する。
   *
   * @return トランザクションの開始時刻[msec]。
   */
  long getBeginTimeMillis();

  /**
   * トランザクションのタイムアウトのリミット時刻を取得する。
   *
   * @return トランザクションのタイムアウトのリミット時刻[msec]。
   */
  long getLimitTimeMillis();

  /**
   * 指定されたコネクションIDに対応づけられたコネクション・オブジェクトを取得
   * する。
   *
   * @param connId 接続先を示すコネクションID。
   * @return コネクション・オブジェクト。
   * @throws ReasonedException コネクション・オブジェクトの取得に失敗した場合。
   */
  QueryConnection getConnection(String connId) throws ReasonedException;

  /**
   * 指定されたコネクションIDに対応づけるコネクションのファクトリ・オブジェクト
   * を設定する。
   *
   * @param connId 接続先を示すコネクションID。
   * @param factory コネクションのファクトリ・オブジェクト。
   */
  void addConnectionFactory(String connId, QueryConnectionFactory factory);

  /**
   * このトランザクションの中で実行されたクエリの実行結果を格納する{@link
   * QueryResultList}オブジェクトを取得する。
   *
   * @return {@link QueryResultList}オブジェクト。
   */
  QueryResultList getQueryResultList();
}
