/*
 * QueryTransactionManager class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.util.Map;
import java.util.HashMap;

/**
 * トランザクションの作成や取得、破棄を管理するためのクラス。
 * <br>
 *
 * @author 佐藤隆之
 * @version $Id: QueryTransactionManager.java,v 1.9 2011-09-18 15:59:39 tayu Exp $
 */
public class QueryTransactionManager
{
  /** このクラスで発生しうるエラーの列挙型。 */
  public enum Error {
    /**
     * スレッド・ローカルなトランザクション・オブジェクトの作成に失敗したことを
     * 示す列挙値。
     */
    ThreadLocalFailToCreate,

    /** スレッド・ローカルなトランザクションが存在しないことを示す列挙値。 */
    ThreadLocalNotExist,

    /**
     * スレッド・ローカルなトランザクションの作成時に、終了していない別の
     * トランザクションが既に存在することを示す列挙値。
     */
    ThreadLocalAlreadyExists,

    /**
     * スレッド・セーフなトランザクション・オブジェクトの作成に失敗したことを
     * 示す列挙値。
     */
    ThreadSafeFailToCreate,

    /** スレッド・セーフなトランザクションが存在しないことを示す列挙値。 */
    ThreadSafeNotExist,

    /**
     * スレッド・セーフなトランザクションの作成時に、同じキーに結びつけられた
     * 終了していない別のトランザクションが既に存在することを示す列挙値。
     */
    ThreadSafeAlreadyExists,
  }

  /**
   * スレッド・ローカルなトランザクション・オブジェクトを格納する{@link
   * ThreadLocal}オブジェクト。
   */
  private final static ThreadLocal<QueryTransaction> THREAD_LOCAL =
    new ThreadLocal<QueryTransaction>();

  /**
   * スレッド・セーフなトランザクション・オブジェクトを格納するためのマップ・
   * オブジェクト。
   */
  private final static Map<String,QueryTransaction> TRANSACTION_MAP =
    new HashMap<String,QueryTransaction>();

  /**
   * デフォルト・コンストラクタ。
   * <br>
   * このクラスはインスタンス化して使用することはないため、アクセス指定を
   * <tt>private</tt>に指定している。
   */
  private QueryTransactionManager()
  {}

  /**
   * デフォルト・トランザクション・クラスのインスタンスをスレッド・ローカル化
   * したトランザクション・オブジェクトを作成する。
   *
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction createThreadLocalTransaction()
    throws ReasonedRuntimeException
  {
    return createThreadLocalTransaction(DefaultQueryTransaction.class);
  }

  /**
   * 指定されたトランザクション・クラスのインスタンスをスレッド・ローカル化した
   * トランザクション・オブジェクトを作成する。
   *
   * @param transactionClass インスタンスを作成するトランザクション・クラス。
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   * @throws ReasonedRuntimeException スレッド・ローカルなトランザクションが
   *           既に存在する場合、又はスレッド・ローカルなトランザクションの
   *           作成に失敗した場合。
   */
  public static QueryTransaction createThreadLocalTransaction(
    Class<? extends QueryTransaction> transactionClass
  ) throws ReasonedRuntimeException
  {
    assert (transactionClass != null) : "@param:transactionClass is null.";

    QueryTransaction tr = THREAD_LOCAL.get();
    if (tr != null && tr.getState() != QueryTransaction.State.Ended) {
      throw new ReasonedRuntimeException(
        QueryTransactionManager.Error.ThreadLocalAlreadyExists,
        tr.getClass().getName());
    }

    try {
      THREAD_LOCAL.set(transactionClass.newInstance());
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(
        QueryTransactionManager.Error.ThreadLocalFailToCreate,
        transactionClass.getName(), e);
    }

    return new ThreadLocalTransaction();
  }


  /**
   * スレッド・ローカルなトランザクション・オブジェクトを作成する。
   *
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction getThreadLocalTransaction()
  {
    return new ThreadLocalTransaction();
  }

  /**
   * デフォルト・トランザクション・クラスのインスタンスをスレッド・セーフ化した
   * トランザクション・オブジェクトを指定したキーに結びつけて作成する。
   *
   * @param key キー。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction createThreadSafeTransaction(String key)
    throws ReasonedRuntimeException
  {
    return createThreadSafeTransaction(key, DefaultQueryTransaction.class);
  }

  /**
   * 指定されたトランザクション・クラスのインスタンスをスレッド・セーフ化した
   * トランザクション・オブジェクトを、指定したキーに結びつけて作成する。
   *
   * @param key キー。
   * @param transactionClass インスタンス化するトランザクション・クラス。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction createThreadSafeTransaction(
    String key, Class<? extends QueryTransaction> transactionClass
  ) throws ReasonedRuntimeException
  {
    synchronized (TRANSACTION_MAP) {
      QueryTransaction tr = TRANSACTION_MAP.get(key);
      if (tr != null && tr.getState() != QueryTransaction.State.Ended) {
        throw new ReasonedRuntimeException(
          QueryTransactionManager.Error.ThreadSafeAlreadyExists,
          key + " - " + tr.getClass().getName());
      }

      try {
        tr = transactionClass.newInstance();
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(
          QueryTransactionManager.Error.ThreadSafeFailToCreate,
          key + " - " + transactionClass.getName(), e);
      }

      TRANSACTION_MAP.put(key, tr);
      return new ThreadSafeTransaction(key);
    }
  }

  /**
   * 指定されたキーに結びつけられたスレッド・セーフなトランザクション・
   * オブジェクトを作成する。
   *
   * @param key キー。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction getThreadSafeTransaction(String key)
  {
    return new ThreadSafeTransaction(key);
  }


  /**
   * トランザクション・オブジェクトをスレッド・ローカル化するための
   * ラップ・クラス。
   */
  private static final class ThreadLocalTransaction implements QueryTransaction
  {
    /**
     * 基になるトランザクション・オブジェクト。
     * このオブジェクトの終了時に基のトランザクション・オブジェクトはスレッド・
     * ローカル・オブジェクトから削除され、そこから取得できなくなってしまう
     * ので、この属性に退避して使用する。
     */
    private QueryTransaction innerTransaction = null;

    /**
     * デフォルト・コンストラクタ。
     *
     * @throws ReasonedRuntimeException スレッド・ローカルなトランザクションが
     *           存在しない場合。
     */
    private ThreadLocalTransaction() throws ReasonedRuntimeException
    {
      QueryTransaction tr = THREAD_LOCAL.get();
      if (tr == null) {
        throw new ReasonedRuntimeException(
          QueryTransactionManager.Error.ThreadLocalNotExist);
      }

      this.innerTransaction = tr;
    }

    /**
     * 基になったトランザクション・オブジェクトを取得する。
     *
     * @return 基になったトランザクション・オブジェクト。
     * @throws ReasonedRuntimeException スレッド・ローカルなトランザクションが
     *           存在しない場合。
     */
    private QueryTransaction getInnerTransaction()
      throws ReasonedRuntimeException
    {
      QueryTransaction tr = THREAD_LOCAL.get();
      if (tr != null) {
        assert (tr == this.innerTransaction);
        return tr;
      }

      if (this.innerTransaction.getState() == QueryTransaction.State.Ended) {
        return this.innerTransaction;
      }

      throw new ReasonedRuntimeException(
        QueryTransactionManager.Error.ThreadLocalNotExist);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin()
    {
      getInnerTransaction().begin();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin(long timeout)
    {
      getInnerTransaction().begin(timeout);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void commit()
    {
      getInnerTransaction().commit();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void rollback()
    {
      getInnerTransaction().rollback();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void end()
    {
      QueryTransaction tr = getInnerTransaction();
      THREAD_LOCAL.remove();
      tr.end();
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public State getState()
    {
      return getInnerTransaction().getState();
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public long getBeginTimeMillis()
    {
      return getInnerTransaction().getBeginTimeMillis();
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public long getLimitTimeMillis()
    {
      return getInnerTransaction().getLimitTimeMillis();
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public QueryConnection getConnection(String connId)
      throws ReasonedException
    {
      return getInnerTransaction().getConnection(connId);
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public void addConnectionFactory(
      String connId, QueryConnectionFactory factory)
    {
      getInnerTransaction().addConnectionFactory(connId, factory);
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public QueryResultList getQueryResultList()
    {
      return getInnerTransaction().getQueryResultList();
    }

    /**
     * このオブジェクトのハッシュ・コードを取得する。
     *
     * @return このオブジェクトのハッシュ・コード。
     */
    @Override
    public int hashCode()
    {
      return getInnerTransaction().hashCode();
    }

    /**
     * 引数のオブジェクトがこのオブジェクトと等しいかどうかを判定する。
     * <br>
     * 引数のオブジェクトがスレッド・セーフなトランザクションであり、その基に
     * なっているトランザクション・オブジェクトが同一の場合に<tt>true</tt>を
     * 返す。
     *
     * @param obj 判定されるオブジェクト。
     * @return 引数のオブジェクトがこのオブジェクトと等しい場合に<tt>true</tt>
     *           を返す。
     */
    @Override
    public boolean equals(Object obj)
    {
      if (obj == null) {
        return false;
      }
      if (! (obj instanceof ThreadLocalTransaction)) {
        return false;
      }
      ThreadLocalTransaction tr = ThreadLocalTransaction.class.cast(obj);
      return getInnerTransaction().equals(tr.getInnerTransaction());
    }
  }


  /**
   * トランザクション・オブジェクトをスレッド・セーフ化するための
   * ラップ・クラス。
   */
  private static final class ThreadSafeTransaction implements QueryTransaction
  {
    /** このトランザクション・オブジェクトに結びつけられたキー。 */
    private final String key;

    /**
     * 終了後のトランザクション・オブジェクト。
     * このオブジェクトの終了時に基のトランザクション・オブジェクトは
     * トランザクション・マップから削除され、そこから取得できなくなってしまう
     * ので、この属性に退避して使用する。
     */ 
    private QueryTransaction innerTransaction = null;

    /**
     * 結びつけるキーを引数にとるコンストラクタ。
     *
     * @param key キー。
     * @throws ReasonedRuntimeException 指定されたキーに結びつけられた
     *           スレッド・セーフなトランザクションが存在しない場合。
     */
    private ThreadSafeTransaction(String key) throws ReasonedRuntimeException
    {
      synchronized (TRANSACTION_MAP) {
        QueryTransaction tr = TRANSACTION_MAP.get(key);
        if (tr == null) {
          throw new ReasonedRuntimeException(
            QueryTransactionManager.Error.ThreadSafeNotExist, key);
        }

        this.key = key;
        this.innerTransaction = tr;
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin()
    {
      synchronized (this.innerTransaction) {
        this.innerTransaction.begin();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin(long timeout)
    {
      synchronized (this.innerTransaction) {
        this.innerTransaction.begin(timeout);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void commit()
    {
      synchronized (this.innerTransaction) {
        this.innerTransaction.commit();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void rollback()
    {
      synchronized (this.innerTransaction) {
        this.innerTransaction.rollback();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void end()
    {
      synchronized (TRANSACTION_MAP) {
        QueryTransaction tr = TRANSACTION_MAP.get(this.key);
        if (tr != null && tr.equals(this.innerTransaction)) {
          TRANSACTION_MAP.remove(this.key);
        }
      }

      synchronized (this.innerTransaction) {
        this.innerTransaction.end();
      }
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public State getState()
    {
      return this.innerTransaction.getState();
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public long getBeginTimeMillis()
    {
      return this.innerTransaction.getBeginTimeMillis();
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public long getLimitTimeMillis()
    {
      return this.innerTransaction.getLimitTimeMillis();
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public QueryConnection getConnection(String connId) throws ReasonedException
    {
      synchronized (this.innerTransaction) {
        return this.innerTransaction.getConnection(connId);
      }
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public void addConnectionFactory(
      String connId, QueryConnectionFactory factory)
    {
      synchronized (this.innerTransaction) {
        this.innerTransaction.addConnectionFactory(connId, factory);
      }
    }

    /**
     * {@inhericDoc}
     */
    @Override
    public QueryResultList getQueryResultList()
    {
      return this.innerTransaction.getQueryResultList().toThreadSafe();
    }

    /**
     * このオブジェクトのハッシュ・コードを取得する。
     *
     * @return このオブジェクトのハッシュ・コード。
     */
    @Override
    public int hashCode()
    {
      return this.innerTransaction.hashCode();
    }

    /**
     * 引数のオブジェクトがこのオブジェクトと等しいかどうかを判定する。
     * <br>
     * 引数のオブジェクトがスレッド・セーフなトランザクションであり、その基に
     * なっているトランザクション・オブジェクトが同一の場合に<tt>true</tt>を
     * 返す。
     *
     * @param obj 判定されるオブジェクト。
     * @return 引数のオブジェクトがこのオブジェクトと等しい場合に<tt>true</tt>
     *           を返す。
     */
    @Override
    public boolean equals(Object obj)
    {
      if (obj == null) {
        return false;
      }
      if (! (obj instanceof ThreadSafeTransaction)) {
        return false;
      }
      ThreadSafeTransaction tr = ThreadSafeTransaction.class.cast(obj);
      return this.innerTransaction.equals(tr.innerTransaction);
    }
  }
}

