package ts.query;

import ts.tester.UnitTest;
import ts.tester.function.ObjectInspector;
import ts.util.*;
import ts.util.table.*;
import java.io.*;
import java.util.*;

public class BatchedQueryExecutionTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(BatchedQueryExecutionTest.class, args);
  }

  enum ENUM { AAA }

  static boolean HAS_RESULT_TABLE = false;
  static boolean FAIL_TO_EXECUTE = false;
  static List<String> SUB_EXECUTE_ID_LIST = new LinkedList<String>();

  static StringBuilder LOG = null;
  static void CLEAR_LOG() { LOG = new StringBuilder(); }

  public static class MyExecution extends BatchedQueryExecution {
    public MyExecution(QueryExecutionConfig config)
    throws ReasonedException, ReasonedRuntimeException { super(config); }
    public MyExecution(QueryExecutionConfig config, IQueryConnection conn)
    { super(config, conn); }
    @Override
    protected void prepareContent(QueryExecutionContent content,
      Map<String,Object> inputMap
    ) throws ReasonedException, ReasonedRuntimeException  {
      if (HAS_RESULT_TABLE) {
        content.addOutput(new QueryExecutionOutput("COL"));
      }
      String subExecId =
        content.getString(BatchedQueryExecution.Attribute.SubExecutionId);
      LOG.append("[prepareContent")
         .append("[connection id=").append(content.getConnectionId())
         .append("][execution id=").append(content.getExecutionId())
         .append("][sub execution id=").append(subExecId)
         .append("]]");
    }
    @Override
    protected void executeContent(QueryExecutionContent content)
    throws ReasonedException, ReasonedRuntimeException {
      String subExecId =
        content.getString(BatchedQueryExecution.Attribute.SubExecutionId);
      LOG.append("[executeContent")
         .append("[connection id=").append(content.getConnectionId())
         .append("][execution id=").append(content.getExecutionId())
         .append("][sub execution id=").append(subExecId)
         .append("]]");
      Table<String,Serializable> table = content.getResultTable();
      if (table != null) table.appendNew().put("COL", subExecId);
      if (FAIL_TO_EXECUTE) throw new ReasonedException(ENUM.AAA);
      try { Thread.sleep(1000L); } catch (Exception e) {}
    }
    @Override
    protected Iterator<String> iterateSubExecutionIds() {
      return SUB_EXECUTE_ID_LIST.iterator();
    }
  }

  static long CONNECTION_LIMIT_TIME = 0L;

  public static class MyConnection implements IQueryConnection {
    private final QueryConnectionConfig cfg;
    private final IQueryHistory history;
    private boolean isClosed = true;
    private long limitTimeMillis;
    public MyConnection(QueryConnectionConfig cfg) {
      this.cfg = cfg;
      this.history = new QueryHistory();
      this.limitTimeMillis = 0L;
    }
    public MyConnection(QueryConnectionConfig cfg, IQueryTransaction tran) {
      this.cfg = cfg;
      this.history = tran.getQueryHistory();
      this.limitTimeMillis = tran.getLimitTimeMillis();
    }
    @Override
    public String getConnectionId() { return this.cfg.getConnectionId(); }
    @Override
    public long getLimitTimeMillis() { return CONNECTION_LIMIT_TIME; }
    @Override
    public IQueryHistory getQueryHistory() { return this.history; }
    @Override
    public void open() throws ReasonedException { this.isClosed = false; }
    @Override
    public void commit() throws ReasonedException {}
    @Override
    public void rollback() {}
    @Override
    public void close() { this.isClosed = true; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public boolean isOpened() { return ! this.isClosed; }
  }

  public static class ErrConnection extends MyConnection {
    public ErrConnection(QueryConnectionConfig cfg) {
      super(cfg);
    }
    public ErrConnection(QueryConnectionConfig cfg, IQueryTransaction tran) {
      super(cfg, tran);
      throw new RuntimeException();
    }
  }


  protected void preInvocation(String method)
  {
    LOG = new StringBuilder();
    HAS_RESULT_TABLE = false;
    FAIL_TO_EXECUTE = false;
    SUB_EXECUTE_ID_LIST = new LinkedList<String>();
  }

  public void constructor_config()
  {
    MSG("実行設定オブジェクトを引数にとるコンストラクタ。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    EQUAL(exec.getConfig(), ecfg);
    EQUAL(exec.getExecutionId(), "");
    TRUE (exec.getQueryConnection() instanceof IQueryConnection);
    TRUE (exec.getQueryConnection() instanceof MyConnection);
    MyConnection conn = exec.getQueryConnection();
    EQUAL(conn.getConnectionId(),
      "BatchedQueryExecutionTest_constructor_config");
  }

  public void constructor_config_Null()
  {
    MSG("引数がヌルの場合。");

    MyExecution exec = null;
    try {
      exec = new MyExecution(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (ReasonedException e) {
      NG(e);
    }
  }

  public void constructor_config_conn()
  {
    MSG("実行設定オブジェクトとコネクションを引数にとるコンストラクタ。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "XXX");

    QueryConnectionConfig ccfg = new QueryConnectionConfig(
      "BatchedQueryExecutionTest_constructor_config");

    MyConnection conn = null;
    try {
      conn = ccfg.create();
    } catch (Exception e) {
      NG(e);
    }

    MyExecution exec = new MyExecution(ecfg, conn);

    EQUAL(exec.getConfig(), ecfg);
    EQUAL(exec.getExecutionId(), "");
    EQUAL(exec.getQueryConnection(), conn);
    EQUAL(exec.getQueryConnection().getConnectionId(),
      "BatchedQueryExecutionTest_constructor_config");

    ecfg = new QueryExecutionConfig(
      "BatchedQueryExecutionTest_constructor_config");

    exec = new MyExecution(ecfg, conn);
    EQUAL(exec.getConfig(), ecfg);
    EQUAL(exec.getExecutionId(),
      "BatchedQueryExecutionTest_constructor_config");
    EQUAL(exec.getQueryConnection(), conn);
    EQUAL(exec.getQueryConnection().getConnectionId(),
      "BatchedQueryExecutionTest_constructor_config");
  }

  public void constructor_config_conn_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "XXX");

    QueryConnectionConfig ccfg = new QueryConnectionConfig(
      "BatchedQueryExecutionTest_constructor_config");

    MyConnection conn = null;
    try {
      conn = ccfg.create();
    } catch (Exception e) {
      NG(e);
    }

    try {
      new MyExecution(null, conn);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }

    try {
      new MyExecution(ecfg, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void prepareResultTable_content()
  {
    MSG("結果テーブルを実行内容オブジェクトに設定するメソッドの確認。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    HashMap<String,Table<String,Serializable>> map =
      new HashMap<String,Table<String,Serializable>>();

    QueryExecutionContent cont = new QueryExecutionContent();
    cont.setExecutionId(exec.getExecutionId());
    cont.setConnectionId(ecfg.getConnectionId());
    cont.put(BatchedQueryExecution.Attribute.ResultTableMap, map);

    MSG("- 結果テーブルなしの場合。");

    FALSE(cont.hasResultTable());
    NULL(cont.getResultTable());

    exec.prepareResultTable(cont);
    NULL(cont.getResultTable());
    NULL(map.get(exec.getExecutionId()));

    MSG("- 結果テーブルありの場合。");
    
    cont.addOutput(new QueryExecutionOutput("COL"));
    TRUE(cont.hasResultTable());

    exec.prepareResultTable(cont);
    Table<String,Serializable> table = cont.getResultTable();
    NOTNULL(table);
    EQUAL(map.get(exec.getExecutionId()), table);

    MSG("- 再度、結果テーブルなしの場合（結果テーブル・クリア）。");

    cont = new QueryExecutionContent();
    cont.setExecutionId(exec.getExecutionId());
    cont.setConnectionId(ecfg.getConnectionId());
    cont.put(BatchedQueryExecution.Attribute.ResultTableMap, map);

    FALSE(cont.hasResultTable());
    NULL(cont.getResultTable());

    exec.prepareResultTable(cont);
    NULL(cont.getResultTable());
    NULL(map.get(exec.getExecutionId()));
  }

  public void prepareResultTable_content_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    try {
      exec.prepareResultTable(null);
      NG();
    } catch (NullPointerException e) {
      OK(e);
    }
  }

  public void disposeContent_content_result()
  {
    MSG("実行内容オブジェクトの後始末を行うメソッドの確認。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    HashMap<String,Table<String,Serializable>> map =
      new HashMap<String,Table<String,Serializable>>();

    Table<String,Serializable> table =
      new ArrayListTable<String,Serializable>();

    QueryResult rslt = new QueryResult("");

    QueryExecutionContent cont = new QueryExecutionContent();
    cont.setExecutionId(exec.getExecutionId());
    cont.setConnectionId(ecfg.getConnectionId());
    cont.put(BatchedQueryExecution.Attribute.ResultTableMap, map);

    MSG("- 結果テーブルなしの場合。");

    cont.setResultTable(null);
    FALSE(cont.hasResultTable());
    NULL(cont.getResultTable());
    exec.disposeContent(cont, rslt);
    NULL(rslt.getResultTable(cont.getExecutionId()));

    cont.setResultTable(table);
    FALSE(cont.hasResultTable());
    EQUAL(cont.getResultTable(), table);
    exec.disposeContent(cont, rslt);
    NULL(rslt.getResultTable(cont.getExecutionId()));

    MSG("- 結果テーブルありの場合。");

    cont.addOutput(new QueryExecutionOutput("COL"));
    cont.setResultTable(null);
    TRUE(cont.hasResultTable());
    NULL(cont.getResultTable());
    exec.disposeContent(cont, rslt);
    NULL(rslt.getResultTable(cont.getExecutionId()));

    cont.setResultTable(table);
    TRUE(cont.hasResultTable());
    EQUAL(cont.getResultTable(), table);
    exec.disposeContent(cont, rslt);
    EQUAL(rslt.getResultTable(cont.getExecutionId()), table);

    MSG("- 再度、結果テーブルなしの場合（結果テーブル・クリア）。");

    cont = new QueryExecutionContent();
    cont.setExecutionId(exec.getExecutionId());
    cont.setConnectionId(ecfg.getConnectionId());
    cont.put(BatchedQueryExecution.Attribute.ResultTableMap, map);

    cont.setResultTable(null);
    FALSE(cont.hasResultTable());
    NULL(cont.getResultTable());
    exec.disposeContent(cont, rslt);
    NULL(rslt.getResultTable(cont.getExecutionId()));

    cont.setResultTable(table);
    FALSE(cont.hasResultTable());
    EQUAL(cont.getResultTable(), table);
    exec.disposeContent(cont, rslt);
    NULL(rslt.getResultTable(cont.getExecutionId()));
  }

  public void disposeContent_content_result_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    HashMap<String,Table<String,Serializable>> map =
      new HashMap<String,Table<String,Serializable>>();

    Table<String,Serializable> table =
      new ArrayListTable<String,Serializable>();

    QueryResult rslt = new QueryResult("");

    QueryExecutionContent cont = new QueryExecutionContent();
    cont.setExecutionId(exec.getExecutionId());
    cont.setConnectionId(ecfg.getConnectionId());
    cont.put(BatchedQueryExecution.Attribute.ResultTableMap, map);

    try {
      exec.disposeContent(null, rslt);
      NG();
    } catch (NullPointerException e) {
      OK(e);
    }

    try {
      exec.disposeContent(cont, null);
      NG();
    } catch (NullPointerException e) {
      OK(e);
    }
  }

  public void execute_content_result_ZeroExecId()
  {
    MSG("このオブジェクトが示す照会・更新処理を実行するメソッドの確認。");
    MSG("- 処理される実行IDがゼロ個の場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    MSG("- - 結果データ・テーブルなし。");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable("e0"));

    EQUAL(LOG.toString(), "");

    CLEAR_LOG();
    HAS_RESULT_TABLE = true;
    MSG("- - 結果データ・テーブルあり。");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable("e0"));

    EQUAL(LOG.toString(), "");
  }

  public void execute_content_result_OneExecId()
  {
    MSG("このオブジェクトが示す照会・更新処理を実行するメソッドの確認。");
    MSG("- 処理される実行IDが一つの場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    SUB_EXECUTE_ID_LIST.add("e0");

    MSG("- - 結果データ・テーブルなし。");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
      e.printStackTrace(System.out);
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
    "");

    CLEAR_LOG();
    HAS_RESULT_TABLE = true;
    MSG("- - 結果データ・テーブルあり。実行IDの設定なし。");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
    "");

    CLEAR_LOG();
    MSG("- - 結果データ・テーブルあり。実行IDの設定と一致しない。");

    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "eX");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
    "");

    CLEAR_LOG();
    MSG("- - 結果データ・テーブルあり。実行IDの設定と一致。");

    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "e0");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds(), Arrays.asList(""));
    EQUAL(rslt.getResultTable("").recordFirst().get("COL"), "e0");

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
    "");
  }

  public void execute_content_result_MultiExecId()
  {
    MSG("このオブジェクトが示す照会・更新処理を実行するメソッドの確認。");
    MSG("- 処理される実行IDが複数の場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    SUB_EXECUTE_ID_LIST.add("e0");
    SUB_EXECUTE_ID_LIST.add("e1");
    SUB_EXECUTE_ID_LIST.add("e2");

    MSG("- - 結果データ・テーブルなし。");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
    "");

    CLEAR_LOG();
    HAS_RESULT_TABLE = true;
    MSG("- - 結果データ・テーブルあり。実行IDの設定なし。");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
    "");

    CLEAR_LOG();
    MSG("- - 結果データ・テーブルあり。実行IDの設定と一致しない。");

    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "eX");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds().size(), 0);
    NULL (rslt.getResultTable(""));

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
    "");

    CLEAR_LOG();
    MSG("- - 結果データ・テーブルあり。実行IDの設定と一致。");

    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "e0");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds(), Arrays.asList(""));
    EQUAL(rslt.getResultTable("").recordFirst().get("COL"), "e0");

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
    "");

    CLEAR_LOG();
    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "e1");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds(), Arrays.asList(""));
    EQUAL(rslt.getResultTable("").recordFirst().get("COL"), "e1");

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
    "");

    CLEAR_LOG();
    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "e2");

    try {
      exec.execute(inputMap, rslt);
    } catch (ReasonedException e) {
      NG(e.toString());
    }

    EQUAL(rslt.getQueryId(), "q0");
    TRUE (rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    NULL (rslt.getException());
    EQUAL(rslt.getAllResultIds(), Arrays.asList(""));
    EQUAL(rslt.getResultTable("").recordFirst().get("COL"), "e2");

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e1]]" +
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e2]]" +
    "");
  }

  public void execute_content_result_Null()
  {
    MSG("引数がヌルの場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("");

    try {
      exec.execute(null, rslt);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (ReasonedException e) {
      NG(e);
    }

    try {
      exec.execute(inputMap, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (ReasonedException e) {
      NG(e);
    }
  }

  public void execute_inputMap_result_Timeout()
  {
    MSG("タイムアウトが発生した場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");
    ecfg.getResource().setFirstValue("ts-query.execution.limit.spenttime",
      "500");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    SUB_EXECUTE_ID_LIST.add("e0");
    SUB_EXECUTE_ID_LIST.add("e1");
    SUB_EXECUTE_ID_LIST.add("e2");

    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "e1");

    try {
      exec.execute(inputMap, rslt);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryExecution.Error.Timeout);
    }

    EQUAL(rslt.getQueryId(), "q0");
    FALSE(rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    EQUAL(rslt.getException().getReason(), IQueryExecution.Error.Timeout);

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
    "");
  }

  public void execute_inputMap_result_FailToExecute()
  {
    MSG("実行中にエラーが発生した場合。");

    QueryExecutionConfig ecfg = new QueryExecutionConfig();
    ecfg.getResource().setFirstValue("ts-query.execution.class",
      "ts.query.BatchedQueryExecutionTest$MyExecution");
    ecfg.getResource().setFirstValue("ts-query.execution.connection.id",
      "BatchedQueryExecutionTest_constructor_config");

    MyExecution exec = null;
    try {
      exec = new MyExecution(ecfg);
    } catch (ReasonedException e) {
      NG(e);
      return;
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    IQueryResult rslt = new QueryResult("q0");

    FAIL_TO_EXECUTE = true;

    SUB_EXECUTE_ID_LIST.add("e0");
    SUB_EXECUTE_ID_LIST.add("e1");
    SUB_EXECUTE_ID_LIST.add("e2");

    ecfg.getResource().setFirstValue("ts-query.execution.result.id", "e1");

    try {
      exec.execute(inputMap, rslt);
      NG();
    } catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ENUM.AAA);
    }

    EQUAL(rslt.getQueryId(), "q0");
    FALSE(rslt.isSuccess());
    POSITIVE(rslt.getBeginTimeMillis());
    POSITIVE(rslt.getEndTimeMillis());
    TRUE (rslt.getSpentTimeMillis() >= 0L);
    EQUAL(rslt.getException().getReason(), ENUM.AAA);

    EQUAL(LOG.toString(),
      "[prepareContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
      "[executeContent[connection id=BatchedQueryExecutionTest_constructor_config][execution id=][sub execution id=e0]]" +
    "");
  }
}
