/*
 * IQueryHistory class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.io.Serializable;
import java.util.List;
import java.util.Map;

/**
 * 実行履歴を格納するクラスのインターフェイス。
 * <br>
 * クエリ・トランザクションやクエリ・コネクションによって使用され、その開始から
 * 終了までの間に処理されたクエリの実行履歴を格納する。
 * 実行履歴として、以下の情報を保持する：
 * <ul>
 *  <li>クエリID</li>
 *  <li>接続先ID</li>
 *  <li>実行ID</li>
 *  <li>実行成否 (true/false)</li>
 *  <li>失敗時にスローされた例外オブジェクト</li>
 *  <li>開始時刻 [msec]</li>
 *  <li>終了時刻 [msec]</li>
 *  <li>処理時間 [msec]</li>
 *  <li>処理件数</li>
 *  <li>実行内容 (実行クラスにより異なる)</li>
 * </ul>
 * さらに、これらの情報をクエリID、接続先ID、実行ID、実行成否、例外オブジェクト
 * をキーに検索して取得することができる。
 *
 * @author 佐藤隆之
 * @version $Id: IQueryHistory.java,v 1.3 2012-02-28 13:25:12 tayu Exp $
 */
public interface IQueryHistory extends Serializable
{
  /** 実行履歴の項目を表す列挙型。 */
  enum Item {
    /** クエリIDを示す列挙値。 */
    QueryId,

    /** 接続先IDを示す列挙値。 */
    ConnectionId,

    /** 実行IDを示す列挙値。 */
    ExecutionId,

    /** 実行成否を示す列挙値。 */
    IsSuccess,

    /** 失敗時にスローされた例外オブジェクトを示す列挙値。 */
    Exception,

    /** 開始時刻を示す列挙値。 */
    BeginTime,

    /** 終了時刻を示す列挙値。 */
    EndTime,

    /** 処理時間を示す列挙値。 */
    SpentTime,

    /** 照会件数を示す列挙値。 */
    FetchCount,

    /** 更新件数を示す列挙値。 */
    UpdateCount,

    /** 実行内容を示す列挙型。 */
    Content,

    /** 登録日時。 */
    EntryDateTime,
  }

  /**
   * 空の実行履歴を一つ追加して返す。
   *
   * @return 実行履歴を表す空のマップ。
   */
  Map<Item,Serializable> appendNew();

  /**
   * 全ての実行履歴を取得する。
   *
   * @return 全ての実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getAll();

  /**
   * クエリIDをキーに実行履歴を取得する。
   *
   * @return 指定されたクエリIDの実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getByQueryId(String queryId);

  /**
   * 接続先IDをキーに実行履歴を取得する。
   *
   * @return 指定された接続先IDの実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getByConnectionId(String connId);

  /**
   * クエリIDと接続先IDをキーに実行履歴を取得する。
   *
   * @return 指定されたクエリIDと接続先IDの実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getByConnectionId(String queryId, String connId);

  /**
   * 実行IDをキーに実行履歴を取得する。
   *
   * @return 指定された実行IDの実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getByExecutionId(String execId);

  /**
   * 接続先IDと実行IDをキーに実行履歴を取得する。
   *
   * @return 指定された接続先IDと実行IDの実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getByExecutionId(String connId, String execId);

  /**
   * クエリIDと接続先ID、実行IDをキーに実行履歴を取得する。
   *
   * @return 指定されたクエリIDと接続先ID、実行IDの実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getByExecutionId(String queryId, String connId,
    String execId);

  /**
   * 実行成否をキーに実行履歴を取得する。
   *
   * @param isSuccess 実行成否。
   * @return 指定された実行成否の実行履歴を格納したリスト。
   */
  List<Map<Item,Serializable>> getByIsSuccess(boolean isSuccess);

  /**
   * 例外オブジェクトをキーに実行履歴を取得する。
   *
   * @return 指定された例外オブジェクトがスローされた実行履歴。
   */
  Map<Item,Serializable> getByException(ReasonedException exc);
}
