/*
 * IQueryTransaction class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;

/**
 * 一括して確定処理されるべきクエリを囲むトランザクション・クラスの
 * インターフェイス。
 * <br>
 * {@link #begin()}メソッドの実行から{@link #end()}メソッドの実行までを一つの
 * トランザクションとして、その間に実行されたクエリによる各接続先への更新の確定
 * または取消をまとめて実行する。
 *
 * @author 佐藤隆之
 * @version $Id: IQueryTransaction.java,v 1.5 2012-01-30 16:41:22 tayu Exp $
 */
public interface IQueryTransaction
{
  /** このクラスで発生しうるエラーの列挙型。 */
  enum Error {
    /**
     * ある操作を実行しようとしたときに、トランザクションがあるべき状態でなか
     * ったことを示す列挙型の値。
     */
    IllegalState,

    /**
     * 指定された接続先IDに対応づけられたクエリ・コネクション・オブジェクトが
     * 存在しないことを示す列挙型の値。
     */
    QueryConnectionNotFound,

    /** ロールバックの処理に何らかの理由で失敗した場合。 */
    FailToRollback,

    /** 終了処理に何らかの理由で失敗した場合。 */
    FailToEnd,
  }

  /** トランザクションの状態を示す列挙型。 */
  enum State {
    /** トランザクションが作成されたという状態を示す列挙型の値。 */
    Created,

    /** トランザクションが開始処理中という状態を示す列挙型の値。 */
    Begining,

    /** トランザクションが開始されたという状態を示す列挙型の値。 */
    Begined,

    /** トランザクションが確定処理中という状態を示す列挙型の値。 */
    Committing,

    /** トランザクションが確定されたという状態を示す列挙型の値。 */
    Committed,

    /** トランザクションが取消処理中という状態を示す列挙型の値。 */
    Rollbacking,

    /** トランザクションが取消されたという状態を示す列挙型の値。 */
    Rollbacked,

    /** トランザクションが終了処理中という状態を示す列挙型の値。 */
    Ending,

    /** トランザクションが終了されたという状態を示す列挙型の値。 */
    Ended,
  }

  /**
   * トランザクションの開始処理を実行する。
   * <br>
   * このメソッドの実行すると、トランザクションの開始時刻が設定され、その開始
   * 時刻とタイムアウト時間を加算して制限時刻が設定される。
   * 制限時間はクエリの実行中のタイムアウト判定に使用される。
   *
   * @throws ReasonedException トランザクションの開始処理に失敗した場合。
   * @throws ReasonedRuntimeException トランザクションの状態が不正な場合。
   */
  void begin() throws ReasonedException, ReasonedRuntimeException;

  /**
   * トランザクション内のクエリの実行結果の確定処理を実行する。
   *
   * @throws ReasonedException トランザクションの確定処理に失敗した場合。
   * @throws ReasonedRuntimeException トランザクションの状態が不正な場合。
   */
  void commit() throws ReasonedException, ReasonedRuntimeException;

  /**
   * トランザクション内のクエリの実行結果の取消処理を実行する。
   * <br>
   * 途中で例外が発生してもこの処理は続行され、この処理が完了した所で例外が
   * スローされる。
   *
   * @throws ReasonedRuntimeException トランザクションの取消処理に失敗した場合、
   *   又はトランザクションの状態が不正な場合。
   */
  void rollback() throws ReasonedRuntimeException;

  /**
   * トランザクションを終了する。
   * <br>
   * トランザクションが確定又は取消されていない場合は、取消を実行してから終了
   * する。
   *
   * @throws ReasonedRuntimeException トランザクションの終了処理に失敗した場合、
   *   又はトランザクションの状態が不正な場合。
   */
  void end() throws ReasonedRuntimeException;

  /**
   * トランザクションの状態を取得する。
   *
   * @return トランザクションの状態。
   */
  State getState();

  /**
   * トランザクションの開始時刻を取得する。
   *
   * @return トランザクションの開始時刻 [msec]。
   */
  long getBeginTimeMillis();

  /**
   * トランザクションの制限時刻を取得する。
   * <br>
   * 値がゼロ以下の場合は制限なしとする。
   *
   * @return トランザクションの制限時刻 [msec]。
   */
  long getLimitTimeMillis();

  /**
   * トランザクションのタイムアウト時間を取得する。
   * <br>
   * 値がゼロ以下の場合は制限なしとする。
   *
   * @return トランザクションのタイムアウト時間 [msec]。
   */
  long getTimeoutMillis();

  /**
   * トランザクションのタイムアウト時間を設定する。
   * <br>
   * 引数の値がゼロ以下の場合は制限なしとする。
   *
   * @param millis トランザクションのタイムアウト時間 [msec]。
   */
  void setTimeoutMillis(long millis);

  /**
   * このトランザクションの中で実行されたクエリの実行履歴を格納する{@link
   * IQueryHistory}オブジェクトを取得する。
   *
   * @return {@link IQueryHistory}オブジェクト。
   */
  IQueryHistory getQueryHistory();

  /**
   * 指定された接続先IDに対応するクエリ・コネクション・オブジェクトを取得する。
   *
   * @param connId 接続先ID。
   * @return コネクション・オブジェクト。
   * @throws ReasonedException コネクションの開始処理に失敗した場合。
   * @throws ReasonedRuntimeException コネクション・オブジェクトの作成に失敗
   *   した場合、又はトランザクションの状態が不正な場合。
   */
  IQueryConnection getQueryConnection(String connId)
    throws ReasonedException, ReasonedRuntimeException;

  /**
   * 指定された実行IDに対応するクエリ実行ブジェクトを取得する。
   * 
   * @param execId 実行ID。
   * @return クエリ実行オブジェクト。
   * @throws ReasonedException クエリが使用するコネクションの開始処理に失敗した
   *   場合。
   * @throws ReasonedRuntimeException クエリ実行オブジェクトの作成に失敗した
   *   場合、又はトランザクションの状態が不正な場合。
   */
  IQueryExecution getQueryExecution(String execId)
    throws ReasonedException, ReasonedRuntimeException;

  /**
   * 指定されたクエリIDに対応するクエリ・オブジェクトを取得する。
   * 
   * @param queryId クエリID。
   * @return クエリ・オブジェクト。
   * @throws ReasonedException クエリが使用するコネクションの開始処理に失敗した
   *   場合。
   * @throws ReasonedRuntimeException クエリ・オブジェクトの作成に失敗した場合、
   *   又はトランザクションの状態が不正な場合。
   */
  IQuery getQuery(String queryId)
    throws ReasonedException, ReasonedRuntimeException;
}
