/*
 * IQueryTransactionManager class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedRuntimeException;

/**
 * トランザクション・オブジェクトの作成や取得、破棄を管理するためのクラスの
 * インターフェイス。
 *
 * @author 佐藤隆之
 * @version $Id: IQueryTransactionManager.java,v 1.9 2012-03-14 07:49:19 tayu Exp $
 */
public interface IQueryTransactionManager
{
  /** このクラスで発生しうるエラーの列挙型。 */
  enum Error {
    /**
     * スレッド・ローカルなトランザクション・オブジェクトの作成に失敗したことを
     * 示す列挙値。
     */
    FailToCreateThreadLocal,

    /**
     * スレッド・ローカルなトランザクション・オブジェクトが存在しないことを示す
     * 列挙値。
     */
    ThreadLocalNotExist,

    /**
     * スレッド・ローカルなトランザクション・オブジェクトの作成時に、終了してい
     * ない別のトランザクションが既に存在することを示す列挙値。
     */
    ThreadLocalAlreadyExists,

    /**
     * スレッド・ローカルなトランザクション・オブジェクトが既に破棄されている
     * ことを示す列挙値。
     */
    ThreadLocalAlreadyDiscarded,

    /** 
     * スレッド・セーフなトランザクション・オブジェクトの作成に失敗したことを
     * 示す列挙値。
     */
    FailToCreateThreadSafe,

    /**
     * 指定されたスレッド・セーフなトランザクション・オブジェクトが存在しない
     * ことを示す列挙値。
     */
    ThreadSafeNotExist,

    /**
     * スレッド・セーフなトランザクション・オブジェクトの作成時に、同じキーに
     * 結びつけられた終了していない別のトランザクションが既に存在することを示す
     * 列挙値。
     */
    ThreadSafeAlreadyExists,
  }

  /** この継承クラスのシングルトン・インスタンス。 */
  public static final IQueryTransactionManager INSTANCE = 
    new QueryTransactionManager();

  /**
   * {@link QueryTransaction}クラスのインスタンスをスレッド・ローカル化した
   * トランザクション・オブジェクトを作成する。
   *
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   * @throws ReasonedRuntimeException スレッド・ローカルなトランザクションが
   *   既に存在する場合、又はスレッド・ローカルなトランザクションの作成に失敗
   *   した場合。
   */
  IQueryTransaction createThreadLocalTransaction()
    throws ReasonedRuntimeException;

  /**
   * 指定されたトランザクション・クラスのインスタンスをスレッド・ローカル化した
   * トランザクション・オブジェクトを作成する。
   *
   * @param tranClass インスタンスを作成するトランザクション・クラス。
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   * @throws ReasonedRuntimeException スレッド・ローカルなトランザクションが
   *   既に存在する場合、又はスレッド・ローカルなトランザクションの作成に失敗
   *   した場合。
   * @throws AsserionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  IQueryTransaction createThreadLocalTransaction(
    Class<? extends IQueryTransaction> tranClass
  ) throws ReasonedRuntimeException;

  /**
   * スレッド・ローカルなトランザクション・オブジェクトを作成する。
   *
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   */
  IQueryTransaction getThreadLocalTransaction();


  /**
   * {@link QueryTransaction}クラスのインスタンスをスレッド・セーフ化した
   * トランザクション・オブジェクトを指定したキーに結びつけて作成する。
   *
   * @param key キー。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   * @throws ReasonedRuntimeException 同じキーに結びつけられたスレッド・セーフ
   *   なトランザクションが 既に存在する場合、又はスレッド・セーフなトランザク
   *   ションの作成に失敗した場合。
   */
  IQueryTransaction createThreadSafeTransaction(String key)
    throws ReasonedRuntimeException;

  /**
   * 指定されたトランザクション・クラスのインスタンスをスレッド・セーフ化した
   * トランザクション・オブジェクトを、指定したキーに結びつけて作成する。
   *
   * @param key キー。
   * @param tranClass インスタンス化するトランザクション・クラス。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   * @throws ReasonedRuntimeException 同じキーに結びつけられたスレッド・セーフ
   *   なトランザクションが 既に存在する場合、又はスレッド・セーフなトランザク
   *   ションの作成に失敗した場合。
   */
  IQueryTransaction createThreadSafeTransaction(String key,
    Class<? extends IQueryTransaction> tranClass
  ) throws ReasonedRuntimeException;

  /**
   * 指定されたキーに結びつけられたスレッド・セーフなトランザクション・オブジェ
   * クトを作成する。
   *
   * @param key キー。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   */
  IQueryTransaction getThreadSafeTransaction(String key);
}
