/*
 * QueryConfig class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.TypedGetter;
import ts.util.AbstractTypedGetter;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.resource.Resource;
import ts.util.resource.PropertyResource;
import ts.util.resource.XmlResource;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.util.List;

/**
 * クエリ設定クラス。
 * <br>
 * {@link IQuery}オブジェクトの処理に関する設定情報を格納するためのクラス。
 * <br>
 * 引数なしのコンストラクタを使った場合は空の設定オブジェクトが作成され、{@link
 * #getResource()}メソッドで取得されるリソース・オブジェクトに設定情報を登録
 * して使用する。
 * <br>
 * 引数にクエリIDを使用するコンストラクタを使った場合は、自動的に{@link 
 * QueryEnvironmentConfig}オブジェクトから取得したディレクトリ下からクエリIDを
 * ファイル・タイトルにもつXMLファイル(.xml)又はJavaプロパティ・ファイル
 * (.properties)をロードするので、その設定情報を使用する。
 * <br>
 * 設定情報は、このオブジェクトを使用するクエリ・クラスによって異なるが、それに
 * 依らない共通の設定情報として、
 * <ul>
 * <li>ts-query.query.class - 使用される{@link IQuery}の派生クラス（必須）</li>
 * </ul>
 * がある。
 * また、{@link Query}オブジェクト固有の設定情報として、
 * <ul>
 * <li>ts-query.query.execution.id - {@link Query}が処理する{@link
 *   IQueryExecution}の実行ID（複数指定可）。</li>
 * </ul>
 * がある。
 * 
 * @author 佐藤隆之
 * @version $Id: QueryConfig.java,v 1.25 2012-03-14 07:49:20 tayu Exp $
 */
public class QueryConfig implements Serializable
{
  /** このクラスで発生しうるエラーを定義する列挙型。 */
  public enum Error {
    /** 指定されたクエリIDがヌル又は空文字列の場合。 */
    QueryIdIsNullOrEmpty,

    /** クエリ設定ファイルが見つからない場合。 */
    QueryConfigFileNotFound,

    /** クエリ設定ファイルのロードに失敗した場合。 */
    FailToLoadQueryConfigFile,

    /** 設定情報のクエリ・クラスが見つからない場合。 */
    QueryClassNotFound,

    /** 設定情報のクエリ・クラスに要求されるコンストラクタが存在しない場合。 */
    QueryConstructorNotFound,

    /** クエリ・オブジェクトの作成に失敗した場合。 */
    FailToCreateQuery,
  }

  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = -41345242524215703L;

  /** クエリID。 */
  private final String queryId;

  /** 設定情報を保持するリソース・オブジェクト。 */
  private final Resource resource;

  /** 設定情報を型変換して取得するための{@link TypedGetter}オブジェクト。 */
  private final AbstractTypedGetter<String,String> typedGetter;

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryConfig()
  {
    this.queryId = "";
    this.resource = new PropertyResource();
    this.typedGetter = newTypedGetter();
  }

  /**
   * クエリIDを引数にとるコンストラクタ。
   *
   * @param queryId クエリID。
   * @throws ReasonedRuntimeException 引数がヌル又は空文字列の場合。
   */
  public QueryConfig(String queryId)
  {
    if (queryId == null) {
      throw new ReasonedRuntimeException(Error.QueryIdIsNullOrEmpty,
        "[query Id=" + queryId + "]");
    }

    queryId = queryId.trim();
    if (queryId.isEmpty()) {
      throw new ReasonedRuntimeException(Error.QueryIdIsNullOrEmpty,
        "[query Id=" + queryId + "]");
    }

    this.queryId = queryId;
    this.resource = loadResource();
    this.typedGetter = newTypedGetter();
  }

  /**
   * クエリIDを取得する。
   *
   * @return クエリID。
   */
  public String getQueryId()
  {
    return this.queryId;
  }

  /**
   * クエリ設定ファイルをロードしたリソース・オブジェクトを作成する。
   *
   * @return クエリ設定ファイルをロードしたリソース・オブジェクト。
   * @throws ReasonedRuntimeException クエリ設定ファイルのロードに失敗した場合。
   */
  protected Resource loadResource() throws ReasonedRuntimeException
  {
    QueryEnvironment env = QueryEnvironment.getInstance();
    File dir = env.getQueryConfigDirectory();

    File xmlFile = new File(dir, getQueryId() + ".xml");
    if (xmlFile.exists()) {
      try {
        XmlResource xres = new XmlResource();
        xres.setValidating(false);
        xres.load(xmlFile.getCanonicalPath());
        return xres;
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadQueryConfigFile,
          "[path=" + xmlFile.getAbsolutePath() + "]", e);
      }
    }

    File propFile = new File(dir, getQueryId() + ".properties");
    if (propFile.exists()) {
      try {
        return new PropertyResource(propFile.getCanonicalPath());
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadQueryConfigFile,
          "[path=" + propFile.getAbsolutePath() + "]", e);
      }
    }

    throw new ReasonedRuntimeException(Error.QueryConfigFileNotFound,
      "[path=" + new File(dir, getQueryId()).getAbsolutePath() +
      ".{xml|properties}]");
  }

  /**
   * 設定情報を型変換して取得するための{@link TypedGetter}オブジェクトを作成
   * する。
   *
   * @return {@link TypedGetter}オブジェクト。
   */
  protected AbstractTypedGetter<String,String> newTypedGetter()
  {
    return new AbstractTypedGetter<String,String>() {
      static final long serialVersionUID = QueryConfig.serialVersionUID + 1L;
      @Override
      public String get(String key) {
        return QueryConfig.this.resource.getFirstValue(key);
      }
      @Override
      public List<String> getList(String key) {
        return QueryConfig.this.resource.getValues(key);
      }
    };
  }

  /**
   * クエリ設定を保持するリソース・オブジェクトを取得する。
   *
   * @return クエリ設定を保持するリソース・オブジェクト。
   */
  protected Resource getResource()
  {
    return this.resource;
  }

  /**
   * 設定情報を型変換して取得するための{@link TypedGetter}オブジェクトを取得
   * する。
   *
   * @return {@link TypedGetter}オブジェクト。
   */
  protected AbstractTypedGetter<String,String> typedGetter()
  {
    return this.typedGetter;
  }

  /**
   * クエリ・オブジェクトを作成する。
   * <br>
   * このオブジェクトが保持するクエリ設定を使って、{@link IQuery}の派生クラスの
   * インスタンスを作成する。
   * 派生クラスの名前は、クエリ設定 <tt>ts-query.query.class</tt>に指定された
   * クラス名が使用される。
   *
   * @return クエリ・オブジェクト。
   * @throws ReasonedException クエリ・オブジェクトの作成に失敗した場合。
   * @throws ReasonedRuntimeException クエリ設定が不正だった場合。
   */
  public <T extends IQuery> T create()
    throws ReasonedException, ReasonedRuntimeException
  {
    String cls = getQueryClass();
    Class<T> queryCls = null;
    try {
      @SuppressWarnings("unchecked")
      Class<T> c = (Class<T>) Class.forName(cls);
      queryCls = c;
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.QueryClassNotFound,
        "[class=" + cls + "]", e);
    }

    Constructor<T> cons = null;
    try {
      cons = queryCls.getConstructor(QueryConfig.class);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.QueryConstructorNotFound,
        "[class=" + cls + "]", e);
    }

    try {
      return cons.newInstance(this);
    }
    catch (Exception e) {
      throw new ReasonedException(Error.FailToCreateQuery,
        "[class=" + cls + "]", e);
    }
  }

  /**
   * クエリ・オブジェクトを作成する。
   * <br>
   * このオブジェクトが保持するクエリ設定と引数のトランザクション・オブジェクト
   * を使って、{@link IQuery}の派生クラスのインスタンスを作成する。
   * 派生クラスの名前は、クエリ設定 <tt>ts-query.query.class</tt>に指定された
   * クラス名が使用される。
   * <br>
   * 引数のトランザクション・オブジェクトからは、その{@link IQueryConnection}
   * オブジェクトや{@link IQueryHistory}オブジェクトを使用する。
   *
   * @param tran クエリ・トランザクション・オブジェクト。
   * @return クエリ・オブジェクト。
   * @throws ReasonedException クエリ・オブジェクトの作成に失敗した場合。
   * @throws ReasonedRuntimeException クエリ設定が不正だった場合。
   */
  public <T extends IQuery> T create(IQueryTransaction tran)
    throws ReasonedException, ReasonedRuntimeException
  {
    String cls = getQueryClass();
    Class<T> queryCls = null;
    try {
      @SuppressWarnings("unchecked")
      Class<T> c = (Class<T>) Class.forName(cls);
      queryCls = c;
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.QueryClassNotFound,
        "[class=" + cls + "]", e);
    }

    Constructor<T> cons = null;
    try {
      cons = queryCls.getConstructor(QueryConfig.class,IQueryTransaction.class);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.QueryConstructorNotFound,
        "[class=" + cls + "]", e);
    }

    try {
      return cons.newInstance(this, tran);
    }
    catch (Exception e) {
      throw new ReasonedException(Error.FailToCreateQuery,
        "[class=" + cls + "]", e);
    }
  }

  /**
   * このオブジェクトから作成されるクエリ・オブジェクトのクラス名を取得する。
   *
   * @return クエリ・クラスの名前。
   */
  protected String getQueryClass()
  {
    return typedGetter().getString("ts-query.query.class");
  }

  /**
   * クエリ・オブジェクトが処理する{@link IQueryExecution}の実行IDのリストを
   * 取得する。
   *
   * @return 実行IDのリスト。
   */
  protected List<String> getExecutionIds()
  {
    return getResource().getValues("ts-query.query.execution.id");
  }

  /**
   * クエリ設定ファイルのXML形式のサンプルを{@link PrintWriter}オブジェクトに
   * 出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleXml(PrintWriter pw, String encoding)
    throws IOException
  {
    pw.println("<?xml version=\"1.0\" encoding=\"" + encoding.toString() +
               "\"?>");
    pw.println("");
    pw.println("<!--");
    pw.println("  || Query Configuration XML File.");
    pw.println("  ||");
    pw.println("  || This file is an XML file which specifys " +
               "some configurations for an ");
    pw.println("  || IQuery object.");
    pw.println("  ||");
    pw.println("  -->");
    pw.println("");
    pw.println("<ts-query>");
    pw.println("  <query>");
    pw.println("    <!-- The query class. -->");
    pw.println("    <class>...</class>");

    outputSampleXmlEntries(pw);

    pw.println("  </query>");
    pw.println("</ts-query>");
    pw.println();
  }

  /**
   * クエリ設定ファイルのXMLエントリのサンプルを{@link PrintWriter}オブジェクト
   * に出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleXmlEntries(PrintWriter pw) throws IOException
  {
    pw.println("    <!-- The execution id list. -->");
    pw.println("    <execution>");
    pw.println("      <id>...</id>");
    pw.println("      <id>...</id>");
    pw.println("      <id>...</id>");
    pw.println("    </execution>");
  }

  /**
   * クエリ設定ファイルのJavaプロパティ形式のサンプルを{@link PrintWriter}
   * オブジェクトに出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleProp(PrintWriter pw) throws IOException
  {
    pw.println("#");
    pw.println("# Query Configuration Property File.");
    pw.println("#");
    pw.println("# This file is a Java property file which specifys " +
               "some configurations for a ");
    pw.println("# IQuery object.");
    pw.println("#");
    pw.println("");
    pw.println("# The query class.");
    pw.println("ts-query.query.class = ...");
    pw.println("");

    outputSamplePropEntries(pw);

    pw.println("");
    pw.println("#.");
  }

  /**
   * クエリ設定ファイルのプロパティ・エントリのサンプルを{@link PrintWriter}
   * オブジェクトに出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSamplePropEntries(PrintWriter pw) throws IOException
  {
    pw.println("# The execution id list.");
    pw.println("ts-query.query.execution.id = ...");
    pw.println("ts-query.query.execution.id = ...");
    pw.println("ts-query.query.execution.id = ...");
    pw.println("");
  }

  /**
   * クエリ設定ファイルのサンプルを出力するコマンドを実行する。
   * <br>
   * コマンドライン引数の配列の構成は以下の通りである：
   * <ol>
   * <li>args[0] - コマンド名</li>
   * <li>args[1] - 出力ファイル・パス</li>
   * <li>args[2] - 出力ファイル形式（<tt>"xml"</tt>:XML形式、<tt>"prop"</tt>:Javaプロパティ形式、但し大文字小文字は区別しない）</li>
   * <li>args[3] - 出力ファイルの文字エンコーディング</li>
   * </ol>
   * また、終了コードは以下の値をとりうる：
   * <ul>
   * <li><tt>0</tt> - 正常終了の場合</li>
   * <li><tt>1</tt> - コマンドライン引数が不正な場合</li>
   * <li><tt>2</tt> - 非対応の文字エンコーディングが指定された場合</li>
   * <li><tt>3</tt> - 上記以外のエラーの場合</li>
   * </ul>
   *
   * @param args コマンドライン引数の配列。
   * @param config クエリ設定オブジェクト。
   * @return コマンドの終了コード。
   */
  protected static int executeCommand(String[] args, QueryConfig config)
  {
    String pgmName = "null", encoding = "";
    File file;
    boolean isXml;
    try {
      if (args != null && args.length > 0) {
        pgmName = args[0];
      }

      file = new File(args[1]);

      if ("xml".equalsIgnoreCase(args[2])) {
        isXml = true;
      }
      else if ("prop".equalsIgnoreCase(args[2])) {
        isXml = false;
      }
      else {
        throw new IllegalArgumentException();
      }

      encoding = args[3];

      if (args.length > 4) {
        throw new IllegalArgumentException();
      }
    }
    catch (Exception e) {
      System.err.println("HELP:");
      System.err.println(
        pgmName + " <output-file-path> {xml|prop} <output-file-encoding>");
      System.err.println();
      return 1;
    }

    try {
      new String(new byte[0], encoding);
    }
    catch (UnsupportedEncodingException e) {
      System.err.println("ERROR:");
      System.err.println("Specified encoding is unsupported: " + encoding);
      System.err.println();
      return 2;
    }

    try {
      PrintWriter pw = null;
      try {
        pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
          encoding));

        if (isXml) {
          config.outputSampleXml(pw, encoding);
        }
        else {
          config.outputSampleProp(pw);
        }
        pw.flush();
        return 0;
      }
      finally {
        if (pw != null) try { pw.close(); } catch (Exception e) {}
      }
    }
    catch (Exception e) {
      System.err.println("ERROR:");
      System.err.println(e.toString());
      System.err.println();
      return 3;
    }
  }
}
