/*
 * QueryConnection class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;

/**
 * クエリを実行する対象への接続を表すコネクション・クラスの抽象クラス。
 *
 * @author 佐藤隆之
 * @version $Id: QueryConnection.java,v 1.20 2012-03-03 20:49:22 tayu Exp $
 */
public abstract class QueryConnection implements IQueryConnection
{
  /** 接続設定オブジェクト。 */
  private final QueryConnectionConfig config;

  /** クエリ実行履歴オブジェクト。 */
  private final QueryHistory history;

  /** 接続制限時刻 [msec]。 */
  private final long limitTimeMillis;

  /**
   * 接続設定オブジェクトを引数にとるコンストラクタ。
   *
   * @param config 接続設定オブジェクト。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected QueryConnection(QueryConnectionConfig config)
  {
    assert (config != null) : "@param:config is null.";

    this.config = config;
    this.history = newQueryHistory();
    this.limitTimeMillis = 0L;
  }

  /**
   * 接続設定オブジェクトとトランザクションを引数にとるコンストラクタ。
   * <br>
   * 引数のトランザクションからは、{@link IQueryHistory}オブジェクトと
   * トランザクション終了時刻の制限値を受け取って使用する。
   *
   * @param config 接続設定オブジェクト。
   * @param tran トランザクション・オブジェクト。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected QueryConnection(QueryConnectionConfig config,
    IQueryTransaction tran)
  {
    assert (config != null) : "@param:config is null.";

    this.config = config;
    this.history = QueryHistory.class.cast(tran.getQueryHistory());
    this.limitTimeMillis = tran.getLimitTimeMillis();
  }

  /**
   * クエリ実行履歴オブジェクトを作成する。
   *
   * @return クエリ実行履歴オブジェクト。
   */
  protected QueryHistory newQueryHistory()
  {
    return new QueryHistory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getConnectionId()
  {
    return this.config.getConnectionId();
  }

  /**
   * 接続設定オブジェクトを取得する。
   *
   * @return 接続設定オブジェクト。
   */
  protected QueryConnectionConfig getConfig()
  {
    return this.config;
  }

  /**
   * 接続開始時刻を取得する。
   *
   * @return 接続開始時刻。
   */
  protected abstract long getOpenTimeMillis();

  /**
   * {@inheritDoc}
   */
  public long getLimitTimeMillis()
  {
    long beginTimeMillis = getOpenTimeMillis();
    long spentTimeMillis = getConfig().getLimitSpentTime();
    long limitTimeMillis = this.limitTimeMillis;

    if (beginTimeMillis <= 0L || spentTimeMillis <= 0L) {
      return limitTimeMillis;
    }
    else if (limitTimeMillis <= 0L) {
      return (beginTimeMillis + spentTimeMillis);
    }
    else {
      return Math.min((beginTimeMillis + spentTimeMillis), limitTimeMillis);
    }
  }

  /**
   * このコネクションを使って実行されたクエリの実行履歴オブジェクトを取得する。
   *
   * @return {@link IQueryHistory}オブジェクト。
   */
  protected QueryHistory getQueryHistoryObject()
  {
    return this.history;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryHistory getQueryHistory()
  {
    return getQueryHistoryObject();
  }
}
