/*
 * QueryEnvironment class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as publicshed by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import ts.util.ReasonedRuntimeException;
import java.io.File;
import java.util.ResourceBundle;
import java.util.MissingResourceException;

/**
 * クエリを実行するための環境を表すクラス。
 * <br>
 * {@link QueryEnvironmentConfig}オブジェクトを内部に持ち、環境設定ファイル
 * からロードした設定データを取得する。
 * <br>
 * このクラスのインスタンスはシングルトン・インスタンスであり、同一プロセス内で
 * インスタンスが一つだけ生成され、どこからでもそのインスタンスを取得することが
 * できる。
 *
 * @author 佐藤隆之
 * @version $Id: QueryEnvironment.java,v 1.11 2012-02-17 17:14:10 tayu Exp $
 */
public final class QueryEnvironment
{
  /** このクラスのシングルトン・インスタンス。 */
  private final static QueryEnvironment INSTANCE = new QueryEnvironment();

  /** 環境設定オブジェクト。*/
  private final QueryEnvironmentConfig config;

  /**
   * このクラスのシングルトン・インスタンスを取得する。
   *
   * @return このクラスのシングルトン・インスタンス。
   */
  public static QueryEnvironment getInstance()
  {
    return INSTANCE;
  }

  /**
   * デフォルト・コンストラクタ。
   */
  private QueryEnvironment()
  {
    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    this.config = cfg;
  }

  /**
   * 環境設定オブジェクトを取得する。
   *
   * @return 環境設定オブジェクト。
   */
  protected QueryEnvironmentConfig getConfig()
  {
    return this.config;
  }

  /**
   * 接続設定ファイルを格納するディレクトリを取得する。
   *
   * 接続設定ファイルを格納するディレクトリを表す{@link File}オブジェクト。
   */
  public File getConnectionConfigDirectory()
  {
    return new File(getConfig().getConnectionConfigDirectory());
  }

  /**
   * 実行設定ファイルを格納するディレクトリを取得する。
   *
   * 実行設定ファイルを格納するディレクトリを表す{@link File}オブジェクト。
   */
  public File getExecutionConfigDirectory()
  {
    return new File(getConfig().getExecutionConfigDirectory());
  }

  /**
   * クエリ設定ファイルを格納するディレクトリを取得する。
   *
   * クエリ設定ファイルを格納するディレクトリを表す{@link File}オブジェクト。
   */
  public File getQueryConfigDirectory()
  {
    return new File(getConfig().getQueryConfigDirectory());
  }
}
