/*
 * QueryEnvironmentConfig class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as publicshed by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import ts.util.ReasonedRuntimeException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ResourceBundle;
import java.util.MissingResourceException;

/**
 * クエリを実行するための環境設定を保持するクラス。
 * <br>
 * リソース・パスが<tt>ts/query/QueryEnvironmentConfig.properties</tt>の環境
 * 設定ファイルからプロパティをロードして保持する。
 * 環境設定ファイルには、他の設定ファイルの配置先ディレクトリなど、アプリケー
 * ションの実行環境によって異なる設定を記述する。
 * <br>
 * 環境設定ファイルは、{@link ResourceBundle}によってロードされるので、アプリ
 * ケーションが参照可能なクラス・パスの下に配置したり、あるいはJARファイルに
 * 固めてアプリケーションが参照可能なライブラリ・パスの中に追加したりすること
 * によってロードすることができる。
 *
 * @author 佐藤隆之
 * @version $Id: QueryEnvironmentConfig.java,v 1.13 2012-03-09 16:12:34 tayu Exp $
 */
public class QueryEnvironmentConfig implements Serializable
{
  /** このクラスで発生しうるエラーを定義する列挙型。 */
  public enum Error {
    /** 環境設定ファイルのロードに失敗した場合。 */
    FailToLoadEnvironmentConfig,

    /** 環境設定ファイルに指定されたプロパティが存在しない場合。 */
    PropertyKeyNotFound,

    /** 環境設定ファイルの指定されたプロパティの取得に失敗した場合。 */
    FailToGetEnvironmentProperty,
  }

  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = -290495901596466072L;

  /** 環境設定を保持する{@link ResourceBundle}オブジェクト。 */
  private final ResourceBundle res = null;

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryEnvironmentConfig()
  {}

  /**
   * 指定されたキーのプロパティの値を取得する。
   *
   * @param key プロパティ・キー。
   * @return プロパティ値。
   * @throws ReasonedRuntimeException 環境設定ファイルのロードに失敗した場合。
   *   又は指定されたプロパティの取得に失敗した場合。
   */
  protected String getProperty(String key) throws ReasonedRuntimeException
  {
    final String BASE = "ts.query.QueryEnvironmentConfig";

    ResourceBundle rb = null;
    try {
      rb = ResourceBundle.getBundle(BASE);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.FailToLoadEnvironmentConfig,
        "[base=" + BASE + "]", e);
    }

    try {
      return rb.getString(key);
    }
    catch (MissingResourceException e) {
      throw new ReasonedRuntimeException(Error.PropertyKeyNotFound,
        "[base=" + BASE + "][key=" + key + "]", e);
    }
    catch (RuntimeException e) {
      throw new  ReasonedRuntimeException(Error.FailToGetEnvironmentProperty,
        "[base=" + BASE + "][key=" + key + "]", e);
    }
  }

  /**
   * 接続設定ファイルを格納するディレクトリ・パスを取得する。
   *
   * @return 接続設定ファイルを格納するディレクトリ・パス。
   */
  public String getConnectionConfigDirectory()
  {
    return getProperty("ts-query.environment.connection.config.directory");
  }

  /**
   * 実行設定ファイルを格納するディレクトリ・パスを取得する。
   *
   * @return 実行設定ファイルを格納するディレクトリ・パス。
   */
  public String getExecutionConfigDirectory()
  {
    return getProperty("ts-query.environment.execution.config.directory");
  }

  /**
   * クエリ設定ファイルを格納するディレクトリ・パスを取得する。
   *
   * @return クエリ設定ファイルを格納するディレクトリ・パス。
   */
  public String getQueryConfigDirectory()
  {
    return getProperty("ts-query.environment.query.config.directory");
  }

  /**
   * 環境設定ファイルのサンプルを{@link PrintWriter}オブジェクトに出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力中に異常が発生した場合。
   */
  protected static void outputSample(PrintWriter pw) throws IOException
  {
    pw.println("#");
    pw.println("# Query Environment Configuration File.");
    pw.println("#");
    pw.println("# This file is a Java property file which specifys " +
               "directories of other");
    pw.println("# configuration files for ts-query modules.");
    pw.println("#");
    pw.println("");
    pw.println("# The directory of query configuration files.");
    pw.println("ts-query.environment.query.config.directory = ...");
    pw.println("");
    pw.println("# The directory of connection configuration files.");
    pw.println("ts-query.environment.connection.config.directory = ...");
    pw.println("");
    pw.println("# The directory of execution configuration files.");
    pw.println("ts-query.environment.execution.config.directory = ...");
    pw.println("");
    pw.println("##.");
  }

  /**
   * 環境設定ファイルのサンプルを作成するコマンドを実行する。
   * <br>
   * サンプル・ファイル名は <tt>"QueryEnvironmentConfig.properties.sample"</tt>
   * である。
   * <br>
   * コマンドライン引数の配列の構成は以下の通りである：
   * <ol>
   *  <li>args[0] - コマンド名</li>
   *  <li>args[1] - 出力先ディレクトリ・パス</li>
   *  <li>args[2] - 出力ファイルの文字エンコーディング</li>
   * </ol>
   *
   * @param args コマンドライン引数の配列。
   * @return コマンドの終了コード。
   */
  static int executeCommand(String[] args)
  {
    String pgmName = "null", encoding;
    File dir;
    try {
      if (args != null && args.length > 0) {
        pgmName = args[0];
      }

      dir = new File(args[1]);
      encoding = args[2];

      if (args.length > 3) {
        throw new IllegalArgumentException();
      }
    }
    catch (Exception e) {
      System.err.println("HELP:");
      System.err.println(pgmName + " <output-dir-path> <output-file-encoding>");
      System.err.println("Create a sample file of Query Environment " +
                         "Configuration.");
      System.err.println();
      return 1;
    }

    try {
      new String(new byte[0], encoding);
    }
    catch (UnsupportedEncodingException e) {
      System.err.println("ERROR:");
      System.err.println("Specified encoding is unsupported: " + encoding);
      System.err.println();
      return 2;
    }

    try {
      File file = new File(dir, "QueryEnvironmentConfig.properties.sample");
      PrintWriter pw = null;
      try {
        pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
          encoding));
        outputSample(pw);
        pw.flush();
        return 0;
      }
      finally {
        if (pw != null) try { pw.close(); } catch (Exception e) {}
      }
    }
    catch (Exception e) {
      System.err.println("ERROR:");
      System.err.println(e.toString());
      System.err.println();
      return 3;
    }
  }

  /**
   * 環境設定ファイルのサンプルを作成するコマンドのメイン関数。
   * <br>
   * サンプル・ファイル名は <tt>"QueryEnvironmentConfig.properties.sample"</tt>
   * である。
   * <br>
   * コマンドライン引数の配列の構成は以下の通りである：
   * <ol>
   *  <li>args[0] - コマンド名</li>
   *  <li>args[1] - 出力先ディレクトリ</li>
   *  <li>args[2] - 出力ファイルの文字エンコーディング</li>
   * </ol>
   *
   * @param args コマンドライン引数。
   */
  public static void main(String[] args) { System.exit(executeCommand(args)); }
}
