/*
 * QueryExecution class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.table.Header;
import ts.util.table.Table;
import ts.util.table.ArrayListTable;
import java.io.Serializable;
import java.util.Map;

/**
 * クエリ処理の中の一回の照会又は更新の実行を表すクラスの抽象クラス。
 * <br>
 * 一回のクエリ処理の中で、異なる接続先に対する複数の照会や更新を実行することが
 * できるが、このクラスはその一つ一つの照会／更新の実行に対応している。
 * どの照会／更新処理を実行するかは、クエリの実行設定を保持する{@link
 * QueryExecutionConfig}オブジェクトから取得する。
 * 
 * @author 佐藤隆之
 * @version $Id: QueryExecution.java,v 1.8 2012-03-14 07:49:20 tayu Exp $
 */
public abstract class QueryExecution implements IQueryExecution
{
  /** 実行設定オブジェクト。 */
  private final QueryExecutionConfig config;

  /** {@link IQueryConnection}オブジェクト。 */
  private final IQueryConnection connection;

  /**
   * 実行設定オブジェクトを引数にとるコンストラクタ。
   * <br>
   * このオブジェクトの中で使用される{@link IQueryConnection}オブジェクトは、
   * 実行設定オブジェクトが保持する接続元IDを使って作成される。
   *
   * @param config {@link QueryExecutionConfig}オブジェクト。
   * @throws ReasonedException このオブジェクトで使用する{@link
   *   IQueryConnection}オブジェクトの作成に失敗した場合。
   * @throws ReasonedRuntimeException このオブジェクトが使用する接続設定が不正
   *   だった場合。
   * @throws AssertionError 引数がヌルの場合。
   */
  public QueryExecution(QueryExecutionConfig config)
    throws ReasonedException, ReasonedRuntimeException
  {
    assert (config != null) : "@param:config is null.";

    this.config = config;

    String connId = config.getConnectionId();
    QueryConnectionConfig connCfg = new QueryConnectionConfig(connId);
    this.connection = connCfg.create();
  }

  /**
   * 実行設定オブジェクトと{@link IQueryConnection}オブジェクトを引数にとる
   * コンストラクタ。
   *
   * @param config {@link QueryExecutionConfig}オブジェクト。
   * @param conn {@link IQueryConnection}オブジェクト。
   * @throws AssertionError 引数がヌルの場合。
   */
  public QueryExecution(QueryExecutionConfig config, IQueryConnection conn)
  {
    assert (config != null && conn != null) : 
      (config != null) ? "@param:config is null." : "@param:conn is null.";

    this.config = config;
    this.connection = conn;
  }

  /**
   * 実行設定オブジェクトを取得する。
   *
   * @return 実行設定オブジェクト。
   */
  protected QueryExecutionConfig getConfig()
  {
    return this.config;
  }

  /**
   * {@link IQueryConnection}オブジェクトを取得する。
   *
   * @return {@link IQueryConnection}オブジェクト。
   */
  protected <T extends IQueryConnection> T getQueryConnection()
  {
    @SuppressWarnings("unchecked")
    T t = (T) this.connection;
    return t;
  }

  /**
   * 実行IDを取得する。
   *
   * @return 実行ID。
   */
  public String getExecutionId()
  {
    return this.config.getExecutionId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException
  {
    String connId = getQueryConnection().getConnectionId();
    QueryResult rslt = new QueryResult("");
    execute(inputMap, rslt);
    return rslt;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException
  {
    assert (inputMap != null && result != null) :
      (inputMap == null) ? "@param:inputMap is null." :
      (result   == null) ? "@param:result is null." : "";

    long beginTm = System.currentTimeMillis();
    long limitTm = calcLimitTimeMillis(beginTm);
    int limitCnt = calcLimitFetchCount();

    QueryResult rslt = QueryResult.class.cast(result);
    if (rslt.getBeginTimeMillis() <= 0 || rslt.getBeginTimeMillis() > beginTm) {
      rslt.setBeginTimeMillis(beginTm);
    }

    QueryExecutionContent content = createContent();
    content.setLimitTimeMillis(limitTm);
    content.setLimitFetchCount(limitCnt);

    try {
      checkTimeout(content);
      prepareContent(content, inputMap);
      prepareResultTable(content);

      checkTimeout(content);
      executeContent(content);
    }
    catch (Exception e) {
      rslt.setException(e);
      throw rslt.getException();
    }
    finally {
      disposeContent(content, rslt);
      rslt.setEndTimeMillis(System.currentTimeMillis());
    }
  }

  /**
   * 終了時刻の制限値を算出する。
   *
   * @param beginTimeMillis 開始時刻 [msec]。
   * @return 終了時刻の制限値 [msec]。
   */
  protected long calcLimitTimeMillis(long beginTimeMillis)
  {
    long spentTimeMillis = getConfig().getLimitSpentTime();
    long limitTimeMillis = getQueryConnection().getLimitTimeMillis();

    if (beginTimeMillis <= 0L || spentTimeMillis <= 0L) {
      return limitTimeMillis;
    }
    else if (limitTimeMillis <= 0L) {
      return (beginTimeMillis + spentTimeMillis);
    }
    else {
      return Math.min(limitTimeMillis, (beginTimeMillis + spentTimeMillis));
    }
  }

  /**
   * 取得データ件数の制限値を算出する。
   *
   * @return 取得データ件数の制限値。
   */
  protected int calcLimitFetchCount()
  {
    return getConfig().getLimitFetchCount();
  }

  /**
   * タイムアウトの判定を行う。
   *
   * @param content 実行内容オブジェクト。
   * @throws ReasonedException タイムアウトの場合。
   */
  protected void checkTimeout(QueryExecutionContent content)
    throws ReasonedException
  {
    long limitTimeMillis = content.getLimitTimeMillis();
    if (limitTimeMillis <= 0L) {
      return;
    }

    long currentTimeMillis = System.currentTimeMillis();
    if (currentTimeMillis > limitTimeMillis) {
      throw new ReasonedException(IQueryExecution.Error.Timeout,
        "[connection id=" + content.getConnectionId() + "]" +
        "[execution id=" + content.getExecutionId() + "]" +
        "[limit time=" + limitTimeMillis + "]" +
        "[current time=" + currentTimeMillis + "]" +
      "");
    }
  }

  /**
   * 取得データ件数オーバーの判定を行う。
   * <br>
   * 実行内容オブジェクトが保持する取得データ件数が制限件数をオーバーしていない
   * 場合はそのままリターンする。
   * 制限件数がゼロ以下の場合や、実行内容が結果データ・テーブルを持たない場合も
   * そのままリターンする。
   * <br>
   * 取得データ件数が制限件数をオーバーした場合は例外をスローする。
   *
   * @param content 実行内容オブジェクト。
   * @param count 実際の取得データ件数。
   * @throws ReasonedException 取得件数オーバーの場合。
   */
  protected void checkFetchCount(QueryExecutionContent content, int count)
    throws ReasonedException
  {
    int limitCount = content.getLimitFetchCount();
    if (limitCount <= 0) {
      return;
    }

    if (count > limitCount) {
      throw new ReasonedException(IQueryExecution.Error.FetchCountOver,
        "[connection id=" + content.getConnectionId() + "]" +
        "[execution id=" + content.getExecutionId() + "]" +
        "[limit count=" + limitCount + "]" +
        "[actual count=" + count + "]" +
      "");
    }
  }

  /**
   * 実行内容オブジェクトを作成する。
   * 
   * @return 実行内容オブジェクト。
   */
  protected QueryExecutionContent createContent()
  {
    QueryExecutionContent content = new QueryExecutionContent();
    content.setExecutionId(getConfig().getExecutionId());
    content.setConnectionId(getQueryConnection().getConnectionId());
    return content;
  }

  /**
   * 実行内容オブジェクトに結果データ・テーブルを設定する。
   * <br>
   * 実行内容オブジェクトの内容が更新処理のみの場合はヌルを設定する。
   * また、引数の実行結果オブジェクトに同じ実行IDの結果データ・テーブルが登録
   * されている場合はそれを設定する。
   * それ以外の場合は、結果データ・テーブルを新規に作成して設定する。
   *
   * @param content 実行内容オブジェクト。
   */
  protected void prepareResultTable(QueryExecutionContent content)
  {
    if (! content.hasResultTable()) {
      content.setResultTable(null);
    }
    else {
      content.setResultTable(newResultTable(content));
    }
  }

  /**
   * 結果データを格納するテーブルを作成する。
   *
   * @return 結果テーブル。
   */
  protected Table<String,Serializable> newResultTable(
    QueryExecutionContent content)
  {
    Header<String> header = new ArrayListTable.Header<String>(
      content.countOutputs());

    for (QueryExecutionOutput output : content.getOutputs()) {
      header.addColumn(output.getName());
    }

    return new ArrayListTable<String,Serializable>(header);
  }

  /**
   * 実行内容オブジェクトの準備処理を実行する。
   * <br>
   * このオブジェクトが保持する実行設定と引数の入力パラメータ、実行結果オブジェ
   * クトを使って実行内容を構築する。
   * <br>
   * 但し、実行結果テーブルの準備処理はこのメソッドの実行直後に呼ばれる{@link
   * #prepareResultTable(QueryExecutionContent)}メソッドによって行われるので、
   * このメソッドで実行結果テーブルの設定を行う必要はない。
   *
   * @param content 実行内容オブジェクト。
   * @param inputMap 入力パラメータ・マップ。
   * @throws ReasonedException 準備処理に失敗した場合。
   * @throws ReasonedRuntimeException 実行設定等が不正だった場合。
   */
  protected abstract void prepareContent(
    QueryExecutionContent content, Map<String,Object> inputMap
  ) throws ReasonedException, ReasonedRuntimeException;

  /**
   * 実行内容オブジェクトの内容を実行する。
   *
   * @param content 実行内容オブジェクト。
   * @throws ReasonedException 実行内容オブジェクトの内容の実行に失敗した場合。
   * @throws ReasonedRuntimeException 実行設定等が不正だった場合。
   */
  protected abstract void executeContent(QueryExecutionContent content)
    throws ReasonedException, ReasonedRuntimeException;

  /**
   * 実行内容オブジェクトの後始末をする。
   * <br>
   * このメソッドは、{@link #executeContent(QueryExecutionContent)}メソッドが
   * 実行された後に呼び出される。
   * {@link #executeContent(QueryExecutionContent)}メソッドの実行途中で例外が
   * 発生した場合も、このメソッドは呼ばれる。
   * <br>
   * 結果データ・テーブルなど実行内容オブジェクトが保持する結果情報を、実行結果
   * オブジェクトに移すなどの処理を行うために用意されている。
   *
   * @param content 実行内容オブジェクト。
   * @param result 実行結果オブジェクト。
   */
  protected void disposeContent(QueryExecutionContent content,
    QueryResult result)
  {
    if (content.hasResultTable() && content.getResultTable() != null) {
      String execId = content.getExecutionId();
      result.getResultTableMap().put(execId, content.getResultTable());
    }
    else {
      String execId = content.getExecutionId();
      result.getResultTableMap().remove(execId);
    }
  }
}
