/*
 * QueryExecutionConfig class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.TypedGetter;
import ts.util.AbstractTypedGetter;
import ts.util.resource.Resource;
import ts.util.resource.PropertyResource;
import ts.util.resource.XmlResource;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.util.List;

/**
 * クエリ実行設定を保持するクラス。
 * <br>
 * 一回の照会又は更新処理を表す{@link IQueryExecution}オブジェクトを作成する
 * ために必要な設定情報を格納する。
 * <br>
 * 引数なしのコンストラクタを使った場合は、空の設定オブジェクトが作成され、
 * {@link #getResource()}メソッドにより得られるリソース・オブジェクトに設定情報
 * を登録して使用する。
 * <br>
 * 引数に実行IDをとるコンストラクタを使った場合は、自動的に{@link
 * QueryEnvironmentConfig}オブジェクトから取得したディレクトリから実行IDを
 * ファイル・タイトルに持つXMLファイル(.xml)又はJavaプロパティ・ファイル
 * (.properties)をロードするので、その設定情報を使用する。
 * <br>
 * 設定情報は各実行クラスによって異なるが、それに依らない共通の設定情報として、
 * <ul>
 * <li>ts-query.execution.class - 使用される{@link IQuery}の派生クラス名(必須)。
 *   </li>
 * <li>ts-query.execution.connection.id - {@link IQueryExecution}オブジェクトが
 *   使用する接続先を表す接続先ID(必須)。</li>
 * <li>ts-query.execution.limit.fetchcount - 取得データ件数の制限値。</li>
 * <li>ts-query.execution.limit.spenttime - 実行時間の制限値 [msec]。</li>
 * </ul>
 * がある。
 *
 * @author 佐藤隆之
 * @version $Id: QueryExecutionConfig.java,v 1.16 2012-03-14 07:49:20 tayu Exp $
 */
public class QueryExecutionConfig implements Serializable
{
  /** このクラスで発生しうるエラーを定義する列挙型。 */
  public enum Error {
    /** 実行IDがヌル又は空文字列の場合。 */
    ExecutionIdIsNullOrEmpty,

    /** 実行設定ファイルが見つからない場合。 */
    ExecutionConfigFileNotFound,

    /** 実行設定ファイルのロードに失敗した場合。 */
    FailToLoadExecutionConfigFile,

    /** 設定情報に指定されたクラスが見つからない場合。 */
    ExecutionClassNotFound,

    /** 設定情報のクラスに要求されるコンストラクタが存在しない場合。 */
    ExecutionConstructorNotFound,

    /** {@link IQueryExecution}オブジェクトの作成に失敗した場合。 */
    FailToCreateExecution,

    /** 渡されたトランザクションがヌルの場合。 */
    TransactionIsNull,

    /** 渡されたコネクションがヌルの場合。 */
    ConnectionIsNull,

    /** 設定情報の取得データ件数の制限値が不正な場合。 */
    IllegalLimitFetchCount,

    /** 設定情報の処理時間の制限値が不正な場合。 */
    IllegalLimitSpentTime,
  }

  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 4161877040444495005L;

  /** 実行ID。 */
  private final String executionId;

  /** 設定情報を保持するリソース・オブジェクト。 */
  private final Resource resource;

  /** 設定情報を型変換して取得するための{@link TypedGetter}オブジェクト。 */
  private final AbstractTypedGetter<String,String> typedGetter;

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryExecutionConfig()
  {
    this.executionId = "";
    this.resource = new PropertyResource();
    this.typedGetter = newTypedGetter();
  }

  /**
   * 実行IDを引数にとるコンストラクタ。
   *
   * @param execId 実行ID。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryExecutionConfig(String execId)
  {
    if (execId == null) {
      throw new ReasonedRuntimeException(Error.ExecutionIdIsNullOrEmpty,
        "[execution Id=" + execId + "]");
    }

    execId = execId.trim();
    if (execId.isEmpty()) {
      throw new ReasonedRuntimeException(Error.ExecutionIdIsNullOrEmpty,
        "[execution Id=" + execId + "]");
    }

    this.executionId = execId;
    this.resource = loadResource();
    this.typedGetter = newTypedGetter();
  }

  /**
   * 実行IDを取得する。
   *
   * @return 実行ID。
   */
  public String getExecutionId()
  {
    return this.executionId;
  }

  /**
   * 実行設定ファイルをロードしたリソース・オブジェクトを作成する。
   *
   * @return 実行設定ファイルをロードしたリソース・オブジェクト。
   * @throws ReasonedRuntimeException 実行設定ファイルのロードに失敗した場合。
   */
  protected Resource loadResource() throws ReasonedRuntimeException
  {
    QueryEnvironment env = QueryEnvironment.getInstance();
    File dir = env.getExecutionConfigDirectory();

    File xmlFile = new File(dir, getExecutionId() + ".xml");
    if (xmlFile.exists()) {
      try {
        XmlResource res = new XmlResource();
        res.setValidating(false);
        res.load(xmlFile.getCanonicalPath());
        return res;
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadExecutionConfigFile,
          "[path=" + xmlFile.getAbsolutePath() + "]", e);
      }
    }

    File propFile = new File(dir, getExecutionId() + ".properties");
    if (propFile.exists()) {
      try {
        return new PropertyResource(propFile.getCanonicalPath());
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadExecutionConfigFile,
          "[path=" + propFile.getAbsolutePath() + "]", e);
      }
    }

    throw new ReasonedRuntimeException(Error.ExecutionConfigFileNotFound,
      "[path=" + new File(dir, getExecutionId()).getAbsolutePath() + 
      ".{xml|properties}]");
  }

  /**
   * 設定情報を型変換して取得するための{@link TypedGetter}オブジェクトを作成
   * する。
   *
   * @return {@link TypedGetter}オブジェクト。
   */
  protected AbstractTypedGetter<String,String> newTypedGetter()
  {
    return new AbstractTypedGetter<String,String>() {
      static final long serialVersionUID =
        QueryExecutionConfig.serialVersionUID + 1L;
      @Override
      public String get(String key) {
        return QueryExecutionConfig.this.resource.getFirstValue(key);
      }
      @Override
      public List<String> getList(String key) {
        return QueryExecutionConfig.this.resource.getValues(key);
      }
    };
  }

  /**
   * 実行設定を保持するリソース・オブジェクトを取得する。
   *
   * @return 実行設定を保持するリソース・オブジェクト。
   */
  protected Resource getResource()
  {
    return this.resource;
  }

  /**
   * 設定情報を型変換して取得するための{@link TypedGetter}オブジェクトを取得
   * する。
   *
   * @return {@link TypedGetter}オブジェクト。
   */
  protected AbstractTypedGetter<String,String> typedGetter()
  {
    return this.typedGetter;
  }

  /**
   * {@link IQueryExecution}オブジェクトを作成する。
   * <br>
   * このオブジェクトが保持する実行設定を使って、{@link IQueryExecution}の派生
   * クラスのインスタンスを作成する。
   * 派生クラスの名前は、実行設定 <tt>ts-query.execution.class</tt>に指定された
   * クラス名が使用される。
   *
   * @return {@link IQueryExecution}オブジェクト。
   * @throws ReasonedException {@link IQueryExecution}オブジェクト又はそれが
   *   使用する{@link IQueryConnection}オブジェクトの作成に失敗した場合。
   * @throws ReasonedRuntimeException 実行設定又は接続設定が不正だった場合。
   */
  public <T extends IQueryExecution> T create()
    throws ReasonedException, ReasonedRuntimeException
  {
    String connId = getConnectionId();
    QueryConnectionConfig connCfg = new QueryConnectionConfig(connId);
    IQueryConnection conn = connCfg.create();

    @SuppressWarnings("unchecked")
    T t = (T) create(conn);
    return t;
  }

  /**
   * {@link IQueryExecution}オブジェクトを作成する。
   * <br>
   * このオブジェクトが保持する実行設定と引数のトランザクションを使って、{@link
   * IQueryExecution}の派生クラスのインスタンスを作成する。
   * 派生クラスの名前は、実行設定 <tt>ts-query.execution.class</tt>に指定された
   * クラス名が使用される。
   * <br>
   * {@link IQueryExecution}オブジェクトが使用するコネクション・オブジェクトは
   * 引数のトランザクション・オブジェクトから取得する。
   *
   * @param tran トランザクション・オブジェクト。
   * @return {@link IQueryExecution}オブジェクト。
   * @throws ReasonedException {@link IQueryExecution}オブジェクト又はそれが
   *   使用する{@link IQueryConnection}オブジェクトの作成に失敗した場合。
   * @throws ReasonedRuntimeException 実行設定又は接続設定が不正だった場合。
   */
  public <T extends IQueryExecution> T create(IQueryTransaction tran)
    throws ReasonedException, ReasonedRuntimeException
  {
    if (tran == null) {
      throw new ReasonedRuntimeException(Error.TransactionIsNull);
    }

    String connId = getConnectionId();
    IQueryConnection conn = tran.getQueryConnection(connId);

    @SuppressWarnings("unchecked")
    T t = (T) create(conn);
    return t;
  }

  /**
   * {@link IQueryExecution}オブジェクトを作成する。
   * <br>
   * このオブジェクトが保持する実行設定と引数のコネクションを使って、{@link
   * IQueryExecution}の派生クラスのインスタンスを作成する。
   * 派生クラスの名前は、実行設定 <tt>ts-query.execution.class</tt>に指定された
   * クラス名が使用される。
   * <br>
   * 引数のコネクション・オブジェクトからは、{@link IQueryHistory}オブジェクト
   * トランザクション開始からの制限時間を受け取って使用する。
   *
   * @param conn コネクション・オブジェクト。
   * @return {@link IQueryExecution}オブジェクト。
   * @throws ReasonedException {@link IQueryExecution}オブジェクトの作成に失敗
   *   した場合。
   * @throws ReasonedRuntimeException 実行設定が不正だった場合。
   */
  public <T extends IQueryExecution> T create(IQueryConnection conn)
    throws ReasonedException, ReasonedRuntimeException
  {
    if (conn == null) {
      throw new ReasonedRuntimeException(Error.ConnectionIsNull);
    }

    String cls = getExecutionClass();
    Class<T> execCls = null;
    try {
      @SuppressWarnings("unchecked")
      Class<T> c = (Class<T>) Class.forName(cls);
      execCls = c;
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.ExecutionClassNotFound,
        "[class=" + cls + "]", e);
    }

    Constructor<T> cons = null;
    try {
      cons = execCls.getConstructor(QueryExecutionConfig.class,
        IQueryConnection.class);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.ExecutionConstructorNotFound,
        "[class=" + cls + "]", e);
    }

    try {
      return cons.newInstance(this, conn);
    }
    catch (Exception e) {
      throw new ReasonedException(Error.FailToCreateExecution,
        "[class=" + cls + "]", e);
    }
  }

  /**
   * このオブジェクトから作成される{@link IQueryExecution}オブジェクトのクラス
   * 名を取得する。
   *
   * @return {@link IQueryExecution}クラスの名前。
   */
  protected String getExecutionClass()
  {
    return typedGetter().getString("ts-query.execution.class");
  }

  /**
   * {@link IQueryExecution}オブジェクトが使用するコネクションの接続先IDを取得
   * する。
   *
   * @return {@link IQueryExecution}オブジェクトが使用するコネクションの
   *   接続先ID。
   */
  protected String getConnectionId()
  {
    return typedGetter().getString("ts-query.execution.connection.id");
  }

  /**
   * このオブジェクトから作成される{@link IQueryExecution}オブジェクトの取得
   * データ件数の制限値を取得する。
   * <br>
   * 取得された制限件数がゼロ以下の場合は、制限がないものとする。
   * <br>
   * このオブジェクトに制限件数の設定がない場合は、制限なしとして<tt>-1</tt>を
   * 返す。
   *
   * @return {@link IQueryExecution}オブジェクトの結果データ件数の制限値。
   */
  protected int getLimitFetchCount()
  {
    final String KEY = "ts-query.execution.limit.fetchcount";

    try {
      return typedGetter().getInteger(KEY);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.IllegalLimitFetchCount,
        "[property=" + KEY + "][value=" + getResource().getFirstValue(KEY) +
        "]", e);
    }
  }

  /**
   * このオブジェクトから作成される{@link IQueryExecution}オブジェクトの
   * 制限時間を取得する。
   * <br>
   * 取得された制限時間がゼロ以下の場合は、制限がないものとする。
   * <br>
   * このオブジェクトに制限時間の設定がない場合は、制限なしとして<tt>-1</tt>を
   * 返す。
   *
   * @return {@link IQueryExecution}オブジェクトの制限時間 [msec]。
   */
  protected long getLimitSpentTime()
  {
    final String KEY = "ts-query.execution.limit.spenttime";

    try {
      return typedGetter().getLong(KEY);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.IllegalLimitSpentTime,
        "[property=" + KEY + "][value=" + getResource().getFirstValue(KEY) +
        "]", e);
    }
  }

  /**
   * 実行設定ファイルのXML形式のサンプルを{@link PrintWriter}オブジェクトに
   * 出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleXml(PrintWriter pw, String encoding)
    throws IOException
  {
    pw.println("<?xml version=\"1.0\" encoding=\"" + encoding.toString() +
               "\"?>");
    pw.println("");
    pw.println("<!--");
    pw.println("  || Query Execution Configuration XML File.");
    pw.println("  ||");
    pw.println("  || This file is an XML file which specifys " +
               "some configuration for an ");
    pw.println("  || IQueryExecution object.");
    pw.println("  ||");
    pw.println("  -->");
    pw.println("");
    pw.println("<ts-query>");
    pw.println(" <execution>");
    pw.println("  <!-- The query execution class -->");
    pw.println("  <class>...</class>");
    pw.println("  <connection>");
    pw.println("    <!-- The query connection id -->");
    pw.println("    <id>...</id>");
    pw.println("  </connection>");
    pw.println("  <limit>");
    pw.println("    <!-- Limit fetch count -->");
    pw.println("    <fetchcount>0</fetchcount>");
    pw.println("");
    pw.println("    <!-- Limit spent time [msec] -->");
    pw.println("    <spenttime>0</spenttime>");
    pw.println("  </limit>");
    pw.println("");

    outputSampleXmlEntries(pw);

    pw.println(" </execution>");
    pw.println("</ts-query>");
    pw.println();
  }

  /**
   * 実行設定ファイルのXMLエントリのサンプルを{@link PrintWriter}オブジェクト
   * に出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleXmlEntries(PrintWriter pw) throws IOException
  {}

  /**
   * 実行設定ファイルのJavaプロパティ形式のサンプルを{@link PrintWriter}
   * オブジェクトに出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleProp(PrintWriter pw) throws IOException
  {
    pw.println("#");
    pw.println("# Query Execution Property File.");
    pw.println("#");
    pw.println("# This file is a Java property file which specifys " +
               "some configurations for an ");
    pw.println("# IQueryExecution object.");
    pw.println("#");
    pw.println("");
    pw.println("# The query execution class.");
    pw.println("ts-query.execution.class = ...");
    pw.println("");
    pw.println("# The query connection id");
    pw.println("ts-query.execution.connection.id = ...");
    pw.println("");
    pw.println("# Limit fetch count");
    pw.println("ts-query.execution.limit.fetchcount = 0");
    pw.println("");
    pw.println("# Limit spent time [msec]");
    pw.println("ts-query.execution.limit.spenttime = 0");
    pw.println("");

    outputSamplePropEntries(pw);

    pw.println("");
    pw.println("#.");
  }

  /**
   * 実行設定ファイルのプロパティ・エントリのサンプルを{@link PrintWriter}
   * オブジェクトに出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSamplePropEntries(PrintWriter pw) throws IOException
  {}

  /**
   * 実行設定ファイルのサンプルを出力するコマンドを実行する。
   * <br>
   * コマンドライン引数の配列の構成は以下の通りである：
   * <ol>
   *  <li>args[0] - コマンド名</li>
   *  <li>args[1] - 出力ファイル・パス</li>
   *  <li>args[2] - 出力ファイル形式 (<tt>"xml"</tt>:XML形式、<tt>"prop"</tt>
   *    :Javaプロパティ形式、但し大文字小文字は区別しない)</li>
   *  <li>args[3] - 出力ファイルの文字エンコーディング</li>
   * </ol>
   * また、終了コードは以下の値をとりうる：
   * <ul>
   * <li><tt>0</tt> - 正常終了の場合</li>
   * <li><tt>1</tt> - コマンドライン引数が不正な場合</li>
   * <li><tt>2</tt> - 非対応の文字エンコーディングが指定された場合</li>
   * <li><tt>3</tt> - 上記以外のエラーの場合</li>
   * </ul>
   *
   * @param args コマンドライン引数の配列。
   * @param config 実行設定オブジェクト。
   * @return コマンドの終了コード。
   */
  protected static int executeCommand(
    String[] args, QueryExecutionConfig config)
  {
    String pgmName = "null", encoding = "";
    File file;
    boolean isXml;
    try {
      if (args != null && args.length > 0) {
        pgmName = args[0];
      }

      file = new File(args[1]);

      if ("xml".equalsIgnoreCase(args[2])) {
        isXml = true;
      }
      else if ("prop".equalsIgnoreCase(args[2])) {
        isXml = false;
      }
      else {
        throw new IllegalArgumentException();
      }

      encoding = args[3];

      if (args.length > 4) {
        throw new IllegalArgumentException();
      }
    }
    catch (Exception e) {
      System.err.println("HELP:");
      System.err.println(
        pgmName + " <output-file-path> {xml|prop} <output-file-encoding>");
      System.err.println();
      return 1;
    }

    try {
      new String(new byte[0], encoding);
    }
    catch (UnsupportedEncodingException e) {
      System.err.println("ERROR:");
      System.err.println("Specified encoding is unsupported: " + encoding);
      System.err.println();
      return 2;
    }

    try {
      PrintWriter pw = null;
      try {
        pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
          encoding));

        if (isXml) {
          config.outputSampleXml(pw, encoding);
        }
        else {
          config.outputSampleProp(pw);
        }
        pw.flush();
        return 0;
      }
      finally {
        if (pw != null) try { pw.close(); } catch (Exception e) {}
      }
    }
    catch (Exception e) {
      System.err.println("ERROR:");
      System.err.println(e.toString());
      System.err.println();
      return 3;
    }
  }
}
