/*
 * QueryExecutionParam class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import ts.util.table.MapIterator;
import ts.util.table.Table;
import java.io.Serializable;
import java.util.Map;
import java.util.HashMap;
import java.util.Collection;
import java.util.Iterator;
import java.util.Arrays;

/**
 * クエリ・パラメータ・クラス。
 * <br>
 * クエリの実行内容に適用するパラメータを表すクラス。
 * パラメータの名前、値を属性に持ち、その他にデータ型などの情報を任意の属性と
 * して保持することができる。
 * <br>
 * {@link #getValue()}メソッドは、値が単数の場合はその値を常に返す。
 * 値が複数の場合はそれらの値を順番に返して最後に至った場合は最後の値を返し続
 * ける。
 *
 * @author 佐藤隆之
 * @version $Id: QueryExecutionParam.java,v 1.5 2012-03-14 07:49:20 tayu Exp $
 */
public class QueryExecutionParam
  extends AbstractTypedGetter<Enum<?>,Serializable>
{
  /** 入出力区分を表す列挙型。 */
  public enum IO {
    /** 入出力区分が入力であることを表す列挙値。 */
    IN,
    /** 入出力区分が出力であることを表す列挙値。 */
    OUT,
    /** 入出力区分が入力兼出力であることを表す列挙値。 */
    INOUT,
  }

  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = -7316624337998775137L;

  /** パラメータ名。 */
  private final String name;

  /** 入出力区分。 */
  private final IO io;

  /** パラメータ値。パラメータ値が複数の場合は最後の要素の値を保持する。 */
  private Serializable value = null;

  /** パラメータ値が複数の場合にそれらを順番に取り出すためのイテレータ。 */
  private Iterator<Serializable> iterator = null;

  /** パラメータ値が複数の場合にその数を数えるオブジェクト。 */
  private Counter counter = null;

  /** 任意の属性を格納するマップ。 */
  private Map<Enum<?>,Serializable> attributeMap = null;

  /**
   * パラメータ名を引数にとるコンストラクタ。
   * <br>
   * 入出力区分は入力、値はヌルに指定される。
   *
   * @param name パラメータ名。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryExecutionParam(String name)
  {
    this(name, IO.IN);
  }

  /**
   * パラメータ名と入出力区分を引数にとるコンストラクタ。
   * <br>
   * 値はヌルに指定される。
   *
   * @param name パラメータ名。
   * @param io 入出力区分。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryExecutionParam(String name, IO io)
  {
    assert (name != null && io != null) :
      (name == null) ? "@param:name is null." :
      (io   == null) ? "@param:io is null." : "";

    this.name = name;
    this.io = io;
  }

  /**
   * パラメータ名を取得する。
   *
   * @return パラメータ名。
   */
  public String getName()
  {
    return this.name;
  }

  /**
   * パラメータの入出力区分を取得する。
   *
   * @return パラメータの入出力区分。
   */
  public IO getIO()
  {
    return this.io;
  }

  /**
   * 指定されたキーに結びつけられた属性値を取得する。
   *
   * @param attrKey キー。
   * @return 属性値。
   */
  @Override
  public Serializable get(Enum<?> attrKey)
  {
    return (this.attributeMap == null) ? null : this.attributeMap.get(attrKey);
  }

  /**
   * 指定されたキーを属性名とする属性値を設定する。
   *
   * @param attrKey キー。
   * @param attrValue 属性値。
   * @return 以前このキーに結びつけられていた属性値。
   */
  public Serializable put(Enum<?> attrKey, Serializable attrValue)
  {
    if (this.attributeMap == null) {
      this.attributeMap = newAttributeMap();
    }

    if (attrValue == null) {
      return this.attributeMap.remove(attrKey);
    }
    else {
      return this.attributeMap.put(attrKey, attrValue);
    }
  }

  /**
   * 現在のパラメータ値を取得する。
   *
   * @return 現在のパラメータ値。
   */
  public Serializable getValue()
  {
    return this.value;
  }

  /**
   * 次のパラメータ値を現在値として設定する。
   * <br>
   * 値が単数の場合はその値を常に設定し、値が複数の場合はそれらの値を順番に設定
   * して最後に至った後は最後の値を設定し続ける。
   */
  public void next()
  {
    if (this.iterator != null && this.iterator.hasNext()) {
      this.value = this.iterator.next();
    }
  }

  /**
   * パラメータの値の数を取得する。
   *
   * @return パラメータの値の数。
   */
  public int countValues()
  {
    return (this.counter != null) ? this.counter.count() : 1;
  }

  /**
   * パラメータ値を設定する。
   * <br>
   * 引数がコレクション・オブジェクト又は配列の場合には、このパラメータが複数の
   * 値を持つものとし、それ以外は単数の値を持つものとして設定する。
   *
   * @param paramValue パラメータ値。
   */
  public void setValue(Object paramValue)
  {
    if (paramValue == null) {
      this.value = null;
      this.counter = null;
      this.iterator = null;
    }
    else if (paramValue instanceof Collection) {
      @SuppressWarnings("unchecked")
      final Collection<Serializable> c = (Collection<Serializable>) paramValue;

      this.value = null;
      this.iterator = c.iterator();
      this.counter = new Counter() { public int count() { return c.size(); } };
    }
    else if (paramValue instanceof Serializable[]) {
      final Serializable[] a = Serializable[].class.cast(paramValue);

      this.value = null;
      this.iterator = Arrays.asList(a).iterator();
      this.counter = new Counter() { public int count() { return a.length; } };
    }
    else {
      this.value = Serializable.class.cast(paramValue);
      this.counter = null;
      this.iterator = null;
    }

    next();
  }

  /**
   * 指定されたテーブルのカラムの値を、パラメータ値として設定する。
   *
   * @param table テーブル。
   * @param column カラム名。
   */
  public void setValue(final Table<String,Serializable> table,
    final String column)
  {
    if (table == null) {
      this.value = null;
      this.counter = null;
      this.iterator = null;
    }
    else {
      final MapIterator<String,Serializable> itr = table.records();

      this.value = null;
      this.iterator = new Iterator<Serializable>() {
        public boolean hasNext() { return itr.hasNext(); }
        public Serializable next() {
          Map<String,Serializable> m = itr.next();
          return  (m != null) ? m.get(column) : null;
        }
        public void remove() {}
      };
      this.counter = new Counter() {
        public int count() { return table.recordCount(); }
      };
    }

    next();
  }

  /**
   * 属性のキーと値の関連付けを格納するマップを作成する。
   *
   * @return 属性マップ。
   */
  protected Map<Enum<?>,Serializable> newAttributeMap()
  {
    return new HashMap<Enum<?>,Serializable>();
  }

  /**
   * 要素数を求めるためのクラスのインターフェイス。
   */
  protected interface Counter
  {
    int count();
  }
}
