/*
 * QueryResult class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.table.Table;
import java.io.Serializable;
import java.util.Collection;
import java.util.Map;
import java.util.HashMap;

/**
 * クエリの実行結果を表すクラス。
 * <br>
 * 実行の成否や実行時刻、失敗時の例外オブジェクトなどを保持する。
 * また、このクラスは結果IDごとに結果データ・テーブルを保持することができる。
 * 結果IDには任意の識別文字列を使用でき、通常は{@link IQueryExecution}を識別
 * する実行IDが使用される。
 *
 * @author 佐藤隆之
 * @version $Id: QueryResult.java,v 1.24 2012-02-27 04:46:12 tayu Exp $
 */
public class QueryResult implements IQueryResult
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = -6094683686497030260L;

  /** クエリID。 */
  private final String queryId;

  /** 開始時刻。 */
  private long beginTimeMillis = 0L;

  /** 終了時刻。 */
  private long endTimeMillis = 0L;

  /** 失敗時にスローされた例外オブジェクト。成功時はヌル。 */
  private ReasonedException exception = null;

  /** 結果IDと結果テーブルの対応を保持するマップ。 */
  private final Map<String,Table<String,Serializable>> resultTableMap =
    newResultTableMap();

  /**
   * クエリIDを引数にとるコンストラクタ。
   *
   * @param queryId クエリID。
   */
  public QueryResult(String queryId)
  {
    assert (queryId != null) : "@param:queryId is null.";

    this.queryId = queryId;
  }

  /**
   * 結果IDと結果テーブルの対応を保持するマップを作成する。
   *
   * @return 結果IDと結果テーブルの対応を保持するマップ。
   */
  protected Map<String,Table<String,Serializable>> newResultTableMap()
  {
    return new HashMap<String,Table<String,Serializable>>();
  }

  /**
   * クエリIDを取得する。
   *
   * @return クエリID。
   */
  public String getQueryId()
  {
    return this.queryId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSuccess()
  {
    return (this.exception != null) ? false : true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ReasonedException getException()
  {
    return this.exception;
  }

  /**
   * 結果が失敗の場合の例外オブジェクトを設定する。
   * <br>
   * 引数の例外オブジェクトを{@link ReasonedException}オブジェクトに型キャスト
   * 又は再作成して、このオブジェクトの属性として設定する。
   *
   * @param exc 例外オブジェクト。
   */
  protected void setException(Exception exc)
  {
    if (exc == null) {
      this.exception = null;
    }
    else if (exc instanceof ReasonedException) {
      this.exception = ReasonedException.class.cast(exc);
    }
    else if (exc instanceof ReasonedRuntimeException) {
      ReasonedRuntimeException rre = ReasonedRuntimeException.class.cast(exc);
      ReasonedException newExc = new ReasonedException(rre.getReason(),
        rre.getMessage(), rre);
      newExc.setStackTrace(rre.getStackTrace());
      this.exception = newExc;
    }
    else {
      ReasonedException newExc = new ReasonedException(Error.FailToExecute,
        exc.getMessage(), exc);
      newExc.setStackTrace(exc.getStackTrace());
      this.exception = newExc;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getBeginTimeMillis()
  {
    return this.beginTimeMillis;
  }

  /**
   * 開始時刻を設定する。
   *
   * @param tm 開始時刻。
   */
  protected void setBeginTimeMillis(long tm)
  {
    assert (tm >= 0L) : "@param:tm is negative.";

    this.beginTimeMillis = tm;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getEndTimeMillis()
  {
    return this.endTimeMillis;
  }

  /**
   * 終了時刻を設定する。
   *
   * @param tm 終了時刻。
   */
  protected void setEndTimeMillis(long tm)
  {
    assert (tm >= this.beginTimeMillis) : "@param:tm is less than begin time.";

    this.endTimeMillis = tm;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getSpentTimeMillis()
  {
    return Math.max((this.endTimeMillis - this.beginTimeMillis), 0L);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Collection<String> getAllResultIds()
  {
    return this.resultTableMap.keySet();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Table<String,Serializable> getResultTable(String rsltId)
  {
    return this.resultTableMap.get(rsltId);
  }

  /**
   * 結果IDと結果テーブルの対応を格納するマップを取得する。
   *
   * @return 結果IDと結果テーブルの対応を格納するマップ。
   */
  protected Map<String,Table<String,Serializable>> getResultTableMap()
  {
    return this.resultTableMap;
  }
}
