package ts.query;

import ts.tester.UnitTest;
import ts.tester.function.ObjectInspector;
import ts.util.*;
import java.util.*;

public class QueryTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryTest.class, args);
  }

  ObjectInspector oi = null;

  static ReasonedException THROW0 = null;
  static ReasonedRuntimeException THROW1 = null;
  static RuntimeException THROW2 = null;
  static StringBuilder BUF = null;
  static enum ENUM { AAA }

  static class MyExecution implements IQueryExecution {
    private QueryExecutionConfig config;
    public MyExecution(QueryExecutionConfig cfg, IQueryConnection conn)
    { this.config = cfg; }
    @Override
    public String getExecutionId() { return this.config.getExecutionId(); }
    @Override 
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException {
      BUF.append("[" + getExecutionId() + ":execute_inputMap]");
      return null;
    }
    @Override
    public void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException {
      BUF.append("[" + getExecutionId() + ":execute_inputMap_result]");
      if (THROW0 != null) throw THROW0;
      if (THROW1 != null) throw THROW1;
      if (THROW2 != null) throw THROW2;
    }
  }

  static class MyConnection implements IQueryConnection {
    private QueryConnectionConfig cfg;
    private boolean isClosed = true;
    public MyConnection(QueryConnectionConfig cfg) { this.cfg = cfg; }
    public MyConnection(QueryConnectionConfig cfg, IQueryTransaction tran)
    { this(cfg); }
    @Override
    public String getConnectionId() { return cfg.getConnectionId(); }
    @Override
    public long getLimitTimeMillis() { return 0L; }
    @Override
    public IQueryHistory getQueryHistory() { return null; }
    @Override
    public void open() throws ReasonedException {
      this.isClosed = false;
    }
    @Override
    public void commit() throws ReasonedException {}
    @Override
    public void rollback() throws ReasonedException {}
    @Override
    public void close() throws ReasonedException { this.isClosed = true; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public boolean isOpened() { return ! this.isClosed; }
  }

  protected void preInvocation(String method)
  {
    oi = new ObjectInspector(this);
    oi.expect("config", null);
    oi.expect("transaction", null);
    oi.expect("isNeededToControlTransaction", true);

    BUF = new StringBuilder();
    THROW0 = null;
    THROW1 = null;
    THROW2 = null;
  }


  public void constructor_config()
  {
    MSG("クエリ設定オブジェクトを引数にとるコンストラクタ。");

    QueryConfig config = new QueryConfig();
    Query query = new Query(config);

    oi.expect("config", config);
    oi.expect("transaction", query.getTransaction());
    oi.expect("isNeededToControlTransaction", true);
    oi.inspect(query);

    EQUAL(query.getConfig(), config);
    NOTNULL(query.getTransaction());
    TRUE(query.isNeededToControlTransaction());
    NOTNULL(query.getQueryHistory());
  }

  public void constructor_config_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      new Query(null);
      NG();
    }
    catch (AssertionError e) {
      OK(e);
    }
  }

  public void constructor_config_tran()
  {
    MSG("クエリ設定オブジェクトとトランザクションを引数にとるコンストラクタ。");

    QueryConfig config = new QueryConfig();
    QueryTransaction tran = new QueryTransaction();

    Query query = new Query(config, tran);

    oi.expect("config", config);
    oi.expect("transaction", tran);
    oi.expect("isNeededToControlTransaction", false);
    oi.inspect(query);

    EQUAL(query.getConfig(), config);
    EQUAL(query.getTransaction(), tran);
    FALSE(query.isNeededToControlTransaction());
    NOTNULL(query.getQueryHistory());
  }

  public void constructor_config_tran_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig config = new QueryConfig();
    QueryTransaction tran = new QueryTransaction();

    try {
      new Query(null, tran);
      NG();
    }
    catch (AssertionError e) {
      OK(e);
    }

    try {
      new Query(config, null);
      NG();
    }
    catch (AssertionError e) {
      OK(e);
    }
  }

  public void executeQuery_inputMap_result()
  {
    MSG("クエリ設定の実行IDに対応するクエリ実行を処理するメソッドの確認。");

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    Map<String,Object> inputMap = new HashMap<String,Object>();
    QueryResult result = new QueryResult(QID);

    QueryConfig config = new QueryConfig(QID);
    List<String> eids = config.getExecutionIds();
    EQUAL(eids.size(), 3);
    EQUAL(eids.get(0), EID0);
    EQUAL(eids.get(1), EID1);
    EQUAL(eids.get(2), EID2);

    IQueryTransaction tran = new QueryTransaction();

    Query query = new Query(config, tran);
    try {
      query.executeQuery(inputMap, result);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }
    catch (ReasonedException e) {
      NG(e);
    }

    try {
      tran.begin();
      query.executeQuery(inputMap, result);
      tran.commit();
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }

    EQUAL(BUF.toString(),
      "[" + EID0 + ":execute_inputMap_result]" +
      "[" + EID1 + ":execute_inputMap_result]" +
      "[" + EID2 + ":execute_inputMap_result]" +
    "");
  }

  public void executeQuery_inputMap_result_Null()
  {
    MSG("引数がヌルの場合。");

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    QueryConfig config = new QueryConfig(QID);
    List<String> eids = config.getExecutionIds();
    EQUAL(eids.size(), 3);
    EQUAL(eids.get(0), EID0);
    EQUAL(eids.get(1), EID1);
    EQUAL(eids.get(2), EID2);

    IQueryTransaction tran = new QueryTransaction();
    Query query = new Query(config, tran);
    try {
      tran.begin();
      query.executeQuery(null, null);
      tran.commit();
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }

    EQUAL(BUF.toString(),
      "[" + EID0 + ":execute_inputMap_result]" +
      "[" + EID1 + ":execute_inputMap_result]" +
      "[" + EID2 + ":execute_inputMap_result]" +
    "");
  }

  public void controlTransaction_inputMap_result()
  {
    MSG("トランザクション制御の確認。");

    IQueryTransaction tran = new QueryTransaction() {
      @Override
      public void begin() throws ReasonedException {
        super.begin();
        BUF.append("[Transaction Begined]");
      }
      @Override
      public void commit() throws ReasonedException {
        super.commit();
        BUF.append("[Transaction Committed]");
      }
      @Override
      public void rollback() throws ReasonedRuntimeException {
        super.rollback();
        BUF.append("[Transaction Rollbacked]");
      }
      @Override
      public void end() throws ReasonedRuntimeException {
        super.end();
        BUF.append("[Transaction Ended]");
      }
    };

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    Map<String,Object> inputMap = new HashMap<String,Object>();
    QueryResult result = new QueryResult(QID);

    QueryConfig config = new QueryConfig(QID);
    Query query = new Query(config, tran);
    try {
      query.controlTransaction(inputMap, result);
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(BUF.toString(),
      "[Transaction Begined]" +
      "[" + EID0 + ":execute_inputMap_result]" +
      "[" + EID1 + ":execute_inputMap_result]" +
      "[" + EID2 + ":execute_inputMap_result]" +
      "[Transaction Committed]" +
      "[Transaction Ended]" +
    "");
  }

  public void execute_inputMap_TransactionIsInternal()
  {
    MSG("トランザクションが内部のものの場合。");
    MSG("- トランザクション制御あり。");

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    Map<String,Object> inputMap = new HashMap<String,Object>();

    QueryConfig config = new QueryConfig(QID);
    IQuery query = new Query(config) {
      @Override
      protected IQueryTransaction createDefaultTransaction() {
        return new QueryTransaction() {
          @Override
          public void begin() throws ReasonedException {
            super.begin();
            BUF.append("[Transaction Begined]");
          }
          @Override
          public void commit() throws ReasonedException {
            super.commit();
            BUF.append("[Transaction Committed]");
          }
          @Override
          public void rollback() throws ReasonedRuntimeException {
            super.rollback();
            BUF.append("[Transaction Rollbacked]");
          }
          @Override
          public void end() throws ReasonedRuntimeException {
            super.end();
            BUF.append("[Transaction Ended]");
          }
        };
      }
    };

    try {
      IQueryResult rslt = query.execute(inputMap);
      EQUAL(rslt.getQueryId(), QID);
      TRUE (rslt.isSuccess());
      POSITIVE(rslt.getBeginTimeMillis());
      TRUE(rslt.getEndTimeMillis() >= rslt.getBeginTimeMillis());
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getException());
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(BUF.toString(),
      "[Transaction Begined]" +
      "[" + EID0 + ":execute_inputMap_result]" +
      "[" + EID1 + ":execute_inputMap_result]" +
      "[" + EID2 + ":execute_inputMap_result]" +
      "[Transaction Committed]" +
      "[Transaction Ended]" +
    "");
  }

  public void execute_inputMap_TransactionIsInternal_ThrowReasonedException()
  {
    MSG("トランザクションが内部のものの場合。");
    MSG("- トランザクション制御あり。");
    MSG("- クエリ実行中にReasonedExceptionが発生。");

    THROW0 = new ReasonedException(ENUM.AAA);

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    Map<String,Object> inputMap = new HashMap<String,Object>();

    QueryConfig config = new QueryConfig(QID);
    IQuery query = new Query(config) {
      @Override
      protected IQueryTransaction createDefaultTransaction() {
        return new QueryTransaction() {
          @Override
          public void begin() throws ReasonedException {
            super.begin();
            BUF.append("[Transaction Begined]");
          }
          @Override
          public void commit() throws ReasonedException {
            super.commit();
            BUF.append("[Transaction Committed]");
          }
          @Override
          public void rollback() throws ReasonedRuntimeException {
            super.rollback();
            BUF.append("[Transaction Rollbacked]");
          }
          @Override
          public void end() throws ReasonedRuntimeException {
            super.end();
            BUF.append("[Transaction Ended]");
          }
        };
      }
    };

    try {
      IQueryResult rslt = query.execute(inputMap);
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ENUM.AAA);
      NOTNULL(query.getQueryHistory());
    }

    EQUAL(BUF.toString(),
      "[Transaction Begined]" +
      "[" + EID0 + ":execute_inputMap_result]" +
      "[Transaction Rollbacked]" +
      "[Transaction Ended]" +
    "");
  }

  public void
    execute_inputMap_TransactionIsInternal_ThrowReasonedRuntimeException()
  {
    MSG("トランザクションが内部のものの場合。");
    MSG("- トランザクション制御あり。");
    MSG("- クエリ実行中にReasonedRuntimeExceptionが発生。");

    THROW1 = new ReasonedRuntimeException(ENUM.AAA);

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    Map<String,Object> inputMap = new HashMap<String,Object>();

    QueryConfig config = new QueryConfig(QID);
    IQuery query = new Query(config) {
      @Override
      protected IQueryTransaction createDefaultTransaction() {
        return new QueryTransaction() {
          @Override
          public void begin() throws ReasonedException {
            super.begin();
            BUF.append("[Transaction Begined]");
          }
          @Override
          public void commit() throws ReasonedException {
            super.commit();
            BUF.append("[Transaction Committed]");
          }
          @Override
          public void rollback() throws ReasonedRuntimeException {
            super.rollback();
            BUF.append("[Transaction Rollbacked]");
          }
          @Override
          public void end() throws ReasonedRuntimeException {
            super.end();
            BUF.append("[Transaction Ended]");
          }
        };
      }
    };

    try {
      IQueryResult rslt = query.execute(inputMap);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ENUM.AAA);
      NOTNULL(query.getQueryHistory());
    }

    EQUAL(BUF.toString(),
      "[Transaction Begined]" +
      "[" + EID0 + ":execute_inputMap_result]" +
      "[Transaction Rollbacked]" +
      "[Transaction Ended]" +
    "");
  }

  public void execute_inputMap_TransactionIsInternal_ThrowRuntimeException()
  {
    MSG("トランザクションが内部のものの場合。");
    MSG("- トランザクション制御あり。");
    MSG("- クエリ実行中にExceptionが発生。");

    THROW2 = new NullPointerException();

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    Map<String,Object> inputMap = new HashMap<String,Object>();

    QueryConfig config = new QueryConfig(QID);
    IQuery query = new Query(config) {
      @Override
      protected IQueryTransaction createDefaultTransaction() {
        return new QueryTransaction() {
          @Override
          public void begin() throws ReasonedException {
            super.begin();
            BUF.append("[Transaction Begined]");
          }
          @Override
          public void commit() throws ReasonedException {
            super.commit();
            BUF.append("[Transaction Committed]");
          }
          @Override
          public void rollback() throws ReasonedRuntimeException {
            super.rollback();
            BUF.append("[Transaction Rollbacked]");
          }
          @Override
          public void end() throws ReasonedRuntimeException {
            super.end();
            BUF.append("[Transaction Ended]");
          }
        };
      }
    };

    try {
      IQueryResult rslt = query.execute(inputMap);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (RuntimeException e) {
      OK(e.toString());
      NOTNULL(query.getQueryHistory());
    }

    EQUAL(BUF.toString(),
      "[Transaction Begined]" +
      "[" + EID0 + ":execute_inputMap_result]" +
      "[Transaction Rollbacked]" +
      "[Transaction Ended]" +
    "");
  }

  public void execute_inputMap_TransactionIsExternal()
  {
    MSG("トランザクションが外部のものの場合。");
    MSG("- トランザクション制御なし。");

    IQueryTransaction tran = new QueryTransaction() {
      @Override
      public void begin() throws ReasonedException {
        super.begin();
        BUF.append("[External Transaction Begined]");
      }
      @Override
      public void commit() throws ReasonedException {
        super.commit();
        BUF.append("[External Transaction Committed]");
      }
      @Override
      public void rollback() throws ReasonedRuntimeException {
        super.rollback();
        BUF.append("[External Transaction Rollbacked]");
      }
      @Override
      public void end() throws ReasonedRuntimeException {
        super.end();
        BUF.append("[External Transaction Ended]");
      }
    };

    String QID  = "QueryTest_executeQuery_inputMap_result";
    String EID0 = "QueryTest_executeQuery_inputMap_result_0";
    String EID1 = "QueryTest_executeQuery_inputMap_result_1";
    String EID2 = "QueryTest_executeQuery_inputMap_result_2";

    Map<String,Object> inputMap = new HashMap<String,Object>();

    QueryConfig config = new QueryConfig(QID);
    IQuery query = new Query(config, tran) {
      @Override
      protected IQueryTransaction createDefaultTransaction() {
        return new QueryTransaction() {
          @Override
          public void begin() throws ReasonedException {
            super.begin();
            BUF.append("[Internal Transaction Begined]");
          }
          @Override
          public void commit() throws ReasonedException {
            super.commit();
            BUF.append("[Internal Transaction Committed]");
          }
          @Override
          public void rollback() throws ReasonedRuntimeException {
            super.rollback();
            BUF.append("[Internal Transaction Rollbacked]");
          }
          @Override
          public void end() throws ReasonedRuntimeException {
            super.end();
            BUF.append("[Internal Transaction Ended]");
          }
        };
      }
    };

    try {
      IQueryResult rslt = query.execute(inputMap);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      tran.begin();
      IQueryResult rslt = query.execute(inputMap);
      EQUAL(rslt.getQueryId(), QID);
      TRUE (rslt.isSuccess());
      POSITIVE(rslt.getBeginTimeMillis());
      TRUE(rslt.getEndTimeMillis() >= rslt.getBeginTimeMillis());
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getException());
      tran.commit();
    }
    catch (Exception e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }

    EQUAL(BUF.toString(),
      "[External Transaction Begined]" +
      "[" + EID0 + ":execute_inputMap_result]" +
      "[" + EID1 + ":execute_inputMap_result]" +
      "[" + EID2 + ":execute_inputMap_result]" +
      "[External Transaction Committed]" +
      "[External Transaction Ended]" +
    "");
  }
}
