package ts.query;

import ts.tester.UnitTest;
import ts.tester.function.ObjectInspector;
import ts.util.*;
import java.util.*;
import java.io.*;

public class QueryTransactionManagerTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryTransactionManagerTest.class, args);
  }

  static StringWriter SW = null;
  static PrintWriter PW = null;
  static String OUTPUT_LOG() { return SW.toString(); }
  static void OPEN_LOG() { SW = new StringWriter(); PW = new PrintWriter(SW); }
  static void CLOSE_LOG() { PW.close(); }
  static void CLEAR_LOG() { CLOSE_LOG(); OPEN_LOG(); }

  public static class MyConnection extends QueryConnection {
    static final long serialVersionUID = -1L;
    private long openTimeMillis = 0L;
    private boolean isClosed = true;
    private long limitTimeMillis;
    public MyConnection(QueryConnectionConfig config) { super(config); }
    public MyConnection(QueryConnectionConfig config, IQueryTransaction tran)
    { super(config, tran); }
    @Override
    protected long getOpenTimeMillis() { return this.openTimeMillis; }
    @Override
    public boolean isOpened() { return ! this.isClosed; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public void open() throws ReasonedException {
      this.openTimeMillis = System.currentTimeMillis();
      this.isClosed = false;
      PW.print("[MyConnection is opened]");
    }
    @Override
    public void commit() throws ReasonedException {
      PW.print("[MyConnection is committed]");
    }
    @Override
    public void rollback() throws ReasonedException {
      PW.print("[MyConnection is rollbacked]");
    }
    @Override
    public void close() throws ReasonedException {
      this.isClosed = true;
      PW.print("[MyConnection is closed]");
    }
  }

  static class BadTransaction extends QueryTransaction {
    public BadTransaction() {
      throw new RuntimeException("!!!");
    }
  }

  final String CID =
  "QueryTransactionManager_createAndGetThreadLocalTransaction_InLocalThread";

  @Override
  protected void preTesting()
  {
    addPart(new QueryTransactionManager_1Test());
    addPart(new QueryTransactionManager_2Test());
    addPart(new QueryTransactionManager_3Test());
  }

  @Override
  protected void preInvocation(String method)
  {
    OPEN_LOG();
  }

  @Override
  protected void postInvocation(String method)
  {
    CLEAR_LOG();
  }

  static long SLEEP_TIME = 0L;
  static void SLEEP() {
    try { Thread.sleep(SLEEP_TIME); } catch (InterruptedException e) {}
  }

  public void createAndGetThreadLocalTransaction_InLocalThread()
  {
    MSG("スレッド・ローカルなトランザクションをローカルなスレッド内で操作。");

    IQueryTransaction tr;

    tr = IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();
    EQUAL(tr.getState(), IQueryTransaction.State.Created);
    EQUAL(tr, IQueryTransactionManager.INSTANCE.getThreadLocalTransaction());

    try {
      tr.begin();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), IQueryTransaction.State.Begined);

    try {
      tr.commit();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), IQueryTransaction.State.Committed);
    EQUAL(tr, IQueryTransactionManager.INSTANCE.getThreadLocalTransaction());

    tr.end();

    try {
      IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadLocalNotExist);
    }
    EQUAL(tr.getState(), IQueryTransaction.State.Ended);
    EQUAL(OUTPUT_LOG(), "");


    tr = IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();
    EQUAL(tr.getState(), IQueryTransaction.State.Created);
    EQUAL(tr, IQueryTransactionManager.INSTANCE.getThreadLocalTransaction());

    try {
      tr.begin();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), IQueryTransaction.State.Begined);

    try {
      tr.commit();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), IQueryTransaction.State.Committed);
    EQUAL(tr, IQueryTransactionManager.INSTANCE.getThreadLocalTransaction());

    tr.end();

    try {
      IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadLocalNotExist);
    }
    EQUAL(tr.getState(), IQueryTransaction.State.Ended);
    EQUAL(OUTPUT_LOG(), "");


    IQueryTransaction tr0, tr1;

    tr0 = IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();
    tr1 = IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();

    EQUAL(tr0, tr1);
    EQUAL(tr0.hashCode(), tr1.hashCode());
    EQUAL(tr0.getState(), IQueryTransaction.State.Created);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);

    try {
      tr1.begin();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr1.getState(), IQueryTransaction.State.Begined);

    try {
      NOTNULL(tr1.getQueryConnection(CID));
      EQUAL(tr1.getState(), IQueryTransaction.State.Begined);
      tr1.commit();
      EQUAL(tr1.getState(), IQueryTransaction.State.Committed);
    }
    catch (ReasonedException e) {
      tr1.rollback();
      EQUAL(tr1.getState(), IQueryTransaction.State.Rollbacked);
    }
    finally {
      tr1.end();
      EQUAL(tr1.getState(), IQueryTransaction.State.Ended);
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection is opened]" +
      "[MyConnection is committed]" +
      "[MyConnection is rollbacked]" +
      "[MyConnection is closed]" +
    "");
  }

  public void createAndGetThreadLocalTransaction_InOtherThread()
  {
    MSG("スレッド・ローカルなトランザクションを別スレッド内で操作。");

    final IQueryTransaction tr =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tr.getState(), IQueryTransaction.State.Created);

    Thread th0 = new Thread() {
      public void run() {
        try {
          tr.begin();
          NG();
        }
        catch (ReasonedException e) {
          NG();
        }
        catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            IQueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th0.start();
      th0.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), IQueryTransaction.State.Created);
    tr.setTimeoutMillis(1000L);
    EQUAL(tr.getTimeoutMillis(), 1000L);

    try {
       tr.begin();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    OK("... transaction begin");
    EQUAL(tr.getState(), IQueryTransaction.State.Begined);
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);


    try {
      EQUAL(tr.getQueryConnection(CID).getConnectionId(), CID);
    }
    catch (ReasonedException e) {
      NG(e);
    }

    Thread th1 = new Thread() {
      public void run() {
        try {
          tr.getQueryConnection(CID);
          NG();
        }
        catch (ReasonedException e) {
          NG();
        }
        catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            IQueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th1.start();
      th1.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    Thread th2 = new Thread() {
      public void run() {
        try {
          tr.rollback();
          NG();
        }
        catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            IQueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th2.start();
      th2.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), IQueryTransaction.State.Begined);

    tr.rollback();
    OK("... transaction rollback");
    EQUAL(tr.getState(), IQueryTransaction.State.Rollbacked);

    Thread th3 = new Thread() {
      public void run() {
        try {
          tr.end();
          NG();
        }
        catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            IQueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th3.start();
      th3.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    tr.end();
    OK("... transaction end");
    EQUAL(tr.getState(), IQueryTransaction.State.Ended);
  }

  public void createAndGetThreadLocalTransaction_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction(null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
    }
  }

  public void createAndGetThreadLocalTransaction_NotFound()
  {
    MSG("スレッド・ローカルなトランザクションを作成前に取得しようとした場合。");

    try {
      Thread th = new Thread() {
        public void run() {
          try {
            IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
            NG();
          }
          catch (ReasonedRuntimeException e) {
            EQUAL(e.getReason(),
              IQueryTransactionManager.Error.ThreadLocalNotExist);
          }
        }
      };
      th.start();
      th.join();
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void createAndGetThreadLocalTransaction_FailToCreate()
  {
    MSG("スレッド・ローカルなトランザクションの作成に失敗した場合。");

    try {
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction(
        BadTransaction.class);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        IQueryTransactionManager.Error.FailToCreateThreadLocal);
      EQUAL(e.getCause().getClass(), RuntimeException.class);
      EQUAL(e.getCause().getMessage(), "!!!");
    }
  }

  public void createAndGetThreadLocalTransaction_AlreadyExists()
  {
    MSG("既にスレッド・ローカルなトランザクションが作成済みの場合。");

    IQueryTransaction tr =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();
    EQUAL(tr.getState(), IQueryTransaction.State.Created);

    try {
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        IQueryTransactionManager.Error.ThreadLocalAlreadyExists);
    }

    tr.end();
    EQUAL(tr.getState(), IQueryTransaction.State.Ended);
  }

  public void equals_ThreadLocalTransaction()
  {
    MSG("スレッド・ローカル・トランザクションの等値判定メソッドの確認。");

    IQueryTransaction tr =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();
    IQueryTransaction tr0 =
      IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
    IQueryTransaction tr1 = new QueryTransaction();

    FALSE(tr.equals(null));
    FALSE(tr.equals(tr1));
    TRUE (tr.equals(tr0));
    TRUE (tr0.equals(tr));

    tr.end();
  }


  public void createAndGetThreadSafeTransaction_InLocalThreads()
  {
    MSG("スレッド・セーフなトランザクションをローカルなスレッド内で操作。");

    IQueryTransaction tr0, tr1;

    tr0 = IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("TH0");
    tr1 = IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("TH1");
    EQUAL(tr0.getState(), IQueryTransaction.State.Created);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);
    EQUAL(tr0,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0"));
    EQUAL(tr1,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH1"));

    try {
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH2");
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadSafeNotExist);
    }
    try {
      IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadLocalNotExist);
    }

    try {
      tr0.begin();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tr0.getState(), IQueryTransaction.State.Begined);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);

    try {
      tr0.commit();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr0.getState(), IQueryTransaction.State.Committed);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);
    EQUAL(tr0,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0"));
    EQUAL(tr1,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH1"));

    try {
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH2");
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadSafeNotExist);
    }
    try {
      IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadLocalNotExist);
    }

    tr0.end();

    try {
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadSafeNotExist);
    }
    EQUAL(tr0.getState(), IQueryTransaction.State.Ended);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);
    EQUAL(OUTPUT_LOG(), "");


    tr0 = IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("TH0");
    EQUAL(tr0.getState(), IQueryTransaction.State.Created);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);
    EQUAL(tr0,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0"));
    EQUAL(tr1,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH1"));

    try {
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH2");
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadSafeNotExist);
    }
    try {
      IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadLocalNotExist);
    }

    try {
      tr0.begin();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr0.getState(), IQueryTransaction.State.Begined);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);

    try {
      tr0.commit();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr0.getState(), IQueryTransaction.State.Committed);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);
    EQUAL(tr0,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0"));
    EQUAL(tr1,
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH1"));

    try {
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH2");
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadSafeNotExist);
    }
    try {
      IQueryTransactionManager.INSTANCE.getThreadLocalTransaction();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadLocalNotExist);
    }

    tr0.end();

    try {
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransactionManager.Error.ThreadSafeNotExist);
    }
    EQUAL(tr0.getState(), IQueryTransaction.State.Ended);
    EQUAL(tr1.getState(), IQueryTransaction.State.Created);
    EQUAL(OUTPUT_LOG(), "");


    IQueryTransaction tr00;

    tr0 = IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("TH0");
    tr00 = IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");

    EQUAL(tr0, tr00);
    NOTEQUAL(tr1, tr00);
    EQUAL(tr0.hashCode(), tr00.hashCode());
    EQUAL(tr0.getState(), IQueryTransaction.State.Created);
    EQUAL(tr00.getState(), IQueryTransaction.State.Created);

    try {
      tr00.begin();
    }
    catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tr00.getState(), IQueryTransaction.State.Begined);

    try {
      NOTNULL(tr00.getQueryConnection(CID));
      EQUAL(tr00.getState(), IQueryTransaction.State.Begined);
      tr00.commit();
      EQUAL(tr00.getState(), IQueryTransaction.State.Committed);
    }
    catch (ReasonedException e) {
      tr00.rollback();
      EQUAL(tr00.getState(), IQueryTransaction.State.Rollbacked);
    }
    finally {
      tr00.end();
      EQUAL(tr00.getState(), IQueryTransaction.State.Ended);
    }

    EQUAL(OUTPUT_LOG(),
     "[MyConnection is opened][MyConnection is committed][MyConnection is rollbacked][MyConnection is closed]");
  }

  public void createAndGetThreadSafeTransaction_InOtherThread()
  {
    MSG("スレッド・セーフなトランザクションを別スレッド内で操作。");

    final IQueryTransaction tr =
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("TH0");

    EQUAL(tr.getState(), IQueryTransaction.State.Created);
    tr.setTimeoutMillis(1000L);
    EQUAL(tr.getTimeoutMillis(), 1000L);

    Thread th0 = new Thread() {
      public void run() {
        try {
          tr.begin();
          EQUAL(tr.getState(), IQueryTransaction.State.Begined);
          EQUAL(tr.getTimeoutMillis(), 1000L);
          EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);
        }
        catch (ReasonedException e) {
          NG();
        }
      }
    };
    try {
      th0.start();
      th0.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), IQueryTransaction.State.Begined);
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);

    try {
      EQUAL(tr.getQueryConnection(CID).getConnectionId(), CID);
    }
    catch (ReasonedException e) {
      NG(e);
    }

    Thread th1 = new Thread() {
      public void run() {
        try {
          IQueryConnection c0 = tr.getQueryConnection(CID);
          EQUAL(c0.getConnectionId(), CID);
          FALSE(c0.isClosed());
          TRUE(c0.isOpened());
        }
        catch (ReasonedException e) {
          NG(e);
        }
      }
    };
    try {
      th1.start();
      th1.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    Thread th2 = new Thread() {
      public void run() {
        tr.rollback();
        EQUAL(tr.getState(), IQueryTransaction.State.Rollbacked);
      }
    };
    try {
      th2.start();
      th2.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    Thread th3 = new Thread() {
      public void run() {
        tr.end();
        EQUAL(tr.getState(), IQueryTransaction.State.Ended);
      }
    };
    try {
      th3.start();
      th3.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    final IQueryTransaction tr0, tr00;

    tr0 = IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("TH0");
    tr00 = IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
    EQUAL(tr0, tr00);
    EQUAL(tr0.hashCode(), tr00.hashCode());
    EQUAL(tr00.getState(), IQueryTransaction.State.Created);

    tr0.setTimeoutMillis(1000L);
    EQUAL(tr0.getTimeoutMillis(), 1000L);

    Thread th10 = new Thread() {
      public void run() {
        IQueryTransaction tr10 =
          IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
        EQUAL(tr0, tr10);
        try {
          tr10.begin();
        }
        catch (ReasonedException e) {
          NG(e);
        }
        EQUAL(tr10.getState(), IQueryTransaction.State.Begined);
      }
    };
    try {
      th10.start();
      th10.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr0.getLimitTimeMillis(), tr0.getBeginTimeMillis() + 1000L);

    try {
      IQueryConnection c0 = tr0.getQueryConnection(CID);
      EQUAL(c0.getConnectionId(), CID);
      FALSE(c0.isClosed());
      TRUE(c0.isOpened());
    }
    catch (ReasonedException e) {
      NG(e);
    }
    Thread th11 = new Thread() {
      public void run() {
        IQueryTransaction tr11 =
          IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
        EQUAL(tr0, tr11);
        EQUAL(tr0.hashCode(), tr11.hashCode());
        try {
          tr11.begin();
          NG();
        }
        catch (ReasonedException e) {
          NG();
        }
        catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
        }
        EQUAL(tr11.getState(), IQueryTransaction.State.Begined);
      }
    };
    try {
      th11.start();
      th11.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    Thread th12 = new Thread() {
      public void run() {
        IQueryTransaction tr12 =
          IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
        EQUAL(tr0, tr12);
        EQUAL(tr0.hashCode(), tr12.hashCode());
        try {
          tr12.commit();
        }
        catch (ReasonedException e) {
          NG(e);
        }
        EQUAL(tr12.getState(), IQueryTransaction.State.Committed);
      }
    };
    try {
      th12.start();
      th12.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    Thread th13 = new Thread() {
      public void run() {
        IQueryTransaction tr13 =
          IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
        EQUAL(tr0, tr13);
        EQUAL(tr0.hashCode(), tr13.hashCode());
        try {
          tr13.end();
        }
        catch (ReasonedRuntimeException e) {
          NG(e);
        }
        EQUAL(tr13.getState(), IQueryTransaction.State.Ended);
      }
    };
    try {
      th13.start();
      th13.join();
    }
    catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr0.getState(), IQueryTransaction.State.Ended);
  }

  public void createAndGetThreadSafeTransaction_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction(null);
      NG();
    }
    catch (AssertionError e) {
      OK(e.toString());
    }

    try {
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction(null,
        QueryTransaction.class);
      NG();
    }
    catch (AssertionError e) {
      OK(e.toString());
    }

    try {
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("tr0",null);
      NG();
    }
    catch (AssertionError e) {
      OK(e.toString());
    }
  }

  public void createAndGetThreadSafeTransaction_AlreadyExists()
  {
    MSG("既にスレッド・セーフなトランザクションが作成済みの場合。");

    IQueryTransaction tr =
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("t0");
    EQUAL(tr.getState(), IQueryTransaction.State.Created);

    try {
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("t0");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        IQueryTransactionManager.Error.ThreadSafeAlreadyExists);
    }

    tr.end();
    EQUAL(tr.getState(), IQueryTransaction.State.Ended);
  }

  public void createAndGetThreadSafeTransaction_FailToCreate()
  {
    MSG("スレッド・セーフなトランザクションの作成に失敗した場合。");

    try {
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("t1",
        BadTransaction.class);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        IQueryTransactionManager.Error.FailToCreateThreadSafe);
      EQUAL(e.getCause().getClass(), RuntimeException.class);
      EQUAL(e.getCause().getMessage(), "!!!");
    }
  }

  public void createAndGetThreadSafeTransaction_NotFound()
  {
    MSG("スレッド・セーフなトランザクションを作成前に取得しようとした場合。");

    try {
      Thread th = new Thread() {
        public void run() {
          try {
            IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("TH0");
            NG();
          }
          catch (ReasonedRuntimeException e) {
            EQUAL(e.getReason(),
              IQueryTransactionManager.Error.ThreadSafeNotExist);
          }
        }
      };
      th.start();
      th.join();
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void equals_ThreadSafeTransaction()
  {
    MSG("スレッド・セーフ・トランザクションの等値判定メソッドの確認。");

    IQueryTransaction tr0 =
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("b0");
    IQueryTransaction tr1 =
      IQueryTransactionManager.INSTANCE.createThreadSafeTransaction("b1");
    IQueryTransaction tr00 =
      IQueryTransactionManager.INSTANCE.getThreadSafeTransaction("b0");
    IQueryTransaction tr2 = new QueryTransaction();

    FALSE(tr0.equals(null));
    FALSE(tr0.equals(tr2));
    FALSE(tr0.equals(tr1));
    TRUE(tr00.equals(tr0));
    TRUE(tr0.equals(tr00));

    tr0.end();
    tr1.end();
  }
}
