package jp.sourceforge.tsukuyomi.openid.impl;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.http.HttpServletResponse;

import jp.sourceforge.tsukuyomi.openid.AuthenticationRequest;
import jp.sourceforge.tsukuyomi.openid.OpenID;
import jp.sourceforge.tsukuyomi.openid.OpenIDRuntimeException;
import jp.sourceforge.tsukuyomi.openid.message.AuthRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * OpenID 認証サーバへの認証リクエストを行うためのクラスです。OpenID4Java
 * の実装を隠蔽するためにあるクラスですので、直接このクラスのインスタンスを生成するべきではありません。
 * 
 * @author asuka
 * 
 */
public class AuthenticationRequestImpl implements AuthenticationRequest {
	private static final Log LOG =
		LogFactory.getLog(AuthenticationRequestImpl.class);

	private static final String LS = System.getProperty("line.separator");

	private AuthRequest authRequest;

	private HttpServletResponse response;

	private String version;

	/**
	 * 新たにインスタンスを生成します。
	 * 
	 * @param response
	 *            HTTP サーブレットレスポンスオブジェクト
	 * @param authRequest
	 *            OpenID4Java の認証リクエスト
	 * @param version
	 *            認証サーバが要求する OpenID のバージョン
	 */
	public AuthenticationRequestImpl(HttpServletResponse response,
			AuthRequest authRequest, String version) {
		this.response = response;
		this.authRequest = authRequest;
		this.version = version;
	}

	/**
	 * 認証リクエストのためのパラメータを取得します。
	 * 
	 * @return 認証リクエストのためのパラメータ
	 */
	public Map<String, String> getParameterMap() {
		Map<?, ?> m = authRequest.getParameterMap();
		Map<String, String> parameter = new HashMap<String, String>();
		for (Entry<?, ?> e : m.entrySet()) {
			parameter.put(e.getKey().toString(), e.getValue().toString());
		}

		return parameter;
	}

	/**
	 * 認証リクエストを POST で送るための URL を取得します。これは OpenID 2.x の場合のみ使用すべきです。
	 * 
	 * @return 認証リクエストを POST で送るための URL
	 */
	public String getPostURL() {
		return authRequest.getDestinationUrl(false);
	}

	/**
	 * 認証リクエストを送るためのリダイレクト URL を取得します。これは OpenID 1.x の場合のみ使用すべきです。
	 * 
	 * @return 認証リクエストを送るためのリダイレクト URL
	 */
	public String getRedirectURL() {
		return authRequest.getDestinationUrl(true);
	}

	/**
	 * HTTP レスポンスとして認証リクエストのリダイレクトを送信します。
	 */
	public void sendRedirect() {
		String url = getRedirectURL();

		if (LOG.isDebugEnabled()) {
			LOG.debug("Redirect to " + url);
		}
		try {
			response.sendRedirect(url);
		} catch (IOException e) {
			throw new OpenIDRuntimeException(e);
		}
	}

	/**
	 * 認証リクエストを POST で送るためのフォームの HTML を HTTP レスポンスに書き込みます。
	 */
	public void writeForm() {
		StringBuilder sb = new StringBuilder();
		sb.append("<!DOCTYPE html "
			+ "PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" "
			+ "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">");
		sb.append(LS);
		sb.append("<html xmlns=\"http://www.w3.org/1999/xhtml\""
			+ " xml:lang=\"ja\" lang=\"ja\">");
		sb.append(LS);
		sb.append("<head>");
		sb.append(LS);
		sb.append("  <meta http-equiv=\"Content-Type\""
			+ " content=\"text/html; charset=UTF-8\" />");
		sb.append(LS);
		sb.append("  <meta http-equiv=\"Content-Script-Type\""
			+ " content=\"text/javascript\" />");
		sb.append(LS);
		sb.append("  <title>OpenID HTML FORM Redirection</title>");
		sb.append(LS);
		sb.append("</head>");
		sb.append(LS);
		sb.append("<body onload=\"document.forms"
			+ "['openid-form-redirection'].submit();\">");
		sb.append(LS);
		sb.append("  <form id=\"openid-form-redirection\""
			+ " name=\"openid-form-redirection\""
			+ " action=\"");
		sb.append(authRequest.getDestinationUrl(false));
		sb.append("\" method=\"post\"" + " accept-charset=\"utf-8\">");
		sb.append(LS);
		for (Entry<String, String> e : getParameterMap().entrySet()) {
			sb.append("    <input type=\"hidden\" name=\"");
			sb.append(e.getKey());
			sb.append("\" value=\"");
			sb.append(e.getValue());
			sb.append("\" />");
			sb.append(LS);
		}
		sb.append("    <button type=\"submit\">Continue...</button>");
		sb.append(LS);
		sb.append(" </form>");
		sb.append(LS);
		sb.append("</body>");
		sb.append(LS);
		sb.append("</html>");
		sb.append(LS);

		String form = sb.toString();

		if (LOG.isDebugEnabled()) {
			LOG.debug("Write to respose" + LS + form);
		}

		response.setContentType("text/html; charset=UTF-8");

		PrintWriter pw;
		try {
			pw = response.getWriter();
		} catch (IOException e) {
			throw new OpenIDRuntimeException(e);
		}
		pw.print(form);
	}

	/**
	 * 認証サーバのバージョンに応じて、 {@link #sendRedirect()} または {@link #writeForm()} を実行します。
	 */
	public void goNext() {
		if (OpenID.isVersion2(version)) {
			writeForm();
		} else {
			sendRedirect();
		}
	}
}
