/*
 * Copyright 2006-2007 Sxip Identity Corporation
 */

package jp.sourceforge.tsukuyomi.openid.message.sreg;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.tsukuyomi.openid.message.MessageException;
import jp.sourceforge.tsukuyomi.openid.message.Parameter;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Implements the extension for Simple Registration fetch requests.
 * 
 * @see SRegMessage Message
 * @author Marius Scurtescu, Johnny Bufu
 */
public class SRegRequest extends SRegMessage {
	private static final Log LOG = LogFactory.getLog(SRegRequest.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	/**
	 * Constructs a SReg Request with an empty parameter list.
	 */
	protected SRegRequest() {
		if (DEBUG) {
			LOG.debug("Created empty SReg request.");
		}
	}

	/**
	 * Constructs a SReg Request with an empty parameter list.
	 */
	public static SRegRequest createFetchRequest() {
		return new SRegRequest();
	}

	/**
	 * Constructs a SRegRequest from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	protected SRegRequest(ParameterList params) {
		parameters = params;
	}

	/**
	 * Constructs a SRegRequest from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	public static SRegRequest createSRegRequest(ParameterList params)
			throws MessageException {
		SRegRequest req = new SRegRequest(params);

		if (!req.isValid()) {
			throw new MessageException("Invalid parameters for a SReg request");
		}

		if (DEBUG) {
			LOG.debug("Created SReg request from parameter list:\n" + params);
		}

		return req;
	}

	/**
	 * Adds an attribute to the SReg request.
	 * 
	 * @param attr
	 *            A requested attribute name.
	 * @param required
	 *            If true, marks the attribute as 'required'; 'if_available'
	 *            otherwise.
	 */
	public void addAttribute(String attr, boolean required) {
		String level = required ? "required" : "optional";

		Parameter levelParam = parameters.getParameter(level);
		Parameter newParam;

		if (levelParam == null) {
			newParam = new Parameter(level, multivalEncode(attr));
		} else {
			newParam =
				new Parameter(level, levelParam.getValue()
					+ ","
					+ multivalEncode(attr));
			parameters.removeParameters(level);
		}

		parameters.set(newParam);

		if (DEBUG) {
			LOG.debug("Added new attribute to SReg request: "
				+ attr
				+ " required: "
				+ required);
		}
	}

	/**
	 * Returns a map with the requested attributes.
	 * 
	 * @param required
	 *            If set to true the list of 'required' attributes is returned,
	 *            otherwise the list of 'optional' attributes.
	 * @return List of attribute names.
	 */
	public List<String> getAttributes(boolean required) {
		List<String> attributes = new ArrayList<String>();

		String level = required ? "required" : "optional";

		Parameter param = parameters.getParameter(level);
		if (param != null) {
			String[] values = param.getValue().split(",");
			for (String element : values) {
				String attr = multivalDecode(element);
				attributes.add(attr);
			}
		}

		return attributes;
	}

	/**
	 * Gets all requested attributes (required and optional).
	 * 
	 * @return List of attribute names.
	 */
	public List<String> getAttributes() {
		List<String> attributes = getAttributes(true);
		attributes.addAll(getAttributes(false));

		return attributes;
	}

	/**
	 * Sets the optional policy URL.
	 * 
	 * @param policyUrl
	 *            A URL which the Consumer provides to give the End User a place
	 *            to read about the how the profile data will be used. The
	 *            Identity Provider SHOULD display this URL to the End User if
	 *            it is given.
	 */
	public void setPolicyUrl(String policyUrl) throws MessageException {
		try {
			new URL(policyUrl);
		} catch (MalformedURLException e) {
			throw new MessageException("Invalid policy_url: " + policyUrl);
		}

		if (DEBUG) {
			LOG.debug("Setting SReg request policy_url: " + policyUrl);
		}

		parameters.set(new Parameter("policy_url", policyUrl));
	}

	/**
	 * Gets the optional policy URL parameter if available, or null otherwise.
	 */
	public String getUpdateUrl() {
		return parameters.hasParameter("policy_url") ? parameters
			.getParameterValue("policy_url") : null;
	}

	/**
	 * Checks the validity of the extension.
	 * <p>
	 * Used when constructing a extension from a parameter list.
	 * 
	 * @return True if the extension is valid, false otherwise.
	 */
	public boolean isValid() {
		if (!parameters.hasParameter("required")
			&& !parameters.hasParameter("optional")) {
			LOG
				.warn("One of 'required' or 'optional' parameters must be present.");
			return false;
		}

		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();
			if (!paramName.equals("required")
				&& !paramName.equals("optional")
				&& !paramName.equals("policy_url")) {
				LOG
					.warn("Invalid parameter name in SReg request: "
						+ paramName);
			}
		}

		return true;
	}
}
