package jp.sourceforge.tsukuyomi.openid.rp.impl;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.Map.Entry;

import javax.crypto.spec.DHParameterSpec;

import jp.sourceforge.tsukuyomi.openid.OpenIDException;
import jp.sourceforge.tsukuyomi.openid.association.Association;
import jp.sourceforge.tsukuyomi.openid.association.AssociationException;
import jp.sourceforge.tsukuyomi.openid.association.AssociationSessionType;
import jp.sourceforge.tsukuyomi.openid.association.DiffieHellmanSession;
import jp.sourceforge.tsukuyomi.openid.discovery.Discovery;
import jp.sourceforge.tsukuyomi.openid.discovery.DiscoveryException;
import jp.sourceforge.tsukuyomi.openid.discovery.DiscoveryInformation;
import jp.sourceforge.tsukuyomi.openid.discovery.Identifier;
import jp.sourceforge.tsukuyomi.openid.discovery.IdentifierException;
import jp.sourceforge.tsukuyomi.openid.discovery.IdentifierParser;
import jp.sourceforge.tsukuyomi.openid.http.HttpClientManager;
import jp.sourceforge.tsukuyomi.openid.message.AssociationError;
import jp.sourceforge.tsukuyomi.openid.message.AssociationRequest;
import jp.sourceforge.tsukuyomi.openid.message.AssociationResponse;
import jp.sourceforge.tsukuyomi.openid.message.AuthFailure;
import jp.sourceforge.tsukuyomi.openid.message.AuthImmediateFailure;
import jp.sourceforge.tsukuyomi.openid.message.AuthRequest;
import jp.sourceforge.tsukuyomi.openid.message.AuthSuccess;
import jp.sourceforge.tsukuyomi.openid.message.DirectError;
import jp.sourceforge.tsukuyomi.openid.message.Message;
import jp.sourceforge.tsukuyomi.openid.message.MessageException;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;
import jp.sourceforge.tsukuyomi.openid.message.VerifyRequest;
import jp.sourceforge.tsukuyomi.openid.message.VerifyResponse;
import jp.sourceforge.tsukuyomi.openid.op.NonceGenerator;
import jp.sourceforge.tsukuyomi.openid.op.RealmVerifier;
import jp.sourceforge.tsukuyomi.openid.rp.ConsumerAssociationStore;
import jp.sourceforge.tsukuyomi.openid.rp.NonceVerifier;
import jp.sourceforge.tsukuyomi.openid.rp.RelayParty;
import jp.sourceforge.tsukuyomi.openid.rp.RelayPartyException;
import jp.sourceforge.tsukuyomi.openid.rp.VerificationResult;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class RelayPartyImpl implements RelayParty {
	private static final Log LOG = LogFactory.getLog(RelayPartyImpl.class);

	public static final boolean DEBUG = LOG.isDebugEnabled();

	private Discovery discovery;

	/**
	 * Private association used for signing consumer nonces when operating in
	 * compatibility (v1.x) mode.
	 */
	private Association _privateAssociation;

	/**
	 * Flag for enabling or disabling stateless mode.
	 */
	private boolean _allowStateless = true;

	/**
	 * The preferred association session type; will be attempted first.
	 */
	private AssociationSessionType _prefAssocSessEnc;

	/**
	 * Maximum number of attmpts for establishing an association.
	 */
	private int _maxAssocAttempts = 4;

	/**
	 * Timeout (in seconds) for keeping track of failed association attempts.
	 * Default 5 minutes.
	 */
	private int _failedAssocExpire = 300;

	/**
	 * Flag for allowing / disallowing no-encryption association session over
	 * plain HTTP.
	 */
	private boolean _allowNoEncHttpSess = false;

	/**
	 * Parameters (modulus and generator) for the Diffie-Hellman sessions.
	 */
	private DHParameterSpec _dhParams =
		DiffieHellmanSession.getDefaultParameter();

	/**
	 * Consumer-side nonce generator, needed for compatibility with OpenID 1.1.
	 */
	private NonceGenerator consumerNonceGenerator;

	private HttpClientManager httpClientManager;

	/**
	 * Used to perform verify realms against return_to URLs.
	 */
	private RealmVerifier _realmVerifier;

	/**
	 * The lowest encryption level session accepted for association sessions.
	 */
	private AssociationSessionType _minAssocSessEnc =
		AssociationSessionType.NO_ENCRYPTION_SHA1MAC;

	/**
	 * Store for keeping track of the established associations.
	 */
	private ConsumerAssociationStore associations;

	/**
	 * Flag for generating checkid_immediate authentication requests.
	 */
	private boolean _immediateAuth = false;

	/**
	 * Verifier for the nonces in authentication responses; prevents replay
	 * attacks.
	 */
	private NonceVerifier nonceVerifier;

	public RelayPartyImpl() throws RelayPartyException {
		_realmVerifier = new RealmVerifier();

		if (Association.isHmacSha256Supported()) {
			_prefAssocSessEnc = AssociationSessionType.DH_SHA256;
		} else {
			_prefAssocSessEnc = AssociationSessionType.DH_SHA1;
		}

		try {
			// initialize the private association for compat consumer nonces
			_privateAssociation =
				Association.generate(
					_prefAssocSessEnc.getAssociationType(),
					"",
					0);
		} catch (AssociationException e) {
			throw new RelayPartyException(
				"Cannot initialize private association, "
					+ "needed for consumer nonces.");
		}
	}

	public List<DiscoveryInformation> discover(String identifier)
			throws DiscoveryException, IdentifierException {
		Identifier id = IdentifierParser.parse(identifier);
		return discovery.discover(id);
	}

	public void setDiscovery(Discovery discovery) {
		this.discovery = discovery;
	}

	public DiscoveryInformation associate(List<DiscoveryInformation> discoveries) {
		DiscoveryInformation discovered;
		Association assoc;

		int attemptsLeft = _maxAssocAttempts;
		Iterator<DiscoveryInformation> itr = discoveries.iterator();
		while (itr.hasNext() && attemptsLeft > 0) {
			discovered = itr.next();
			attemptsLeft -= associate(discovered, attemptsLeft);

			// check if an association was established
			assoc = associations.load(discovered.getIdpEndpoint().toString());

			if (assoc != null
				&& !Association.FAILED_ASSOC_HANDLE.equals(assoc.getHandle())) {
				return discovered;
			}
		}

		if (discoveries.size() > 0) {
			// no association established, return the first service endpoint
			DiscoveryInformation d0 = discoveries.get(0);
			LOG.warn("Association failed; using first entry: "
				+ d0.getIdpEndpoint());

			return d0;
		} else {
			LOG
				.error("Association attempt, but no discovey endpoints provided.");
			return null;
		}
	}

	/**
	 * Tries to establish an association with the OpenID Provider.
	 * <p>
	 * The resulting association information will be kept on storage for later
	 * use at verification stage.
	 * 
	 * @param discovered
	 *            DiscoveryInformation obtained during the discovery
	 * @return The number of association attempts performed.
	 */
	private int associate(DiscoveryInformation discovered, int maxAttempts) {
		if (_maxAssocAttempts == 0) {
			return 0; // associations disabled
		}

		URL idpUrl = discovered.getIdpEndpoint();
		String idpEndpoint = idpUrl.toString();

		LOG.info("Trying to associate with "
			+ idpEndpoint
			+ " attempts left: "
			+ maxAttempts);

		// check if there's an already established association
		Association a = associations.load(idpEndpoint);
		if (a != null && a.getHandle() != null) {
			LOG.info("Found an existing association.");
			return 0;
		}

		String handle = Association.FAILED_ASSOC_HANDLE;

		// build a list of association types, with the preferred one at the end
		// LinkedHashMap<AssociationSessionType, String> requests = new
		// LinkedHashMap<AssociationSessionType, String>();
		Set<AssociationSessionType> requests =
			new HashSet<AssociationSessionType>();

		if (discovered.isVersion2()) {
			requests.add(AssociationSessionType.NO_ENCRYPTION_SHA1MAC);
			requests.add(AssociationSessionType.NO_ENCRYPTION_SHA256MAC);
			requests.add(AssociationSessionType.DH_SHA1);
			requests.add(AssociationSessionType.DH_SHA256);
		} else {
			requests.add(AssociationSessionType.NO_ENCRYPTION_COMPAT_SHA1MAC);
			requests.add(AssociationSessionType.DH_COMPAT_SHA1);
		}

		if (_prefAssocSessEnc.isVersion2() == discovered.isVersion2()) {
			requests.add(_prefAssocSessEnc);
		}

		// build a stack of Association Request objects
		// and keep only the allowed by the configured preferences
		// the most-desirable entry is always at the top of the stack
		Stack<AssociationRequest> reqStack = new Stack<AssociationRequest>();
		for (AssociationSessionType type : requests) {
			// create the appropriate Association Request
			AssociationRequest newReq = createAssociationRequest(type, idpUrl);
			if (newReq != null) {
				reqStack.push(newReq);
			}
		}

		// perform the association attempts
		int attemptsLeft = maxAttempts;
		Set<AssociationSessionType> alreadyTried =
			new HashSet<AssociationSessionType>();
		while (attemptsLeft > 0 && !reqStack.empty()) {
			try {
				attemptsLeft--;
				AssociationRequest assocReq =
					(AssociationRequest) reqStack.pop();

				if (DEBUG) {
					LOG.debug("Trying association type: " + assocReq.getType());
				}

				// was this association / session type attempted already?
				if (alreadyTried.contains(assocReq.getType())) {
					if (DEBUG) {
						LOG.debug("Already tried.");
					}
					continue;
				}

				// mark the current request type as already tried
				alreadyTried.add(assocReq.getType());

				ParameterList respParams = new ParameterList();
				int status = call(idpEndpoint, assocReq, respParams);

				// process the response
				if (status == HttpStatus.SC_OK) // success response
				{
					AssociationResponse assocResp;

					assocResp =
						AssociationResponse
							.createAssociationResponse(respParams);

					// valid association response
					Association assoc =
						assocResp.getAssociation(assocReq.getDHSess());
					handle = assoc.getHandle();

					AssociationSessionType respType = assocResp.getType();
					if (respType.equals(assocReq.getType()) ||
					// v1 IdPs may return a success no-encryption resp
						(!discovered.isVersion2()
							&& respType.getHAlgorithm() == null && createAssociationRequest(
							respType,
							idpUrl) != null)) {
						// store the association and do no try alternatives
						associations.save(idpEndpoint, assoc);
						LOG.info("Associated with "
							+ discovered.getIdpEndpoint()
							+ " handle: "
							+ assoc.getHandle());
						break;
					} else {
						LOG.info("Discarding, not matching consumer criteria");
					}
				} else if (status == HttpStatus.SC_BAD_REQUEST) // error
				// response
				{
					LOG.info("Association attempt failed.");

					// retrieve fallback sess/assoc/encryption params set by IdP
					// and queue a new attempt
					AssociationError assocErr =
						AssociationError.createAssociationError(respParams);

					AssociationSessionType idpType =
						AssociationSessionType.create(
							assocErr.getSessionType(),
							assocErr.getAssocType());

					if (alreadyTried.contains(idpType)) {
						continue;
					}

					// create the appropriate Association Request
					AssociationRequest newReq =
						createAssociationRequest(idpType, idpUrl);

					if (newReq != null) {
						if (DEBUG) {
							LOG.debug("Retrieved association type "
								+ "from the association error: "
								+ newReq.getType());
						}

						reqStack.push(newReq);
					}
				}
			} catch (OpenIDException e) {
				LOG.error("Error encountered during association attempt.", e);
			}
		}

		// store IdPs with which an association could not be established
		// so that association attempts are not performed with each auth request
		if (Association.FAILED_ASSOC_HANDLE.equals(handle)
			&& _failedAssocExpire > 0) {
			associations.save(idpEndpoint, Association
				.getFailedAssociation(_failedAssocExpire));
		}

		return maxAttempts - attemptsLeft;
	}

	/**
	 * Constructs an Association Request message of the specified session and
	 * association type, taking into account the user preferences (encryption
	 * level, default Diffie-Hellman parameters).
	 * 
	 * @param type
	 *            The type of the association (session and association)
	 * @param idpUrl
	 *            The IdP for which the association request is created
	 * @return An AssociationRequest message ready to be sent back to the OpenID
	 *         Provider, or null if an association of the requested type cannot
	 *         be built.
	 */
	private AssociationRequest createAssociationRequest(
			AssociationSessionType type, URL idpUrl) {
		try {
			if (_minAssocSessEnc.isBetter(type)) {
				return null;
			}

			AssociationRequest assocReq = null;

			DiffieHellmanSession dhSess;
			if (type.getHAlgorithm() != null) // DH session
			{
				dhSess = DiffieHellmanSession.create(type, _dhParams);
				if (DiffieHellmanSession.isDhSupported(type)
					&& Association.isHmacSupported(type.getAssociationType())) {
					assocReq =
						AssociationRequest.createAssociationRequest(
							type,
							dhSess);
				}
			} else // no-enc session
			{
				if ((_allowNoEncHttpSess || idpUrl
					.getProtocol()
					.equals("https"))
					&& Association.isHmacSupported(type.getAssociationType())) {
					assocReq =
						AssociationRequest.createAssociationRequest(type);
				}
			}

			return assocReq;
		} catch (OpenIDException e) {
			LOG.error("Error trying to create association request.", e);
			return null;
		}
	}

	/**
	 * Makes a HTTP call to the specified URL with the parameters specified in
	 * the Message.
	 * 
	 * @param url
	 *            URL endpoint for the HTTP call
	 * @param request
	 *            Message containing the parameters
	 * @param response
	 *            ParameterList that will hold the parameters received in the
	 *            HTTP response
	 * @return the status code of the HTTP call
	 */
	private int call(String url, Message request, ParameterList response)
			throws MessageException {
		int responseCode = -1;

		// build the post message with the parameters from the request
		PostMethod post = new PostMethod(url);

		try {
			// can't follow redirects on a POST (w/o user intervention)
			// post.setFollowRedirects(true);
			post.setRequestEntity(new StringRequestEntity(
				request.wwwFormEncoding(),
				"application/x-www-form-urlencoded",
				"UTF-8"));

			// place the http call to the IdP
			if (DEBUG) {
				LOG.debug("Performing HTTP POST on " + url);
			}

			HttpClient httpClient = httpClientManager.getHttpClient();
			responseCode = httpClient.executeMethod(post);

			String postResponse = post.getResponseBodyAsString();
			response.copyOf(ParameterList.createFromKeyValueForm(postResponse));

			if (DEBUG) {
				LOG.debug("Retrived response:\n" + postResponse);
			}
		} catch (IOException e) {
			LOG.error("Error talking to "
				+ url
				+ " response code: "
				+ responseCode, e);
		} finally {
			post.releaseConnection();
		}

		return responseCode;
	}

	/**
	 * Builds a authentication request message for the user specified in the
	 * discovery information provided as a parameter.
	 * 
	 * @param discovered
	 *            A DiscoveryInformation endpoint from the list obtained by
	 *            performing dicovery on the User-supplied OpenID identifier.
	 * @param returnToUrl
	 *            The URL on the Consumer site where the OpenID Provider will
	 *            return the user after generating the authentication response.
	 *            <br>
	 *            Null if the Consumer does not with to for the End User to be
	 *            returned to it (something else useful will have been performed
	 *            via an extension). <br>
	 *            Must not be null in OpenID 1.x compatibility mode.
	 * @return Authentication request message to be sent to the OpenID Provider.
	 */
	public AuthRequest authenticate(DiscoveryInformation discovered,
			String returnToUrl) throws MessageException, RelayPartyException {
		return authenticate(discovered, returnToUrl, returnToUrl);
	}

	/**
	 * Builds a authentication request message for the user specified in the
	 * discovery information provided as a parameter.
	 * 
	 * @param discovered
	 *            A DiscoveryInformation endpoint from the list obtained by
	 *            performing dicovery on the User-supplied OpenID identifier.
	 * @param returnToUrl
	 *            The URL on the Consumer site where the OpenID Provider will
	 *            return the user after generating the authentication response.
	 *            <br>
	 *            Null if the Consumer does not with to for the End User to be
	 *            returned to it (something else useful will have been performed
	 *            via an extension). <br>
	 *            Must not be null in OpenID 1.x compatibility mode.
	 * @param realm
	 *            The URL pattern that will be presented to the user when he/she
	 *            will be asked to authorize the authentication transaction.
	 *            Must be a super-set of the
	 * @returnToUrl.
	 * @return Authentication request message to be sent to the OpenID Provider.
	 */
	public AuthRequest authenticate(DiscoveryInformation discovered,
			String returnToUrl, String realm) throws MessageException,
			RelayPartyException {
		if (discovered == null) {
			throw new RelayPartyException("Authentication cannot continue: "
				+ "no discovery information provided.");
		}

		associate(discovered, _maxAssocAttempts);

		Association assoc =
			associations.load(discovered.getIdpEndpoint().toString());
		String handle =
			assoc != null ? assoc.getHandle() : Association.FAILED_ASSOC_HANDLE;

		// get the Claimed ID
		String claimedId;
		if (discovered.hasClaimedIdentifier()) {
			claimedId = discovered.getClaimedIdentifier().getIdentifier();
		} else {
			claimedId = AuthRequest.SELECT_ID;
		}

		// set the Delegate ID (aka OP-specific identifier)
		String delegate = claimedId;
		if (discovered.hasDelegateIdentifier()) {
			delegate = discovered.getDelegateIdentifier();
		}

		// stateless mode disabled ?
		if (!_allowStateless && Association.FAILED_ASSOC_HANDLE.equals(handle)) {
			throw new RelayPartyException(
				"Authentication cannot be performed: "
					+ "no association available and stateless mode is disabled");
		}

		LOG.info("Creating authentication request for"
			+ " OP-endpoint: "
			+ discovered.getIdpEndpoint()
			+ " claimedID: "
			+ claimedId
			+ " OP-specific ID: "
			+ delegate);

		AuthRequest authReq =
			AuthRequest.createAuthRequest(claimedId, delegate, !discovered
				.isVersion2(), returnToUrl, handle, realm, _realmVerifier);

		authReq.setOPEndpoint(discovered.getIdpEndpoint());

		if (!discovered.isVersion2()) {
			authReq.setReturnTo(insertConsumerNonce(authReq.getReturnTo()));
		}

		// ignore the immediate flag for OP-directed identifier selection
		if (!AuthRequest.SELECT_ID.equals(claimedId)) {
			authReq.setImmediate(_immediateAuth);
		}

		if (!authReq.isValid()) {
			throw new MessageException("Invalid AuthRequest: "
				+ authReq.wwwFormEncoding());
		}

		return authReq;
	}

	/**
	 * Inserts a consumer-side nonce as a custom parameter in the return_to
	 * parameter of the authentication request.
	 * <p>
	 * Needed for preventing replay attack when running compatibility mode.
	 * OpenID 1.1 OpenID Providers do not generate nonces in authentication
	 * responses.
	 * 
	 * @param returnTo
	 *            The return_to URL to which a custom nonce parameter will be
	 *            added.
	 * @return The return_to URL containing the nonce.
	 */
	public String insertConsumerNonce(String returnTo) {
		String nonce = consumerNonceGenerator.next();

		returnTo += (returnTo.indexOf('?') != -1) ? '&' : '?';

		try {
			returnTo += "openid.rpnonce=" + URLEncoder.encode(nonce, "UTF-8");

			returnTo +=
				"&openid.rpsig="
					+ URLEncoder.encode(
						_privateAssociation.sign(returnTo),
						"UTF-8");

			LOG.info("Inserted consumer nonce.");

			if (DEBUG) {
				LOG.debug("return_to:" + returnTo);
			}
		} catch (Exception e) {
			LOG.error("Error inserting consumre nonce.", e);
			return null;
		}

		return returnTo;
	}

	public void setConsumerNonceGenerator(NonceGenerator consumerNonceGenerator) {
		this.consumerNonceGenerator = consumerNonceGenerator;
	}

	/**
	 * Performs verification on the Authentication Response (assertion) received
	 * from the OpenID Provider.
	 * <p>
	 * Three verification steps are performed:
	 * <ul>
	 * <li> nonce: the same assertion will not be accepted more than once
	 * <li> signatures: verifies that the message was indeed sent by the OpenID
	 * Provider that was contacted earlier after discovery
	 * <li> discovered information: the information contained in the assertion
	 * matches the one obtained during the discovery (the OpenID Provider is
	 * authoritative for the claimed identifier; the received assertion is not
	 * meaningful otherwise
	 * </ul>
	 * 
	 * @param receivingUrl
	 *            The URL where the Consumer (Relying Party) has accepted the
	 *            incoming message.
	 * @param response
	 *            ParameterList of the authentication response being verified.
	 * @param discovered
	 *            Previously discovered information (which can therefore be
	 *            trusted) obtained during the discovery phase; this should be
	 *            stored and retrieved by the RP in the user's session.
	 * 
	 * @return A VerificationResult, containing a verified identifier; the
	 *         verified identifier is null if the verification failed).
	 * @throws IdentifierException
	 */
	public VerificationResult verify(String receivingUrl,
			ParameterList response, DiscoveryInformation discovered)
			throws MessageException, DiscoveryException, AssociationException,
			IdentifierException {
		VerificationResult result = new VerificationResult();
		LOG.info("Verifying authentication response...");

		// non-immediate negative response
		if ("cancel".equals(response.getParameterValue("openid.mode"))) {
			result.setAuthResponse(AuthFailure.createAuthFailure(response));
			LOG.info("Received auth failure.");
			return result;
		}

		// immediate negative response
		if ("setup_needed".equals(response.getParameterValue("openid.mode"))
			|| ("id_res".equals(response.getParameterValue("openid.mode")) && response
				.hasParameter("openid.user_setup_url"))) {
			AuthImmediateFailure fail =
				AuthImmediateFailure.createAuthImmediateFailure(response);
			result.setAuthResponse(fail);
			result.setIdpSetupUrl(fail.getUserSetupUrl());
			LOG.info("Received auth immediate failure.");
			return result;
		}

		AuthSuccess authResp = AuthSuccess.createAuthSuccess(response);
		LOG.info("Received positive auth response.");

		if (!authResp.isValid()) {
			throw new MessageException("Invalid Authentication Response: "
				+ authResp.wwwFormEncoding());
		}
		result.setAuthResponse(authResp);

		// [1/4] return_to verification
		if (!verifyReturnTo(receivingUrl, authResp)) {
			result.setStatusMsg("Return_To URL verification failed.");
			LOG.error("Return_To URL verification failed.");
			return result;
		}

		// [2/4] : discovered info verification
		discovered = verifyDiscovered(authResp, discovered);
		if (discovered == null || !discovered.hasClaimedIdentifier()) {
			result.setStatusMsg("Discovered information verification failed.");
			LOG.error("Discovered information verification failed.");
			return result;
		}

		// [3/4] : nonce verification
		if (!verifyNonce(authResp, discovered)) {
			result.setStatusMsg("Nonce verificaton failed.");
			LOG.error("Nonce verificaton failed.");
			return result;
		}

		// [4/4] : signature verification
		return (verifySignature(authResp, discovered, result));
	}

	/**
	 * Verifies that the URL where the Consumer (Relying Party) received the
	 * authentication response matches the value of the "openid.return_to"
	 * parameter in the authentication response.
	 * 
	 * @param receivingUrl
	 *            The URL where the Consumer received the authentication
	 *            response.
	 * @param response
	 *            The authentication response.
	 * @return True if the two URLs match, false otherwise.
	 */
	public boolean verifyReturnTo(String receivingUrl, AuthSuccess response) {
		if (DEBUG) {
			LOG.debug("Verifying return URL; receiving: "
				+ receivingUrl
				+ "\nmessage: "
				+ response.getReturnTo());
		}

		URL receiving;
		URL returnTo;
		try {
			receiving = new URL(receivingUrl);
			returnTo = new URL(response.getReturnTo());
		} catch (MalformedURLException e) {
			LOG.error("Invalid return URL.", e);
			return false;
		}

		// [1/2] schema, authority (includes port) and path

		// deal manually with the trailing slash in the path
		StringBuffer receivingPath = new StringBuffer(receiving.getPath());
		if (receivingPath.length() > 0
			&& receivingPath.charAt(receivingPath.length() - 1) != '/') {
			receivingPath.append('/');
		}

		StringBuffer returnToPath = new StringBuffer(returnTo.getPath());
		if (returnToPath.length() > 0
			&& returnToPath.charAt(returnToPath.length() - 1) != '/') {
			returnToPath.append('/');
		}

		if (!receiving.getProtocol().equals(returnTo.getProtocol())
			|| !receiving.getAuthority().equals(returnTo.getAuthority())
			|| !receivingPath.toString().equals(returnToPath.toString())) {
			if (DEBUG) {
				LOG.debug("Return URL schema, authority or "
					+ "path verification failed.");
			}
			return false;
		}

		// [2/2] query parameters
		try {
			Map<String, List<String>> returnToParams =
				extractQueryParams(returnTo);
			Map<String, List<String>> receivingParams =
				extractQueryParams(receiving);

			if (returnToParams == null) {
				return true;
			}

			if (receivingParams == null) {
				if (DEBUG) {
					LOG
						.debug("Return URL query parameters verification failed.");
				}
				return false;
			}

			for (Entry<String, List<String>> returnToParam : returnToParams
				.entrySet()) {
				List<String> receivingValues =
					receivingParams.get(returnToParam.getKey());
				List<String> returnToValues = returnToParam.getValue();

				if (receivingValues == null
					|| receivingValues.size() != returnToValues.size()
					|| !receivingValues.containsAll(returnToValues)) {
					if (DEBUG) {
						LOG
							.debug("Return URL query parameters verification failed.");
					}
					return false;
				}
			}
		} catch (UnsupportedEncodingException e) {
			LOG.error("Error verifying return URL query parameters.", e);
			return false;
		}

		return true;
	}

	/**
	 * Returns a Map(key, List(values)) with the URL's query params, or null if
	 * the URL doesn't have a query string.
	 */
	public Map<String, List<String>> extractQueryParams(URL url)
			throws UnsupportedEncodingException {
		if (url.getQuery() == null) {
			return null;
		}

		Map<String, List<String>> paramsMap =
			new HashMap<String, List<String>>();

		for (String keyValue : url.getQuery().split("&")) {
			int equalPos = keyValue.indexOf("=");

			String key =
				equalPos > -1 ? URLDecoder.decode(keyValue.substring(
					0,
					equalPos), "UTF-8") : URLDecoder.decode(keyValue, "UTF-8");
			String value;
			if (equalPos <= -1) {
				value = null;
			} else if (equalPos + 1 > keyValue.length()) {
				value = "";
			} else {
				value =
					URLDecoder
						.decode(keyValue.substring(equalPos + 1), "UTF-8");
			}

			List<String> existingValues = (List<String>) paramsMap.get(key);
			if (existingValues == null) {
				List<String> newValues = new ArrayList<String>();
				newValues.add(value);
				paramsMap.put(key, newValues);
			} else {
				existingValues.add(value);
			}
		}

		return paramsMap;
	}

	/**
	 * Verifies the signature in a authentication response message.
	 * 
	 * @param authResp
	 *            Authentication response to be verified.
	 * @param discovered
	 *            The discovery information obtained earlier during the
	 *            discovery stage.
	 * @return True if the verification succeeded, false otherwise.
	 */
	private VerificationResult verifySignature(AuthSuccess authResp,
			DiscoveryInformation discovered, VerificationResult result)
			throws AssociationException, MessageException {
		if (discovered == null || authResp == null) {
			LOG.error("Can't verify signature: "
				+ "null assertion or discovered information.");

			result.setStatusMsg("Can't verify signature: "
				+ "null assertion or discovered information.");

			return result;
		}

		String handle = authResp.getHandle();
		URL idp = discovered.getIdpEndpoint();
		Association assoc = associations.load(idp.toString(), handle);

		if (assoc != null) // association available, local verification
		{
			LOG.info("Found association: "
				+ assoc.getHandle()
				+ " verifying signature locally...");
			String text = authResp.getSignedText();
			String signature = authResp.getSignature();

			if (assoc.verifySignature(text, signature)) {
				result.setVerifiedId(discovered.getClaimedIdentifier());
				if (DEBUG) {
					LOG.debug("Local signature verification succeeded.");
				}
			} else if (DEBUG) {
				LOG.debug("Local signature verification failed.");
			}

		} else // no association, verify with the IdP
		{
			LOG.info("No association found, "
				+ "contacting the OP for direct verification...");

			VerifyRequest vrfy = VerifyRequest.createVerifyRequest(authResp);

			ParameterList responseParams = new ParameterList();

			int respCode = call(idp.toString(), vrfy, responseParams);
			if (HttpStatus.SC_OK == respCode) {
				VerifyResponse vrfyResp =
					VerifyResponse.createVerifyResponse(responseParams);

				if (vrfyResp.isValid() && vrfyResp.isSignatureVerified()) {
					// process the optional invalidate_handle first
					String invalidateHandle = vrfyResp.getInvalidateHandle();
					if (invalidateHandle != null) {
						associations.remove(idp.toString(), invalidateHandle);
					}

					result.setVerifiedId(discovered.getClaimedIdentifier());
					if (DEBUG) {
						LOG.debug("Direct signature verification succeeded "
							+ "with OP: "
							+ idp);
					}
				} else {
					if (DEBUG) {
						LOG.debug("Direct signature verification failed "
							+ "with OP: "
							+ idp);
					}
					result
						.setStatusMsg("Direct signature verification failed.");
				}
			} else {
				DirectError err = DirectError.createDirectError(responseParams);

				if (DEBUG) {
					LOG.debug("Error verifying signature with the OP: "
						+ idp
						+ " error message: "
						+ err.keyValueFormEncoding());
				}

				result.setStatusMsg("Error verifying signature with the OP: "
					+ err.getErrorMsg());
			}
		}

		Identifier verifiedID = result.getVerifiedId();
		if (verifiedID != null) {
			LOG.info("Verification succeeded for: " + verifiedID);
		} else {
			LOG.error("Verification failed for: null reason: "
				+ result.getStatusMsg());
		}

		return result;
	}

	/**
	 * Verifies the dicovery information matches the data received in a
	 * authentication response from an OpenID Provider.
	 * 
	 * @param authResp
	 *            The authentication response to be verified.
	 * @param discovered
	 *            The discovery information obtained earlier during the
	 *            discovery stage, associated with the identifier(s) in the
	 *            request. May be null for OpenID 2.0; must not be null for
	 *            OpenID 1.x.
	 * @return The discovery information associated with the claimed identifier,
	 *         that can be used further in the verification process. Null if the
	 *         discovery on the claimed identifier does not match the data in
	 *         the assertion.
	 * @throws IdentifierException
	 */
	private DiscoveryInformation verifyDiscovered(AuthSuccess authResp,
			DiscoveryInformation discovered) throws DiscoveryException,
			IdentifierException {
		if (authResp == null || authResp.getIdentity() == null) {
			LOG.info("Assertion is not about an identifier");
			return null;
		}

		if (authResp.isVersion2()) {
			return verifyDiscovered2(authResp, discovered);
		} else {
			return verifyDiscovered1(authResp, discovered);
		}
	}

	/**
	 * Verifies the discovered information associated with a OpenID 1.x
	 * response.
	 * 
	 * @param authResp
	 *            The authentication response to be verified.
	 * @param discovered
	 *            The discovery information obtained earlier during the
	 *            discovery stage, associated with the identifier(s) in the
	 *            request. Must not be null, and must contain a claimed
	 *            identifier.
	 * @return The discovery information associated with the claimed identifier,
	 *         that can be used further in the verification process. Null if the
	 *         discovery on the claimed identifier does not match the data in
	 *         the assertion.
	 */
	private DiscoveryInformation verifyDiscovered1(AuthSuccess authResp,
			DiscoveryInformation discovered) throws DiscoveryException {
		if (authResp == null
			|| authResp.isVersion2()
			|| authResp.getIdentity() == null
			|| discovered == null
			|| discovered.getClaimedIdentifier() == null
			|| discovered.isVersion2()) {
			if (DEBUG) {
				LOG.debug("Discovered information doesn't match "
					+ "auth response / version");
			}
			return null;
		}

		// asserted identifier in the AuthResponse
		String assertId = authResp.getIdentity();

		// claimed identifier
		Identifier claimedId = discovered.getClaimedIdentifier();

		if (DEBUG) {
			LOG.debug("Verifying discovered information for OpenID1 assertion "
				+ "about ClaimedID: "
				+ claimedId.getIdentifier());
		}

		// OP-specific ID
		String opSpecific =
			discovered.hasDelegateIdentifier() ? discovered
				.getDelegateIdentifier() : claimedId.getIdentifier();

		// does the asserted ID match the OP-specific ID from the discovery?
		if (opSpecific.equals(assertId)) {
			return discovered; // success
		}

		// discovered info verification failed
		if (DEBUG) {
			LOG.debug("Identifier in the assertion doesn't match "
				+ "the one in the discovered information.");
		}
		return null;
	}

	/**
	 * Verifies the discovered information associated with a OpenID 2.0
	 * response.
	 * 
	 * @param authResp
	 *            The authentication response to be verified.
	 * @param discovered
	 *            The discovery information obtained earlier during the
	 *            discovery stage, associated with the identifier(s) in the
	 *            request. May be null, in which case discovery will be
	 *            performed on the claimed identifier in the response.
	 * @return The discovery information associated with the claimed identifier,
	 *         that can be used further in the verification process. Null if the
	 *         discovery on the claimed identifier does not match the data in
	 *         the assertion.
	 * @throws IdentifierException
	 */
	private DiscoveryInformation verifyDiscovered2(AuthSuccess authResp,
			DiscoveryInformation discovered) throws DiscoveryException,
			IdentifierException {
		if (authResp == null
			|| !authResp.isVersion2()
			|| authResp.getIdentity() == null
			|| authResp.getClaimed() == null) {
			if (DEBUG) {
				LOG.debug("Discovered information doesn't match "
					+ "auth response / version");
			}
			return null;
		}

		// asserted identifier in the AuthResponse
		String assertId = authResp.getIdentity();

		// claimed identifier in the AuthResponse
		Identifier respClaimed = IdentifierParser.parse(authResp.getClaimed());

		// the OP endpoint sent in the response
		String respEndpoint = authResp.getOpEndpoint();

		if (DEBUG) {
			LOG.debug("Verifying discovered information for OpenID2 assertion "
				+ "about ClaimedID: "
				+ respClaimed.getIdentifier());
		}

		// was the claimed identifier in the assertion previously discovered?
		if (discovered != null
			&& discovered.hasClaimedIdentifier()
			&& discovered.getClaimedIdentifier().equals(respClaimed)) {
			// OP-endpoint, OP-specific ID and protocol version must match
			String opSpecific =
				discovered.hasDelegateIdentifier() ? discovered
					.getDelegateIdentifier() : discovered
					.getClaimedIdentifier()
					.getIdentifier();

			if (opSpecific.equals(assertId)
				&& discovered.isVersion2()
				&& discovered.getIdpEndpoint().toString().equals(respEndpoint)) {
				if (DEBUG) {
					LOG
						.debug("ClaimedID in the assertion was previously discovered: "
							+ respClaimed);
				}
				return discovered;
			}
		}

		// stateless, bare response, or the user changed the ID at the OP
		DiscoveryInformation firstServiceMatch = null;

		// perform discovery on the claim identifier in the assertion
		if (DEBUG) {
			LOG
				.debug("Performing discovery on the ClaimedID in the assertion: "
					+ respClaimed);
		}
		List<DiscoveryInformation> discoveries =
			discovery.discover(respClaimed);

		// find the newly discovered service endpoint that matches the assertion
		// - OP endpoint, OP-specific ID and protocol version must match
		// - prefer (first = highest priority) endpoint with an association
		if (DEBUG) {
			LOG.debug("Looking for a service element to match "
				+ "the ClaimedID and OP endpoint in the assertion...");
		}
		for (DiscoveryInformation service : discoveries) {
			if (DiscoveryInformation.OPENID2_OP.equals(service.getVersion())) {
				continue;
			}

			String opSpecific =
				service.hasDelegateIdentifier() ? service
					.getDelegateIdentifier() : service
					.getClaimedIdentifier()
					.getIdentifier();

			if (!opSpecific.equals(assertId)
				|| !service.isVersion2()
				|| !service.getIdpEndpoint().toString().equals(respEndpoint)) {
				continue;
			}

			// keep the first endpoint that matches
			if (firstServiceMatch == null) {
				if (DEBUG) {
					LOG.debug("Found matching service: " + service);
				}
				firstServiceMatch = service;
			}

			Association assoc =
				associations.load(service.getIdpEndpoint().toString(), authResp
					.getHandle());

			// don't look further if there is an association with this endpoint
			if (assoc != null) {
				if (DEBUG) {
					LOG.debug("Found existing association, "
						+ "not looking for another service endpoint.");
				}
				return service;
			}
		}

		if (firstServiceMatch == null) {
			LOG.error("No service element found to match "
				+ "the ClaimedID / OP-endpoint in the assertion.");
		}

		return firstServiceMatch;
	}

	/**
	 * Verifies the nonce in an authentication response.
	 * 
	 * @param authResp
	 *            The authentication response containing the nonce to be
	 *            verified.
	 * @param discovered
	 *            The discovery information associated with the authentication
	 *            transaction.
	 * @return True if the nonce is valid, false otherwise.
	 */
	public boolean verifyNonce(AuthSuccess authResp,
			DiscoveryInformation discovered) {
		String nonce = authResp.getNonce();

		if (nonce == null) {
			nonce = extractConsumerNonce(authResp.getReturnTo());
		}

		if (nonce == null) {
			return false;
		}

		// using the same nonce verifier for both server and consumer nonces
		return (NonceVerifier.OK == nonceVerifier.seen(discovered
			.getIdpEndpoint()
			.toString(), nonce));
	}

	/**
	 * Extracts the consumer-side nonce from the return_to parameter in
	 * authentication response from a OpenID 1.1 Provider.
	 * 
	 * @param returnTo
	 *            return_to URL from the authentication response
	 * @return The nonce found in the return_to URL, or null if it wasn't found.
	 */
	public String extractConsumerNonce(String returnTo) {
		if (DEBUG) {
			LOG.debug("Extracting consumer nonce...");
		}

		String nonce = null;
		String signature = null;

		URL returnToUrl;
		try {
			returnToUrl = new URL(returnTo);
		} catch (MalformedURLException e) {
			LOG.error("Invalid return_to: " + returnTo, e);
			return null;
		}

		String query = returnToUrl.getQuery();

		String[] params = query.split("&");

		for (String element : params) {
			String keyVal[] = element.split("=", 2);

			try {
				if (keyVal.length == 2 && "openid.rpnonce".equals(keyVal[0])) {
					nonce = URLDecoder.decode(keyVal[1], "UTF-8");
					if (DEBUG) {
						LOG.debug("Extracted consumer nonce: " + nonce);
					}
				}

				if (keyVal.length == 2 && "openid.rpsig".equals(keyVal[0])) {
					signature = URLDecoder.decode(keyVal[1], "UTF-8");
					if (DEBUG) {
						LOG.debug("Extracted consumer nonce signature: "
							+ signature);
					}
				}
			} catch (UnsupportedEncodingException e) {
				LOG.error("Error extracting consumer nonce / signarure.", e);
				return null;
			}
		}

		// check the signature
		if (signature == null) {
			LOG.error("Null consumer nonce signature.");
			return null;
		}

		String signed =
			returnTo.substring(0, returnTo.indexOf("&openid.rpsig="));
		if (DEBUG) {
			LOG.debug("Consumer signed text:\n" + signed);
		}

		try {
			if (_privateAssociation.verifySignature(signed, signature)) {
				LOG.info("Consumer nonce signature verified.");
				return nonce;
			}

			else {
				LOG.error("Consumer nonce signature failed.");
				return null;
			}
		} catch (AssociationException e) {
			LOG.error("Error verifying consumer nonce signature.", e);
			return null;
		}
	}

	public void setNonceVerifier(NonceVerifier nonceVerifier) {
		this.nonceVerifier = nonceVerifier;
	}

	public void setAssociations(ConsumerAssociationStore associations) {
		this.associations = associations;
	}

	public HttpClientManager getHttpClientManager() {
		return httpClientManager;
	}

	public void setHttpClientManager(HttpClientManager httpClientManager) {
		this.httpClientManager = httpClientManager;
	}
}
