import threading,subprocess,os,signal,itertools,datetime,functools,sys
from Queue import Queue,Empty
from favoritelistdb import FavoriteListDataBase,default_dbname
from protocol import shoutcast,icecast
import config
import gettext
import gc

exec_dir = os.path.dirname(os.path.abspath(__file__))
locale_dir = os.path.join(exec_dir,'locale')
db_path = os.path.join(exec_dir,default_dbname)
gettext.install('tuxshout',locale_dir,unicode=True)
try:
    import urwid, urwid.raw_display
except:
    print _('urwid module could not import!')
    print _('if urwid is not installed,please install!')
    if not config.is_module_installed('easy_install'):
        print _('you can not use "easy_install" ?')
        print _('please install them,and...')
    print _('type below')
    print _('easy_install urwid')
    sys.exit(-1)

version ='0.0.5'

class ListBoxItem(urwid.Text):
    def __init__(self, label, user_data):
        self.__super = super(self.__class__,self)
        self.__super.__init__(label)
        self.user_data = user_data
        self.__is_active = False
        
    def set_focus(self,is_focused):
        if self.__is_active:return
        if is_focused:
            self.__set_color(SharedObject.palette.ITEM_FOCUS)
        else:
            self.__set_color(None)

    def leave_focus(self):
        if self.__is_active:
            self.__set_color(SharedObject.palette.ITEM_LEAVE_REV)
        else:
            self.__set_color(SharedObject.palette.ITEM_LEAVE)
    def enter_focus(self):
        if self.__is_active:
            self.__set_color(SharedObject.palette.ITEM_REVERSE)
        else:
            self.__set_color(SharedObject.palette.ITEM_FOCUS)

    def __set_color(self,color):
        if color:
            self.set_text(
                (color,
                 self.get_text()[0])
                )
        else:
            self.set_text(self.get_text()[0])

    @property
    def label(self):
        return self.get_text()[0]

    @property
    def is_active(self):
        return self.__is_active

    def activate(self,is_active=True):
        self.__is_active = is_active
        if is_active:
            self.__set_color(SharedObject.palette.ITEM_REVERSE)
        else:
            self.__is_already_focused = False
            self.__set_color(None)

    def selectable(self):
        return True

    def keypress(self,size,key):return key

class ListBoxItemWalker(urwid.SimpleListWalker):
    def __init__(self,contents,focus_callback):
        self._super = super(self.__class__,self)
        self._super.__init__(contents)
        self.focus_callback = focus_callback

    def _focus_item(self,widget,is_active):
        if widget:
            widget.set_focus(is_active)

    def set_focus(self,position):
        focus_out_item,cur_pos = self.get_focus()

        #Only different item can focus out
        if cur_pos != position:
            self._focus_item(focus_out_item,False)
            self._super.set_focus(position)

        focus_in_item = self.get_focus()[0]
        self._focus_item(focus_in_item,True)
        self.focus_callback(focus_in_item)
    
class MouseDrivenListBox(urwid.ListBox):
    def __init__(self,items,key_event_handler,on_focus_callback):
        self.__super = super(self.__class__,self)
        self.__super.__init__(ListBoxItemWalker(items,self.__on_focus))
        self.__on_focus_callback = on_focus_callback
        self.__key_event_handler = key_event_handler
        self.__active_item = None
        self.__prev_focused = None
        self.__focused_timedelta = None
        self.__double_click_timedelta = datetime.timedelta(
            milliseconds=config.double_click_speed)

    def keypress(self, size, key):
        handler = self.__key_event_handler.get(key,None)
        if handler:
            focused_item = self.get_focus()[0]
            focused_item.activate()
            self._toggle_active_item(focused_item)
            handler(focused_item)
        else:
            return self.__super.keypress(size,key)

    def _toggle_active_item(self,item):
        if item and item.is_active and \
                self.__active_item != item:
            if self.__active_item:
                self.__active_item.activate(False)
            self.__active_item = item

    def __is_double_click(self,item):
        is_double_click = False
        if self.__prev_focused == item:
            if self.__focused_timedelta:
                timedelta = datetime.datetime.now() - self.__focused_timedelta
                if timedelta < self.__double_click_timedelta:
                    is_double_click = True
        else:
            self.__prev_focused = item
        self.__focused_timedelta = datetime.datetime.now()
        return is_double_click

    def leave_focus(self):
        if not self.get_focus():
            #this flow has not detected yet!
            return
        item = self.get_focus()[0]
        if item:
            item.leave_focus()

    def enter_focus(self):
        item = self.get_focus()[0]
        if item:
            item.enter_focus()

    def __on_focus(self,item):
        if item:
            is_double_click = self.__is_double_click(item)
            if is_double_click:
                item.activate()
                self._toggle_active_item(item)
            self.__on_focus_callback(item,is_double_click)

    def mouse_event(self, size, event, button, col, row, focus):
        is_handled = False
        if button == config.scroll_back:
            widget,pos = self.get_focus()
            if widget and pos > 0:
                self.set_focus(pos -1)
                is_handled = True
        elif button == config.scroll_next:
            widget,pos = self.get_focus()
            if widget and pos < len(self.body) -1:
                self.set_focus(pos + 1)
                is_handled = True
        else:
            is_handled = self.__super.mouse_event(size,
                                                  event,
                                                  button,
                                                  col,
                                                  row,
                                                  focus)
        return is_handled

class ListBoxedFrame(urwid.Frame):
    def __init__(self,listbox,header_text=None):
        self.listbox = listbox
        self.__super.__init__(urwid.LineBox( self.listbox ))
        if header_text:
            self.__super.set_header(
                urwid.Text(header_text)
                )

    def set_body(self,listbox):
        self.listbox = listbox
        self.__super.set_body(urwid.LineBox( self.listbox ))

    @property
    def has_item(self):
        return len(self.listbox.body) > 0

    def leave_focus(self):
        self.listbox.leave_focus()

    def enter_focus(self):
        self.listbox.enter_focus()

class StationListBox(ListBoxedFrame):
    def __init__(self,protocol):
        self._station_cache =[]
        self._station_info = {}
        self.__thread = None
        self._need_swip_cache = False
        self.__prev_focused = None
        self._protocol = protocol
        super(StationListBox,self).__init__(None,_('Stations'))

    def add_station(self,name,url,info):
        if self._need_swip_cache:self.__swip_cache()
        self._station_cache.append((name,url))
        self._station_info[url]=info

    def __swip_cache(self):
        del self._station_cache[:]
        self._station_info.clear()
        self._need_swip_cache = False

    def _on_focus(self,item,is_double_click):
        if SharedObject.notify and item:
            SharedObject.notify.print_info(
                self._protocol.translate_info(
                    self._station_info[item.user_data])
                )

            if item.is_active and is_double_click:
                self.on_player_activate(item)

    def on_player_activate(self,item):
        PlayerDialog(self._protocol).build(item.label,
                                           item.user_data)

    def build(self,event_handler):
        if self._need_swip_cache:
            self.__swip_cache()

        listbox =  MouseDrivenListBox(
            [ListBoxItem(name,url)
             for name ,url in self._station_cache]
            ,event_handler,self._on_focus)

        if self.body:
            if isinstance(self.body.w,urwid.Pile):
                del self.body.w.widget_list[:]
            else:
                del self.body.w.body.contents[:]

        self.set_body(listbox)
        SharedObject.notify.clean()
        listbox.set_focus(0)
        self.leave_focus()
        self._need_swip_cache = True

class FavoriteStationListBox(StationListBox):
    def __init__(self,remove_event_callback):
        self.__super = super(self.__class__,self)
        self.__remove_event_callback = remove_event_callback
        self.__super.__init__(None)
        self.__protocol_dict = {}

    def _on_focus(self,item,is_double_click):
        self.__switch_protocol(item)
        self.__super._on_focus(item,is_double_click)

    def on_player_activate(self,item):
        self.__switch_protocol(item)
        self.__super.on_player_activate(item)

    def __switch_protocol(self,item):
        for protocol_name,url_list in self.__protocol_dict.iteritems():
            if item.user_data in url_list:
                if protocol_name.endswith('shoutcast'):
                    self._protocol = shoutcast
                elif protocol_name.endswith('icecast'):
                    self._protocol = icecast
                break

    def add_station(self,name,url,protocol,info):
        self.__super.add_station(name,url,info)
        if protocol not in self.__protocol_dict:
            self.__protocol_dict[protocol] = []
        self.__protocol_dict[protocol].append(url)

    def on_remove_favorite(self,item):
        SharedObject.notify.print_info(_('Remove from Favorite: %s') % item.label) 
        FavoriteListDataBase(db_path).remove_station(item.user_data)
        self.__remove_event_callback()

    def build(self):
        event_handler ={'enter':self.on_player_activate,
                        'p':self.on_player_activate,
                        'delete':self.on_remove_favorite,
                        'r':self.on_remove_favorite}
        return self.__super.build(event_handler)

class SearchedStationListBox(StationListBox):
    def __init__(self,protocol):
        self.__super = super(self.__class__,self)
        self.__super.__init__(protocol)

    def on_add_favorite(self,item):
        SharedObject.notify.print_info(_('Add to Favorite: %s') % item.label)
        codec,bitrate,genre = self._station_info[item.user_data][0:3]
        FavoriteListDataBase(db_path).add_station(item.user_data,
                                           item.label,
                                           max(self._protocol.__name__.split()),
                                           codec,bitrate,genre)

    def build(self):
        event_handler ={'enter':self.on_player_activate,
                        'p':self.on_player_activate,
                        'a':self.on_add_favorite}
        return self.__super.build(event_handler)

class GenreListBox(ListBoxedFrame):
    def __init__(self,callback):
        self.__active_item = None
        self.build(callback)
        
    def __on_focus(self,item,is_double_click):
        if item.is_active and is_double_click:
            item.user_data(item)

    def build(self,callback):
        event_handler ={'enter':callback}
        listbox =  MouseDrivenListBox(
            [ListBoxItem(name,callback)           
             for name in config.major_genre]
            ,event_handler,self.__on_focus)

        listbox.set_focus(0)
        self.__super = super(self.__class__,self)
        self.__super.__init__(listbox,_('Genres'))

class AbstractGenreBrowser(urwid.Columns):
    _thread = None
    def __init__(self,protocol):
        self.protocol = protocol
        self.__super = super(AbstractGenreBrowser,self)
        self.__super.__init__(self.get_content())
        
    def terminate(self):
        if self._thread:
            self._thread.terminate()

    def get_content(self):
        self._genrebox = GenreListBox(self.genre_callback)
        self._stationbox = SearchedStationListBox(self.protocol)
        self._stationbox.build()
        return [ ('fixed',25,
                 self._genrebox),
                 self._stationbox
                 ]

    def mouse_event(self, size, event, button, col, row, focus):
        if not self.get_cursor_coords(size):
            cur_index = self.get_focus_column()
            self.move_cursor_to_coords(size,col,row)
            focused = self.get_focus_column()
            self.__super.set_focus_column(cur_index)            
            self.set_focus_column(focused)
        return self.__super.mouse_event(size, event, button, col, row, focus)

    def set_focus_column(self,num):
        if self.widget_list[num].has_item:
            self.get_focus().leave_focus()
            self.__super.set_focus_column(num)
            self.get_focus().enter_focus()

    def genre_callback(self,item):
        genre = item.label
        if not self._thread:
            self._thread = ThreadQueueDispatcher()
        self._thread.regist( self.create_request(genre) )
        SharedObject.notify.print_warn(
            _('Station-list of %s Downloading...') % genre)

    def create_request(self,genre):
        request = functools.partial(self.protocol.get_stations,genre,
                                    config.request_limit)
        return RequestCommand(self._stationbox,
                              request,
                              self.protocol.parse_station_list)

class SHOUTcastBrowser(AbstractGenreBrowser):
    def __init__(self):
        super(self.__class__,self).__init__(shoutcast)

class IcecastBrowser(AbstractGenreBrowser):
    def __init__(self):
        super(self.__class__,self).__init__(icecast)

class FavoriteStation(urwid.Frame):
    def terminate(self):
        del self.body.w.widget_list[:]

    def __init__(self):
        station_frame = self.get_content()
        super(self.__class__,self).__init__(station_frame.get_body(),
                                            header=station_frame.get_header())

    def __update_content(self):
        SharedObject.window.set_body(self.get_content())

    def get_content(self):
        stations = FavoriteStationListBox(self.__update_content)

        FavoriteListDataBase(db_path).get_stations(stations.add_station)
        stations.build()
        return stations

class AbstractStationSearch(urwid.Columns):
    def __init__(self,protocol):
        self.__thread = None
        self._protocol = protocol
        self.__keyword = ''
        self.__entry = config.request_limit
        self.__bitrate = config.request_bitrate
        self.__codec = self._get_default_codec()
        self.__super = super(AbstractStationSearch,self)
        self.__super.__init__(self.get_content())
        self.__super.set_focus_column(0)

    def terminate(self):
        if self.__thread:
            self.__thread.terminate()

    def _get_default_codec(self):
        match_codecs = filter(lambda item:item.endswith(config.request_codec.lower()),
                              self._protocol.get_codec_mimes())
        if match_codecs:
            return match_codecs[0]
        else:
            return self._protocol.get_codec_mimes()[0]

    def _on_bitrate_toggled(self,radio,state,rate):
        if state:
            self.__bitrate = rate

    def _on_codec_toggled(self,radio,state,codec):
        if state:
            self.__codec = codec

    def _verify_search_form(self):
        is_valid = False
        if not self.__keyword:
            SharedObject.notify.print_error(_('Please set keyword at least'))
        elif not self.__entry:
            SharedObject.notify.print_error(_('Please set entry more than 0'))
        else:
            is_valid = True
        return is_valid

    def _on_search(self,button):
        self.__entry = self.__entry_box.value()
        self.__keyword = self.__keyword_box.get_text()[0]
        if not self._verify_search_form():
            return
        self.__update_content()

    def _create_field_label(self,text):
        return urwid.Text(text+':')

    def _create_radio(self,group,label,value,callback,state=False):
        return urwid.RadioButton(group,
                                 label,
                                 state,
                                 callback,
                                 value)

    def _build_keyword_box(self):
        self.__keyword_box = urwid.Edit(edit_text=self.__keyword)
        yield self._create_field_label(_('keyword'))
        yield self._decorate_textbase_box(self.__keyword_box)

    def _build_limit_box(self):
        self.__entry_box = urwid.IntEdit(default=self.__entry)
        yield self._create_field_label(_('entry'))
        yield self._decorate_textbase_box(self.__entry_box)

    def _decorate_textbase_box(self,widget):
        return urwid.AttrWrap(widget,
                              SharedObject.palette.TEXTBOX)

    def _toggle_state(self,group,value):
        for radio in group:
            if radio.user_data == value:
                radio.set_state(True,False)
            else:
                radio.set_state(False,False)

    def _build_bitrate_group(self):
        bitrates = config.request_bitrate_range
        group = []
        event = self._on_bitrate_toggled
        yield self._create_radio(group, _('All'), 'All',event)
        for rate in bitrates:
            yield self._create_radio(group, rate+'kb', rate,event)
        self._toggle_state(group,self.__bitrate)

    def _build_bitrate_box(self):
        yield self._create_field_label(_('bitrate'))
        yield urwid.GridFlow(
            [x for x in self._build_bitrate_group()]
            ,9,1,1,'left')

    def _translate_codec(self,codec):
        return codec.split('/')[1]

    def _build_codec_group(self):
        group = []
        event = self._on_codec_toggled
        for codec in self._protocol.get_codec_mimes():
            label = self._translate_codec(codec).upper()
            yield self._create_radio(group,label,codec,event)
        self._toggle_state(group,self.__codec)

    def _build_codec_box(self):
        yield self._create_field_label(_('format'))
        yield urwid.GridFlow(
            [x for x in self._build_codec_group()]
            ,9,1,1,'left')

    def _build_search_button(self):
        yield urwid.Padding(
            urwid.Button(_('Search'),self._on_search),
            'center',10)

    def _build_search_form(self):
        body = urwid.ListBox([
                widget for widget in itertools.chain(
                    self._build_keyword_box(),
                    self._build_bitrate_box(),
                    self._build_codec_box(),
                    self._build_limit_box(),
                    self._build_search_button()
                    )])

        return urwid.Frame(urwid.LineBox(body),
                           header=urwid.Text(_('Search Form')))

    def get_content(self,*args):
        self.__stationbox = SearchedStationListBox(self._protocol)
        self.__stationbox.build()
        return [ ('fixed',25,
                 self._build_search_form()),
                 self.__stationbox
                 ]

    def __update_content(self):
        if not self.__thread:
            self.__thread = ThreadQueueDispatcher()

        request = functools.partial(self._protocol.search_stations,
                                    self.__keyword,
                                    self.__bitrate,
                                    self.__codec,
                                    self.__entry_box.value()
                                    )
        self.__thread.regist(RequestCommand(self.__stationbox,
                                            request,
                                            self._protocol.parse_station_list)
                             )
        SharedObject.notify.print_warn(_('Stations Searching...'))

    def mouse_event(self, size, event, button, col, row, focus):
        if not self.get_cursor_coords(size):
            cur = self.get_focus_column()
            self.move_cursor_to_coords(size,col,row)
            focused = self.get_focus_column()
            self.__super.set_focus_column(cur)
            self.set_focus_column(focused)
        return self.__super.mouse_event(size, event, button, col, row, focus)

    def set_focus_column(self,num):
        cur =self.get_focus()
        focused = self.widget_list[num]
        if cur == focused:return
        if isinstance(focused,SearchedStationListBox):
            if focused.has_item:
                focused.enter_focus()
                self.__super.set_focus_column(num)
        else:
            cur.leave_focus()
            self.__super.set_focus_column(num)

class SHOUTcastStationSearch(AbstractStationSearch):
    def __init__(self):
        super(self.__class__,self).__init__(shoutcast)

class IcecastStationSearch(AbstractStationSearch):
    def __init__(self):
        super(self.__class__,self).__init__(icecast)

class TerminatableThread(threading.Thread):
    _need_terminate = False
    def __init__(self,target=None,args=()):
        super(TerminatableThread,self).__init__(target=target,args=args)

    def terminate(self):
        self._need_terminate = True

class RequestCommand(TerminatableThread):
    def __init__(self,stationbox,download_func,parse_func):
        self._box = stationbox
        self._download = download_func
        self._parse = parse_func
        super(self.__class__,self).__init__()

    def run(self):
        resource = None
        try:
            resource = self._download()
        except:
            SharedObject.notify.print_warn(_('Requests are failed!'))
            return
        self._add_to_box(resource)

    def _add_to_box(self,resource):
        if not self._need_terminate:
            try:
                self._parse(resource,self._box.add_station)
                self._box.build()
            except:
                import sys
                SharedObject.notify.print_warn('error!')

class ThreadQueueDispatcher(TerminatableThread):
    __q = Queue()
    __worker = None
    def __init__(self):
        super(self.__class__,self).__init__()
        self.start()

    def regist(self,thread_entry):
        self.__q.put(thread_entry)

    def run(self):
        while not self._need_terminate:
            #single request can run at the same time.
            #queued request have priority over the running.
            try:
                entry = self.__q.get(timeout=0.1)
            except Empty:
                continue
            if self.__worker and self.__worker.isAlive():
                self.__worker.terminate()
            self.__worker = entry
            self.__worker.start()

        if self.__worker and self.__worker.isAlive():
            self.__worker.terminate()

class DialogBroker(object):
    def __init__(self):
        self.__dialog = None
        self.__prev_size = None
        self.__top = None

    def connect(self,widget,size):
        if self.__top:
            raise Exception
        self.__top = widget,size

    def disconnect(self):
        del self.__top ; del self.__dialog
        self.__top = None ; self.__dialog = None

    def propagate_key(self,size,key):
        if self.__dialog:
            if urwid.is_mouse_event(key):
                event,button,col,row = key
                self.__dialog.mouse_event(size, event, button, col, row, False)
            else:
                self.__dialog.keypress(size,key)

    @property
    def has_connection(self):
        return None != self.__dialog

    def __create_dialog(self,size):
        widget,widget_size = self.__top

        #calcurate position to locate top-widget as center
        left,top = [ (size[i] - widget_size[i])/2 for i in range(2)]
        wrapped_widget = urwid.Filler(
            urwid.AttrWrap(widget,
                           SharedObject.palette.DIALOG)
            )

        self.__dialog = urwid.Overlay(wrapped_widget,
                                      SharedObject.window,
                                      ('fixed left',left),
                                      widget_size[0],
                                      ('fixed top',top),
                                      widget_size[1])
    def get_canvas(self,size):
        canvas = None
        if self.__top:
            if self.__prev_size != size:
                self.__create_dialog(size)
            canvas = self.__dialog.render(size,True)
        return canvas

class AbstractDialog(object):
    def build(self,content,size):
        SharedObject.broker.connect(content,size)

    def close(self):
        SharedObject.broker.disconnect()

class RadioPlayer(TerminatableThread):
    def __init__(self,protocol,id):
        self.__proc = None
        self._temp = None
        self._protocol = protocol
        self.__id = id
        self.__super = super(self.__class__,self)
        self.__super.__init__()

    def run(self):
        location = self._get_playlist()

        if self._need_terminate:return

        command = [param if param !='%s' else location 
                   for param in config.player_command]

        SharedObject.notify.print_warn(_("Execute Player now!"))
        self.__proc = subprocess.Popen(command,
                                       stdout=subprocess.PIPE,
                                       stderr=subprocess.PIPE)
        self.__proc.wait()
        err = self.__proc.stderr.read()
        if err:
            SharedObject.notify.print_error(err)

    def _get_playlist(self):
        station_links = self._protocol.get_playlist(self.__id)
        if self._need_terminate:raise SystemExit()

        if not config.is_player_support_playlist:
            #protocol handles station_links as only simple url?
            if not self._protocol.is_support_playlist():return station_links

            parser = self._protocol.PlayListParser(station_links)
            if self._need_terminate:raise SystemExit()

            if not config.is_player_support_activity_check:
                SharedObject.notify.print_warn(_("Station's Activity Asking..."))
                return parser.get_active_item()
            else:
                return parser.iteritems().next()
        else:
            self._write_playlist_to_tempfile(station_links)
            return self._temp.name

    def terminate(self):
        self.__super.terminate()
        if self.__proc and self.__proc.returncode == None:
            os.kill(self.__proc.pid,signal.SIGTERM)
        self._cleanup_tempfile()

    def _cleanup_tempfile(self):
        if self._temp:
            if not self._temp.closed:
                self._temp.close()
            os.remove(self._temp.name)

    def _write_playlist_to_tempfile(self,playlist):
        import tempfile
        self._temp = open(
            os.path.join( tempfile.gettempdir(),
                          '%s.%s' % (tempfile.gettempprefix(),id)),
            'w')

        if isinstance(playlist,file):
            self._temp.write(playlist.read())
        else:
            self._temp.wirte(playlist)

        if not self._temp.closed:
            self._temp.close()

class PlayerDialog(AbstractDialog):
    def __init__(self,protocol):
        self.__protocol = protocol
        self.__super = super(self.__class__,self)
        self.__super.__init__()
        self.__thread = None

    def build(self,name,id):
        terminate_button = urwid.Padding(
            urwid.Button(_('Terminate'),self._close),
            'center',('relative',22)
            )
        content = urwid.Pile([urwid.Text(_('Player calling...')),
                              urwid.Divider('-'),
                              urwid.Text(name),
                              terminate_button
                              ])
        self.__super.build(content,(60,6))
        self._run_worker(id)

    def _run_worker(self,id):
        SharedObject.notify.print_warn(_('Playlist Downloading...'))
        self.__thread = RadioPlayer(self.__protocol,id)
        self.__thread.start()

    def _cleanup_worker(self):
        if self.__thread:
            self.__thread.terminate()
            self.__thread.join()

    def _close(self,button):
        SharedObject.notify.print_warn(_('Player Terminating...'))
        self._cleanup_worker()
        SharedObject.notify.print_warn(_('Player Terminated!'))
        self.__super.close()

class HelpDialog(AbstractDialog):
    __about_message =\
'''tuxshout version %s
(c)2009 s.yamada
Release under the LGPL
tuxshout is SHOUTcast/Icecast Radio tuner''' % version

    __help_message =\
_('''To run below method, type shortcut-key (i.e. S-key for SHOUTcast)
Help     : display this messages
SHOUTcast: supplys listing of SHOUTcast stations
Icecast  : supplys listing of Icecast stations
Favorite : supplys listing of registered stations
Exit     : exit program''')

    __method_message =\
_('''if station selected,below shortcut-key available
some shortcut are available at below mode
Browser=B Search=S Favorite=F
p [B|S|F]: play the station
enter    : same as p-key
a [B|S]  : add the station to favorite list
r [F]    : remove the station from favorite list
delete   : same as r-key''')

    __config_message =\
_('''configurable parameter is store in the config.py
please edit it to adapt to you environment''')

    def __init__(self):
        self.__super = super(self.__class__,self)
        self.__super.__init__()

    def build(self):
        content = urwid.Pile([urwid.Text(self.__about_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__help_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__method_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__config_message),
                              urwid.Padding(
                    urwid.Button(_('Close'),self._close),
                    'center',
                    ('relative',16)
                    )
                              ])
        self.__super.build(content,(65,24))

    def _close(self,button):
        self.__super.close()

class AbstractMenuItem(urwid.AttrWrap):
    def __init__(self,key,text,func,need_toggle=True):
        self._shortcut = key
        label = self.__create_label(key,text)
        super(AbstractMenuItem,self).__init__(
            ListBoxItem(label,None),None)
        self.activate(False)
        self.need_toggle = need_toggle
        self.func = func

    def __create_label(self,key,text):
        if key:
            readable_key = key.upper()
            return '%s: %s' % (readable_key,text)
        else:
            return text

    def activate(self,is_active=True):
        if is_active:
            self.func()
            if self.need_toggle:
                self.set_attr(SharedObject.palette.ITEM_FOCUS)
        else:
            self.set_attr(SharedObject.palette.ITEM_LEAVE)
        
    def is_shortcut(self,key):
        return self._shortcut == key

    def keypress(self, size, key):
        if self.is_shortcut(key):
            self.activate()
        else:
            return key

class NotebookItem(AbstractMenuItem):
    def __init__(self,update_func,text,pageobj_data):
        self.__pageobj_data = pageobj_data
        self.__content = None
        self.update = update_func
        super(self.__class__,self).__init__(None,text,
                                            self.draw_content)
        
    def draw_content(self):
        self.update(self._get_page_content())

    def _get_page_content(self):
        if not self.__content:
            self.__content = self.__pageobj_data()
        return self.__content

    def activate(self,is_active=True):
        if is_active:
            self.func()
            if self.need_toggle:
                self.set_attr(SharedObject.palette.NOTE_FOCUS)
        else:
            self.set_attr(SharedObject.palette.NOTE_LEAVE)

class ApplicationMenuItem(AbstractMenuItem):
    def __init__(self,key,label,application):
        self._super = super(self.__class__,self)
        self._super.__init__(key,
                             label,
                             self._switch_application)
        self._application = application

    def _switch_application(self):
        body = SharedObject.window.get_body()
        if body:
            body.terminate()
            del body
        SharedObject.window.set_body(
            self._application()
            )
        SharedObject.notify.clean()

class MenuBox(urwid.Columns):
    def __init__(self):
        self.__super = super(MenuBox,self)

    def _build(self): 
        self.__super.__init__(self._create_menu())
        self._active_item = self.widget_list[0]
        self._active_item.activate()
       
    def _create_menu(self):
        return []

    def set_focus(self,item):
        if self._active_item == item:return

        item.activate()
        if item.need_toggle:
            self._active_item.activate(False)
            self._active_item = item

    def keypress(self,size,key):
        is_propagated = False
        for cell in self.widget_list:
            if cell.is_shortcut(key):
                self.set_focus(cell)
                break
        if not is_propagated:return key

class NotebookItemBox(MenuBox):
    def __init__(self,target,item_inf):
        self.__target = target
        self.__item_inf = item_inf
        super(self.__class__,self).__init__()
        self._build()

    def set_page(self,pos):
        if len(self.widget_list) > pos:
            self.set_focus(self.widget_list[pos])

    def set_next_page(self,rotate=True):
        pos = self.widget_list.index(self._active_item) + 1
        if len(self.widget_list) == pos:
            pos = 0

        self.set_page(pos)

    def get_page(self):
        return self._active_item

    def update_content(self):
        self.get_page().draw_content()

    def __draw_content(self,content):
        if hasattr(self.__target,'body') and self.__target.body:
            self.__target.body.terminate()
        self.__target.set_body(content)

    def _create_menu(self):
        menus = [ 
            NotebookItem(self.__draw_content,label,classobj) 
            for label,classobj in self.__item_inf ]
        del self.__item_inf
        return menus

class ApplicationMenuBox(MenuBox):
    def __init__(self,is_simple_mode):
        self.__is_simple_mode = is_simple_mode
        super(self.__class__,self).__init__()
        self._build()

    def _create_menu(self):
        entries = itertools.chain(self.__application_generator(),(
            AbstractMenuItem('?',_('Help'),self.__show_help,False),
            AbstractMenuItem('x',_('Exit'),self.__exit_program,False)))
        items = []
        for entry in entries:
            items.append(entry)
        return items
    
    def __application_generator(self):
        if not self.__is_simple_mode:
            yield ApplicationMenuItem('s',_('SHOUTcast'),SHOUTcastService)
            yield ApplicationMenuItem('i',_('Icecast'),IcecastService)
        yield ApplicationMenuItem('f',_('Favorite'),FavoriteStation)
        
    def __show_help(self):
        HelpDialog().build()

    def __exit_program(self):
        import sys
        sys.exit(0)

class Notebook(urwid.Frame):
    def __init__(self,pages):
        menu = NotebookItemBox(self,pages)
        self.__super = super(Notebook,self)
        self.__super.__init__(
            None,
            header=menu)

        menu.update_content()

    def keypress(self, size, key):
        if key == 'tab':
            self.header.set_next_page()
        else:
            return self.__super.keypress(size,key)

class AbstractService(Notebook):
    def __init__(self,pages):
        self.__super = super(AbstractService,self)
        self.__super.__init__(pages)

    def terminate(self):
        self.body.terminate()

class SHOUTcastService(AbstractService):
    def __init__(self):
        super(self.__class__,self).__init__(
            (
                (_('Browse'),SHOUTcastBrowser),
                (_('Search'),SHOUTcastStationSearch)
                )
            )

class IcecastService(AbstractService):
    def __init__(self):
        from protocol import icecast
        super(self.__class__,self).__init__(
            (
                (_('Browse'),IcecastBrowser),
                (_('Search'),IcecastStationSearch)
                )
            )

class TuxShout:
    __worker_thread = None
    def __init__(self,is_simple_mode=False):
        SharedObject.regist_main_window(urwid.Frame( None ))
        SharedObject.window.set_header(
            urwid.AttrWrap(ApplicationMenuBox(is_simple_mode),
                           SharedObject.palette.ITEM_LEAVE))

    def set_status(self,text,state):
        info = urwid.AttrWrap(
            urwid.Text(text),state)
        SharedObject.window.set_footer(info)

    def main(self):
        space = (65,24)
        if space > SharedObject.screen.get_cols_rows():
            print _(\
'''tuxshout can not run at this nallow screen!
please allocate more wider screen than 
width :%s
height:%s''') % space
            return
        SharedObject.screen.run_wrapper( self.run )

    def run(self):
        SharedObject.screen.set_mouse_tracking()
        size = SharedObject.screen.get_cols_rows()
        broker = SharedObject.broker
        while 1:
            self.draw_screen(size,
                             broker.get_canvas(size))
            keys = SharedObject.screen.get_input()
            for k in keys:
                if k == "window resize":
                    size = SharedObject.screen.get_cols_rows()
                    continue

                #key-action of dialog prioir than other widgets
                if broker.has_connection:
                    broker.propagate_key(size,k)
                    continue

                #mouse event-handler
                if urwid.is_mouse_event(k):
                    event,button,col,row = k
                    SharedObject.window.mouse_event(size,event,button,
                                                  col,row,True)
                    continue

                #key-action of any other widget prioir than menu
                if not SharedObject.window.keypress( size, k ):
                    continue
                else:
                    SharedObject.window.header.keypress(size,k)

    def draw_screen(self, size,canvas=None):
        if not canvas:
            canvas = SharedObject.window.render( size, focus=True )
        SharedObject.screen.draw_screen( size, canvas )

class Notification:
    def print_info(self,text):
        self._set_status(text,SharedObject.palette.MSG_INFO)

    def print_warn(self,text):
        self._set_status(text,SharedObject.palette.MSG_WARN)

    def print_error(self,text):
        self._set_status(text,SharedObject.palette.MSG_ERR)

    def _set_status(self,text,state):
        info = SharedObject.window.footer
        if not info:
            info =urwid.AttrWrap(
                urwid.Text(text),state)
            SharedObject.window.set_footer(info)
        else:
            info.w.set_text(text)
            info.set_attr(state)

    def clean(self):
        SharedObject.window.set_footer(None)

class ColorPalette(object):
    def __init__(self,screen):
        screen.register_palette(self.__palette_generator())

    def __palette_generator(self):
        yield self.DEFAULT, 'white', 'black', 'standout'
        yield self.DIALOG,  'black', 'light gray', 'standout'
        yield self.ITEM_LEAVE, 'black', 'dark cyan', 'bold'
        yield self.ITEM_LEAVE_REV, 'black', 'dark magenta', 'bold'
        yield self.ITEM_FOCUS, 'white', 'dark blue', 'bold'
        yield self.NOTE_FOCUS, 'white', 'dark green', 'bold'
        yield self.NOTE_LEAVE, 'black', 'brown', 'bold'
        yield self.ITEM_REVERSE, 'white', 'dark red', 'bold'
        yield self.TEXTBOX,  'black', 'light gray','underline'
        yield self.MSG_INFO, 'light green', 'black', 'underline'
        yield self.MSG_WARN, 'yellow', 'black','bold'
        yield self.MSG_ERR,  'light red', 'black',('bold','underline')

    @property
    def MSG_INFO(self):
        return 'info'

    @property
    def MSG_WARN(self):
        return 'warn'

    @property
    def MSG_ERR(self):
        return 'error'

    @property
    def ITEM_LEAVE(self):
        return 'leave'

    @property
    def ITEM_LEAVE_REV(self):
        return 'leave_rev'

    @property
    def ITEM_FOCUS(self):
        return 'focus'

    @property
    def ITEM_REVERSE(self):
        return 'reverse'

    @property
    def TEXTBOX(self):
        return 'tbox'

    @property
    def DEFAULT(self):
        return 'default'

    @property
    def NOTE_FOCUS(self):
        return 'note_focus'

    @property
    def NOTE_LEAVE(self):
        return 'note_leave'

    @property
    def DIALOG(self):
        return 'dialog'


class SharedObjectManager(object):
    def __init__(self):
        self.screen = urwid.raw_display.Screen()
        self.palette = ColorPalette(self.screen)
        self.notify = Notification()

    def regist_main_window(self,frame):
        self.window = frame
        self.broker = DialogBroker()

SharedObject = SharedObjectManager()

if __name__ == '__main__':
    gc.enable()
    gc.set_threshold(5)
    is_simple_mode = "-s" in sys.argv
    TuxShout(is_simple_mode).main()
