#!/usr/bin/env python
import sys, time, locale, os, glob, csv
from PyQt4 import QtCore, QtGui, uic
from twitturnConfigDialog import TwitturnConfigDialog
from twitturnTimeLineWidget import TwitturnTimeLineWidget
from twitturnSearchingLineEdit import TwitturnSeachingLineEdit
from twitturnShortcutKeyManager import TwitturnShortcutKeyManager
from twitturnTabWidget import TwitturnTabWidget
from twitturnCore import TwitturnCore
from iconCacheManager import IconChacheManager


TWITTER_MAX_STRING_LENGTH = 140
class ExtendedQTextEdit(QtGui.QTextEdit):
    def keyPressEvent(self, e):
        if e.key() in (QtCore.Qt.Key_Return, QtCore.Qt.Key_Enter):
            self.emit(QtCore.SIGNAL("ReteurnOrEnterPressed"))
        QtGui.QTextEdit.keyPressEvent(self, e)


form_class, base_class = uic.loadUiType("mainwindow.ui")
class Twitturn(base_class, form_class):
    def __init__(self, *args):
        base_class.__init__(self, *args)

        self.setupUi(self)

        self.textEdit.__class__ = ExtendedQTextEdit
        self.tabWidget = TwitturnTabWidget(self)
        self.verticalLayoutCenter.addWidget(self.tabWidget)
        self.mainTimeLine = TwitturnTimeLineWidget(self.tabWidget)
        self.tabWidget.addTab(self.mainTimeLine, self.tr("&TimeLine"))
        self.treeWidget = self.mainTimeLine.treeWidget
        self.lineEdit.__class__ = TwitturnSeachingLineEdit
        self.lineEdit.setParentForm(self)
        self.lineEdit.setVisible(False)
        self.searchTab = None

        self.labelApiLimitStatus = QtGui.QLabel(self)
        self.labelApiLimitStatus.setContentsMargins(6,0,0,0)
        self.statusBar().addWidget(self.labelApiLimitStatus)
        self.labelTimeLineSpeed = QtGui.QLabel(self)
        self.labelTimeLineSpeed.setContentsMargins(4,0,0,0)
        self.statusBar().addWidget(self.labelTimeLineSpeed)
        self.newPostCount = 0

        self.shortcutkeyManager = TwitturnShortcutKeyManager()
        self.registerShortcutKeys()

        self.loadingIcon = QtGui.QMovie("images/loader.mng")
        self.labelLoading.setMovie(self.loadingIcon)
        self.loadingIcon.start()
        self.loadingIcon.setPaused(True)

        self.systemTrayIcon = QtGui.QSystemTrayIcon(
                QtGui.QIcon("images/twitturn-icon"),
                self)
        self.systemTrayIcon.show()
        self.systemTrayIconMenu = QtGui.QMenu(self)
        self.actionBlockPopupNotification = self.systemTrayIconMenu.addAction( 
                self.tr("Block Popup Notification"))
        self.actionBlockPopupNotification.setCheckable(True)
        actionQuit = self.systemTrayIconMenu.addAction(
                self.tr("&Quit"), 
                self.exit)
        self.systemTrayIconMenu.insertSeparator(actionQuit)
        self.systemTrayIcon.setContextMenu( self.systemTrayIconMenu)
        self.systemTrayIconMessageDictionary = {}

        self.updatePostProcessingFunctionQueue = []
        self.autoReplieFetchCounter = 0
        self.autoDirectMessageFetchCounter = 0

        self.iconCache = IconChacheManager()
        self.appendingStatusesQeue = None
        self.replyToStatusId = None
        self.replyToUserId = None
        self.userNick = None
        self.timer = QtCore.QTimer(self)
        self.timer.setInterval(60 * 1000)
        self.timer2 = QtCore.QTimer(self)
        self.timer2.setInterval(3 * 60 * 1000)
        self.rxReply = QtCore.QRegExp("@([a-zA-Z0-9_]+)")
        # If URL includes '@', this regexp will not accept.
        self.rxURL = QtCore.QRegExp("(https?:\/\/[-_.!~*'()a-zA-Z0-9;/?:&=+$,%#]+)")
        self.settings = QtCore.QSettings("Twitturn")

        self.core = TwitturnCore(self)
        self.connectActions()

        if not self.settings.contains("configured"): 
            self.importDefaultSettings()
            self.showConfigDialog()

        self.restoreGUISettings()
        self.applySettings()

        self.timer.start()
        self.timer2.start()
        if self.settings.value('twitter/fetchDMandRepliesOnStartup').toBool():
            self.updatePostProcessingFunctionQueue.append( 
                    self.core.queryReplies)
            self.updatePostProcessingFunctionQueue.append(
                    self.core.queryDirectMessages)
        self.updatePostProcessingFunctionQueue.append(
                self.core.queryApiLimitStatus)
        self.update()

    def connectActions(self):
        self.connect(self.action_Import_Settings,
                QtCore.SIGNAL("triggered()"),
                self.importSettings)
        self.connect(self.action_Export_Settings,
                QtCore.SIGNAL("triggered()"),
                self.exportSettings)
        self.connect(self.actionBlockPopupNotification,
                QtCore.SIGNAL("triggered()"),
                self.togglePopupNotificationEnabled)
        self.connect(self.actionFocus_Imput_Erea, 
                QtCore.SIGNAL("triggered()"),
                self.textEdit, QtCore.SLOT("setFocus()"))
        self.connect(self.action_Quit, QtCore.SIGNAL("triggered()"),
                self.exit)
        self.connect(self.actionSelect_Top, QtCore.SIGNAL("triggered()"),
                self.selectTopPost)
        self.connect(self.action_Configure, QtCore.SIGNAL("triggered()"),
                self.showConfigDialog)
        self.connect(self.actionFocus_Time_Line, 
                QtCore.SIGNAL("triggered()"),
                self.focusTimeLine)
        self.connect(self.action_TimeLine, QtCore.SIGNAL("triggered()"),
                self.fetchTimeline)
        self.connect(self.action_Replies, QtCore.SIGNAL("triggered()"),
                self.fetchReplies)
        self.connect(self.action_DirectMessages, 
                QtCore.SIGNAL("triggered()"),
                self.fetchDirectMessages)
        self.connect(self.actionSearch,
                QtCore.SIGNAL("triggered()"),
                self.beginSearching)
        self.connect(self.actionSearch_Relative,
                QtCore.SIGNAL("triggered()"),
                self.searchRelative)
        self.connect(
                self.mainTimeLine, 
                QtCore.SIGNAL('currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)'),
                self.setCurrentPost)
        self.connect(
                self.mainTimeLine.treeWidget,
                QtCore.SIGNAL('itemDoubleClicked(QTreeWidgetItem *,int)'),
                self.putReplyHeader)
        self.connect(self.textEdit, QtCore.SIGNAL("textChanged()"),
                self.countText)
        self.connect(
                self.textEdit, 
                QtCore.SIGNAL("ReteurnOrEnterPressed"),
                self.update)
        self.connect(self.pushButtonWeb, QtCore.SIGNAL("clicked()"),
                self.openWeb)
        self.connect(self.toolButtonTime, QtCore.SIGNAL("clicked()"),
                self.openPost)
        self.connect(self.pushButtonUpdate, QtCore.SIGNAL("clicked()"),
                self.update)
        self.connect(self.core, QtCore.SIGNAL("friendsTimelineUpdated"),
                self.updateTimeline)
        self.connect(self.core, QtCore.SIGNAL("apiLimitStatusReady"),
                self.updateApiLimitStatus)
        self.connect(self.core, QtCore.SIGNAL("done(bool)"), self.networkCheck)
        self.connect(
                self.core, 
                QtCore.SIGNAL("responseHeaderReceived (const QHttpResponseHeader&)"),
                self.printStatusCode)
        self.connect(self.core, QtCore.SIGNAL("statusUpdated"),
                self.doPostingPhase2)
        self.connect(self.timer, QtCore.SIGNAL("timeout()"),
                self.autoFetchTimeline)
        self.connect(self.timer2, QtCore.SIGNAL("timeout()"),
                self.updateTimeLineSpeed)
        self.connect(
                self.iconCache, 
                QtCore.SIGNAL("iconReady(QIcon)"),
                self.appendStatusWithIcon)
        self.connect(
                self.tabWidget,
                QtCore.SIGNAL("currentChanged(int)"),
                self.changeCurrentTab)
        self.connect(
                self.systemTrayIcon,
                QtCore.SIGNAL("activated(QSystemTrayIcon::ActivationReason)"),
                self.systemTrayIconActivatedEventHandler)
        self.connect(
                self.systemTrayIcon,
                QtCore.SIGNAL("messageClicked()"),
                self.showAndActivateWindow)
        return

    def registerShortcutKeys(self):
        functionList  = (
            ( lambda: self.toggleFavo(self.treeWidget.currentItem()),
                "toggleFavo" ),
            ( self.focusBellowPost, 'focusBellowPost' ),
            ( self.focusAbovePost, 'focusAbovePost' ),
            ( self.searchNick, 'searchNick'  ),
            ( self.openWeb, 'openWeb' ),
            ( self.openHome, 'openHome' ),
            ( self.openPost, 'openPost' ),
            ( self.openLinks, 'openLinks' ),
            ( self.putReplyHeader, 'putReplyHeader'), 
            ( self.appendMultiReplyHeader, 'appendMultiReplyHeader' ), 
            ( self.putDirectMessageHeader, 'putDirectMessageHeader' ),
            ( self.textEdit.clear, 'clearTextEdit'), 
            ( lambda: self.treeWidget.setCurrentItem(
                self.treeWidget.topLevelItem(0)),
                'focusTopItem' ),
            ( self.beginSearching, 'beginSearching'),
            ( self.searchRelative, 'searchRelative' ),
            ( self.textEdit.setFocus, 'focusTextEdit' ),
            ( self.mainTimeLine.changeThisTab, 
                'changeToTheMainTimeLineTab'),
            )

        for entry in functionList:
            self.shortcutkeyManager.addFunction(*entry)

    def updateTimeLineSpeed(self):
        intervalSec = self.timer2.interval()/1000
        timeLineSpeed = self.newPostCount*60/intervalSec
        maxTimeLineSpeed = self.retrievingStatusCount*60*3/intervalSec
        self.labelTimeLineSpeed.setText(
                "TL Speed: " + str(timeLineSpeed) + "/" + str(maxTimeLineSpeed) + " Log count: " + str(self.treeWidget.topLevelItemCount()))
        self.newPostCount = 0
        return

    def importSettings(self):
        fileName = QtGui.QFileDialog.getOpenFileName(
                self,
                self.tr("Import File"),
                QtCore.QString(),
                self.tr("Twitturn Setting Files (*.ini *.conf);;All Files (*)"))
        self.importSettingFile(fileName)

    def importSettingFile(self, fileName):
        fromSettings = QtCore.QSettings(
                fileName, QtCore.QSettings.IniFormat, self)
        toSettings = self.settings
        allKeys = fromSettings.allKeys()
        for key in allKeys:
            toSettings.setValue(key, fromSettings.value(key))
        toSettings.sync()

        self.restoreGUISettings()
        self.applySettings()

    def exportSettings(self):
        fileName = QtGui.QFileDialog.getSaveFileName(
                self,
                self.tr("Export File"),
                "Twitturn.ini",
                self.tr("Twitturn Setting Files (*.ini *.conf);;All Files (*)"))
        self.exportSettingFile(fileName)

    def doSearching(self):
        self.mainTimeLine.swapHighlighting()

#       DO NOT CALL "self.searchTab.treeWidget.clear()"
        self.searchTab.clear()

        rx = QtCore.QRegExp( 
                self.lineEdit.text(), 
                QtCore.Qt.CaseInsensitive,
                QtCore.QRegExp.RegExp2
                )
        for i in range(self.mainTimeLine.treeWidget.topLevelItemCount()):
            item = self.mainTimeLine.treeWidget.topLevelItem(i)
            for c in [1,2]:
                if not (-1 == rx.indexIn( item.text(c))):
                    self.searchTab.treeWidget.addTopLevelItem(
                            item.clone())
                    break

        self.mainTimeLine.unswapHighlighting()
        self.lineEdit.setVisible(False)
        self.searchTab.changeThisTab()

    def exportSettingFile(self, fileName):
        fromSettings = self.settings
        toSettings = QtCore.QSettings(
                fileName, QtCore.QSettings.IniFormat, self)
        allKeys = fromSettings.allKeys()
        for key in allKeys:
            toSettings.setValue(key, fromSettings.value(key))
        toSettings.sync()
    
    def togglePopupNotificationEnabled(self):
        self.settings.setValue("notification/popup/disabled",
                QtCore.QVariant(self.actionBlockPopupNotification.isChecked()))
    def systemTrayIconActivatedEventHandler(self, reason):
        if reason == QtGui.QSystemTrayIcon.Trigger:
            self.toggleWindowVisibility()
    def toggleWindowVisibility(self):
        if self.isVisible():
            if self.isMinimized():
                self.showNormal()
            else:
                self.hide()
        else:
            self.show()
            self.activateWindow()
            self.raise_()
    def showAndActivateWindow(self):
        if self.isVisible():
            if self.isMinimized():
                self.showNormal()
        else:
            self.show()
            self.activateWindow()
            self.raise_()

    def changeCurrentTab(self, tabIdx):
        self.treeWidget = self.tabWidget.currentWidget().treeWidget

        if self.treeWidget.currentItem() == None:
            self.treeWidget.setCurrentItem(
                self.treeWidget.topLevelItem(0))
        self.treeWidget.setFocus()

        current = self.treeWidget.currentItem()
        if not current == None:
            self.setCurrentPost(current, None)

    def applySettings(self):
        if not self.settings.contains("configured"): return
        self.applyTwitturnCoreSettings()
        self.userNick = self.settings.value("twitter/user").toString()
        self.timer.setInterval(
                self.settings.value("twitter/interval").toInt()[0] * 1000)
        self.timer2.setInterval(
                3*self.settings.value("twitter/interval").toInt()[0] * 1000)
        self.retrievingStatusCount = self.settings.value("twitter/count").toInt()[0]
        self.applyIconChacheSettings()
        self.actionBlockPopupNotification.setChecked(
                self.settings.value("notification/popup/disabled").toBool())
    def applyTwitturnCoreSettings(self):
        self.settings.beginGroup("twitter")
        self.core.setUser(
                self.settings.value("user").toString(),
                self.settings.value("password").toString())
        if self.settings.value("useGETMethod").toBool():
            self.core.setQueryMethod("GET")
        else:
            self.core.setQueryMethod("POST")
        self.settings.endGroup()
        self.settings.beginGroup("proxy")
        if self.settings.value("useProxy").toBool():
            self.core.setProxy(
                    self.settings.value("server").toString(),
                    self.settings.value("port").toInt())
        self.settings.endGroup()
    def applyIconChacheSettings(self):
        self.settings.beginGroup("proxy")
        if self.settings.value("useProxy").toBool():
            self.iconCache.setProxy(
                    self.settings.value("server").toString(),
                    self.settings.value("port").toInt())
        self.settings.endGroup()

    def autoFetchTimeline(self):
        if 4 < self.autoReplieFetchCounter:
            self.updatePostProcessingFunctionQueue.append( 
                    self.core.queryReplies)
            self.autoReplieFetchCounter = 0
        else:
            self.autoReplieFetchCounter += 1
        if 5 < self.autoDirectMessageFetchCounter:
            self.updatePostProcessingFunctionQueue.append(
                    self.core.queryDirectMessages)
            self.autoDirectMessageFetchCounter = 0
        else: 
            self.autoDirectMessageFetchCounter += 1
        self.updatePostProcessingFunctionQueue.append(
                self.core.queryApiLimitStatus)
        self.fetchTimeline()


    def fetchTimeline(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        self.core.queryFriendsTimeline(
                None, None, self.retrievingStatusCount)

    def doPostingPhase2(self):
        self.textEdit.clear()
        self.replyToStatusId = None
        self.replyToUserId = None
        if self.settings.value("twitter/fetchTimeLineWhenPosting").toBool():
            self.core.queryFriendsTimeline(
                    None, None, self.retrievingStatusCount)
        else:
            self.escapeFetchingMode()
        
    def networkCheck(self, error):
        if error:
            self.escapeFetchingMode(True)
            time.sleep(1)
            if self.core.error() == TwitturnCore.AuthenticationRequiredError:
                QtGui.QMessageBox.warning(self,
                        self.tr("Twitturn"),
                        self.tr("""Twitter authentication error!
Incorrect user name or password.
Could you check your settings?"""))
                self.showConfigDialog()
            if self.core.error() == TwitturnCore.Aborted:
                QtGui.QMessageBox.warning(self,
                        self.tr("Twitturn"),
                        self.tr(self.core.errorString()))

            print self.core.errorString()
            return

    def printStatusCode(self, resp):
        if not resp.statusCode() == 200:
            self.escapeFetchingMode(True)
        print resp.statusCode()
    def update(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        if self.textEdit.toPlainText() == "":
            self.core.queryFriendsTimeline( 
                    None, None, self.retrievingStatusCount )
        else:
            if not self.replyToStatusId == None:
                if not self.textEdit.toPlainText().startsWith(
                        '@' + self.replyToUserId + ' '):
                    self.replyToStatusId = None
            self.core.queryUpdate(
                    str(QtCore.QUrl.toPercentEncoding(
                        self.textEdit.toPlainText())),
                    self.replyToStatusId )
    def fetchReplies(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        self.core.queryReplies()
    def fetchDirectMessages(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        self.core.queryDirectMessages()

    def updateTimeline(self, statuses):
        if len(statuses) == 0:
            self.escapeFetchingMode()
            return
        self.appendingStatusesQeue = statuses
        self.iconCache.queryIcon(
                self.appendingStatusesQeue[0][0],
                self.appendingStatusesQeue[0][1])
    def updateApiLimitStatus(self, status):
        if status == None: return
        self.labelApiLimitStatus.setText(
                "Remaining API: "+status[0]+"/"+status[1]+" Resets: "+ QtCore.QDateTime.fromTime_t(QtCore.QString(status[3]).toULong()[0]).toString("h:mm"))
        self.escapeFetchingMode()

    def appendStatusWithIcon(self, icon):
        self.addPost(icon, self.appendingStatusesQeue.pop(0))
        if len(self.appendingStatusesQeue) == 0:
            self.escapeFetchingMode()
            return
        self.iconCache.queryIcon(
                self.appendingStatusesQeue[0][0],
                self.appendingStatusesQeue[0][1])
        return
        
    def addPost(self, icon, post):
        treeEntry = QtGui.QTreeWidgetItem()
        treeEntry.setIcon(0, icon)
        for i, v in enumerate(post[1: ]):
            if i == 2:
                # Twitter API response sample: 
                # Wed Dec 17 02:30:10 +0000 2008
                # Sun Dec 07 22:25:26 +0000 2008 
                dt = QtCore.QLocale( 
                        QtCore.QLocale.English,
                        ).toDateTime(
                    v, 
                    "ddd MMM dd hh:mm:ss +0000 yyyy")
                dt.setTimeSpec(QtCore.Qt.UTC)
                v =  dt.toLocalTime().toString("yyyy/MM/dd hh:mm:ss")
            treeEntry.setText(i+1,v)

        self.emit(QtCore.SIGNAL("newPostCame"), treeEntry)
        if self.mainTimeLine.addPost(treeEntry, False):
            self.newPostCount += 1

            if self.isSelfPost(post[1], self.userNick):
                treeEntry.setForeground( 
                        1, QtGui.QBrush(QtGui.QColor("darkcyan")))
                treeEntry.setForeground( 
                        2, QtGui.QBrush(QtGui.QColor("darkcyan")))
                return

            if self.isReply(post[2], self.userNick):
                treeEntry.setForeground(
                        1, QtGui.QBrush(QtGui.QColor("red")))
                treeEntry.setForeground(
                        2, QtGui.QBrush(QtGui.QColor("red")))
                return

    def isReply(self, text, nick):
        return QtCore.QString(text).contains(nick)
    def isSelfPost(self, postNick, yourNick):
        return postNick == yourNick;

    def enterFetchingMode(self):
        if not self.pushButtonUpdate.isEnabled(): return
        self.pushButtonUpdate.setEnabled(False)
        self.loadingIcon.setPaused(False)
        for i in range(self.tabWidget.count()):
            self.tabWidget.setTabIcon(i, QtGui.QIcon())
    def escapeFetchingMode(self, force=False):
        if not force:
            if 0 < len(self.updatePostProcessingFunctionQueue):
                print "escapeFetchingMode reverce"
                self.updatePostProcessingFunctionQueue.pop(0)()
                return 
        if self.pushButtonUpdate.isEnabled(): return
        self.pushButtonUpdate.setEnabled(True)
        self.loadingIcon.setPaused(True)
        if not self.settings.value("notification/popup/disabled").toBool():
            keys = self.systemTrayIconMessageDictionary.keys()
            if 0 < len(keys):
                str = ""
                keys.sort()
                for key in keys:
                    str += self.systemTrayIconMessageDictionary[key] + '\n'
                self.systemTrayIcon.showMessage(
                        QtCore.QString(),
                        str,
                        self.systemTrayIcon.NoIcon)
        self.systemTrayIconMessageDictionary.clear()
        del self.updatePostProcessingFunctionQueue[:]

    def showConfigDialog(self):
        self.timer.stop()
        dialog = TwitturnConfigDialog(self)
        dialog.exec_()
        self.applySettings()
        self.timer.start()

    def lineEditKeyPressEventHandler(self, keyEvent):
        if keyEvent.key() == QtCore.Qt.Key_Escape:
            self.lineEdit.setVisible(True)

    def keyPressEvent(self, keyEvent):
        if self.treeWidget.hasFocus():
            function = self.shortcutkeyManager.keyToFunction(keyEvent.key())
            if not function == None:
                function()


    def beginSearching(self):
        self.lineEdit.selectAll()
        self.lineEdit.setVisible(True)
        self.lineEdit.setFocus()
    def searchRelative(self):
        current = self.treeWidget.currentItem()
        nicks = self.pickupNicks(current.text(2))
        nicks.append(current.text(1))
        str = QtCore.QString()
        for i, nick in enumerate(nicks):
            if i == 0:
                str.append("("+nick+")")
            else:
                str.append("|("+nick+")")

        self.lineEdit.setText(str)
        self.doSearching()
    def searchNick(self):
        current = self.treeWidget.currentItem()
        self.lineEdit.setText("^"+current.text(1)+"$")
        self.doSearching()


    def focusBellowPost(self):
        currentItem = self.treeWidget.currentItem()
        bellowItem = self.treeWidget.itemBelow(currentItem)
        if not bellowItem == None:
            self.treeWidget.setCurrentItem(bellowItem)
    def focusAbovePost(self):
        currentItem = self.treeWidget.currentItem()
        aboveItem = self.treeWidget.itemAbove(currentItem)
        if not aboveItem == None:
            self.treeWidget.setCurrentItem(aboveItem)

    def toggleFavo(self, post):
        if not self.pushButtonUpdate.isEnabled():
            return
        if post.text(9) == "":
            self.core.queryFavorite(post.text(4), True)
            post.setText(9, "F")
        else:
            self.core.queryFavorite(post.text(4), False)
            post.setText(9, "")
    def toggleSpam(self, post):
        if post.text(10) == "True":
            post.setText(10, "False")
        else:
            post.setText(10, "True")
    def openWeb(self):
        post = self.treeWidget.currentItem()
        if not post.text(6) == "":
            QtGui.QDesktopServices.openUrl(QtCore.QUrl(post.text(6)))
    def openHome(self):
        post = self.treeWidget.currentItem()
        if not post.text(1) == "":
            QtGui.QDesktopServices.openUrl(
                    QtCore.QUrl('http://twitter.com/' + post.text(1)))
        pass
    def openPost(self):
        post = self.treeWidget.currentItem()
        if not post.text(1) == "" and not post.text(4) == "":
            QtGui.QDesktopServices.openUrl(
                    QtCore.QUrl('http://twitter.com/' + post.text(1) + '/status/' + post.text(4)))
    def openLinks(self):
        str = self.treeWidget.currentItem().text(2)
        pos = 0
        while True:
            pos = self.rxURL.indexIn(str, pos)
            if pos == -1:
                break;
            QtGui.QDesktopServices.openUrl( QtCore.QUrl( self.rxURL.cap()))
            pos += self.rxURL.matchedLength()
            time.sleep(1)
        return
    def putReplyHeader(self):
        self.replyToStatusId = self.treeWidget.currentItem().text(4)
        self.replyToUserId = self.treeWidget.currentItem().text(1)
        self.textEdit.setText(
                '@' + self.treeWidget.currentItem().text(1) + ' ')
        self.textEdit.moveCursor(QtGui.QTextCursor.End)
        self.textEdit.setFocus()
    def putDirectMessageHeader(self):
        self.textEdit.setText(
                'D ' + self.treeWidget.currentItem().text(1) + ' ')
        self.textEdit.moveCursor(QtGui.QTextCursor.End)
        self.textEdit.setFocus()
    def appendMultiReplyHeader(self):
        str = self.textEdit.toPlainText()
        if str == "":
            str = '.'
        str += '@' + self.treeWidget.currentItem().text(1) + ' '
        self.textEdit.setText(str)
        self.textEdit.moveCursor(QtGui.QTextCursor.End)

    def countText(self):
        self.labelTextCount.setText(
                '%s' % (TWITTER_MAX_STRING_LENGTH - self.textEdit.toPlainText().size()))

    def closeEvent(self, event):
        self.saveGUISettings()

    def exit(self):
        self.saveGUISettings()
        self.systemTrayIcon.hide()
        QtGui.qApp.quit()

    def importDefaultSettings(self):
        self.importSettingFile( "TwitturnDefaultSettings.ini")

    def restoreGUISettings(self):

        self.settings.beginGroup("GUI")
        self.treeWidget.header().restoreState(
                self.settings.value("treeWidget/header").toByteArray())
        self.restoreGeometry(
                self.settings.value("mainwindow/geometory").toByteArray())
        self.setVisible(
                not self.settings.value("mainwindow/hide").toBool())
        self.settings.endGroup()


    def saveGUISettings(self):
#        self.generateDefaultSettingFile()

        self.settings.beginGroup("GUI")
        self.settings.setValue("treeWidget/header",
                QtCore.QVariant(self.treeWidget.header().saveState()))
        self.settings.setValue("mainwindow/geometory",
                QtCore.QVariant(self.saveGeometry()))
        self.settings.setValue("mainwindow/hide",
                QtCore.QVariant(not self.isVisible()))
        self.settings.endGroup()
        self.settings.sync()

# This function is used to (re)generate "TwitturnDefaultSettings.ini" file.
    def generateDefaultSettingFile(self):
        defaultSettings = QtCore.QSettings(
                "TwitturnDefaultSettings.ini",
                QtCore.QSettings.IniFormat)
        defaultSettings.beginGroup("GUI")
        defaultSettings.setValue("treeWidget/header",
                QtCore.QVariant(self.treeWidget.header().saveState()))
#        defaultSettings.setValue("mainwindow/geometory",
#                QtCore.QVariant(self.saveGeometry()))
#        defaultSettings.setValue("mainwindow/hide",
#                QtCore.QVariant(not self.isVisible()))
        defaultSettings.endGroup()
        defaultSettings.sync()
        defaultSettings.setValue("GUI/treeWidget/header",
                QtCore.QVariant(self.treeWidget.header().saveState()))
        print defaultSettings.fileName()

    def appendTab(self, 
            widget, tabName, filter=None, tabIdx=-1):
        self.tabWidget.insertTab(
                tabIdx,
                widget,
                QtCore.QString(tabName))
        if not filter==None:
            self.connect(self, QtCore.SIGNAL("newPostCame"),
                    filter)
        self.connect(
                widget, 
                QtCore.SIGNAL('currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)'),
                self.setCurrentPost)
        self.connect(
                widget.treeWidget,
                QtCore.SIGNAL('itemDoubleClicked(QTreeWidgetItem *,int)'),
                self.putReplyHeader)
        return

    def selectTopPost(self):
        self.treeWidget.setCurrentItem(
            self.treeWidget.topLevelItem(0))
        self.treeWidget.setFocus()
    def focusTimeLine(self):
        self.treeWidget.setFocus()

    def setCurrentPost(self, current, previous):
        self.textBrowser.setHtml( 
                self.convertNickTextToLink( 
                    self.convetUriToLink( 
                        current.text(2))))
        self.pushButtonWeb.setEnabled(not current.text(6) == "")
        self.label.setPixmap(current.icon(0).pixmap(48,48))
        self.labelNick.setText(
                current.text(1) + '/' + current.text(5))
        self.toolButtonTime.setText(current.text(3))

    def convetUriToLink(self, str):
        if not str.contains(self.rxURL):
            return str
        retStr = QtCore.QString()
        head = self.rxURL.indexIn(str, 0)
        retStr += str.mid(0, head)
        while True:
            if head == -1:
                break;
            url = self.rxURL.cap()
            retStr += '<a href=\"' + url + '\">'+url+'</a>'
            tail = head + self.rxURL.matchedLength()

            head = self.rxURL.indexIn(str, tail)
            len = head - tail
            if len < -1:
                retStr += str.mid(tail, -1)
            else:
                retStr += str.mid(tail, len)
        return retStr

    def convertNickTextToLink(self, str):
        if not str.contains(self.rxReply):
            return str
        retStr = QtCore.QString()
        list = self.splitString(str)
        for s in list:
            if s.startsWith('@'):
                tmp = s.mid(1)
                s = '<a href=\"http://twitter.com/'+tmp+'\">@'+tmp+'</a>'
            retStr += s
        return retStr

    def splitString(self, str):
        list = []
        head = self.rxReply.indexIn(str, 0)
        list.append(str.mid(0, head))
        while True:
            if head == -1:
                break;
            list.append(self.rxReply.cap())
            tail = head + self.rxReply.matchedLength()

            head = self.rxReply.indexIn(str, tail)
            len = head - tail
            if len < -1:
                list.append(str.mid(tail, -1))
            else:
                list.append(str.mid(tail, len))
        return list

    def pickupNicks(self, str):
        list = []
        pos = 0
        while True:
            pos = self.rxReply.indexIn(str, pos)
            if pos == -1:
                break;
            list.append(self.rxReply.cap(1))
            pos += self.rxReply.matchedLength()
        return list


def replyTabFilter(widget, post, 
        userNick, systemTrayIconMessageDictionary):
    if post.text(2).contains(userNick) and post.text(7) == "stat":
        if widget.addPost(post, True):
            widget.setNotify(True)
            systemTrayIconMessageDictionary[post.text(4)] = (
                    post.text(1) + ": " + post.text(2))

def directMessageTabFilter(widget, post,
        systemTrayIconMessageDictionary):
    if post.text(7) == "DM":
        if widget.addPost(post, True):
            widget.setNotify(True)
            systemTrayIconMessageDictionary[post.text(4)] = (
                    post.text(1) + ": " + post.text(2))

if __name__=="__main__":
    app = QtGui.QApplication(sys.argv)
    form = Twitturn()
    systemTrayIcon = QtGui.QSystemTrayIcon(
            QtGui.QIcon("images/twitturn-icon.png"),
            form)
    repliesTimeLineWidget = TwitturnTimeLineWidget(form.tabWidget)
    form.appendTab(
            repliesTimeLineWidget,
            form.tr("&Replies"), 
            lambda x: replyTabFilter(
                repliesTimeLineWidget,
                x,
                form.userNick, form.systemTrayIconMessageDictionary))

    dmTimeLineWidget = TwitturnTimeLineWidget(form.tabWidget)
    form.appendTab(
            dmTimeLineWidget,
            form.tr("&DM box"),
            lambda x:directMessageTabFilter(
                dmTimeLineWidget,
                x,
                form.systemTrayIconMessageDictionary))
    plugins = glob.glob(os.path.join("plugins",'*.py'))
    for plugin in plugins:
        print plugin
        execfile(plugin)
    searchTimeLineWidget = TwitturnTimeLineWidget(form.tabWidget)
    form.appendTab(
            searchTimeLineWidget,
            form.tr("&Search"))
    form.searchTab = searchTimeLineWidget

    form.shortcutkeyManager.loadKeyConfigFile('default.key')

    app.exec_()
