/* Copyright (C) 2022 Momi-g

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

/*SH_doc
title=twl section=3 repnl=\040
@name twl
@_brief toml-like data format and parser
@_syno
 typedef struct twl_tag {
 	int rc;
 	char* emsg;
 	double cnt;
 	//..opaque data
 } twl_t;
 
 typedef struct twlval_tag {
 	const char* k;	//key
 	const char* t;	//3 types, "n"um/"s"tr/"d"ate
 	double n;		//num val
 	const char* s;	//str val
 	const char* date;	//date tostr: rfc3339 format
 	double y,m,d,hh,mm,ss,z,zh,zm; //date: z==zone 0/1
 } twl_vt;
 
 //--read twl format file/str
 twl_t* twl_new();
 int twl_pushfile(twl_t* obj, const char* flname);
 int twl_pushstr(twl_t* obj, const char* s [, int ssz] );
 twl_t* twl_clear(twl_t* obj);
 void* twl_free(twl_t* obj);
 int twl_dbgout(twl_t* obj, FILE* dbgfp);

 //--getter
 twl_vt twl_get(twl_t* obj, const char* id);
 twl_foreach(const char* key, twl_vt val, twl_t* obj){..}	//macro
@tl_dr
		@(code)--@
 #include "twl.h"

 int main(int argc, char** argv){
     twl_t* obj = twl_new();
     twl_pushstr(obj, "[hw] \n twl = 10.1");
     twl_pushstr(obj, NULL);	//EOF
     
     twl_vt val = twl_get(obj, "hw.twl");
     printf("%s, %f\n", val.k, val.n);
     twl_free(obj);
     return 0;
 }
 //~$ cc hw.c libtwl.a -Wall -pedantic -std=c99 -D_POSIX_C_SOURCE=200112
 //see EXSAMPLE for more detail code
	@()--
 
@params
	@(list)
	flname: (char*) read file if set. "stdin" reads FILE* stdin
	obj: (twl_t*) core obj. holds parser, hash map etc
	s: (char*) send str/bin to twl push-parser. send EOF if set NULL
	ssz: (int) send str/bin size if set. use strlen(s) if noset/-1
	id: (char*) keyname for search
	key:(char*) twl_foreach() ID for liner search
	val:(twl_vt) twl_foreach() ID holding data
	dbgfp: (FILE*) fp for output dbginfo. noout if set NULL(==dfl)
	@()
@return
	@(list)
	obj: twl_new() rtns core. rtn NULL if failed. obj->cnt holds keycnt.
	rc : twl_pushfile() rtns int 0/rc<0 --
		twl_pushstr() rtns int 0/1/rc<0 == complete/morestr/err --
		err == ERR_TKN / ERR_STX / ERR_INN and emsg is 'obj->emsg'.
		'obj->rc' holds the same int.
	val: all data is double/char*. all data is init with 0/"". --
		never fails *val.t =='n', puts(val.date) etc  --
		--
		val.k @: char*, ID/keyname, "aa.bb.c" etc --
		val.t @: char*, valtype. "n/s/d" or "", num/str/date --
		--
		val.n @: dbl, val.t=="n"  data.  0.11,  12.00 etc --
		val.s @: char*, val.t=="s" data. "hw" etc --
		val.date @: char*, val.t=="d" tostr data. follows rfc3339 --
		--
		val.ssz@: dbl, val.s size. val.s can holds binary, \0 etc. --
		val.y/m/d/hh/mm/ss/z/zh/zm@: dbl, 'z'one is 0/1 == noset/set --
	@()

@error
	twl_pushfile/str() may rtns rc<0.
	@(code)--@
	ERR_TKN: lex err
	ERR_STX: yacc err
	ERR_INN: other inner err
	@()--@
	'obj->emsg' may holding detail msgstr. --
	exit() if fatal err.

@_desc
	twl is alt-toml data format. omit unnecessary syntax from toml.
	the differences is the below. see NOTES for details:
		@(code)--@
	- no nest with rval: deep nest causes troubles.
	- lowercase ID only: ok: ab.xy=100  NG: Ab.xy=100
	- no quote lval    : ID is no quoting alnum+[._] as c-lang
	- no table array   : silver bullet for javascript specific
	- add mulitline cmt: #*..*# etc
	- add parallel set : a,b = 10,20 >> a=10; b=10
		@()--@
	twl format target is:
		@(code)--@
	- non-programmer can use with low learning cost
	- a few syntax rule
	- pay cost if you need to send complex data (or fix data structure)
	- respect to: ini, shell, c-lang, utf8
	- avoid to  : XML, XSLT, python, c++, lisp, markdown
		@()--@
	
	see samples for twl data read api fundtions. input data is saved to hashtb.
	all value is 'const char* / double' except rc, obj->rc (int).
	char* data is init with blank str "", so never fails to put(val.s) etc.
@_eg
	@(code)--@
 //--data.twl
 # cmtline
 [ab]
 a=10	#ab.a=10
 [aa.bb]
 b="hw"	#aa.bb.b="hw"

 //--src.c
 #include "twl.h"
 
 int main(int argc, char** argv){
 	int rc=0;
 	twl_t* obj;
 	twl_vt val;

 	// read from file
 	if(0){
 	obj = twl_new();
 	; rc = twl_dbgout(obj, stderr);  //dispinfo. rc<0 >> badfp+nochange
 	rc = twl_pushfile(obj, "data.twl");	//suc/fail == 0/rc<0
 	; rc = twl_dbgout(obj, NULL);	// noout(==dfl)
 	val = twl_get(obj, "aa.bb.b");
 	; if( *val.t == 0 ){ puts("bad key"); return 1; }
 	printf("%s, %s, %f\n", val.k, val.s, obj->cnt); //key,val,cntkey
 	twl_free(obj);
 	}

 	//read using push-style parsing
 	obj = twl_new();
 	const char* s = "[ab]\n a = 10 ; d = 2000-01-01";
 	rc = twl_pushstr(obj, s);	//rc=1/0 (more/complete) err<0
 	// (obj,s, -1) >> use -1 == strlen(s) if ag3 isnt
 	; if(rc<0){ puts(obj->emsg); return rc; }
 	rc = twl_pushstr(obj, NULL);	//NULL >> send EOF
 
 	val = twl_get(obj, "ab.a");
 	printf("%s, %f\n", val.t, val.n);	// "n", 10.0
 	
 	twl_foreach(k, v, obj){
 		puts(k);	//== v.k, char* key/id, "ab.a" etc
 		if( strcmp(v.t, "n")==0 ){ printf("num: %f\n", v.n); }
 		if( *v.t == 's' ){ printf("str : %s\n", v.s); }
 		if( *v.t == 'd' ){ printf("date: %s\n", v.date); }
 	}
 	obj = twl_clear(obj); //read other file/str if needs
 	printf("%f\n", obj->cnt);	//==0, obj->cnt == holding keycnt
 	
 	twl_free(obj);
 	return 0;
 }
 //$ cc src.c libtwl.a -Wall -pedantic -std=c99 -D_POSIX_C_SOURCE=200112
		@()--

@notes
		@(code)--@
	---bench mark: 
	code: a[n]=num / twl_pushstr(obj, "key=10 ") etc
	
	set : 8.3ms <<<< 8842ms
	FAST: a[n] (1) <<<< twl_set (1000) :SLOW
	
	get : 14.4ms << 216.8ms 
	FAST: a[n] (1)  <<  twl_get (15)   :SLOW

	-O0
	real	332.267 ms  : ./twl.tmp.c 4674: msg:sprintf() ovh 1000*1000
	real	341.507 ms  : ./twl.tmp.c 4686: msg:a[n] set 1000*1000
	real	14.469 ms   : ./twl.tmp.c 4693: msg:a[n] get 1000*1000
	real	14608.907 ms: ./twl.tmp.c 4702: msg:twl_set 1000*1000
	real	449.135 ms  : ./twl.tmp.c 4712: msg:twl_get 1000*1000
	-O3
	real	356.416 ms  : ./twl.tmp.c 4674: msg:sprintf() ovh 1000*1000
	real	341.656 ms  : ./twl.tmp.c 4686: msg:a[n] set 1000*1000
	real	0.001 ms    : ./twl.tmp.c 4693: msg:a[n] get 1000*1000
	real	8842.322 ms : ./twl.tmp.c 4702: msg:twl_set 1000*1000
	real	216.870 ms  : ./twl.tmp.c 4712: msg:twl_get 1000*1000
	--- 
		@()--@
	@(code)--@
	--- twl syntax info
# twl syntax. linecmt is '#' or '//', allow not BOS
#* multiline cmt is #*...* # or / *..* /, toml >> #...(nl) only
 - preprocess \
   \+(nl) marges lines then tokenize. same as c-lang
 - newline: NL is '\n' or '\r\n'
 - utf8   : file format allows only utf8 and (nl)(EOF)
 - ab.x_y : ID(left hand val) is alnum + dot.ul_
 - Ab.xy  : NG. ID allows only lowercase
 - valtype: VAL(right hand val) is 3types: floating num, str, date
 - [ab.xy]: LABEL [..] add prefix name to ID
 
 - a=1;a=2: (twl_ext) 2nd assign overwrites the before:  >> a==2
 - sep ;  : (twl_ext) semicolon ';' is ignored (or works as token sep)
 - a,b=1,2: (twl_ext) parallel assign: >> a=1 b=2
 - 1_0   : (twl_ext) NUM ignores '_' except token head: >> a=10
 - #_twl: : (twl_ext) magic cmt token '#_twl:' is reserved
*#

#_twl:v1.0.0 //file syntax is for v1.0.0 (currently magic is only this)
//_twl:v1.0.0 //this is not magic

[num]	#cmt
n0 = 10     //num.a0 = 10.0 (floating num only)
n0 = 10.1   //num.a0 = 10.1 (twl:overwrite toml:error)
n1 = -_.11  //-0.11: token >> del '_' >> conv with strtod() 
n2 =        //allow noset == ignored
n3=1 n4=2   //valid. free format 
n5=1;n6=2   //';' == '\n' in most

[str_bin]
s0 = "\u0000h\0w" //s0 size == 4: treat as utf8 sequences 
s1 = "\100s\ntr"  //"" conv c-lang esc except \xhh. lit can holds raw nl
s2 = '@s
tr'	  //'' holds rawlit. '' cant holds single-quote (same as shell)

#* hex esc \xhh.. is non-portable (c99: 6.4.4.4 Character constants)
 - \ooo is 1-3digits(1byte 0-255) but nolim with \xhh..(1-100digits etc)
 - \xhh.. val depends on endian:  0x0a11 / 0x110a etc
 - use \ooo for binary and \u,\U for i18n charactors
*#

//heredoc
s3 = '''alnum_end	//''' or """, 3 or more quote. '''''' is valid.
stop hdoc with (nl)headID(nl). ID chars is alnum+[._], same as lval.
quotes follows ending ID is valid. (nl)headID'"''"(nl) works as end.
this is help you to adjust syntax highlight in manually.

start with s-quote takes completely rawstr.
set blankline if you want EOS nl "str\n".
 
alnum_end"''

//hdoc with c-lang esc
s4 = """end
	w-quote allows esc\n \u0060 etc. \n
end

s5 = """-end
	"""- or '''- removes leading raw tabs '\t', same as posix-shell '<<-'
\t\t123 ...esc char "\t,\u0009" isnt stripped
end"""

_='''___
	use assign syntax as cmt like shell, :<<'CMT'...CMT
___'''


[date.time.etc]	//date format: follows rfc3339
d.a0 = 0020-01-01 12:00:12.345        //year: more than 4digits
d.a1 = 2000-01-01T12:00:23.456+00:00  //date.time.etc.d.a1 = ...

d.a2 = 2000-01-01    //(twl_ext) date only >> others are init with 0
d.a3 = 12:00:22.3345 //(twl_ext) time only
d.a4 = -20000-01-01T00:00:00.000Z //(twl_ext) sign and over 4digits year

#*(twl_ext)
 lowercase sep 't/z' is invalid. T/(sp)/Z only.
 y,m,d,hh,mm,ss,z,zh,zm holds double(64bit floating point) num
*#

[]	//blank LABEL set no prefix
base.num=10	//(no_prefix)base.num == 10

//(twl_ext) parallel assign (ID list)=(VAL list). ignore uneven ID/VAL
  a,    b,       c,     d
=10.9, "hw", 2000-01-01		//'d' is ignored

a = 10.9, "hw", 2000-01-01	//str, date is ignored

# ---
#*---appendix: not collect BNF, but helps you
//yacc
stmt: expr
    | stmt expr ;

expr: LB
    | lve
    | ass ;

ass : lve rv
    | ass ',' rv ;	//VAL list

lve  : lv '=' ;

lv   : ID
     | lv ',' ID ;	//ID list

rv: num | lit | date ;

num  : NUM ;
lit  : DLIT | SLIT | DDOC | SDOC ;
date : DATE_
     | DATE_ DATET  
     | DATE_ DATET DATEZ  
     | TIME ;	
	
//lex:  marge \(nl) lines with preprocess
NL   : \n|\r\n
LABEL: '[' ID ']'
ID   : ([a-z_][a-z0-9_]*)(\.[a-z_][a-z0-9_]*)*
NUM  : [-+.0-9][_xXoO0-9a-fA-F.lL]*   //del '_' and suc to conv strtod()

DLIT : ["]([^"]|\\.)*["}		//conv esc
SLIT : [']([^'])*[']
DDOC : """+\(ID\).* NL \1 ["']* NL     //BRE+shortest. conv esc
SDOC : '''+\(ID\).* NL \1 ["']* NL

DATE_: [-+]?[0-9][0-9][0-9][0-9]+-[0-1][0-9]-[0-3][0-9]
DATET: [T ][0-2][0-9]:[0-6][0-9]:[0-6][0-9](\.[0-9]+)?
DATEZ: [Z]|[\-+][0-2][0-9]\:[0-6][0-9]
TIME: [0-2][0-9]:[0-6][0-9]:[0-6][0-9](\.[0-9]+)?

 1979-05-27T07:32:00Z
 1979-05-27T00:32:00-07:00
 1979-05-27T00:32:00.999999-07:00
 1979-05-27 07:32:00.1234+09:00

 2000-05-27 07:32:00 
 2000-01-01
 12:00:00
 12:00:00.12345

*#
	@()--@
@conforming_to POSIX.1-2001+ (-D_POSIX_C_SOURCE=200112L etc)
@COPYRIGHT Copyright 2022 momi-g, GPLv3+
@_ver 2022-07-12 v1.0.2 (2022-06-13 v1.0.0)
@_see https://en.wikipedia.org/wiki/Configuration_file
//SH_docE*/

/*SH_CBRT*/
#ifndef ERRact
#include <stdio.h>
 #if (199901L <= __STDC_VERSION__ +0)	/* nealy 200112L, _POSIX_C_SOURCE	c99*/
	#include <sys/types.h>
	#include <unistd.h>
	#define ERRactag	__func__, getpid()
 #else
	#define ERRactag	"func:c99+", 0
 #endif
 #include <string.h>
 #include <errno.h>
 #define ERRact(xpr, msg, act)	if(xpr){ int en_=errno; fprintf(stderr, \
	"ERR: %s %d %s() pid:%d %s msg:%s sys:%s\n",__FILE__,__LINE__, ERRactag \
	, "hit(" #xpr ")", msg, strerror(en_) ); act; }
 #define STOP(xpr, msg)	ERRact(xpr, msg, fputs("STOP\n",stderr);exit(1) )
#endif
#define loop(a)		for(int lpcnt=1;lpcnt<=a;lpcnt++)

#include <stdio.h>
#include <stdlib.h>
#include "twl.h"
# include "stredit.h"	/*SH_co*	-D_IMPL_stredit	 */
# include "clit.h"		/*SH_co*	-D_IMPL_clit	 */
# include "hfl.h"		/*SH_co*	-D_IMPL_hfl		 */
# include "msgp.h"		/*SH_co*	-D_IMPL_msgp	 */



/*pre-code---*/
//typedef struct pos_tag { int l, c; } pos_t;

typedef double dbl;
typedef struct td_tag {
	int l;
	int c;
	//
	se_t* se;
	int idx;
	int ssz;
	int tp;
	//
	dbl y;
	dbl m;
	dbl d;
	dbl hh;
	dbl mm;
	dbl ss;
	dbl z;
	dbl zh;
	dbl zm;
} td_t;
//token data

// major, minor, revision
typedef struct ver_tag{
	int rc;
	int m;
	int n;
	int r;
} ver_t;

static int ck_magic(twl_t* obj, const char* cmt);
int twl_dbgout(twl_t* obj, FILE* fp){
	int rc=0;
	MSGP_t yo = pfset(PF_NOCHANGE, PF_NOCHANGE, PF_NOCHANGE, fp);
	yo.rc ? rc = -1: 0;
	return rc;
}

// sync to %prefix YY_: YP(a) >> YY_a,  YP(_,a) >> _YY_a
#define PFIX YY_
#define YP(...) YP1(PFIX, __VA_ARGS__, __VA_ARGS__, , )
#define YP1(z,a,b,c, ...) YP2(c,z,b)
#define YP2(a,m,b) a ## m ## b

/*lex---*/

static int err_l2g(int ec);
static ver_t parse_ver(const char* cmt);
static int try_ver(const char* cmt);
static int seterr(twl_t* obj, const char* emsg, int errcode);

#include <strings.h>
static 
ver_t parse_ver(const char* cmt){
sbg(cmt);
	const char* s = cmt;
	ver_t res = {0};
	; if(0){ lb_RTN:;
idbg(res.rc, "verck end, suc:res==1");
		return res;
	}
	//
	int sz = 0;
	strncasecmp(s, "v", 1)==0 ? sz=1 :0;
	strncasecmp(s, "ver", 3)==0 ? sz=3 :0;
	if(sz==0){	goto lb_RTN; }
	s+=sz;
sbg(s, "num pos");
	for(int i=0;i<3;i++){
		char* rp = NULL;
		int n = strtol(s, &rp, 10);
idbg(i, n, s, rp, "idx, convnum, src, ep");
		if(s==rp){ goto lb_RTN; }
		i==0 ? res.m = n :0;
		i==1 ? res.n = n :0;
		i==2 ? res.r = n :0;
		//
		if(i==0){res.rc = -1;}	//set syx err
//		if(i==2 && *rp!=0){ goto lb_RTN;}
		if(i<2  && *rp!='.'){ goto lb_RTN; }
		s=rp;
		s++;
	}
	res.rc = 1;
	goto lb_RTN;
	//unreachable code
	return res;
}

static
int try_ver(const char* cmt){
sbg(cmt, TWL_VER, ":ver ck, req / TWL_VER");
	//rc 0/1 <0 == not ver/suc/err
	ver_t cver = parse_ver(TWL_VER);
	ver_t fver = parse_ver(cmt);
idbg(fver.rc, fver.m, fver.n, fver.r, cver.m, cver.n, cver.r, "cmp");
	if(fver.rc==0){ return 0; }
	if(fver.rc<0){ return ERR_TKN; }
	// cmp ver
	if(cver.m > fver.m){ return 1; }
	if(cver.m < fver.m){ return ERR_STX; }
	//
	if(cver.n > fver.n){ return 1; }
	if(cver.n < fver.n){ return ERR_STX; }
	//
	if(cver.r >=fver.r){ return 1; }
	if(cver.r < fver.r){ return ERR_STX; }
	//unreachable
	; STOP(1, "fatal err");
	return 0;
}

static
int ck_magic(twl_t* obj, const char* cmt){
	const char* s = cmt;
	int rc =0;
	const char* emsg = NULL;
	; if(0){ lb_RTN:;
idbg(rc, "magic rtn, rc==0 == suc");
		if(rc==0){return rc;}
		seterr(obj, emsg, rc);
		return err_l2g(rc);
	}
	//
	rc = strncmp(s, TWL_MGCMT, strlen(TWL_MGCMT) );
idbg(rc, TWL_MGCMT);
sbg(TWL_MGCMT);
	if(rc){ rc=0; goto lb_RTN; }	//normal cmt
sbg("hit magic");
	//hit #_twl:...
	int cnt=0;
	s += strlen(TWL_MGCMT);
	
	//--test magic list: rc==0 >> delective nohit
	rc = try_ver(s);
	if(rc>0){ cnt++; }
idbg(rc, "magic rc, err == rc<0");
	if(rc==ERR_TKN){
		emsg = "magic cmt token is invalid, 'v1.0.0' etc";
		goto lb_RTN;
	}
	if(rc==ERR_STX){
		emsg = "twl reader " TWL_VER " is oloder than req ver";
		goto lb_RTN;
	}

/*	//
	rc = try_pragma(s);
	if(rc>0){ cnt++; }
	if(rc<0){ goto lb_RTN; }
	//
	rc = try_option(s);
	if(rc>0){ cnt++; }
	if(rc<0){ goto lb_RTN; }
*/	
	//if any magic isnt work
idbg(cnt, "cnt==0 magic but not support");
	if(cnt==0){ rc = ERR_TKN; }
	rc=0;
	goto lb_RTN;
	//unreachable
	; STOP(1, "fatal err");
	return 0;	
}

static
char* twl_makekey(twl_t* obj, const char* k, const char* fix){
	se_t* se = obj->setmp;
	se_clear(se);
//dbg(obj->lb, k, fix, se->sz);
	if(obj->lb[0]){ se_insf(se, -1, "%s=%s.%s%c", fix, obj->lb, k, 0); }
	else{ se_insf(se, -1, "%s=%s%c", fix, k, 0); }
	char* rp = se_ptr(se);
//dbg(rp, se->sz);
	return rp;
}


static 
int twl_as(twl_t* obj, const char* bkey, td_t rval_){
	td_t* rval = &rval_;
	char* hkey = NULL;
idbg( ((se_t*)(obj->seyc))->sz, rval->se, rval->idx, bkey);
sbg( bkey);
//conv style
	char* vp = se_ptr(rval->se) + rval->idx;
	int vpsz = rval->ssz;
	int tp = rval->tp;
	hkey = twl_makekey(obj, bkey, "t");
idbg("asin_vals:", bkey, hkey, vp, vpsz, tp);
sbg(bkey, hkey, vp);
	int cc = obj->hdb->cnt;	//current cnt
	while(tp == 'n'){
		hfl_nt* nd = hfl_set_s(obj->hdb, hkey, "n");	//hfl_nt* nd = hfl_set_s(obj->hdb, key, "n");
		; STOP(nd==NULL, "fatal memerr");
		int nc = obj->hdb->cnt;	//next cnt
		cc!=nc ? obj->cnt++ :0;
	
		//k,v == t=a.b.d , "n"		
		//--val: skip 1_0 >> 10
//pf1("%s, %s, %d", bkey, nd->s, nd->ssz);sleep(1);
		
		double d;
		char* p = calloc(vpsz+1, 1);
		int n = 0;
		for(int i=0;vp[i];i++){
			if(vp[i]=='_'){continue;}
			p[n] = vp[i];
			n++;
		}
		char* rp=NULL;
		errno=0;
		d = strtod(p, &rp);	//endptr
		int flg = errno||*rp;
		free(p);
		if(flg){
			char sbuf[128] = {0};
			snprintf(sbuf, 127, "strtod() failed: %s = %s", bkey, vp);
			free(obj->emsg);
			seterr(obj, sbuf, ERR_TKN);
			return -1;
		}
		//--ckend
		
//dbg(1, vp);
		hkey = twl_makekey(obj, bkey, "n");
//dbg(2, vp);
idbg( d, "valkey");
sbg(hkey, vp, "k,v");
//		double d = strtod(vp, &rp);	//endptr
		hfl_set_d(obj->hdb, hkey, d);
		break;
	}
	while(tp == 's'){
		hfl_set_s(obj->hdb, hkey, "s");
		int nc = obj->hdb->cnt;	//next cnt
		cc!=nc ? obj->cnt++ :0;

		hkey = twl_makekey(obj, bkey, "s");
		//make val from arg s, sz
		int dlit = 0;
		if(vp[0] == '"'){ dlit=1; }
		*(vp+vpsz-1)=0;
		vp++;
		vpsz-=2;	// "abc" >> abc|\0  5>>4 -1 -1
idbg(hkey, vp, vpsz, "valkey");
sbg(hkey, vp, ":del qu");
		//esc
		if(dlit){ vp = clitv(vp, &vpsz); }
		//set string to hfl
		hfl_set_s(obj->hdb, hkey, vp, vpsz+1);	//binもありえるので安全敵に\0追加
		if(dlit){ free(vp); }
		// maybe needs strsz
		hkey = twl_makekey(obj, bkey, "z");	//sz
		hfl_set_d(obj->hdb, hkey, (double)vpsz);
idbg(vpsz, "val strsz");
sbg(hkey);
		break;
	}
	while(tp == 'd'){
		hfl_set_s(obj->hdb, hkey, "d");
		int nc = obj->hdb->cnt;	//next cnt
		cc!=nc ? obj->cnt++ :0;

		double dd;
		double y  = dd=rval->y;  hkey=twl_makekey(obj,bkey,"dy");	hfl_set_d(obj->hdb,hkey,dd);
		double m  = dd=rval->m;  hkey=twl_makekey(obj,bkey,"dm");	hfl_set_d(obj->hdb,hkey,dd);
		double d  = dd=rval->d;  hkey=twl_makekey(obj,bkey,"dd");	hfl_set_d(obj->hdb,hkey,dd);
		double hh = dd=rval->hh; hkey=twl_makekey(obj,bkey,"dhh"); 	hfl_set_d(obj->hdb,hkey,dd);
		double mm = dd=rval->mm; hkey=twl_makekey(obj,bkey,"dmm"); 	hfl_set_d(obj->hdb,hkey,dd);
		double ss = dd=rval->ss; hkey=twl_makekey(obj,bkey,"dss"); 	hfl_set_d(obj->hdb,hkey,dd);
		double z  = dd=rval->z;  hkey=twl_makekey(obj,bkey,"dz");	hfl_set_d(obj->hdb,hkey,dd);
		double zh= dd=rval->zh;  hkey=twl_makekey(obj,bkey,"dzh");	hfl_set_d(obj->hdb,hkey,dd);
		double zm= dd=rval->zm;  hkey=twl_makekey(obj,bkey,"dzm");	hfl_set_d(obj->hdb,hkey,dd);
		//tostr
		hkey = twl_makekey(obj,bkey,"d");
		//
		z = zh<0 ? '-' : '+';
		char* sn = "";
		y<0 ? sn="-", y*=-1 :0;
		char sbuf[128] = {0};
		sprintf(sbuf,
			"%s%04.f-%02.f-%02.fT%02.f:%02.f:%02d.%03d%c%02.f:%02.f%c"
			 ,sn, y ,m ,d ,hh, mm ,(int)ss, (int)(ss*1000)%1000, (int)z, zh,zm, 0
		);
		hfl_set_s(obj->hdb, hkey, sbuf);
sbg(hkey, sbuf, "date tostr");
//dbg(se_ptr(yo) + idx);
		break;
	}
	return 0;
}

int twl_ycc(twl_t* obj, td_t einfo){
idbg(obj->svcnt, obj->ldcnt, "hit ycc");
// del used token
	se_t* se = einfo.se;
// 	se_t* se = obj->seyc;
	int tdx = einfo.idx;
//	int tdx = obj->tdx;
idbg(tdx, se->p, se->p + tdx, "del until nexthead");
sbg(se->p, se->p + tdx);
//	int tdxsz = einfo.ssz;
//	int tdxsz = obj->tdxsz;

	se_del(se, 0, tdx);	//add \0 to all str
idbg(se->p, se->sz, "after del preycc, nexthead");
sbg(se->p);
//*/
//remove hfl mlv data
	char sbuf[128]={0};
	if(obj->ldcnt!=0){
		for(int i=0; i<obj->svcnt; i++){
			sprintf(sbuf, "%d", i);
			hfl_unset(obj->hdb, sbuf);
		}
		obj->ldcnt = 0;
		obj->svcnt = 0;
	}
	obj->tdx = 0;
	obj->tdxsz = 0;
	return 0;
}

static int twl_mlvsv(twl_t* obj, td_t td){
	//clear saved data
	char sbuf[128]={0};
	sprintf(sbuf, "%d", obj->svcnt);
	char* k = se_ptr(td.se) + td.idx;
idbg(k, obj->svcnt, obj->ldcnt, "hit lval ID");
sbg(k);
	hfl_set_s(obj->hdb, sbuf, k);

	obj->svcnt++;
	return 0;
}

static char* twl_mlvld(twl_t* obj){
	char sbuf[128] = {0};
	sprintf(sbuf, "%d", obj->ldcnt);
	hfl_nt* nd = hfl_find(obj->hdb, sbuf);
idbg((nd?nd->s: 0), obj->svcnt, obj->ldcnt, "hit multi VAL ass");
	if(nd==NULL){ return NULL; }
	obj->ldcnt++;
	return nd->s;
}

/* --------- START OF GENERATED CODE ------------ */
#include <stdlib.h> /* realloc(), free(), NULL, size_t */
#include <string.h> /* memcpy() */
#include <stddef.h> /* size_t */
#include <stdint.h> /* SIZE_MAX */
struct YY_sym_data {
  int state_;
  td_t common_;
  union {
    char * uv1_;
  } v_;
};
static const size_t YY_scan_table_rex[] = {
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  6,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  8,  9,  0,  0,  0, 10,  0,  0,  0, 11, 12, 13, 14, 15, 16, 16, 16, 17, 17, 17, 17, 17, 17, 17,  0,  5,  0, 18,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 19, 20,  0,  0,  0, 21,  0, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,
   22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 23, 22, 22, 24, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22,  2,  2,  2,  2,
   25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 26, 25, 25, 27, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25,  3,  3,  3,  3,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 28,  0,  0, 29,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  4,  4,  4,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  6,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  5,  5,  5,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  6,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  6,  6,  6,  6,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  6,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  7,  7,  7,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 33, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30,  8,  8,  8,  8,
   35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 36, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 37, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35,  9,  9,  9,  9,
   38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 38, 38, 40, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 41, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 10, 10, 10, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 42, 42, 42,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 11, 11, 11, 11,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12, 12, 12, 12,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 13, 13, 13, 13,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 14, 14, 14, 14,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 46,  0,  0,  0,  0, 47,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 15, 15, 15, 15,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16, 16, 16, 16,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 17, 17, 17, 17,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 18, 18, 18, 18,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 19, 19, 19, 19,
    0,  0,  0,  0,  0,  0,  0,  0,  0, 50,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 50,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 51,  0, 52,  0, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 20, 20, 20, 20,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 53,  0, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 54,  0, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 21, 21, 21, 21,
   22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 23, 22, 22, 24, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 23, 23, 23, 23,
   22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 55, 22, 22, 24, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 24, 24, 24, 24,
   25, 25, 25, 25, 25, 25, 25, 25, 25, 25,  0, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 26, 26, 26, 26,
   25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 56, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 27, 27, 27, 27,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 28, 28, 28, 28,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 57,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 29, 29, 29, 29,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 58, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 58, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 31, 31, 31,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 59, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 58, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 32, 32, 32, 32,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 60,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 33, 33, 33, 33,
   61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 31, 61, 61, 62, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 63, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 64, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 34, 34, 34, 34,
   35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 36, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 36, 36, 36, 36,
   65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 65, 65, 67, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 68, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 37, 37, 37, 37,
   38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 38, 38, 40, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 69, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38,
   38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 38, 38, 40, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 69, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 39, 39, 39,
   38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 70, 38, 38, 40, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 69, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 40, 40, 40, 40,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 71,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 41, 41, 41, 41,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 42, 42, 42, 42,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 44, 44, 44, 44,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 45, 45, 45, 45,
   74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 74, 74, 75, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 76, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 46, 46, 46, 46,
   77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 78, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 47, 47, 47, 47,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 48, 48, 48, 48,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 49, 49, 49, 49,
    0,  0,  0,  0,  0,  0,  0,  0,  0, 50,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 50,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 51,  0, 52,  0, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 50, 50, 50, 50,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 51, 51, 51, 51,
    0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 82,  0, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 84,  0, 83,  0, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 52, 52, 52, 52,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 85,  0, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 53, 53, 53, 53,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 53,  0, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 54,  0, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 54, 54, 54, 54,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 55, 55, 55, 55,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 56, 56, 56, 56,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 57, 57, 57, 57,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 58, 58, 58, 58,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 58, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 59, 59, 59, 59,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 60,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 86,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 87,  0, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 60, 60, 60, 60,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 58, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 61, 61, 61, 61,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 59, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 58, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 62, 62, 62, 62,
   30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 30, 30, 32, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 58, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 34, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 63, 63, 63, 63,
   61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 31, 61, 61, 62, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 63, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 64, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 64, 64, 64, 64,
   65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 65, 65, 67, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 68, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65,
   88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 88, 88, 89, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 90, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 66, 66, 66, 66,
   65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 65, 65, 67, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 68, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 67, 67, 67, 67,
   91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 92, 91, 91, 93, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 94, 91, 91, 91, 91, 91, 91, 95, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 68, 68, 68, 68,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 69, 69, 69, 69,
   38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 38, 38, 40, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 69, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 70, 70, 70, 70,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 71,  0,  0,  0,  0,  0, 86,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 87,  0, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 71, 71, 71, 71,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 96,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 72, 72, 72, 72,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 96,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 73, 73, 73, 73,
   74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 74, 74, 75, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 76, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
   74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 74, 74, 75, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 76, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 75, 75, 75,
   97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 98, 97, 97, 98, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 99, 97, 97, 97, 97,100, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 76, 76, 76, 76,
   77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 78, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 78, 78, 78, 78,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,101,101,101,101,101,101,101,101,101,101,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 79, 79, 79, 79,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,102,102,102,102,102,102,102,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 80, 80, 80, 80,
    0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 84,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 81, 81, 81, 81,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,103,  0,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 82, 82, 82, 82,
    0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 82,  0, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 84,  0, 83,  0, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 83, 83, 83, 83,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 84, 84, 84, 84,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 53,  0,104,104,104,104,104,104,104,104,104,104,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,104,  0,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 85, 85, 85, 85,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 87,  0, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 86, 86, 86, 86,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,105,  0,106,106,106,106,106,106,106,106,106,106,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,106,  0,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 87, 87, 87, 87,
   88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 88, 88, 89, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 90, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88,
   88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 88, 88, 89, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 90, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 89, 89, 89,
  107,107,107,107,107,107,107,107,107,107,108,107,107,108,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,109,107,107,107,107,107,107,110,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107, 90, 90, 90, 90,
   65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 65, 65, 67, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 68, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 91, 91, 91, 91,
   88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 88, 88, 89, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 90, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 92, 92, 92, 92,
   65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 65, 65, 67, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 68, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 93, 93, 93, 93,
   35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 36, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 94, 94, 94, 94,
   91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 92, 91, 91, 93, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 35, 91, 91, 91, 91, 91, 91, 95, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 95, 95, 95, 95,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,111,111,111,111,111,111,111,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 96, 96, 96, 96,
   74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 74, 74, 75, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 76, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 97, 97, 97, 97,
   74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 74, 74, 75, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 76, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 98, 98, 98, 98,
   97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 98, 97, 97, 98, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 99, 97, 97, 97, 97,  0, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 99, 99, 99, 99,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,100,100,100,100,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,112, 43,  0,101,101,101,101,101,101,101,101,101,101,  0,  0,  0,  0,  0,  0,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0, 43,  0, 43, 43, 43, 43, 43, 43,  0,  0,  0,  0,  0, 43,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,101,101,101,101,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,113,113,113,113,113,113,113,113,113,113,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,102,102,102,102,
    0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 82,  0,114,114,114,114,114,114,114,114,114,114,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 84,  0,114,  0,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,103,103,103,103,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 53,  0,104,104,104,104,104,104,104,104,104,104,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,104,  0,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,104,104,104,104,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,115,  0,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,105,105,105,105,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,105,  0,106,106,106,106,106,106,106,106,106,106,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,106,  0,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,106,106,106,106,
   88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 88, 88, 89, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 90, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88,107,107,107,107,
   88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 88, 88, 89, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 90, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88,108,108,108,108,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,109,109,109,109,
  107,107,107,107,107,107,107,107,107,107,108,107,107,108,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,  0,107,107,107,107,107,107,110,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,110,110,110,110,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,116,116,116,116,116,116,116,116,116,116,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,111,111,111,111,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,117,117,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,112,112,112,112,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,118,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,113,113,113,113,
    0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 81,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 82,  0,114,114,114,114,114,114,114,114,114,114,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 84,  0,114,  0,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,114,114,114,114,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,105,  0,119,119,119,119,119,119,119,119,119,119,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,119,  0,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,115,115,115,115,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,116,116,116,116,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,120,120,120,120,120,120,120,120,120,120,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,117,117,117,117,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,121,121,121,121,121,121,121,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,118,118,118,118,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,105,  0,119,119,119,119,119,119,119,119,119,119,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,119,  0,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,119,119,119,119,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,122,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,120,120,120,120,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,123,123,123,123,123,123,123,123,123,123,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,121,121,121,121,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,124,124,124,124,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,122,122,122,122,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,125,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,123,123,123,123,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,126,126,126,126,126,126,126,126,126,126,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,124,124,124,124,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,127,127,127,127,127,127,127,127,127,127,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,125,125,125,125,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,126,126,126,126,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,127,127,127,127,127,127,127,127,127,127,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,127,127,127,127,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,129,129,129,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,128,128,128,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,130,130,130,130,130,130,130,130,130,130,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,129,129,129,129,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,131,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,130,130,130,130,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,132,132,132,132,132,132,132,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,131,131,131,131,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,133,133,133,133,133,133,133,133,133,133,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,132,132,132,132,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,134,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,133,133,133,133,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,135,135,135,135,135,135,135,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,134,134,134,134,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,136,136,136,136,136,136,136,136,136,136,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,135,135,135,135,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,137,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,136,136,136,136,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,138,138,138,138,138,138,138,138,138,138,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,137,137,137,137,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,138,138,138,138,138,138,138,138,138,138,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,138,138,138,138
};
static const size_t YY_scan_actions_rex[] = { 0, 0, 0, 10, 0, 5, 5, 5, 0, 0, 0, 0, 12, 20, 20, 0, 20, 20, 11, 17, 0, 15, 0, 8, 0, 10, 9, 10, 6, 0, 0, 0, 0, 21, 0, 0, 4, 0, 0, 0, 0, 22, 0, 20, 20, 20, 0, 0, 20, 20, 0, 14, 0, 0, 15, 8, 9, 6, 21, 0, 0, 0, 0, 21, 0, 0, 4, 0, 0, 22, 0, 0, 0, 20, 0, 0, 0, 0, 3, 20, 0, 0, 0, 0, 13, 15, 0, 7, 0, 0, 0, 0, 4, 0, 1, 0, 0, 0, 0, 0, 2, 20, 0, 0, 15, 0, 7, 0, 0, 1, 0, 0, 0, 0, 0, 7, 17, 0, 0, 7, 0, 0, 0, 19, 0, 0, 18, 19, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 16 };
static const int YY_minimum_sym = 3;
static const size_t YY_num_columns = 27;
static const size_t YY_num_rows = 28;
static const size_t YY_num_productions = 24;
static const int YY_parse_table[] = {
    0,  0,  7, 27, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,  0, 2, 3, 6,22,26, 0, 0, 0, 0, 0, 0, 0,
    0,  0, -3, -3, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, -3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,  0,  7, 27, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, -1, 0, 1, 6,22,26, 0, 0, 0, 0, 0, 0, 0,
    0,  0, -2, -2, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, -2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, -6, -6, -6, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, -6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,  0,  0,  0, 9,10,11,12,19,20,  0,21, 0, 0,  0, 0, 0, 0, 0, 0, 4,13,17,15,16,18, 0,
    0,  5, -4, -4, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, -4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, -5, -5, -5, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, -5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, -7, -7, -7, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, -7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-12,-12,-12, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-13,-13,-13, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-14,-14,-14, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-15,-15,-15, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-15, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-16,-16,-16, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-22,-22,-22, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-17,-17,-17, 0, 0, 0, 0, 0, 0, 14, 0, 0, 0,-17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-18,-18,-18, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-19,-19,-19, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-19, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-20,-20,-20, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-23,-23,-23, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-21,-21,-21, 0, 0, 0, 0, 0, 0,-21, 0, 0, 0,-21, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,-24,-24,-24, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,-24, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, -8, -8, -8, 9,10,11,12,19,20,  0,21, 0, 0, -8, 0, 0, 0, 0, 0, 8,13,17,15,16,18, 0,
    0, -9, -9, -9,-9,-9,-9,-9,-9,-9,  0,-9, 0, 0, -9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  -10,-10,  0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,  0,  0, 24, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   23, 25,  0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  -11,-11,  0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
};
static const size_t YY_production_lengths[] = {
  1,
  1,
  2,
  1,
  1,
  3,
  2,
  1,
  2,
  3,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  2,
  1,
  1
};
static const int YY_production_syms[] = {
  29,
  18,
  18,
  19,
  20,
  20,
  20,
  20,
  21,
  22,
  22,
  23,
  23,
  23,
  23,
  23,
  24,
  24,
  24,
  24,
  26,
  27,
  25,
  28
};

#ifndef CARB_YY_CBFT_H_INCLUDED
struct YY_stack {
  unsigned int error_recovery_:1;
  unsigned int pending_reset_:1;
  unsigned int discard_remaining_actions_:1;
  unsigned int slot_1_has_sym_data_:1;
  unsigned int slot_1_has_common_data_:1;
  unsigned int top_of_stack_has_sym_data_:1;
  unsigned int top_of_stack_has_common_data_:1;
  unsigned int need_sym_:1;
  unsigned int is_final_input_:1;
  unsigned int slot_0_has_current_sym_data_:1;
  unsigned int slot_0_has_common_data_:1;
  int current_sym_;
  size_t input_size_;
  const char *input_;
  int slot_1_sym_;
  int continue_at_;
  int mute_error_turns_;
  size_t pos_, num_stack_allocated_;
  struct YY_sym_data *stack_;
  struct YY_sym_data *sym_data_;
  size_t current_production_length_;
  int current_production_nonterminal_;
  size_t scan_state_;
  size_t current_mode_start_state_;
  size_t match_index_;
  size_t best_match_action_;
  size_t best_match_size_;
  size_t input_index_;
  size_t input_offset_;
  size_t match_buffer_size_;
  size_t match_buffer_size_allocated_;
  /* offset, line and column at the start of match_buffer_ */
  size_t match_offset_;
  int match_line_;
  int match_col_;
  /* offset, line and column at the zero-termination (best_match_size_) of a match 
  * in match_buffer_ (the actual buffer may well be longer.) */
  size_t best_match_offset_;
  int best_match_line_;
  int best_match_col_;
  size_t token_size_;
  char *match_buffer_;
  char terminator_repair_;
  int input_line_;
  int input_col_;
};
#define _YY_FINISH 0
#define _YY_MATCH 1
#define _YY_OVERFLOW 2
#define _YY_NO_MEMORY 3
#define _YY_FEED_ME 4
#define _YY_END_OF_INPUT 5
#define _YY_SYNTAX_ERROR 6
#define _YY_LEXICAL_ERROR 7
#define _YY_INTERNAL_ERROR 8

#define YY_AS 3
#define YY_CM 4
#define YY_LB 5
#define YY_ID 6
#define YY_NUM 7
#define YY_DLIT 8
#define YY_SLIT 9
#define YY_HDOC 10
#define YY_DATE 11
#define YY_DATET 12
#define YY_DATEZ 13
#define YY_TIME 14
#define YY_BAD_TOKEN 15
#define YY_ERROR 16
#define YY_INPUT_END 17

#define YY_STMT 18
#define YY_EXPR 19
#define YY_ASS 20
#define YY_ASS_ 21
#define YY_LV 22
#define YY_RV 23
#define YY_DVAL 24
#define YY_DATE_ 25
#define YY_DATET_ 26
#define YY_DATEZ_ 27
#define YY_TIME_ 28

#define M_YY_HDOC_S 2
#define M_YY_HDOC_P 3
#define M_YY_HDOC_E 4
#define M_YY_DEFAULT 1

#endif /* CARB_YY_CBFT_H_INCLUDED */

void YY_stack_init(struct YY_stack *stack) {
  stack->error_recovery_ = 0;
  stack->pending_reset_ = 1;
  stack->discard_remaining_actions_ = 0;
  stack->need_sym_ = 1;
  stack->current_sym_ = 0;
  stack->input_ = NULL;
  stack->input_size_ = 0;
  stack->is_final_input_ = 0;
  stack->slot_1_has_sym_data_ = stack->slot_1_has_common_data_ = 0;
  stack->slot_1_sym_ = 0;
  stack->top_of_stack_has_sym_data_ = stack->top_of_stack_has_common_data_ = 0;
  stack->continue_at_ = 0;
  stack->mute_error_turns_ = 0;
  stack->pos_ = 0;
  stack->num_stack_allocated_ = 0;
  stack->stack_ = NULL;
  stack->sym_data_ = NULL;
  stack->current_production_length_ = 0;
  stack->current_production_nonterminal_ = 0;
  stack->slot_0_has_current_sym_data_ = stack->slot_0_has_common_data_ = 0;
  stack->current_mode_start_state_ = M_YY_DEFAULT;
  stack->scan_state_ = stack->current_mode_start_state_;
  stack->input_index_ = 0;
  stack->input_offset_ = 0;
  stack->input_line_ = 1;
  stack->input_col_ = 1;
  stack->match_index_ = 0;
  stack->match_buffer_ = NULL;
  stack->match_buffer_size_ = 0;
  stack->match_buffer_size_allocated_ = 0;
  stack->terminator_repair_ = '\0';
  stack->token_size_ = 0;
  stack->match_offset_ = 0;
  stack->match_line_ = 1;
  stack->match_col_ = 1;
  stack->best_match_action_ = 0;
  stack->best_match_size_ = 0;
  stack->best_match_offset_ = 0;
  stack->best_match_line_ = 1;
  stack->best_match_col_ = 1;
}

void YY_stack_cleanup(struct YY_stack *stack) {
  size_t n;
  for (n = 2; n < stack->pos_; ++n) {
    if ((n != (stack->pos_ - 1)) || stack->top_of_stack_has_sym_data_) {
      switch (stack->stack_[n].state_) {
      } /* switch */
    }
  }
  if (stack->slot_1_has_sym_data_) {
    switch (stack->slot_1_sym_) {
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 9:
      case 10:
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      break;
    } /* switch */
  }
  if (stack->slot_0_has_current_sym_data_) {
    /* Deconstruct placeholder location for terminal not yet shifted */
    switch (stack->current_sym_) {
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 9:
      case 10:
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      break;
    } /* switch */
  }
  
  if (stack->stack_) free(stack->stack_);
  if (stack->match_buffer_) free(stack->match_buffer_);
}

static int YY_push_state(struct YY_stack *stack, int state) {
  if (stack->num_stack_allocated_ == stack->pos_) {
    size_t new_num_allocated = 0;
    if (stack->num_stack_allocated_) {
      new_num_allocated = stack->num_stack_allocated_ * 2;
      if (new_num_allocated <= stack->num_stack_allocated_) {
        /* Overflow in allocation */
        return _YY_OVERFLOW;
      }
    }
    else {
      new_num_allocated = 16;
    }
    
    if (new_num_allocated > (SIZE_MAX / sizeof(struct YY_sym_data))) {
      /* Overflow in allocation */
      return _YY_OVERFLOW;
    }
    
    void *p = realloc(stack->stack_, new_num_allocated * sizeof(struct YY_sym_data));
    if (!p) {
      /* Out of memory */
      return _YY_NO_MEMORY;
    }
    stack->stack_ = (struct YY_sym_data *)p;
    stack->num_stack_allocated_ = new_num_allocated;
  }
  stack->stack_[stack->pos_++].state_ = state;
  return 0;
}

int YY_stack_reset(struct YY_stack *stack) {
  size_t n;
  stack->pending_reset_ = 0;
  stack->discard_remaining_actions_ = 0;
  for (n = 2; n < stack->pos_; ++n) {
    if ((n != (stack->pos_ - 1)) || stack->top_of_stack_has_sym_data_) {
      switch (stack->stack_[n].state_) {
      } /* switch */
    }
  }
  if (stack->slot_1_has_sym_data_) {
    switch (stack->slot_1_sym_) {
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 9:
      case 10:
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      break;
    } /* switch */
  }
  stack->slot_1_has_sym_data_ = stack->slot_1_has_common_data_ = 0;
  stack->slot_1_sym_ = 0;
  stack->top_of_stack_has_sym_data_ = 0;
  stack->top_of_stack_has_common_data_ = 0;
  if (stack->slot_0_has_current_sym_data_) {
    /* Deconstruct placeholder location for terminal not yet shifted */
    switch (stack->current_sym_) {
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 9:
      case 10:
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      break;
    } /* switch */
  }
  stack->slot_0_has_current_sym_data_ = stack->slot_0_has_common_data_ = 0;
  stack->sym_data_ = NULL;
  stack->current_production_length_ = 0;
  stack->current_production_nonterminal_ = 0;
  stack->pos_ = 0;
  stack->error_recovery_ = 0;
  stack->need_sym_ = 1;
  stack->current_sym_ = 0;
  stack->continue_at_ = 0;
  stack->mute_error_turns_ = 0;
  /* reserve slot [0] for the token sym data */
  switch (YY_push_state(stack, 0)) {
    case _YY_OVERFLOW:{
      return _YY_OVERFLOW;
    }
    break;
    case _YY_NO_MEMORY: {
      return _YY_NO_MEMORY;
    }
    break;
  }
  /* reserve slot [1] for the initial state 0, corresponding to the synthetic S production
  ** sym_data in slot [1] is used for temporary nonterminal data storage */
  switch (YY_push_state(stack, 0)) {
    case _YY_OVERFLOW:{
      return _YY_OVERFLOW;
    }
    break;
    case _YY_NO_MEMORY: {
      return _YY_NO_MEMORY;
    }
    break;
  }
  stack->scan_state_ = stack->current_mode_start_state_;
  stack->input_offset_ = 0;
  stack->input_line_ = 1;
  stack->input_col_ = 1;
  stack->match_index_ = 0;
  stack->match_buffer_size_ = 0;
  stack->terminator_repair_ = '\0';
  stack->token_size_ = 0;
  stack->match_offset_ = 0;
  stack->match_line_ = 1;
  stack->match_col_ = 1;
  stack->best_match_action_ = 0;
  stack->best_match_size_ = 0;
  stack->best_match_offset_ = 0;
  stack->best_match_line_ = 1;
  stack->best_match_col_ = 1;
  return 0;
}

int YY_stack_can_recover(struct YY_stack *stack) {
  return !!stack->error_recovery_;
}

int YY_stack_accepts(struct YY_stack *stack, int sym) {
  if (!stack->pos_) return 0;
  return 0 != YY_parse_table[YY_num_columns * stack->stack_[stack->pos_ - 1].state_ + (sym - YY_minimum_sym)];}

void YY_set_mode(struct YY_stack *stack, int mode) {
  if (stack->current_mode_start_state_ == stack->scan_state_) {
    stack->scan_state_ = mode;
  }
  stack->current_mode_start_state_ = mode;
}
static int YY_append_match_buffer(struct YY_stack *stack, const char *s, size_t len) {
  size_t size_needed = len;
  size_needed += stack->match_buffer_size_;
  if (size_needed < stack->match_buffer_size_) {
    return _YY_OVERFLOW;
  }
  if (size_needed == SIZE_MAX) {
    /* cannot fit null terminator */
    return _YY_OVERFLOW;
  }
  size_needed++; /* null terminator */
  if (size_needed < 128) {
    size_needed = 128;
  }
  if (size_needed > stack->match_buffer_size_allocated_) {
    /* intent of code: grow buffer size by powers of 2-1, unless our needs require more now. */
    size_t size_to_allocate = stack->match_buffer_size_allocated_ * 2 + 1;
    if (size_to_allocate <= stack->match_buffer_size_allocated_) {
      return _YY_OVERFLOW;
    }
    if (size_to_allocate < size_needed) {
      size_to_allocate = size_needed;
    }
    void *buf = realloc(stack->match_buffer_, size_to_allocate);
    if (!buf) {
      return _YY_NO_MEMORY;
    }
    stack->match_buffer_ = (char *)buf;
    stack->match_buffer_size_allocated_ = size_to_allocate;
  }
  
  memcpy(stack->match_buffer_ + stack->match_buffer_size_, s, len);
  stack->match_buffer_size_ += len;
  stack->match_buffer_[stack->match_buffer_size_] = '\0';
  return 0;
}

void YY_set_input(struct YY_stack *stack, const char *input, size_t input_size, int is_final_input) {
  stack->input_ = input;
  stack->input_size_ = input_size;
  stack->is_final_input_ = is_final_input;
  stack->input_index_ = 0;
}

void YY_set_location(struct YY_stack *stack, int line, int col, size_t offset) {
  if (stack->token_size_) {
    /* Parsing of next token not in progress, set end location of this token as
    ** it will be the start of the next token. */
    stack->input_line_ = stack->input_line_ - stack->best_match_line_ + line;
    stack->input_col_ = stack->input_col_ - stack->best_match_col_ + col;
    stack->input_offset_ = stack->input_offset_ - stack->best_match_offset_ + offset;
    
    stack->best_match_line_ = line;
    stack->best_match_col_ = col;
    stack->best_match_offset_ = offset;
    return;
  }
  /* Parsing of token in progress, dynamically move the start of the token, as
  ** well as the relative current partial end of the token, to the desired location. */
  stack->input_line_ = stack->input_line_ - stack->match_line_ + line;
  stack->input_col_ = stack->input_col_ - stack->match_col_ + col;
  stack->input_offset_ = stack->input_offset_ - stack->match_offset_ + offset;
  
  stack->best_match_line_ = stack->best_match_line_ - stack->match_line_ + line;
  stack->best_match_col_ = stack->best_match_col_ - stack->match_col_ + col;
  stack->best_match_offset_ = stack->best_match_offset_ - stack->match_offset_ + offset;
  stack->match_line_ = line;
  stack->match_col_ = col;
  stack->match_offset_ = offset;
}

const char *YY_text(struct YY_stack *stack) {
  return stack->match_buffer_;
}

size_t YY_len(struct YY_stack *stack) {
  return stack->token_size_;
}

int YY_line(struct YY_stack *stack) {
  return stack->match_line_;
}

int YY_column(struct YY_stack *stack) {
  return stack->match_col_;
}

size_t YY_offset(struct YY_stack *stack) {
  return stack->match_offset_;
}

int YY_endline(struct YY_stack *stack) {
  return stack->best_match_line_;
}

int YY_endcolumn(struct YY_stack *stack) {
  return stack->best_match_col_;
}

size_t YY_endoffset(struct YY_stack *stack) {
  return stack->best_match_offset_;
}

int YY_lex(struct YY_stack *stack) {
  int r;
  unsigned char c;
  const char *input = stack->input_;
  size_t input_size = stack->input_size_;
  int is_final_input = !!stack->is_final_input_;
  size_t scan_state = stack->scan_state_;
  const size_t *transition_table = YY_scan_table_rex;
  const size_t *actions = YY_scan_actions_rex;
  const size_t row_size = 260;
  const size_t default_action = 0;
  const size_t start_state = 1;
  const size_t start_action = 0;
  
  size_t match_index = stack->match_index_;
  
  size_t best_match_action = stack->best_match_action_;
  size_t best_match_size = stack->best_match_size_;
  size_t best_match_offset = stack->best_match_offset_;
  int best_match_line = stack->best_match_line_;
  int best_match_col = stack->best_match_col_;
  
  size_t input_index = stack->input_index_;
  size_t input_offset = stack->input_offset_;
  int input_line = stack->input_line_;
  int input_col = stack->input_col_;
  
  /* Move any prior token out of the way */
  if (stack->token_size_) {
    stack->match_buffer_[stack->token_size_] = stack->terminator_repair_;
    
    memcpy(stack->match_buffer_, stack->match_buffer_ + stack->token_size_, stack->match_buffer_size_ - stack->token_size_);
    stack->match_buffer_size_ -= stack->token_size_;
    stack->match_offset_ = stack->best_match_offset_;
    stack->match_line_ = stack->best_match_line_;
    stack->match_col_ = stack->best_match_col_;
    
    /* Reset scanner to get ready for next token */
    stack->match_index_ = 0;
    stack->best_match_action_ = best_match_action = start_action;
    stack->best_match_size_ = best_match_size = 0;
    stack->scan_state_ = scan_state = stack->current_mode_start_state_;
    stack->token_size_ = 0;
    
  }
  
  size_t at_match_index_offset = stack->match_offset_;
  int at_match_index_line = stack->match_line_;
  int at_match_index_col = stack->match_col_;
  while (match_index < stack->match_buffer_size_) {
    c = (unsigned char)stack->match_buffer_[match_index];
    for (;;) {
      /* Check for start of input */
      if ((transition_table[row_size * scan_state + 256] != scan_state) && (!at_match_index_offset)) {
        scan_state = transition_table[row_size * scan_state + 256];
      }
      /* Check for start of line */
      else if ((transition_table[row_size * scan_state + 257] != scan_state) && (at_match_index_col == 1)) {
        scan_state = transition_table[row_size * scan_state + 257];
      }
      /* Check for end of line */
      else if ((transition_table[row_size * scan_state + 258] != scan_state) && ('\n' == c)) {
        scan_state = transition_table[row_size * scan_state + 258];
      }
      /* (No need to check for end of input; we have at least 1 character ahead) */
      else {
        break;
      }
    }
    size_t state_action = actions[scan_state];
    if (state_action != default_action) /* replace with actual */ {
      best_match_action = state_action;
      best_match_size = match_index;
      best_match_offset = at_match_index_offset;
      best_match_line = at_match_index_line;
      best_match_col = at_match_index_col;
    }
    scan_state = transition_table[row_size * scan_state + c];
    if (scan_state) {
      at_match_index_offset++;
      if (c != '\n') {
        at_match_index_col++;
      }
      else {
        at_match_index_col = 1;
        at_match_index_line++;
      }
      
      match_index++;
    }
    else {
      /* error, or, end of token, depending on whether we have a match before */
      if (best_match_action == default_action) {
        goto syntax_error;
      }
      
      /* Ensure token match is null terminated */
      stack->terminator_repair_ = stack->match_buffer_[best_match_size];
      stack->match_buffer_[best_match_size] = '\0';
      stack->token_size_ = best_match_size;
      stack->best_match_action_ = best_match_action;
      stack->best_match_size_ = best_match_size;
      stack->best_match_offset_ = best_match_offset;
      stack->best_match_line_ = best_match_line;
      stack->best_match_col_ = best_match_col;
      
      stack->input_index_ = input_index;
      stack->input_offset_ = input_offset;
      stack->input_line_ = input_line;
      stack->input_col_ = input_col;
      return _YY_MATCH;
    }
  }
  
  while (input_index < input_size) {
    c = (unsigned char)input[input_index];
    for (;;) {
      /* Check for start of input */
      if ((transition_table[row_size * scan_state + 256] != scan_state) && (!input_offset)) {
        scan_state = transition_table[row_size * scan_state + 256];
      }
      /* Check for start of line */
      else if ((transition_table[row_size * scan_state + 257] != scan_state) && (input_col == 1)) {
        scan_state = transition_table[row_size * scan_state + 257];
      }
      /* Check for end of line */
      else if ((transition_table[row_size * scan_state + 258] != scan_state) && ('\n' == c)) {
        scan_state = transition_table[row_size * scan_state + 258];
      }
      /* (No need to check for end of input; we have at least 1 character ahead) */
      else {
        break;
      }
    }
    size_t state_action = actions[scan_state];
    if (state_action != default_action) /* replace with actual */ {
      best_match_action = state_action;
      best_match_size = stack->match_buffer_size_ + input_index - stack->input_index_;
      best_match_offset = input_offset;
      best_match_col = input_col;
      best_match_line = input_line;
    }
    scan_state = transition_table[row_size * scan_state + c];
    if (scan_state) {
      input_offset++;
      if (c != '\n') {
        input_col++;
      }
      else {
        input_col = 1;
        input_line++;
      }
      input_index++;
    }
    else {
      /* Append from stack->input_index_ to input_index, excluding input_index itself */
      r = YY_append_match_buffer(stack, input + stack->input_index_, input_index - stack->input_index_);
      if (r) return r;
      
      if (best_match_action == default_action) {
        goto syntax_error;
      }
      
      /* Ensure token match is null terminated, note that the size we just appended may
      * (likely) be longer than the last section we matched. */
      stack->terminator_repair_ = stack->match_buffer_[best_match_size];
      stack->match_buffer_[best_match_size] = '\0';
      stack->token_size_ = best_match_size;
      stack->best_match_action_ = best_match_action;
      stack->best_match_size_ = best_match_size;
      stack->best_match_offset_ = best_match_offset;
      stack->best_match_line_ = best_match_line;
      stack->best_match_col_ = best_match_col;
      
      stack->input_index_ = input_index;
      stack->input_offset_ = input_offset;
      stack->input_line_ = input_line;
      stack->input_col_ = input_col;
      return _YY_MATCH;
    }
  }
  
  r = YY_append_match_buffer(stack, input + stack->input_index_, input_index - stack->input_index_);
  if (r) return r;
  
  /* Resume scanning after the input already processed. */
  match_index = stack->match_buffer_size_;
  
  if (!is_final_input) {
    /* Need more input */
    stack->scan_state_ = scan_state;
    stack->token_size_ = 0; /* no match yet */
    stack->input_index_ = input_index;
    stack->input_offset_ = input_offset;
    stack->input_line_ = input_line;
    stack->input_col_ = input_col;
    
    stack->best_match_action_ = best_match_action;
    stack->best_match_size_ = best_match_size;
    stack->best_match_offset_ = best_match_offset;
    stack->best_match_line_ = best_match_line;
    stack->best_match_col_ = best_match_col;
    
    stack->match_index_ = match_index;
    
    return _YY_FEED_ME;
  }
  for (;;) {
    /* Check for start of input */
    if ((transition_table[row_size * scan_state + 256] != scan_state) && (!input_offset)) {
      scan_state = transition_table[row_size * scan_state + 256];
    }
    /* Check for start of line */
    else if ((transition_table[row_size * scan_state + 257] != scan_state) && (input_col == 1)) {
      scan_state = transition_table[row_size * scan_state + 257];
    }
    /* Check for end of line (always true at end of input) */
    else if (transition_table[row_size * scan_state + 258] != scan_state) {
      scan_state = transition_table[row_size * scan_state + 258];
    }
    /* Check for end of input (always true) */
    else if (transition_table[row_size * scan_state + 259] != scan_state) {
      scan_state = transition_table[row_size * scan_state + 259];
    }
    /* (No need to check for end of input; we have at least 1 character ahead) */
    else {
      break;
    }
  }
  size_t state_action = actions[scan_state];
  if (state_action != default_action) /* replace with actual */ {
    best_match_action = state_action;
    best_match_size = stack->match_buffer_size_;
    best_match_offset = input_offset;
    best_match_col = input_col;
    best_match_line = input_line;
  }
  
  if (!stack->match_buffer_size_ && (stack->input_index_ == input_size)) {
    /* Exhausted all input - leave stack in a state where we can
    * immediately re-use it in its initial state */
    stack->match_index_ = 0;
    stack->best_match_action_ = best_match_action = start_action;
    stack->best_match_size_ = best_match_size;
    stack->best_match_offset_ = best_match_offset;
    stack->best_match_line_ = best_match_line;
    stack->best_match_col_ = best_match_col;
    stack->scan_state_ = scan_state = stack->current_mode_start_state_;
    
    stack->token_size_ = 0;
    stack->input_index_ = 0;
    stack->input_offset_ = input_offset;
    stack->input_line_ = input_line;
    stack->input_col_ = input_col;
    return _YY_END_OF_INPUT;
  }
  
  if (best_match_action == default_action) {
    goto syntax_error;
  }
  
  /* Ensure token match is null terminated */
  stack->terminator_repair_ = stack->match_buffer_[best_match_size];
  stack->match_buffer_[best_match_size] = '\0';
  stack->token_size_ = best_match_size;
  stack->best_match_action_ = best_match_action;
  stack->best_match_size_ = best_match_size;
  stack->best_match_offset_ = best_match_offset;
  stack->best_match_line_ = best_match_line;
  stack->best_match_col_ = best_match_col;
  
  stack->input_index_ = input_index;
  stack->input_offset_ = input_offset;
  stack->input_line_ = input_line;
  stack->input_col_ = input_col;
  return _YY_MATCH;
  syntax_error:
  if (stack->match_buffer_size_) {
    stack->best_match_offset_ = stack->match_offset_ + 1;
    if (stack->match_buffer_[0] != '\n') {
      stack->best_match_line_ = stack->match_line_;
      stack->best_match_col_ = stack->match_col_ + 1;
    }
    else {
      stack->best_match_line_ = stack->match_line_ + 1;
      stack->best_match_col_ = 1;
    }
  }
  else {
    /* Append the single character causing the syntax error */
    r = YY_append_match_buffer(stack, input + stack->input_index_, 1);
    if (r) return r;
    
    input_offset++;
    if (input[stack->input_index_] != '\n') {
      input_col++;
    }
    else {
      input_col = 1;
      input_line++;
    }
    input_index = stack->input_index_ + 1;
    stack->best_match_offset_ = input_offset;
    stack->best_match_line_ = input_line;
    stack->best_match_col_ = input_col;
  }
  
  /* Reset scanner to get ready for next token */
  stack->token_size_ = 1;
  stack->terminator_repair_ = stack->match_buffer_[1];
  stack->match_buffer_[1] = '\0';
  
  stack->input_index_ = input_index;
  stack->input_offset_ = input_offset;
  stack->input_line_ = input_line;
  stack->input_col_ = input_col;
  
  return _YY_LEXICAL_ERROR;
}

int YY_scan(struct YY_stack *stack, twl_t* obj) {
  if (stack->pending_reset_) {
    int r;
    r = YY_stack_reset(stack);
    if (r) return r;
  }
    switch (stack->continue_at_) {
    case 0: break; /* fall through to regular code path */
    case 1: goto C1;
    case 2: goto C2;
    case 3: goto C3;
    case 4: goto C4;
    case 5: goto C5;
    case 6: goto C6;
    case 7: goto C7;
    case 8: goto C8;
    case 9: goto C9;
    case 10: goto C10;
    case 11: goto C11;
    case 12: goto C12;
    case 13: goto C13;
    case 14: goto C14;
    case 15: goto C15;
    case 16: goto C16;
    case 17: goto C17;
    case 18: goto C18;
    case 19: goto C19;
    case 20: goto C20;
    case 21: goto C21;
    case 22: goto C22;
    case 23: goto C23;
    case 24: goto C24;
    case 25: goto C25;
    case 26: goto C26;
    case 27: goto C27;
    case 28: goto C28;
    case 29: goto C29;
    case 30: goto C30;
    case 31: goto C31;
    case 32: goto C32;
    case 33: goto C33;
    case 34: goto C34;
    case 35: goto C35;
    case 36: goto C36;
    case 37: goto C37;
    case 38: goto C38;
    case 39: goto C39;
    case 40: goto C40;
    case 41: goto C41;
    case 42: goto C42;
    case 43: goto C43;
    case 44: goto C44;
    case 45: goto C45;
    case 46: goto C46;
    case 47: goto C47;
    case 48: goto C48;
    case 49: goto C49;
    case 50: goto C50;
    case 51: goto C51;
    case 52: goto C52;
    case 53: goto C53;
    case 54: goto C54;
    case 55: goto C55;
    case 56: goto C56;
    case 57: goto C57;
    case 58: goto C58;
    case 59: goto C59;
    case 60: goto C60;
    case 61: goto C61;
    case 62: goto C62;
    case 63: goto C63;
    case 64: goto C64;
    case 65: goto C65;
    case 66: goto C66;
    case 67: goto C67;
    case 68: goto C68;
    case 69: goto C69;
    case 70: goto C70;
    case 71: goto C71;
    case 72: goto C72;
    case 73: goto C73;
    case 74: goto C74;
    case 75: goto C75;
    case 76: goto C76;
    case 77: goto C77;
    case 78: goto C78;
    case 79: goto C79;
    case 80: goto C80;
    case 81: goto C81;
    case 82: goto C82;
    case 83: goto C83;
    case 84: goto C84;
    case 85: goto C85;
    case 86: goto C86;
    case 87: goto C87;
    case 88: goto C88;
    case 89: goto C89;
    case 90: goto C90;
    case 91: goto C91;
    case 92: goto C92;
    case 93: goto C93;
    case 94: goto C94;
    case 95: goto C95;
    case 96: goto C96;
    case 97: goto C97;
    case 98: goto C98;
    case 99: goto C99;
    case 100: goto C100;
    case 101: goto C101;
    case 102: goto C102;
    case 103: goto C103;
    case 104: goto C104;
    case 105: goto C105;
    case 106: goto C106;
    case 107: goto C107;
    case 108: goto C108;
    case 109: goto C109;
    case 110: goto C110;
    case 111: goto C111;
    case 112: goto C112;
    case 113: goto C113;
    case 114: goto C114;
  } /* continuation switch */
for (;;) {
    stack->continue_at_ = 0;
    if (stack->need_sym_) {
      switch (YY_lex(stack)) {
        case _YY_MATCH:
        stack->need_sym_ = 0;
        stack->discard_remaining_actions_ = 0;
        stack->current_sym_ = YY_INPUT_END; /* no sym specific type associated, changed when we successfully construct it below */
        if (stack->mute_error_turns_) stack->mute_error_turns_--;
        stack->slot_0_has_common_data_ = 1;
        stack->continue_at_ = 1;
        {
           (stack->stack_[0].common_) = (td_t){0};
        }
        C1:;
        switch (stack->best_match_action_) {
          case 1: /* #\*([^\*]|[\r\n]|(\*+([^\*#]|[\r\n])))*\*# */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 2;
          {
            
	sbg((stack->match_buffer_), "mcmt#");

          }
          C2:;
          break;
          case 2: /* /\*([^\*]|[\r\n]|(\*+([^\* //]|[\r\n])))*\* // */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 3;
          {
            
	sbg((stack->match_buffer_), "mcmt/");

          }
          C3:;
          break;
          case 3: /* [/][/][^\n]*[\n] */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 4;
          {
            
	sbg((stack->match_buffer_), "lcmt/");

          }
          C4:;
          break;
          case 4: /* #[^\n]*[\n] */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 5;
          {
            
	sbg((stack->match_buffer_), "lcmt#");
	int rc = ck_magic(obj, (stack->match_buffer_));
	if(rc){return rc;}	//err

          }
          C5:;
          break;
          case 5: /* ([\ \t\n\;]|[\r\n])+ */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          break;
          case 6: /* \n|\r\n */
          stack->current_sym_ = YY_HDOC;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 6;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C6:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 7;
            {
              
	//cbrtはmodeがigだと$chmを使っても無視されるロジックなので
	//専用tokenからタグ付きで送らないとダメ
sbg("HDOC_E");
	//hdocからyaccに詰め直し
	se_t* se = obj->sehd;
	(stack->stack_[0].common_).l = obj->line;
	(stack->stack_[0].common_).c = obj->clm;
	char* p = se_ptr(se);
	int sz = se_sz(se);
sbg(p, "hdptr");
	se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(obj->seyc, -1, p, -2);
	(stack->stack_[0].common_).ssz = sz;
sbg((stack->stack_[0].common_).se->p + (stack->stack_[0].common_).idx, ":repack sehd >> seyc");
	              YY_set_mode(stack, M_YY_DEFAULT);
sbg("rtn2DFL");

            }
            C7:;
          }
          break;
          case 7: /* (""\042+|''\047+)[\-]?[a-z_][a-z0-9_]*(\.[a-z_][a-z0-9_]*)* */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 8;
          {
            
sbg((stack->match_buffer_), "hdoc_head");
	se_t* se = obj->sehd;
	se_clear(se);
	char* p = (stack->match_buffer_);
	while(*p=='"'||*p=='\''){p++;}
	p--;
	se_ins(se, 0, p, -2);
	            YY_set_mode(stack, M_YY_HDOC_S);
	obj->igtab=0;
	if( *(p+1) == '-'){obj->igtab = 1;}
	obj->line = (stack->match_line_);
	obj->clm = (stack->match_col_);

          }
          C8:;
          break;
          case 8: /* .*(\n|\r\n) */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 9;
          {
             YY_set_mode(stack, M_YY_HDOC_P); 
          }
          C9:;
          break;
          case 9: /* \n|\r\n */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 10;
          {
             se_ins(obj->sehd, -1, (stack->match_buffer_)); 
          }
          C10:;
          break;
          case 10: /* .* */
          /* Pattern does not have a symbol */
          stack->current_sym_ = YY_INPUT_END;
          stack->need_sym_ = 1; /* keep scanning */
          stack->continue_at_ = 11;
          {
            
	se_t* se = obj->sehd;
	int eflg =0;
	char* bp = se->p;
	char* p = se->p+1;
	*p == '-' ? p++ :0;
	char* np = bp+strlen(bp);
	
	//hdoc end id
	if(strncmp(p, (stack->match_buffer_), np-p) == 0){ eflg=1;}
	if(eflg){
sbg((stack->match_buffer_), "hdoc_ehit?");
		// ""''skip
		while(*np=='"'||*np=='\''){np++;}
idbg(*np, "hdocend_ck non-blank(nl)== token end == 0");
		if(*np!=0){ break; }	//. isnt nl
		//hit end
		eflg=2;
		char cs[] = "'";
		cs[0]=se->p[0];
idbg(cs[0], "ck slit/dlit");
		int sz = strlen(se->p);	//headerに\0を追加してる endのstrcmp()のため
idbg(se->p, sz, "ck head");
sbg(se->p);
		se_del(se, 0, sz+1);		//\0も併せて削除
		se_del(se, -1, -1);			//del end \n, \r\n, 
		if( se_eptr(se)[0] == '\r'){ se_del(se, -1, -1); }
sbg(se->p, "trim");
		//add "" ''
		se_ins(se, 0, cs);
		se_ins(se, -1, cs);
sbg(se->p, "add qu");
		//
		            YY_set_mode(stack, M_YY_HDOC_E);
idbg(obj->line, obj->clm, p,sz,"hdoc lexend");
sbg(p);
		// return; //dont use return
	}
	if(eflg!=2){
		//not end. skip leading tab
		char* p = (stack->match_buffer_);
		if(obj->igtab){	while(*p=='\t'){ p++; }	}
		se_ins(se, -1, p);
	}

          }
          C11:;
          break;
          case 11: /* [=] */
          stack->current_sym_ = YY_AS;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 12;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C12:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 13;
            {
              ;
            }
            C13:;
          }
          break;
          case 12: /* [\,] */
          stack->current_sym_ = YY_CM;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 14;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C14:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 15;
            {
              ;
            }
            C15:;
          }
          break;
          case 13: /* \[[\ \t]*([a-z_][a-z0-9_]*)(\.[a-z_][a-z0-9_]*)*[\ \t]*\] */
          stack->current_sym_ = YY_LB;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 16;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C16:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 17;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C17:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 18;
            {
              ;
            }
            C18:;
          }
          break;
          case 14: /* \[[\ \t]*\] */
          stack->current_sym_ = YY_LB;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 19;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C19:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 20;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C20:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 21;
            {
              ;
            }
            C21:;
          }
          break;
          case 15: /* ([a-z_][a-z0-9_]*)(\.[a-z_][a-z0-9_]*)* */
          stack->current_sym_ = YY_ID;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 22;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C22:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 23;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C23:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 24;
            {
              ;
            }
            C24:;
          }
          break;
          case 16: /* [\-]?[0-9][0-9][0-9][0-9]+\-[0-1][0-9]\-[0-3][0-9][T\ ][0-2][0-9]\:[0-6][0-9]\:[0-6][0-9](\.[0-9]+)? */
          stack->current_sym_ = YY_DATET;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 25;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C25:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 26;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C26:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 27;
            {
              ;
            }
            C27:;
          }
          break;
          case 17: /* [Z]|[\-\+][0-2][0-9]\:[0-6][0-9] */
          stack->current_sym_ = YY_DATEZ;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 28;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C28:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 29;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C29:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 30;
            {
              ;
            }
            C30:;
          }
          break;
          case 18: /* [\-]?[0-9][0-9][0-9][0-9]+\-[0-1][0-9]\-[0-3][0-9] */
          stack->current_sym_ = YY_DATE;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 31;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C31:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 32;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C32:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 33;
            {
              ;
            }
            C33:;
          }
          break;
          case 19: /* [0-2][0-9]\:[0-6][0-9]\:[0-6][0-9](\.[0-9]+)? */
          stack->current_sym_ = YY_TIME;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 34;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C34:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 35;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C35:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 36;
            {
              ;
            }
            C36:;
          }
          break;
          case 20: /* [\-\.0-9][_xXoO0-9a-fA-F\.lL]* */
          stack->current_sym_ = YY_NUM;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 37;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C37:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 38;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C38:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 39;
            {
              ;
            }
            C39:;
          }
          break;
          case 21: /* "([^\"]|\n|\r\n|\\.)*" */
          stack->current_sym_ = YY_DLIT;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 40;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C40:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 41;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C41:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 42;
            {
              ;
            }
            C42:;
          }
          break;
          case 22: /* '([^\047]|\n|\r\n)*' */
          stack->current_sym_ = YY_SLIT;
          stack->slot_0_has_current_sym_data_ = 1;
          stack->continue_at_ = 43;
          {
            	(stack->stack_[0].v_.uv1_) = 0;
          }
          C43:;
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 44;
            {
              
	//pre-act 
	(stack->stack_[0].common_) = (td_t){0};
	(stack->stack_[0].common_).l = (stack->match_line_);
	(stack->stack_[0].common_).c = (stack->match_col_);
	se_t* se = obj->seyc;
	(stack->stack_[0].common_).se = se;
	(stack->stack_[0].common_).idx = se_sz(se);
	se_ins(se, -1, (stack->match_buffer_), -2);
	(stack->stack_[0].common_).ssz = (int)(stack->token_size_);
idbg("lex token", (stack->match_buffer_), (stack->match_line_), (stack->match_col_), (int)(stack->token_size_) );
sbg((stack->match_buffer_));


            }
            C44:;
          }
          if (!stack->discard_remaining_actions_) {
            stack->continue_at_ = 45;
            {
              ;
            }
            C45:;
          }
          break;
        } /* switch */
        break;
        case _YY_OVERFLOW:
        stack->continue_at_ = 0;
        return _YY_OVERFLOW;
        case _YY_NO_MEMORY:
        stack->continue_at_ = 0;
        return _YY_NO_MEMORY;
        case _YY_FEED_ME:
        stack->continue_at_ = 0;
        /* Need more input */
        return _YY_FEED_ME;
        case _YY_END_OF_INPUT:
        stack->current_sym_ = YY_INPUT_END;
        stack->need_sym_ = 0;
        if (stack->mute_error_turns_) stack->mute_error_turns_--;
        break;
        case _YY_LEXICAL_ERROR:
        stack->continue_at_ = 0;
        /* Lexical error */
        return _YY_LEXICAL_ERROR;
      } /* switch */
    } /* if (need_sym_) */
    else {
      int sym = stack->current_sym_;
      if (!stack->error_recovery_) {
        int action = YY_parse_table[YY_num_columns * stack->stack_[stack->pos_ - 1].state_ + (sym - YY_minimum_sym)];
        if (action > 0) {
          switch (YY_push_state(stack, action)) {
            case _YY_OVERFLOW: {
              stack->continue_at_ = 0;
              return _YY_OVERFLOW;
            }
            break;
            case _YY_NO_MEMORY: {
              stack->continue_at_ = 0;
              return _YY_NO_MEMORY;
            }
            break;
          } /* switch */
          
          /* Fill in the sym from the tokenizer */
          stack->need_sym_ = 1;
          stack->stack_[stack->pos_ - 1].common_ = stack->stack_[0].common_;
          stack->stack_[stack->pos_ - 1].v_ = stack->stack_[0].v_;
          stack->slot_0_has_current_sym_data_ = 0;
          stack->slot_0_has_common_data_ = 0;
          stack->top_of_stack_has_sym_data_ = 1;
          stack->top_of_stack_has_common_data_ = 1;
        } /* action > 0 */
        else if (action < 0) {
          int production = -action - 1;
          stack->discard_remaining_actions_ = 0;
          stack->current_production_length_ = YY_production_lengths[production];
          stack->current_production_nonterminal_ = YY_production_syms[production];
          if (0 == production) {
            stack->pending_reset_ = 1;
            return _YY_FINISH;
          }
          
          memset(&stack->stack_[1], 0, sizeof(struct YY_sym_data));
          { /* scope guard */
            stack->sym_data_ = stack->stack_ + stack->pos_ - stack->current_production_length_;
            switch (production) {
              /* stmt: expr */
              case 1: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 46;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C46:;
                stack->continue_at_ = 47;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C47:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 48;
                  {
                    {;}
                  }
                  C48:;
                }
              }
              break;
              /* stmt: stmt expr */
              case 2: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 49;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C49:;
                stack->continue_at_ = 50;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C50:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 51;
                  {
                    {;}
                  }
                  C51:;
                }
              }
              break;
              /* expr: ass */
              case 3: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 52;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C52:;
                stack->continue_at_ = 53;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C53:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 54;
                  {
                    { sbg(":expr==ass cmpl");  }
                  }
                  C54:;
                }
              }
              break;
              /* ass: LB */
              case 4: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 55;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C55:;
                stack->continue_at_ = 56;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C56:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 57;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	free(obj->lb);
	const char* p = se_ptr((stack->sym_data_[0].common_).se)+(stack->sym_data_[0].common_).idx ;
	obj->lb = strdup(p+1);	//[]
	char* np = obj->lb;
	np[strlen(np)-1] = 0;		//kill ']'
//
//forerr, rawpos
	obj->tdx = (stack->sym_data_[0].common_).idx;
	obj->tdxsz = (stack->sym_data_[0].common_).ssz;
}
                  }
                  C57:;
                }
              }
              break;
              /* ass: ass CM rv */
              case 5: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 58;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C58:;
                stack->continue_at_ = 59;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C59:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 60;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[2].common_);

	char* k = twl_mlvld(obj);
idbg("hit 2nd or more rv ass", k);
sbg(k);
	if(k){
		int rc = twl_as(obj, k, (stack->sym_data_[2].common_));
		if(rc<0){ return YP(_, SYNTAX_ERROR); }
	}
	obj->tdx =   (stack->sym_data_[2].common_).idx;
	obj->tdxsz = (stack->sym_data_[2].common_).ssz;
}
                  }
                  C60:;
                }
              }
              break;
              /* ass: ass_ rv */
              case 6: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 61;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C61:;
                stack->continue_at_ = 62;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C62:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 63;
                  {
                    {
	obj->ldcnt=0;
sbg("hit 1st rv ass");
	char* k = twl_mlvld(obj);
idbg("ass after rval", obj->svcnt, obj->ldcnt, k);
sbg(k);
	if(k){
		int rc = twl_as(obj, k, (stack->sym_data_[1].common_) );
		if(rc<0){ return YP(_, SYNTAX_ERROR); }
	}
	(stack->stack_[1].common_) = (stack->sym_data_[1].common_);
	obj->tdx =   (stack->sym_data_[1].common_).idx;
	obj->tdxsz = (stack->sym_data_[1].common_).ssz;
}
                  }
                  C63:;
                }
              }
              break;
              /* ass: ass_ */
              case 7: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 64;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C64:;
                stack->continue_at_ = 65;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C65:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 66;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx, "hit void ass");
	obj->tdx =   (stack->sym_data_[0].common_).idx;
	obj->tdxsz = (stack->sym_data_[0].common_).ssz;
}
                  }
                  C66:;
                }
              }
              break;
              /* ass_: lv AS */
              case 8: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 67;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C67:;
                stack->continue_at_ = 68;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C68:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 69;
                  {
                    {
sbg("hit lv_as yc");
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	//tdxはlastのtailになる可能性がある場合のみ更新する
	//asはセパレータなのでinfo不要
	obj->tdx =   (stack->sym_data_[0].common_).idx;
	obj->tdxsz = (stack->sym_data_[0].common_).ssz;
}
                  }
                  C69:;
                }
              }
              break;
              /* lv: lv CM ID */
              case 9: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 70;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C70:;
                stack->continue_at_ = 71;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C71:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 72;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[2].common_);
	twl_mlvsv( obj, (stack->sym_data_[2].common_));
}
                  }
                  C72:;
                }
              }
              break;
              /* lv: ID */
              case 10: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 73;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C73:;
                stack->continue_at_ = 74;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C74:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 75;
                  {
                    {
//dbg(${0}.se->p, ${0}.idx, se_ptr(${0}.se) );
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
sbg((stack->sym_data_[0].common_).se->p, (stack->sym_data_[0].common_).se->p + (stack->sym_data_[0].common_).idx, "yscc, hit 1st ID");
sbg("del prebuff, reset");
	twl_ycc(obj, (stack->sym_data_[0].common_));
	(stack->sym_data_[0].common_).idx = 0;	//removeの影響修正. 1stは常に0
	twl_mlvsv( obj, (stack->sym_data_[0].common_));
}
                  }
                  C75:;
                }
              }
              break;
              /* rv: NUM */
              case 11: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 76;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C76:;
                stack->continue_at_ = 77;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C77:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 78;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='n'; }
                  }
                  C78:;
                }
              }
              break;
              /* rv: DLIT */
              case 12: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 79;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C79:;
                stack->continue_at_ = 80;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C80:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 81;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='s'; sbg( (stack->stack_[1].common_).se->p, (stack->stack_[1].common_).se->p +(stack->stack_[1].common_).idx, "yacc DLIT"); }
                  }
                  C81:;
                }
              }
              break;
              /* rv: SLIT */
              case 13: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 82;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C82:;
                stack->continue_at_ = 83;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C83:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 84;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='s'; sbg( (stack->stack_[1].common_).se->p +(stack->stack_[1].common_).idx, "yacc SLIT"); }
                  }
                  C84:;
                }
              }
              break;
              /* rv: HDOC */
              case 14: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 85;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C85:;
                stack->continue_at_ = 86;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C86:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 87;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='s'; sbg( (stack->stack_[1].common_).se->p +(stack->stack_[1].common_).idx, "yacc HDOC"); }
                  }
                  C87:;
                }
              }
              break;
              /* rv: dval */
              case 15: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 88;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C88:;
                stack->continue_at_ = 89;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C89:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 90;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='d'; }
                  }
                  C90:;
                }
              }
              break;
              /* dval: datet_ */
              case 16: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 91;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C91:;
                stack->continue_at_ = 92;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C92:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 93;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                  }
                  C93:;
                }
              }
              break;
              /* dval: datez_ */
              case 17: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 94;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C94:;
                stack->continue_at_ = 95;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C95:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 96;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                  }
                  C96:;
                }
              }
              break;
              /* dval: date_ */
              case 18: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 97;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C97:;
                stack->continue_at_ = 98;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C98:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 99;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                  }
                  C99:;
                }
              }
              break;
              /* dval: time_ */
              case 19: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 100;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C100:;
                stack->continue_at_ = 101;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C101:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 102;
                  {
                    { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                  }
                  C102:;
                }
              }
              break;
              /* datet_: DATET */
              case 20: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 103;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C103:;
                stack->continue_at_ = 104;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C104:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 105;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	char* p = se_ptr((stack->sym_data_[0].common_).se) + (stack->sym_data_[0].common_).idx ;
	(stack->stack_[1].common_).y = strtod(p, &p);
	(stack->stack_[1].common_).m = strtod(p+1, &p);
	(stack->stack_[1].common_).d = strtod(p+1, &p);
	(stack->stack_[1].common_).hh = strtod(p+1, &p);
	(stack->stack_[1].common_).mm = strtod(p+1, &p);
	(stack->stack_[1].common_).ss = strtod(p+1, &p);
idbg("DATET yctk", (stack->stack_[1].common_).y, (stack->stack_[1].common_).m, (stack->stack_[1].common_).d, (stack->stack_[1].common_).hh, (stack->stack_[1].common_).mm, (stack->stack_[1].common_).ss);
}
                  }
                  C105:;
                }
              }
              break;
              /* datez_: datet_ DATEZ */
              case 21: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 106;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C106:;
                stack->continue_at_ = 107;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C107:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 108;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	(stack->stack_[1].common_).z = 1;
	(stack->stack_[1].common_).idx = (stack->sym_data_[1].common_).idx;
	char* p = se_ptr((stack->sym_data_[1].common_).se) + (stack->sym_data_[1].common_).idx ;
	if(*p=='Z'){;}
	else{
		(stack->stack_[1].common_).zh = strtod(p, &p);
		(stack->stack_[1].common_).zm = strtod(p+1, &p);
	}
}
                  }
                  C108:;
                }
              }
              break;
              /* date_: DATE */
              case 22: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 109;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C109:;
                stack->continue_at_ = 110;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C110:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 111;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	(stack->stack_[1].common_).tp='d';
	char* p = se_ptr((stack->sym_data_[0].common_).se) + (stack->sym_data_[0].common_).idx ;
	(stack->stack_[1].common_).y = strtod(p, &p);
	(stack->stack_[1].common_).m = strtod(p+1, &p);
	(stack->stack_[1].common_).d = strtod(p+1, &p);
}
                  }
                  C111:;
                }
              }
              break;
              /* time_: TIME */
              case 23: {
                stack->slot_1_has_common_data_ = 1;
                stack->continue_at_ = 112;
                {
                   (stack->stack_[1].common_) = (td_t){0};
                }
                C112:;
                stack->continue_at_ = 113;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
                }
                C113:;
                if (!stack->discard_remaining_actions_) {
                  stack->continue_at_ = 114;
                  {
                    {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	(stack->stack_[1].common_).tp='d';
	char* p = se_ptr((stack->sym_data_[0].common_).se) + (stack->sym_data_[0].common_).idx ;
	(stack->stack_[1].common_).hh = strtod(p, &p);
	(stack->stack_[1].common_).mm = strtod(p+1, &p);
	(stack->stack_[1].common_).ss = strtod(p+1, &p);
}
                  }
                  C114:;
                }
              }
              break;
            } /* switch */
          } /* scope guard */
          
          /* Free symdata for every symbol in the production, including the first slot where we will soon
          * push nonterminal_data_reduced_to */
          size_t YY_sym_idx;
          for (YY_sym_idx = stack->pos_ - stack->current_production_length_; YY_sym_idx < stack->pos_; ++YY_sym_idx) {
            switch (stack->stack_[YY_sym_idx].state_) {
            } /* switch */
          } /* for */
          stack->pos_ -= stack->current_production_length_;
          action = YY_parse_table[YY_num_columns * stack->stack_[stack->pos_ - 1].state_ + (stack->current_production_nonterminal_ - YY_minimum_sym)];
          if (action <= 0) {
            stack->continue_at_ = 0;
            /* Internal error */
            return _YY_INTERNAL_ERROR;
          }
          switch (YY_push_state(stack, action /* action for a "goto" shift is the ordinal */)) {
            case _YY_OVERFLOW: {
              stack->continue_at_ = 0;
              return _YY_OVERFLOW;
            }
            break;
            case _YY_NO_MEMORY: {
              stack->continue_at_ = 0;
              return _YY_NO_MEMORY;
            }
            break;
          } /* switch */
          struct YY_sym_data *sd = stack->stack_ + stack->pos_ - 1;
          *sd = stack->stack_[1];
          sd->state_ = action;
          stack->slot_1_has_common_data_ = 0;
          stack->slot_1_has_sym_data_ = 0;
          stack->top_of_stack_has_sym_data_ = 1;
          stack->top_of_stack_has_common_data_ = 1;
        } /* action < 0 */
        else /* action == 0 */ {
          /* check if we can recover using an error token. */
          size_t n;
          for (n = 0; n < stack->pos_; ++n) {
            int err_action = YY_parse_table[YY_num_columns * stack->stack_[n].state_ + (16 /* error token */ - YY_minimum_sym)];
            if (err_action > 0) {
              /* we can transition on the error token somewhere on the stack */
              break;
            }
          }
          if (n != stack->pos_) {
            /* Enter error-token recovery mode given that such a recovery is possible */
            stack->error_recovery_ = (n != stack->pos_);
          }
          else {
            if (sym != YY_INPUT_END) {
              /* Retain EOF but discard any other sym so we make progress */
              stack->need_sym_ = 1;
              /* Deconstruct placeholder location for discarded symbol */
              switch (stack->current_sym_) {
                case 3:
                case 4:
                case 5:
                case 6:
                case 7:
                case 8:
                case 9:
                case 10:
                case 11:
                case 12:
                case 13:
                case 14:
                case 15:
                break;
              } /* switch */
            }
          }
          /* Issue the error here */
          if (!stack->mute_error_turns_) {
            stack->mute_error_turns_ = 3;
            stack->continue_at_ = 0;
            /* Syntax error */
            return _YY_SYNTAX_ERROR;
          }
          else {
            stack->mute_error_turns_--;
            if (sym == YY_INPUT_END) {
              /* EOF means we cannot shift to recover, and errors are muted, so return completion */
              stack->pending_reset_ = 1;
              return _YY_FINISH;
            }
          }
        }
      } /* !stack->error_recovery_ */
      if (stack->error_recovery_) {
        size_t n;
        n = stack->pos_;
        if (n) {
          do {
            --n;
            /* Can we shift an error token? */
            int err_action = YY_parse_table[YY_num_columns * stack->stack_[n].state_ + (16 /* error token */ - YY_minimum_sym)];
            if (err_action > 0) {
              /* Does the resulting state accept the current symbol? */
              int err_sym_action = YY_parse_table[YY_num_columns * err_action + (sym - YY_minimum_sym)];
              if (err_sym_action) {
                /* Current symbol is accepted, recover error condition by shifting the error token and then process the symbol as usual */
                /* Free symdata for every symbol up to the state where we will shift the error token */
                size_t YY_sym_idx;
                for (YY_sym_idx = n + 1; YY_sym_idx < stack->pos_; ++YY_sym_idx) {
                  switch (stack->stack_[YY_sym_idx].state_) {
                  } /* switch */
                } /* for */
                stack->pos_ = n + 1;
                switch (YY_push_state(stack, err_action)) {
                  case _YY_OVERFLOW: {
                    stack->continue_at_ = 0;
                    return _YY_OVERFLOW;
                  }
                  break;
                  case _YY_NO_MEMORY: {
                    stack->continue_at_ = 0;
                    return _YY_NO_MEMORY;
                  }
                  break;
                } /* switch */
                stack->error_recovery_ = 0;
                /* Break out of do { .. } while loop, we've recovered */
                break;
              } /* if (err_sym_action) (if the current sym can continue after an error transition) */
            } /* if (err_action) (if the state at position 'n' can accept an error transition) */
          } while (n);
        }
        if (stack->error_recovery_) {
          /* Did not yet recover, discard current sym and get next */
          if (stack->current_sym_ == YY_INPUT_END) {
            /* EOF means we cannot shift to recover, so return completion */
            stack->pending_reset_ = 1;
            return _YY_FINISH;
          }
          stack->need_sym_ = 1;
          /* Deconstruct placeholder location for discarded symbol */
          switch (stack->current_sym_) {
            case 3:
            case 4:
            case 5:
            case 6:
            case 7:
            case 8:
            case 9:
            case 10:
            case 11:
            case 12:
            case 13:
            case 14:
            case 15:
            break;
          } /* switch */
        }
      } /* stack->error_recovery_ */
    }
  } /* for (;;) */
}

int YY_parse(struct YY_stack *stack, int sym, twl_t* obj) {
  if (stack->pending_reset_) {
    int r;
    r = YY_stack_reset(stack);
    if (r) return r;
  }
    switch (stack->continue_at_) {
    case 0: break; /* fall through to regular code path */
    case 1: goto C1;
    case 2: goto C2;
    case 3: goto C3;
    case 4: goto C4;
    case 5: goto C5;
    case 6: goto C6;
    case 7: goto C7;
    case 8: goto C8;
    case 9: goto C9;
    case 10: goto C10;
    case 11: goto C11;
    case 12: goto C12;
    case 13: goto C13;
    case 14: goto C14;
    case 15: goto C15;
    case 16: goto C16;
    case 17: goto C17;
    case 18: goto C18;
    case 19: goto C19;
    case 20: goto C20;
    case 21: goto C21;
    case 22: goto C22;
    case 23: goto C23;
    case 24: goto C24;
    case 25: goto C25;
    case 26: goto C26;
    case 27: goto C27;
    case 28: goto C28;
    case 29: goto C29;
    case 30: goto C30;
    case 31: goto C31;
    case 32: goto C32;
    case 33: goto C33;
    case 34: goto C34;
    case 35: goto C35;
    case 36: goto C36;
    case 37: goto C37;
    case 38: goto C38;
    case 39: goto C39;
    case 40: goto C40;
    case 41: goto C41;
    case 42: goto C42;
    case 43: goto C43;
    case 44: goto C44;
    case 45: goto C45;
    case 46: goto C46;
    case 47: goto C47;
    case 48: goto C48;
    case 49: goto C49;
    case 50: goto C50;
    case 51: goto C51;
    case 52: goto C52;
    case 53: goto C53;
    case 54: goto C54;
    case 55: goto C55;
    case 56: goto C56;
    case 57: goto C57;
    case 58: goto C58;
    case 59: goto C59;
    case 60: goto C60;
    case 61: goto C61;
    case 62: goto C62;
    case 63: goto C63;
    case 64: goto C64;
    case 65: goto C65;
    case 66: goto C66;
    case 67: goto C67;
    case 68: goto C68;
    case 69: goto C69;
    case 70: goto C70;
    case 71: goto C71;
  } /* continuation switch */
if (stack->mute_error_turns_) stack->mute_error_turns_--;
  for (;;) {
    if (!stack->error_recovery_) {
      int action = YY_parse_table[YY_num_columns * stack->stack_[stack->pos_ - 1].state_ + (sym - YY_minimum_sym)];
      if (action > 0) {
        switch (YY_push_state(stack, action)) {
          case _YY_OVERFLOW: {
            stack->continue_at_ = 0;
            return _YY_OVERFLOW;
          }
          break;
          case _YY_NO_MEMORY: {
            stack->continue_at_ = 0;
            return _YY_NO_MEMORY;
          }
          break;
        } /* switch */
        
        stack->sym_data_ = stack->stack_ + stack->pos_ - 1;
        stack->top_of_stack_has_sym_data_ = 0;
        stack->top_of_stack_has_common_data_ = 1;
        stack->continue_at_ = 1;
        {
           (stack->sym_data_->common_) = (td_t){0};
        }
        C1:;
        switch (sym) {
          case YY_AS:
          case YY_CM:
          case YY_LB:
          case YY_ID:
          case YY_NUM:
          case YY_DLIT:
          case YY_SLIT:
          case YY_HDOC:
          case YY_DATE:
          case YY_DATET:
          case YY_DATEZ:
          case YY_TIME:
          case YY_BAD_TOKEN:
          stack->top_of_stack_has_sym_data_ = 1;
          stack->continue_at_ = 2;
          {
            	(stack->sym_data_->v_.uv1_) = 0;
          }
          C2:;
          break;
        } /* switch */
        stack->continue_at_ = 0;
        /* Next token */
        return _YY_FEED_ME;
      } /* action > 0 */
      else if (action < 0) {
        int production = -action - 1;
        stack->discard_remaining_actions_ = 0;
        stack->current_production_length_ = YY_production_lengths[production];
        stack->current_production_nonterminal_ = YY_production_syms[production];
        if (0 == production) {
          stack->pending_reset_ = 1;
          /* Synth S we're done */
          return _YY_FINISH;
        }
        
        memset(&stack->stack_[1], 0, sizeof(struct YY_sym_data));
        { /* scope guard */
          stack->sym_data_ = stack->stack_ + stack->pos_ - stack->current_production_length_;
          switch (production) {
            /* stmt: expr */
            case 1: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 3;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C3:;
              stack->continue_at_ = 4;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C4:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 5;
                {
                  {;}
                }
                C5:;
              }
            }
            break;
            /* stmt: stmt expr */
            case 2: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 6;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C6:;
              stack->continue_at_ = 7;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C7:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 8;
                {
                  {;}
                }
                C8:;
              }
            }
            break;
            /* expr: ass */
            case 3: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 9;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C9:;
              stack->continue_at_ = 10;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C10:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 11;
                {
                  { sbg(":expr==ass cmpl");  }
                }
                C11:;
              }
            }
            break;
            /* ass: LB */
            case 4: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 12;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C12:;
              stack->continue_at_ = 13;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C13:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 14;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	free(obj->lb);
	const char* p = se_ptr((stack->sym_data_[0].common_).se)+(stack->sym_data_[0].common_).idx ;
	obj->lb = strdup(p+1);	//[]
	char* np = obj->lb;
	np[strlen(np)-1] = 0;		//kill ']'
//
//forerr, rawpos
	obj->tdx = (stack->sym_data_[0].common_).idx;
	obj->tdxsz = (stack->sym_data_[0].common_).ssz;
}
                }
                C14:;
              }
            }
            break;
            /* ass: ass CM rv */
            case 5: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 15;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C15:;
              stack->continue_at_ = 16;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C16:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 17;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[2].common_);

	char* k = twl_mlvld(obj);
idbg("hit 2nd or more rv ass", k);
sbg(k);
	if(k){
		int rc = twl_as(obj, k, (stack->sym_data_[2].common_));
		if(rc<0){ return YP(_, SYNTAX_ERROR); }
	}
	obj->tdx =   (stack->sym_data_[2].common_).idx;
	obj->tdxsz = (stack->sym_data_[2].common_).ssz;
}
                }
                C17:;
              }
            }
            break;
            /* ass: ass_ rv */
            case 6: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 18;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C18:;
              stack->continue_at_ = 19;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C19:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 20;
                {
                  {
	obj->ldcnt=0;
sbg("hit 1st rv ass");
	char* k = twl_mlvld(obj);
idbg("ass after rval", obj->svcnt, obj->ldcnt, k);
sbg(k);
	if(k){
		int rc = twl_as(obj, k, (stack->sym_data_[1].common_) );
		if(rc<0){ return YP(_, SYNTAX_ERROR); }
	}
	(stack->stack_[1].common_) = (stack->sym_data_[1].common_);
	obj->tdx =   (stack->sym_data_[1].common_).idx;
	obj->tdxsz = (stack->sym_data_[1].common_).ssz;
}
                }
                C20:;
              }
            }
            break;
            /* ass: ass_ */
            case 7: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 21;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C21:;
              stack->continue_at_ = 22;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C22:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 23;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx, "hit void ass");
	obj->tdx =   (stack->sym_data_[0].common_).idx;
	obj->tdxsz = (stack->sym_data_[0].common_).ssz;
}
                }
                C23:;
              }
            }
            break;
            /* ass_: lv AS */
            case 8: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 24;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C24:;
              stack->continue_at_ = 25;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C25:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 26;
                {
                  {
sbg("hit lv_as yc");
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	//tdxはlastのtailになる可能性がある場合のみ更新する
	//asはセパレータなのでinfo不要
	obj->tdx =   (stack->sym_data_[0].common_).idx;
	obj->tdxsz = (stack->sym_data_[0].common_).ssz;
}
                }
                C26:;
              }
            }
            break;
            /* lv: lv CM ID */
            case 9: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 27;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C27:;
              stack->continue_at_ = 28;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C28:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 29;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[2].common_);
	twl_mlvsv( obj, (stack->sym_data_[2].common_));
}
                }
                C29:;
              }
            }
            break;
            /* lv: ID */
            case 10: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 30;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C30:;
              stack->continue_at_ = 31;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C31:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 32;
                {
                  {
//dbg(${0}.se->p, ${0}.idx, se_ptr(${0}.se) );
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
sbg((stack->sym_data_[0].common_).se->p, (stack->sym_data_[0].common_).se->p + (stack->sym_data_[0].common_).idx, "yscc, hit 1st ID");
sbg("del prebuff, reset");
	twl_ycc(obj, (stack->sym_data_[0].common_));
	(stack->sym_data_[0].common_).idx = 0;	//removeの影響修正. 1stは常に0
	twl_mlvsv( obj, (stack->sym_data_[0].common_));
}
                }
                C32:;
              }
            }
            break;
            /* rv: NUM */
            case 11: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 33;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C33:;
              stack->continue_at_ = 34;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C34:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 35;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='n'; }
                }
                C35:;
              }
            }
            break;
            /* rv: DLIT */
            case 12: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 36;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C36:;
              stack->continue_at_ = 37;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C37:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 38;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='s'; sbg( (stack->stack_[1].common_).se->p, (stack->stack_[1].common_).se->p +(stack->stack_[1].common_).idx, "yacc DLIT"); }
                }
                C38:;
              }
            }
            break;
            /* rv: SLIT */
            case 13: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 39;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C39:;
              stack->continue_at_ = 40;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C40:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 41;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='s'; sbg( (stack->stack_[1].common_).se->p +(stack->stack_[1].common_).idx, "yacc SLIT"); }
                }
                C41:;
              }
            }
            break;
            /* rv: HDOC */
            case 14: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 42;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C42:;
              stack->continue_at_ = 43;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C43:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 44;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='s'; sbg( (stack->stack_[1].common_).se->p +(stack->stack_[1].common_).idx, "yacc HDOC"); }
                }
                C44:;
              }
            }
            break;
            /* rv: dval */
            case 15: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 45;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C45:;
              stack->continue_at_ = 46;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C46:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 47;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); (stack->stack_[1].common_).tp='d'; }
                }
                C47:;
              }
            }
            break;
            /* dval: datet_ */
            case 16: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 48;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C48:;
              stack->continue_at_ = 49;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C49:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 50;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                }
                C50:;
              }
            }
            break;
            /* dval: datez_ */
            case 17: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 51;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C51:;
              stack->continue_at_ = 52;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C52:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 53;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                }
                C53:;
              }
            }
            break;
            /* dval: date_ */
            case 18: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 54;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C54:;
              stack->continue_at_ = 55;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C55:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 56;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                }
                C56:;
              }
            }
            break;
            /* dval: time_ */
            case 19: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 57;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C57:;
              stack->continue_at_ = 58;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C58:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 59;
                {
                  { (stack->stack_[1].common_) = (stack->sym_data_[0].common_); }
                }
                C59:;
              }
            }
            break;
            /* datet_: DATET */
            case 20: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 60;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C60:;
              stack->continue_at_ = 61;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C61:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 62;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	char* p = se_ptr((stack->sym_data_[0].common_).se) + (stack->sym_data_[0].common_).idx ;
	(stack->stack_[1].common_).y = strtod(p, &p);
	(stack->stack_[1].common_).m = strtod(p+1, &p);
	(stack->stack_[1].common_).d = strtod(p+1, &p);
	(stack->stack_[1].common_).hh = strtod(p+1, &p);
	(stack->stack_[1].common_).mm = strtod(p+1, &p);
	(stack->stack_[1].common_).ss = strtod(p+1, &p);
idbg("DATET yctk", (stack->stack_[1].common_).y, (stack->stack_[1].common_).m, (stack->stack_[1].common_).d, (stack->stack_[1].common_).hh, (stack->stack_[1].common_).mm, (stack->stack_[1].common_).ss);
}
                }
                C62:;
              }
            }
            break;
            /* datez_: datet_ DATEZ */
            case 21: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 63;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C63:;
              stack->continue_at_ = 64;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C64:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 65;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	(stack->stack_[1].common_).z = 1;
	(stack->stack_[1].common_).idx = (stack->sym_data_[1].common_).idx;
	char* p = se_ptr((stack->sym_data_[1].common_).se) + (stack->sym_data_[1].common_).idx ;
	if(*p=='Z'){;}
	else{
		(stack->stack_[1].common_).zh = strtod(p, &p);
		(stack->stack_[1].common_).zm = strtod(p+1, &p);
	}
}
                }
                C65:;
              }
            }
            break;
            /* date_: DATE */
            case 22: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 66;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C66:;
              stack->continue_at_ = 67;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C67:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 68;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	(stack->stack_[1].common_).tp='d';
	char* p = se_ptr((stack->sym_data_[0].common_).se) + (stack->sym_data_[0].common_).idx ;
	(stack->stack_[1].common_).y = strtod(p, &p);
	(stack->stack_[1].common_).m = strtod(p+1, &p);
	(stack->stack_[1].common_).d = strtod(p+1, &p);
}
                }
                C68:;
              }
            }
            break;
            /* time_: TIME */
            case 23: {
              stack->slot_1_has_common_data_ = 1;
              stack->continue_at_ = 69;
              {
                 (stack->stack_[1].common_) = (td_t){0};
              }
              C69:;
              stack->continue_at_ = 70;
              {
                { (stack->stack_[1].common_) = (stack->sym_data_[0].common_);
idbg("yacc_allactck", (stack->stack_[1].common_).idx);
	sbg((stack->stack_[1].common_).se->p + (stack->stack_[1].common_).idx);
	}
              }
              C70:;
              if (!stack->discard_remaining_actions_) {
                stack->continue_at_ = 71;
                {
                  {
	(stack->stack_[1].common_) = (stack->sym_data_[0].common_);
	(stack->stack_[1].common_).tp='d';
	char* p = se_ptr((stack->sym_data_[0].common_).se) + (stack->sym_data_[0].common_).idx ;
	(stack->stack_[1].common_).hh = strtod(p, &p);
	(stack->stack_[1].common_).mm = strtod(p+1, &p);
	(stack->stack_[1].common_).ss = strtod(p+1, &p);
}
                }
                C71:;
              }
            }
            break;
          } /* switch */
        } /* scope guard */
        
        /* Free symdata for every symbol in the production, including the first slot where we will soon
        * push nonterminal_data_reduced_to */
        size_t YY_sym_idx;
        for (YY_sym_idx = stack->pos_ - stack->current_production_length_; YY_sym_idx < stack->pos_; ++YY_sym_idx) {
          switch (stack->stack_[YY_sym_idx].state_) {
          } /* switch */
        } /* for */
        stack->pos_ -= stack->current_production_length_;
        action = YY_parse_table[YY_num_columns * stack->stack_[stack->pos_ - 1].state_ + (stack->current_production_nonterminal_ - YY_minimum_sym)];
        if (action <= 0) {
          stack->continue_at_ = 0;
          /* Internal error */
          return _YY_INTERNAL_ERROR;
        }
        switch (YY_push_state(stack, action /* action for a "goto" shift is the ordinal */)) {
          case _YY_OVERFLOW: {
            stack->continue_at_ = 0;
            return _YY_OVERFLOW;
          }
          break;
          case _YY_NO_MEMORY: {
            stack->continue_at_ = 0;
            return _YY_NO_MEMORY;
          }
          break;
        } /* switch */
        struct YY_sym_data *sd = stack->stack_ + stack->pos_ - 1;
        *sd = stack->stack_[1];
        sd->state_ = action;
        stack->slot_1_has_common_data_ = 0;
        stack->slot_1_has_sym_data_ = 0;
        stack->top_of_stack_has_common_data_ = 1;
        stack->top_of_stack_has_sym_data_ = 1;
      } /* action < 0 */
      else /* action == 0 */ {
        /* check if we can recover using an error token. */
        size_t n;
        for (n = 0; n < stack->pos_; ++n) {
          int err_action = YY_parse_table[YY_num_columns * stack->stack_[n].state_ + (16 /* error token */ - YY_minimum_sym)];
          if (err_action > 0) {
            /* we can transition on the error token somewhere on the stack */
            break;
          }
        }
        if (n != stack->pos_) {
          /* Enter error-token recovery mode given that such a recovery is possible */
          stack->error_recovery_ = (n != stack->pos_);
        }
        else {
          if (sym != YY_INPUT_END) {
            /* Retain EOF but discard any other sym so we make progress */
            stack->continue_at_ = 0;
            /* Next token */
            return _YY_FEED_ME;
          }
        }
        /* Issue the error here */
        if (!stack->mute_error_turns_) {
          stack->mute_error_turns_ = 3;
          stack->continue_at_ = 0;
          /* Syntax error */
          return _YY_SYNTAX_ERROR;
        }
        else {
          stack->mute_error_turns_--;
          if (sym == YY_INPUT_END) {
            /* EOF means we cannot shift to recover, and errors are muted, so return completion */
            stack->pending_reset_ = 1;
            return _YY_FINISH;
          }
        }
      }
    } /* !stack->error_recovery_ */
    if (stack->error_recovery_) {
      size_t n;
      n = stack->pos_;
      if (n) {
        do {
          --n;
          /* Can we shift an error token? */
          int err_action = YY_parse_table[YY_num_columns * stack->stack_[n].state_ + (16 /* error token */ - YY_minimum_sym)];
          if (err_action > 0) {
            /* Does the resulting state accept the current symbol? */
            int err_sym_action = YY_parse_table[YY_num_columns * err_action + (sym - YY_minimum_sym)];
            if (err_sym_action) {
              /* Current symbol is accepted, recover error condition by shifting the error token and then process the symbol as usual */
              /* Free symdata for every symbol up to the state where we will shift the error token */
              size_t YY_sym_idx;
              for (YY_sym_idx = n + 1; YY_sym_idx < stack->pos_; ++YY_sym_idx) {
                switch (stack->stack_[YY_sym_idx].state_) {
                } /* switch */
              } /* for */
              stack->pos_ = n + 1;
              /* Push the state of the error transition */
              switch (YY_push_state(stack, err_action)) {
                case _YY_OVERFLOW: {
                  stack->continue_at_ = 0;
                  return _YY_OVERFLOW;
                }
                break;
                case _YY_NO_MEMORY: {
                  stack->continue_at_ = 0;
                  return _YY_NO_MEMORY;
                }
                break;
              } /* switch */
              stack->error_recovery_ = 0;
              /* Break out of do { .. } while loop, we've recovered */
              break;
            } /* if (err_sym_action) (if the current sym can continue after an error transition) */
          } /* if (err_action) (if the state at position 'n' can accept an error transition) */
        } while (n);
      }
      if (stack->error_recovery_) {
        /* Did not yet recover, discard current sym and get next */
        if (sym == YY_INPUT_END) {
          /* EOF means we cannot shift to recover, so return completion */
          stack->pending_reset_ = 1;
          return _YY_FINISH;
        }
        stack->continue_at_ = 0;
        /* Next token */
        return _YY_FEED_ME;
      }
    } /* stack->error_recovery_ */
  } /* for (;;) */
}
/* --------- END OF GENERATED CODE ------------ */
//post c-code %%-

typedef struct YP(stack) pvm_t;

twl_t* twl_new_impl(const char* flname){
	twl_t* obj = calloc(1, sizeof(twl_t) );
	hfl_t* hdb = hfl_new();
	obj->hdb = hdb;
	void* pvm = calloc(1, sizeof(pvm_t) );
	YP(stack_init)(pvm);
//	YY_stack_init(pvm);
	obj->pvm = pvm;
	// hfl_find(hobj, ".");
	// eq_p(hobj->p, pvm);
	// printf("%p == %p\n", hobj->p, pvm);
//	obj->hdb = hdb;
//	obj->pvm = pvm;
	obj->sehd = se_new();
	obj->sepp = se_new();
	obj->setmp = se_new();
	obj->seyc  = se_new();
	obj->lb = strdup("");
	obj->lex0 = NULL;
	obj->lex1 = NULL;
//
	obj->svpf = pfgetfp(MSGP_FP_DBG);
	pfsetfp(MSGP_FP_DBG, NULL);
	return obj;
}

void* twl_free(twl_t* obj){
//	restore
	pfsetfp(MSGP_FP_DBG, obj->svpf);
//
	hfl_free(obj->hdb);
	YP(stack_cleanup)(obj->pvm);
//	YY_stack_cleanup(obj->pvm);
	free(obj->pvm);
	free(obj->emsg);
	se_free(obj->sehd);
	se_free(obj->sepp);
	se_free(obj->setmp);
	se_free(obj->seyc);
	free(obj->lb);
//return NULL;
	free(obj->lex0);
	free(obj->lex1);
	//
	free(obj);
	*obj = (twl_t){0};
	return NULL;
}

twl_t* twl_clear(twl_t* obj){
//	twl_t* obj = calloc(1, sizeof(twl_t));
//	hfl_t* hdb = hfl_new();
//	obj->hdb = hdb;
	hfl_clear(obj->hdb);
	YP(stack_reset)(obj->pvm);
	obj->rc=0;
	obj->cnt=0;

	se_clear(obj->sehd); 
	se_clear(obj->sepp); 
	se_clear(obj->setmp);
	se_clear(obj->seyc); 
	
	free(obj->lb);
	free(obj->emsg);
	obj->emsg=NULL;

	obj->lb = strdup("");
	free(obj->lex0);
	free(obj->lex1);
	obj->lex0 = NULL;
	obj->lex1 = NULL;
	return obj;
}



int twl_pushfile_impl(twl_t* obj, const char* flname){
	int rc = 0;
	int sverr=0;
	FILE* fp = stdin;
idbg(fp);
sbg(flname);

	if(strcmp(flname, "stdin") ){
		errno=0;
		fp = fopen(flname, "r");
sbg("fopen suc/fail");
idbg(fp, errno);
		sverr=errno;
		if(fp==NULL){  goto lb_ERR; }
	}
sbg("fp suc, start readloop");
idbg(fp, stdin);
	while(1){
		char buf[128]={0};
		rc = fread(buf, 1, 127, fp);	//1x128 = 128byte  
sbg("fread suc?");
idbg(rc);
		if(rc ==0 && ferror(fp) ) { sverr = EIO; goto lb_ERR; }
		buf[rc]=0;
		char* p = buf;
		if(rc==0){ p=NULL; }
sbg(p, "fread, push2pp");
		rc = twl_pushstr(obj, p, rc);
sbg(obj->emsg, "pushck, err,nx,cpl == rc<0,1,0");
idbg(rc);
		if(rc<0){ goto lb_ERR;}
		if(rc==0){ break; }
	}
sbg("fileEOF, close");
	if(fp!=stdin){ fclose(fp);}
	YP(stack_reset)(obj->pvm);

lb_RTN:;
	return rc;

lb_ERR:;
	if(obj->emsg){ goto lb_RTN; }
	int sz = snprintf(NULL, 0, "%s: %s",  strerror(sverr), flname );
	char* ep = malloc(sz+1);
	sprintf(ep, "%s: %s", strerror(errno), flname );
	rc = 1;
	obj->emsg = ep;
	obj->rc = rc;
	return rc;
}

static
int seterr(twl_t* obj, const char* emsg, int errcode){
	int sz=0;
	void* mp = NULL;
	if(obj->emsg){ return 0;}
	// cbrt errline start with 0? >> add +1 to fix
	// err clm maybe next(yacc) or just(lex) so reverse pos -1? 0?
	int l = YP(line)(obj->pvm);
	int c = YP(column)(obj->pvm);
	const char* p = YP(text)(obj->pvm);
//	l==0 ? l=1 :0;
	while(1){
		sz = snprintf(mp, sz, "%d:%d: (l/c around) %s: %.10s"
		, l, c, emsg, p );
		if(mp){break; }
		sz++;
		mp = malloc(sz);
	}
	free(obj->emsg);
	obj->emsg = mp;
	obj->rc = errcode;	//lex err;
sbg(obj->emsg, "hit err");
	return 0;
}

static
int err_l2g(int ec){
	if(ec==ERR_TKN){ ec = YP(_,LEXICAL_ERROR); }
	else if(ec==ERR_STX){ ec = YP(_,SYNTAX_ERROR); }
	return ec;
}

static 
char* twl_pp(twl_t* obj, const char* s, size_t ssz, se_t* se, int stopflg){
	//update buff, until \0
	if(se->sz){
		//not init >> del with \0 , utf8 == no \0 char
		int idx = strlen(se->p);
		se_del(se, 0, idx+1);
	}
	//
	if(stopflg){ s=""; ssz=0; }
	se_ins(se, -1, s);
sbg(s, se->p, -10, ":sv2se");

	//pp concat
	for(int i=0;i<se->sz;){
		char* p1 = strstr(se->p + i, "\\\r\n");
		char* p2 = strstr(se->p + i, "\\\n");
		if(p1==NULL && p2==NULL){break; }
		p1==NULL ? p1 = p2+1 :0;
		p2==NULL ? p2 = p1+1 :0;
sbg(p1, p2, se->p, -10, ":hit \\+nl");
		if(p1<p2){
			i = p1 - se->p;
			se_del(se, i, 3);
			i+=3;
		}
		else{
			i = p2 - se->p;
			se_del(se, i, 2);
			i += 2;
		}
sbg(p1, p2, se->p, -10, ":hit_cat");
	}
sbg(":msg: ckend pp_cat >> cktail \\(eos), \\\\r(eos)\n");
	int edx = se->sz -1;
	char* ep = se->p + edx;
	if( *ep == '\\' && !stopflg){ se_ins(se, edx, "\0", 1); }
	if( strcmp(ep-1, "\\\\r") == 0 && !stopflg){ se_ins(se, edx-1, "\0", 1); }
sbg(ep, se->p + edx, ":ck_eos, ep==\\ etc");

	ssz = strlen(se->p);
	if(stopflg){ ssz = se->sz; }
	obj->rc = ssz;
sbg(s, se->p, -10, ":pp exit");
	return se->p;
}

int twl_pushstr_impl(twl_t* obj, const char* s, int ssz, ...){
sbg(s, ":pushstr head");
idbg(obj->rc);
	if(obj->rc<0){return obj->rc;}
	//; STOP(obj->rc<0, "parse is already failed. renew parser");

	int rc = 0;
	int stopflg = 0;
	if(s==NULL){
		stopflg = 1;
		s="";
		ssz=0;
	}
	ssz < 0 ? ssz=strlen(s): ssz;	
	//ppstr 第一バッファ。\n系と\eofを始末する

sbg(s, "pp_bf");
idbg(ssz);
	s = twl_pp(obj, s, ssz, obj->sepp, stopflg);
	ssz = obj->rc;	//tmp intbuff
sbg(s, "pp_af");
idbg(ssz);
	
	//keep mem. 第二バッファinput()のapiがクソなのでこっちで保持しないといけない。
	if(obj->lex0==NULL){
		s = obj->lex0 = strdup(s);
		free(obj->lex1);
		obj->lex1=NULL;
	}else{
		s = obj->lex1 = strdup(s);
		free(obj->lex0);
		obj->lex0=NULL;
	}
	YP(set_input)(obj->pvm, s, ssz, stopflg);
	rc = YP(scan)(obj->pvm, obj);		//ag2 == %params int* final_result
idbg(s,rc,_YY_LEXICAL_ERROR,_YY_SYNTAX_ERROR,_YY_INTERNAL_ERROR,_YY_FEED_ME,_YY_FINISH,
"lex, stx, inn, feed, fin");
	switch(rc){
		case YP(_,LEXICAL_ERROR	): rc= ERR_TKN; seterr(obj, "bad token", rc); break;
		case YP(_,SYNTAX_ERROR	): rc= ERR_STX; seterr(obj, "bad syntax", rc); break;
		case YP(_,INTERNAL_ERROR): rc= ERR_INN; seterr(obj, "fatal inner err", rc); break;
		case YP(_,OVERFLOW		): rc= ERR_INN; seterr(obj, "fatal nest err", rc); break;
		case YP(_,NO_MEMORY		): rc= ERR_INN; seterr(obj, "fatal mem err", rc); break;

		case YP(_,FEED_ME		): rc = 1; break;
		case YP(_,FINISH		): rc = 0; break;
		default : STOP(1, "dead code"); break;
	}
	obj->rc = rc;
	return rc;
	//rc==1 feed, 0:end -1:err
}


twl_vt twl_get(twl_t* obj, const char* key){
	int sz = strlen(key)+10;	//dhh + \0
	char* p = calloc(sz, sizeof(char));
	twl_vt rj = {0};
	rj.t = "";
	rj.s = "(nostr)";
	rj.date = "(nodate)";
	if(0){
		; lb_RTN:;
		free(p);
		return rj;
	}
	hfl_nt* nd = NULL;
	//
	char* k = p+3;	//___t=abc
	*k = 't';
	*(k+1) = '=';
	strcpy(k+2, key);
	//
	nd = hfl_find(obj->hdb, k);
	if(nd==NULL){ goto lb_RTN;}
	//tp
	if( *(nd->s)=='n') {
		rj.t = "n";
		*k = 'n';
		nd = hfl_find(obj->hdb, k);
		rj.k = nd->key +2;
		rj.n = nd->d;
		goto lb_RTN;
	}
	else if( *(nd->s)=='s') {
		rj.t = "s";
		*k = 's';
		nd = hfl_find(obj->hdb, k);
		rj.k = nd->key +2;
		rj.s = nd->s;
		*k = 'z';
		nd = hfl_find(obj->hdb, k);
		rj.ssz = nd->d;
		goto lb_RTN;
	}
	else if( *(nd->s)=='d') {
		//d単体はstrなので先に始末
		rj.t = "d";
		*k = 'd';
		nd = hfl_find(obj->hdb, k);
		rj.k = nd->key +2;
		rj.date = nd->s;
		//
		const char* pfix[] =
		{ "dy"  ,"dm"  ,"dd"  ,"dhh"  ,"dmm"  ,"dss"  ,"dz"  ,"dzh"  ,"dzm", NULL };
		double* dst[] =
		{ &rj.y ,&rj.m ,&rj.d ,&rj.hh ,&rj.mm ,&rj.ss ,&rj.z ,&rj.zh ,&rj.zm, NULL};
		
		for(int i=0;pfix[i];i++){
			char* sp = (char*)pfix[i];
			int sz = strlen(sp);
			char* dp = k - sz +1;
			memcpy(dp, sp, sz);
			nd = hfl_find(obj->hdb, dp);
			*(dst[i]) = nd->d;
		}
		goto lb_RTN;
	}
	; STOP(1, "unreachable code");
	return rj;
}








// ~$ carburetta src.cbrt --c
// ~$ carburetta src.cbrt --c src.c
// ~$ carburetta src.cbrt --c src.c --h src.h
//etc

/*SH_CBRTE*/

/*SH_SMP
 #include "twl.h"

 int main(int argc, char** argv){
     twl_t* obj = twl_new();
     twl_pushstr(obj, "[hw] \n twl = 10.1");
     twl_pushstr(obj, NULL);	//EOF
     
     twl_vt val = twl_get(obj, "hw.twl");
     printf("%s, %f\n", val.k, val.n);
     twl_free(obj);
     return 0;
 }
 //~$ cc hw.c libtwl.a -Wall -pedantic -std=c99 -D_POSIX_C_SOURCE=200112
//SH_SMPE*/


/*
 change log
 --
2022-07-12	Momi-g	<dmy@dmy.dmy>

	* twl.sh.c(all): release pkg. v1.0.2

2022-06-29	Momi-g	<dmy@dmy.dmy>

	* twl.sh.c(funcs): add static toolfuncs, twl_svcnt() etc

2022-06-17	Momi-g	<dmy@dmy.dmy>

	* twl.sh.c: init. v1.0.0

*/

