/* Copyright (C) 2022 Momi-g

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

/*SH_doc
title=twl section=3 repnl=\040
@name twl
@_brief toml-like data format and parser
@_syno
 typedef struct twl_tag {
 	int rc;
 	char* emsg;
 	double cnt;
 	//..opaque data
 } twl_t;
 
 typedef struct twlval_tag {
 	const char* k;	//key
 	const char* t;	//3 types, "n"um/"s"tr/"d"ate
 	double n;		//num val
 	const char* s;	//str val
 	const char* date;	//date tostr: rfc3339 format
 	double y,m,d,hh,mm,ss,z,zh,zm; //date: z==zone 0/1
 } twl_vt;
 
 //--read twl format file/str
 twl_t* twl_new();
 int twl_pushfile(twl_t* obj, const char* flname);
 int twl_pushstr(twl_t* obj, const char* s [, int ssz] );
 twl_t* twl_clear(twl_t* obj);
 void* twl_free(twl_t* obj);
 int twl_dbgout(twl_t* obj, FILE* dbgfp);

 //--getter
 twl_vt twl_get(twl_t* obj, const char* id);
 twl_foreach(const char* key, twl_vt val, twl_t* obj){..}	//macro
@tl_dr
		@(code)--@
 #include "twl.h"

 int main(int argc, char** argv){
     twl_t* obj = twl_new();
     twl_pushstr(obj, "[hw] \n twl = 10.1");
     twl_pushstr(obj, NULL);	//EOF
     
     twl_vt val = twl_get(obj, "hw.twl");
     printf("%s, %f\n", val.k, val.n);
     twl_free(obj);
     return 0;
 }
 //~$ cc hw.c libtwl.a -Wall -pedantic -std=c99 -D_POSIX_C_SOURCE=200112
 //see EXSAMPLE for more detail code
	@()--
 
@params
	@(list)
	flname: (char*) read file if set. "stdin" reads FILE* stdin
	obj: (twl_t*) core obj. holds parser, hash map etc
	s: (char*) send str/bin to twl push-parser. send EOF if set NULL
	ssz: (int) send str/bin size if set. use strlen(s) if noset/-1
	id: (char*) keyname for search
	key:(char*) twl_foreach() ID for liner search
	val:(twl_vt) twl_foreach() ID holding data
	dbgfp: (FILE*) fp for output dbginfo. noout if set NULL(==dfl)
	@()
@return
	@(list)
	obj: twl_new() rtns core. rtn NULL if failed. obj->cnt holds keycnt.
	rc : twl_pushfile() rtns int 0/rc<0 --
		twl_pushstr() rtns int 0/1/rc<0 == complete/morestr/err --
		err == ERR_TKN / ERR_STX / ERR_INN and emsg is 'obj->emsg'.
		'obj->rc' holds the same int.
	val: all data is double/char*. all data is init with 0/"". --
		never fails *val.t =='n', puts(val.date) etc  --
		--
		val.k @: char*, ID/keyname, "aa.bb.c" etc --
		val.t @: char*, valtype. "n/s/d" or "", num/str/date --
		--
		val.n @: dbl, val.t=="n"  data.  0.11,  12.00 etc --
		val.s @: char*, val.t=="s" data. "hw" etc --
		val.date @: char*, val.t=="d" tostr data. follows rfc3339 --
		--
		val.ssz@: dbl, val.s size. val.s can holds binary, \0 etc. --
		val.y/m/d/hh/mm/ss/z/zh/zm@: dbl, 'z'one is 0/1 == noset/set --
	@()

@error
	twl_pushfile/str() may rtns rc<0.
	@(code)--@
	ERR_TKN: lex err
	ERR_STX: yacc err
	ERR_INN: other inner err
	@()--@
	'obj->emsg' may holding detail msgstr. --
	exit() if fatal err.

@_desc
	twl is alt-toml data format. omit unnecessary syntax from toml.
	the differences is the below. see NOTES for details:
		@(code)--@
	- no nest with rval: deep nest causes troubles.
	- lowercase ID only: ok: ab.xy=100  NG: Ab.xy=100
	- no quote lval    : ID is no quoting alnum+[._] as c-lang
	- no table array   : silver bullet for javascript specific
	- add mulitline cmt: #*..*# etc
	- add parallel set : a,b = 10,20 >> a=10; b=10
		@()--@
	twl format target is:
		@(code)--@
	- non-programmer can use with low learning cost
	- a few syntax rule
	- pay cost if you need to send complex data (or fix data structure)
	- respect to: ini, shell, c-lang, utf8
	- avoid to  : XML, XSLT, python, c++, lisp, markdown
		@()--@
	
	see samples for twl data read api fundtions. input data is saved to hashtb.
	all value is 'const char* / double' except rc, obj->rc (int).
	char* data is init with blank str "", so never fails to put(val.s) etc.
@_eg
	@(code)--@
 //--data.twl
 # cmtline
 [ab]
 a=10	#ab.a=10
 [aa.bb]
 b="hw"	#aa.bb.b="hw"

 //--src.c
 #include "twl.h"
 
 int main(int argc, char** argv){
 	int rc=0;
 	twl_t* obj;
 	twl_vt val;

 	// read from file
 	if(0){
 	obj = twl_new();
 	; rc = twl_dbgout(obj, stderr);  //dispinfo. rc<0 >> badfp+nochange
 	rc = twl_pushfile(obj, "data.twl");	//suc/fail == 0/rc<0
 	; rc = twl_dbgout(obj, NULL);	// noout(==dfl)
 	val = twl_get(obj, "aa.bb.b");
 	; if( *val.t == 0 ){ puts("bad key"); return 1; }
 	printf("%s, %s, %f\n", val.k, val.s, obj->cnt); //key,val,cntkey
 	twl_free(obj);
 	}

 	//read using push-style parsing
 	obj = twl_new();
 	const char* s = "[ab]\n a = 10 ; d = 2000-01-01";
 	rc = twl_pushstr(obj, s);	//rc=1/0 (more/complete) err<0
 	// (obj,s, -1) >> use -1 == strlen(s) if ag3 isnt
 	; if(rc<0){ puts(obj->emsg); return rc; }
 	rc = twl_pushstr(obj, NULL);	//NULL >> send EOF
 
 	val = twl_get(obj, "ab.a");
 	printf("%s, %f\n", val.t, val.n);	// "n", 10.0
 	
 	twl_foreach(k, v, obj){
 		puts(k);	//== v.k, char* key/id, "ab.a" etc
 		if( strcmp(v.t, "n")==0 ){ printf("num: %f\n", v.n); }
 		if( *v.t == 's' ){ printf("str : %s\n", v.s); }
 		if( *v.t == 'd' ){ printf("date: %s\n", v.date); }
 	}
 	obj = twl_clear(obj); //read other file/str if needs
 	printf("%f\n", obj->cnt);	//==0, obj->cnt == holding keycnt
 	
 	twl_free(obj);
 	return 0;
 }
 //$ cc src.c libtwl.a -Wall -pedantic -std=c99 -D_POSIX_C_SOURCE=200112
		@()--

@notes
		@(code)--@
	---bench mark: 
	code: a[n]=num / twl_pushstr(obj, "key=10 ") etc
	
	set : 8.3ms <<<< 8842ms
	FAST: a[n] (1) <<<< twl_set (1000) :SLOW
	
	get : 14.4ms << 216.8ms 
	FAST: a[n] (1)  <<  twl_get (15)   :SLOW

	-O0
	real	332.267 ms  : ./twl.tmp.c 4674: msg:sprintf() ovh 1000*1000
	real	341.507 ms  : ./twl.tmp.c 4686: msg:a[n] set 1000*1000
	real	14.469 ms   : ./twl.tmp.c 4693: msg:a[n] get 1000*1000
	real	14608.907 ms: ./twl.tmp.c 4702: msg:twl_set 1000*1000
	real	449.135 ms  : ./twl.tmp.c 4712: msg:twl_get 1000*1000
	-O3
	real	356.416 ms  : ./twl.tmp.c 4674: msg:sprintf() ovh 1000*1000
	real	341.656 ms  : ./twl.tmp.c 4686: msg:a[n] set 1000*1000
	real	0.001 ms    : ./twl.tmp.c 4693: msg:a[n] get 1000*1000
	real	8842.322 ms : ./twl.tmp.c 4702: msg:twl_set 1000*1000
	real	216.870 ms  : ./twl.tmp.c 4712: msg:twl_get 1000*1000
	--- 
		@()--@
	@(code)--@
	--- twl syntax info
# twl syntax. linecmt is '#' or '//', allow not BOS
#* multiline cmt is #*...* # or / *..* /, toml >> #...(nl) only
 - preprocess \
   \+(nl) marges lines then tokenize. same as c-lang
 - newline: NL is '\n' or '\r\n'
 - utf8   : file format allows only utf8 and (nl)(EOF)
 - ab.x_y : ID(left hand val) is alnum + dot.ul_
 - Ab.xy  : NG. ID allows only lowercase
 - valtype: VAL(right hand val) is 3types: floating num, str, date
 - [ab.xy]: LABEL [..] add prefix name to ID
 
 - a=1;a=2: (twl_ext) 2nd assign overwrites the before:  >> a==2
 - sep ;  : (twl_ext) semicolon ';' is ignored (or works as token sep)
 - a,b=1,2: (twl_ext) parallel assign: >> a=1 b=2
 - 1_0   : (twl_ext) NUM ignores '_' except token head: >> a=10
 - #_twl: : (twl_ext) magic cmt token '#_twl:' is reserved
*#

#_twl:v1.0.0 //file syntax is for v1.0.0 (currently magic is only this)
//_twl:v1.0.0 //this is not magic

[num]	#cmt
n0 = 10     //num.a0 = 10.0 (floating num only)
n0 = 10.1   //num.a0 = 10.1 (twl:overwrite toml:error)
n1 = -_.11  //-0.11: token >> del '_' >> conv with strtod() 
n2 =        //allow noset == ignored
n3=1 n4=2   //valid. free format 
n5=1;n6=2   //';' == '\n' in most

[str_bin]
s0 = "\u0000h\0w" //s0 size == 4: treat as utf8 sequences 
s1 = "\100s\ntr"  //"" conv c-lang esc except \xhh. lit can holds raw nl
s2 = '@s
tr'	  //'' holds rawlit. '' cant holds single-quote (same as shell)

#* hex esc \xhh.. is non-portable (c99: 6.4.4.4 Character constants)
 - \ooo is 1-3digits(1byte 0-255) but nolim with \xhh..(1-100digits etc)
 - \xhh.. val depends on endian:  0x0a11 / 0x110a etc
 - use \ooo for binary and \u,\U for i18n charactors
*#

//heredoc
s3 = '''alnum_end	//''' or """, 3 or more quote. '''''' is valid.
stop hdoc with (nl)headID(nl). ID chars is alnum+[._], same as lval.
quotes follows ending ID is valid. (nl)headID'"''"(nl) works as end.
this is help you to adjust syntax highlight in manually.

start with s-quote takes completely rawstr.
set blankline if you want EOS nl "str\n".
 
alnum_end"''

//hdoc with c-lang esc
s4 = """end
	w-quote allows esc\n \u0060 etc. \n
end

s5 = """-end
	"""- or '''- removes leading raw tabs '\t', same as posix-shell '<<-'
\t\t123 ...esc char "\t,\u0009" isnt stripped
end"""

_='''___
	use assign syntax as cmt like shell, :<<'CMT'...CMT
___'''


[date.time.etc]	//date format: follows rfc3339
d.a0 = 0020-01-01 12:00:12.345        //year: more than 4digits
d.a1 = 2000-01-01T12:00:23.456+00:00  //date.time.etc.d.a1 = ...

d.a2 = 2000-01-01    //(twl_ext) date only >> others are init with 0
d.a3 = 12:00:22.3345 //(twl_ext) time only
d.a4 = -20000-01-01T00:00:00.000Z //(twl_ext) sign and over 4digits year

#*(twl_ext)
 lowercase sep 't/z' is invalid. T/(sp)/Z only.
 y,m,d,hh,mm,ss,z,zh,zm holds double(64bit floating point) num
*#

[]	//blank LABEL set no prefix
base.num=10	//(no_prefix)base.num == 10

//(twl_ext) parallel assign (ID list)=(VAL list). ignore uneven ID/VAL
  a,    b,       c,     d
=10.9, "hw", 2000-01-01		//'d' is ignored

a = 10.9, "hw", 2000-01-01	//str, date is ignored

# ---
#*---appendix: not collect BNF, but helps you
//yacc
stmt: expr
    | stmt expr ;

expr: LB
    | lve
    | ass ;

ass : lve rv
    | ass ',' rv ;	//VAL list

lve  : lv '=' ;

lv   : ID
     | lv ',' ID ;	//ID list

rv: num | lit | date ;

num  : NUM ;
lit  : DLIT | SLIT | DDOC | SDOC ;
date : DATE_
     | DATE_ DATET  
     | DATE_ DATET DATEZ  
     | TIME ;	
	
//lex:  marge \(nl) lines with preprocess
NL   : \n|\r\n
LABEL: '[' ID ']'
ID   : ([a-z_][a-z0-9_]*)(\.[a-z_][a-z0-9_]*)*
NUM  : [-+.0-9][_xXoO0-9a-fA-F.lL]*   //del '_' and suc to conv strtod()

DLIT : ["]([^"]|\\.)*["}		//conv esc
SLIT : [']([^'])*[']
DDOC : """+\(ID\).* NL \1 ["']* NL     //BRE+shortest. conv esc
SDOC : '''+\(ID\).* NL \1 ["']* NL

DATE_: [-+]?[0-9][0-9][0-9][0-9]+-[0-1][0-9]-[0-3][0-9]
DATET: [T ][0-2][0-9]:[0-6][0-9]:[0-6][0-9](\.[0-9]+)?
DATEZ: [Z]|[\-+][0-2][0-9]\:[0-6][0-9]
TIME: [0-2][0-9]:[0-6][0-9]:[0-6][0-9](\.[0-9]+)?

 1979-05-27T07:32:00Z
 1979-05-27T00:32:00-07:00
 1979-05-27T00:32:00.999999-07:00
 1979-05-27 07:32:00.1234+09:00

 2000-05-27 07:32:00 
 2000-01-01
 12:00:00
 12:00:00.12345

*#
	@()--@
@conforming_to POSIX.1-2001+ (-D_POSIX_C_SOURCE=200112L etc)
@COPYRIGHT Copyright 2022 momi-g, GPLv3+
@_ver 2022-07-12 v1.0.2 (2022-06-13 v1.0.0)
@_see https://en.wikipedia.org/wiki/Configuration_file
//SH_docE*/

#ifndef twl_61a25e678cdd
#define twl_61a25e678cdd

#include <features.h> 	//SH_co* -D_XOPEN_SOURCE=600 -std=c99 */
#if ( _POSIX_C_SOURCE +0 < 200112L )
	#include "stop cc: needs compiler posix-2001 or upper(c99+)"
#endif

#include <stdio.h>
#include <string.h>
/*--copyfrom hfl.h*/
/* Copyright (C) 2022 Momi-g

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

// this code is rewrite of flat_hash_map(cpp boost license) for c-lang.
// https://github.com/skarupke/flat_hash_map/blob/master/flat_hash_map.hpp

/*SH_doc
title=hfl section=3 repnl=\040

@name hfl
@_brief hashtb api for C-lang
@_syno
 typedef struct hfl_node_tag {
 	int tp;		// 'i'nt, 'd'bl, 'p'tr, 's'tr(or 's'truct)
 	char* key;
 	int i;
 	double d;
 	void* p;
 	char* s;
 	int ssz;
 	//..opaque data
 } hfl_nt;
 
 typedef struct hfl_tag{
 	uint32_t cnt;
 	//..opaque data
 } hfl_t;
 
 hfl_t*	hfl_new([uint32_t sz]);
 void*	hfl_free(hfl_t* obj);
 hfl_nt*	hfl_unset(hfl_t* obj, char* key);
 hfl_nt*	hfl_find(hfl_t* obj, char* key);
 hfl_t*	hfl_clear(hfl_t* obj);
 
 #define	hfl_set_i(obj,k,v)	
 #define	hfl_set_d(obj,k,v)
 #define	hfl_set_p(obj,k,v)
 #define	hfl_set_s(obj,k,ptr,[sz])
 #define	hfl_foreach(node, obj)
	
@tl_dr
		@(code)
 #include "hfl.h"
 
 int main(int argc, char** argv){
 	hfl_t* obj = hfl_new();
 	hfl_nt* np = hfl_set_i(obj, "k1", 12);
 	puts(rp->key);	// "k1"
 	int tp = rp->tp;	// int == 'i'
 	hfl_unset(obj, "k1");	//del
 
 	np = hfl_set_s(obj, "k2", "hw");
 	puts(np->s);	// "hw"
 	np = hfl_find(obj, "k2");
 	puts(np->s);	// "hw", np->ssz == 0 (nochange, ag4 isnt)

 	hfl_set_s(obj, "k3", "hi\0gw", 5);	//ag4 isnt/is == strdup/malloc
 	np = hfl_find(obj, "k3");
 	strncmp(np->s, "hi\0gw", np->ssz);	// == 0, np->ssz==5
 	
 	hfl_foreach(p, obj){
 		puts(p->key);
 		hfl_set_i(obj, p->key, 2);	//ok, replace/update
 		hfl_unset(obj, p->key);	//ok
 		// hfl_set_i(obj, "new", 2);	//NG, add newkey
 		p->i = 10;	// allow direct set, not recommend
 	}
 	
 	np = hfl_unset(obj, "k2");
 	np = hfl_find(obj, "k2");	//np == NULL;
 	int cnt = obj->cnt;	//holding keys
 	
 	obj = hfl_clear(obj);	//del allkeys, obj->cnt == 0
 	
 	hfl_free(obj);
 	return 0;
 }
 //~$ cc src.c libhfl.a
		@()
@_desc
	c99 hsearch() allows only 1 hashtb in 1 pg. 
	hfl allows to hold many hashtb.
		@(code)
	--- bench mark: c_arr, c99hash, hfl, gtkhash
	  FAST: c_arr(1) < c99hash == hfl == ghash (20) :SLOW
		@()
	consume time of get/set is almost the same. --
	--
	hfl_set_s(obj, key, vptr, [memsz]) saves mem using strdup()/malloc().
	use malloc() if ag4 exist and not 0. buffmem is freed when hfl_free().

@_eg
	-
@return_value
	suc/fail == ptr, int 0 / NULL, int not0
@errors
	output emsg and exit(1) if fatal err
@notes -
@conforming_to POSIX.1-2001+ (-D_XOPEN_SOURCE=600/_POSIX_C_SOURCE=200112L)
@COPYRIGHT Copyright 2022 momi-g, GPLv3+
@_ver 2022-07-11 v1.0.6 (2022-05-29 v1.0.0)

@_see
https://probablydance.com/2017/02/26/i-wrote-the-fastest-hashtable/ --
https://github.com/skarupke/flat_hash_map --
//SH_docE*/
#ifndef hfl_61a25e678cdd
#define hfl_61a25e678cdd

#include <features.h> 	//SH_co* -D_XOPEN_SOURCE=600 -std=c99 */
#if ( _POSIX_C_SOURCE +0 < 200112L )
	#include "stop cc: needs compiler posix-2001 or upper(c99+)"
#endif

#ifdef hfl_p_
# include <emsg: detect conflict varname hfl.h/'_hfl_p' >
#endif

#include <stdio.h>
#include <string.h>
#include <stdint.h>
typedef struct hfl_node_tag {
	int tp;
	char* key;
	int i;
	double d;
	void* p;
	char* s;
	int ssz;
	//
	uint32_t slot;
	struct hfl_node_tag* pre;
	struct hfl_node_tag* next;
} hfl_nt;

typedef uint32_t (s2u_ft)(const char* key);

typedef struct hfl_tag{
	uint32_t cnt;
	//
	uint32_t mcnt;
	uint32_t lcnt;
	s2u_ft* hashcb;
	hfl_nt** arr;
	hfl_nt* enode;
} hfl_t;

#define hfl_foreach(node, obj)	for(hfl_nt* (node)=obj->enode->pre, *hfl_p_; \
	 hfl_p_=(node)->pre, (node) != obj->enode; (node)=hfl_p_ )

#define hfl_set_i(obj,k,v)	hfl_set_cmn(obj,k,'i',(int)(v), 0)
#define hfl_set_d(obj,k,v)	hfl_set_cmn(obj,k,'d',(double)(v), 0)
#define hfl_set_p(obj,k,v)	hfl_set_cmn(obj,k,'p',(void*)(v), 0)

#define hfl_set_s(obj,k,...)	hfl_set_s_(obj,k, __VA_ARGS__, -1, -1)
#define hfl_set_s_(obj,k,v,sz,...)	hfl_set_s__(obj,k,v,sz)
#define hfl_set_s__(obj,k,v,sz)	hfl_set_cmn(obj,k,'s',(char*)(v), (int)(sz))

#define hfl_new(...)	hfl_new_impl(__VA_ARGS__ +0)
hfl_t* hfl_new_impl(uint32_t sz);
void*	hfl_free(hfl_t* obj);
hfl_nt* hfl_set_cmn(hfl_t* obj, char* k, int tp, ...);
hfl_nt* hfl_unset(hfl_t* obj, char* k);
hfl_nt* hfl_find(hfl_t* obj, char* k);
hfl_t* hfl_clear(hfl_t* obj);

//hpp
#ifdef _IMPL_hfl		/*SH_c o* -D_IMPL_hfl	*/
/*--copyfrom hfl.c*/
/* Copyright (C) 2022 Momi-g

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

// this code is rewrite of flat_hash_map(cpp boost license) for c-lang.
// https://github.com/skarupke/flat_hash_map/blob/master/flat_hash_map.hpp

/*SH_doc
title=hfl section=3 repnl=\040

@name hfl
@_brief hashtb api for C-lang
@_syno
 typedef struct hfl_node_tag {
 	int tp;		// 'i'nt, 'd'bl, 'p'tr, 's'tr(or 's'truct)
 	char* key;
 	int i;
 	double d;
 	void* p;
 	char* s;
 	int ssz;
 	//..opaque data
 } hfl_nt;
 
 typedef struct hfl_tag{
 	uint32_t cnt;
 	//..opaque data
 } hfl_t;
 
 hfl_t*	hfl_new([uint32_t sz]);
 void*	hfl_free(hfl_t* obj);
 hfl_nt*	hfl_unset(hfl_t* obj, char* key);
 hfl_nt*	hfl_find(hfl_t* obj, char* key);
 hfl_t*	hfl_clear(hfl_t* obj);
 
 #define	hfl_set_i(obj,k,v)	
 #define	hfl_set_d(obj,k,v)
 #define	hfl_set_p(obj,k,v)
 #define	hfl_set_s(obj,k,ptr,[sz])
 #define	hfl_foreach(node, obj)
	
@tl_dr
		@(code)
 #include "hfl.h"
 
 int main(int argc, char** argv){
 	hfl_t* obj = hfl_new();
 	hfl_nt* np = hfl_set_i(obj, "k1", 12);
 	puts(rp->key);	// "k1"
 	int tp = rp->tp;	// int == 'i'
 	hfl_unset(obj, "k1");	//del
 
 	np = hfl_set_s(obj, "k2", "hw");
 	puts(np->s);	// "hw"
 	np = hfl_find(obj, "k2");
 	puts(np->s);	// "hw", np->ssz == 0 (nochange, ag4 isnt)

 	hfl_set_s(obj, "k3", "hi\0gw", 5);	//ag4 isnt/is == strdup/malloc
 	np = hfl_find(obj, "k3");
 	strncmp(np->s, "hi\0gw", np->ssz);	// == 0, np->ssz==5
 	
 	hfl_foreach(p, obj){
 		puts(p->key);
 		hfl_set_i(obj, p->key, 2);	//ok, replace/update
 		hfl_unset(obj, p->key);	//ok
 		// hfl_set_i(obj, "new", 2);	//NG, add newkey
 		p->i = 10;	// allow direct set, not recommend
 	}
 	
 	np = hfl_unset(obj, "k2");
 	np = hfl_find(obj, "k2");	//np == NULL;
 	int cnt = obj->cnt;	//holding keys
 	
 	obj = hfl_clear(obj);	//del allkeys, obj->cnt == 0
 	
 	hfl_free(obj);
 	return 0;
 }
 //~$ cc src.c libhfl.a
		@()
@_desc
	c99 hsearch() allows only 1 hashtb in 1 pg. 
	hfl allows to hold many hashtb.
		@(code)
	--- bench mark: c_arr, c99hash, hfl, gtkhash
	  FAST: c_arr(1) < c99hash == hfl == ghash (20) :SLOW
		@()
	consume time of get/set is almost the same. --
	--
	hfl_set_s(obj, key, vptr, [memsz]) saves mem using strdup()/malloc().
	use malloc() if ag4 exist and not 0. buffmem is freed when hfl_free().

@_eg
	-
@return_value
	suc/fail == ptr, int 0 / NULL, int not0
@errors
	output emsg and exit(1) if fatal err
@notes -
@conforming_to POSIX.1-2001+ (-D_XOPEN_SOURCE=600/_POSIX_C_SOURCE=200112L)
@COPYRIGHT Copyright 2022 momi-g, GPLv3+
@_ver 2022-07-11 v1.0.6 (2022-05-29 v1.0.0)

@_see
https://probablydance.com/2017/02/26/i-wrote-the-fastest-hashtable/ --
https://github.com/skarupke/flat_hash_map --
//SH_docE*/
/* tool macros */
#ifndef ERRact
#include <stdio.h>	//SH_co* -D_XOPEN_SOURCE=600 -std=c99 */
 #if (199901L <= __STDC_VERSION__ +0)	/* nealy 200112L, _POSIX_C_SOURCE	c99*/
	#include <sys/types.h>
	#include <unistd.h>
	#define ERRactag	__func__, getpid()
 #else
	#define ERRactag	"func:c99+", 0
 #endif
 #include <string.h>
 #include <errno.h>
 #define ERRact(xpr, msg, act)	if(xpr){ int en_=errno; fprintf(stderr, \
	"ERR: %s %d %s() pid:%d %s msg:%s sys:%s\n",__FILE__,__LINE__, ERRactag \
	, "hit(" #xpr ")", msg, strerror(en_) ); act; }
 #define STOP(xpr, msg)	ERRact(xpr, msg, fputs("STOP\n",stderr);exit(1) )
#endif
#define loop(a)		for(int lpcnt=1;lpcnt<=a;lpcnt++)
/*tool end*/

#include <stdlib.h>
#include <string.h>
#include "hfl.h"

#ifndef HAVE_CONFIG_H
# include "msgp.h"		/*SH_co* -D_IMPL_msgp	*/
#endif



//https://softwareengineering.stackexchange.com/questions/49550
#ifndef HFL_HASHCB
# define HFL_HASHCB			hash_fnv1a
#endif

typedef uint32_t uint;
// <sys/types.h> << omitted uint,  from c99 >> c11
// https://pubs.opengroup.org/onlinepubs/9699919799/

static uint hash_djb2(const char *str);
static uint hash_fnv1a(const char *str);

static uint hfl_nextsz(uint n);
static int hfl_delnode(hfl_t* obj, hfl_nt* p);
static hfl_t* hfl_renew(hfl_t* obj);
static hfl_nt* hfl_insnode(hfl_t* obj, char* key, hfl_nt* p);
static uint getslot(s2u_ft hashcb, uint hashsz, char* key);

//djb2: http://www.cse.yorku.ca/~oz/hash.html
static uint32_t hash_djb2(const char *str){
	unsigned char* s = (unsigned char*)str;
	uint32_t hash = 5381;	//init vec
	//hash * 33 + c
	for(; *s; s++){	hash = ((hash << 5) + hash) + *s; }
	return hash;
}



//fnv-1a: http://www.isthe.com/chongo/tech/comp/fnv/index.html#FNV-param
#define FNV32_INIT ((uint32_t)0x811c9dc5)
#define FNV32_PRIME	((uint32_t)0x01000193)
static uint32_t hash_fnv1a(const char *str){
	unsigned char* s = (unsigned char*)str;
	uint32_t h = FNV32_INIT;	//init vec
	for(; *s; s++) { h ^= *s; h *= FNV32_PRIME; }
    return h;
}


//https://stackoverflow.com/questions/14069737/
//only 6:  const int, enum, charint 'c', sizeof(), _Alignof(), (int)float/dbl
enum prime_etag {
 P0 =1			,P1 =2			,P2 =3			,P3 =7			,P4 =13
,P5 =31			,P6 =61			,P7 =127		,P8 =251		,P9 =509
,P10=1021		,P11=2039		,P12=4093		,P13=8191		,P14=16381
,P15=32749		,P16=65521		,P17=131071		,P18=262139		,P19=524287
,P20=1048573	,P21=2097143	,P22=4194301	,P23=8388593	,P24=16777213
,P25=33554393	,P26=67108859	,P27=134217689	,P28=268435399	,P29=536870909
,P30=1073741789	,P31=2147483647
};

static uint prime_arr[32+1] = {
 P0  ,P1  ,P2  ,P3	,P4		,P5  ,P6  ,P7  ,P8  ,P9  
,P10 ,P11 ,P12 ,P13 ,P14	,P15 ,P16 ,P17 ,P18 ,P19
,P20 ,P21 ,P22 ,P23 ,P24	,P25 ,P26 ,P27 ,P28 ,P29
,P30 ,P31 ,0
};
#ifndef HFL_SLOTDFL
# define HFL_SLOTDFL	31
#endif

#ifndef HFL_SLOTMAX
# define HFL_SLOTMAX 31
#endif
static uint hfl_nextsz(uint n){
	uint* arr = prime_arr;
	uint lim = prime_arr[HFL_SLOTMAX];
	lim < HFL_SLOTDFL ? lim=HFL_SLOTDFL: lim;
	for(int i=0;; i++){
		uint nn = arr[i];
		if( nn!=0 && nn<=lim && nn<=n ){ continue; }
		n = nn;
		break;
	}
	return n;
}

#ifndef HFL_RATIO
# define HFL_RATIO	0.8
#endif

hfl_t* hfl_new_impl(uint sz){
	//if(sz==HFL_SLOTDFL){sz=HFL_SLOTDFL; }
	if(sz==0){sz=HFL_SLOTDFL; }
	hfl_t* obj = calloc(1, sizeof(hfl_t) );
	obj->hashcb = HFL_HASHCB;
	obj->cnt = 0;
	obj->mcnt = sz;
	obj->lcnt = sz * HFL_RATIO;
	obj->arr  = calloc(sz, sizeof(hfl_nt*) );
	obj->enode= calloc(1, sizeof(hfl_nt) );
	obj->enode->pre = obj->enode;
	obj->enode->next = obj->enode;
	
//idx系の管理になるのでf_rqからとってくる。arr[0]で固定位置
	return obj;

//deadcode
	hash_djb2("");
}
void* hfl_free(hfl_t* obj){
	hfl_nt* p = obj->enode;
	if(p){
		void* sv = p->pre;
		for(;;){
			p = sv;
			if(p == obj->enode){break;}
			free(p->key);
			free(p->s);
			sv=p->pre;
			free(p);
		}
	}
	free(obj->arr);
	free(obj->enode);
	free(obj);
	hfl_t buf ={0};
	*obj = buf;
	return obj;
}

// ISO99
// #define strdup(a)	(memcpy(malloc(strlen(a)+1),a ,strlen(a)+1))
static
hfl_nt* hfl_insnode(hfl_t* obj, char* key, hfl_nt* p){
	if(obj==NULL|| key==NULL){ return NULL; }
	//編集するのはpのみ。xp,ppはobjのデータなのでp->系ではない
	if(obj->cnt > obj->lcnt){ hfl_renew(obj); }
	if(p==NULL){
		p = calloc(1, sizeof(hfl_nt) );
		p->key = strdup(key);
	}
	uint pos = getslot(obj->hashcb, obj->mcnt, key);
	hfl_nt* pp = obj->arr[pos];
	hfl_nt* xp;
	if(pp==NULL){
		pp = obj->enode->pre;
		xp = obj->enode;
	}
	else { xp = pp->next; }
	
	//insnode
	p->slot = pos;
	p->pre = pp;	//arr先頭に挿入。initはtailにセットされてる
	p->next = xp;
	
	pp->next = p;
	xp->pre = p;
	
	obj->arr[pos] = p;
	obj->cnt++;
	return p;
}

static
hfl_t* hfl_renew(hfl_t* obj){
	if(obj==NULL){ return hfl_new(); }
	uint nsz = hfl_nextsz(obj->mcnt);
	if(nsz==0){
		obj->lcnt = obj->mcnt;
		return NULL;
	}
	
	hfl_t* nobj = hfl_new(nsz);
	uint ck=0;
//トリッキーだけどforは降順なのでxdiは見ない。逆方向にメモって修正すればいけそう
	hfl_foreach(p, obj){
		hfl_insnode(nobj, p->key, p);
		ck++;
	}
	STOP(ck != obj->cnt, "cnt unmatch");

	//rep
	free(obj->arr);
	free(obj->enode);
	*obj = *nobj;
	
	nobj->arr = NULL;
	nobj->enode =NULL;
	hfl_free(nobj);
	return obj;
}


static int hfl_delnode(hfl_t* obj, hfl_nt* p){
	STOP(p==obj->enode, "try to del enode");

	hfl_nt*	pp = p->pre;
	hfl_nt* xp = p->next;

	xp->pre = pp;
	pp->next = xp;
	
	//slot
	uint pos = p->slot;
	if(obj->arr[pos] == p){
		if(pp->slot==pos){ obj->arr[pos] = pp; }
		else{ obj->arr[pos] = NULL; }
	}
	free(p->key);
	free(p->s);
	hfl_nt buf = {0};
	*p = buf;
	free(p);
	obj->cnt--;
	return 0;
}

static uint getslot(s2u_ft hashcb, uint hsz, char* key){
	uint idx = hashcb(key);
	// cc optimize: mod Strength reduction, case flow control
	// https://stackoverflow.com/questions/14069737/
	switch(hsz){
		case P0 : idx %= P0 ; break;
		case P1 : idx %= P1 ; break;
		case P2 : idx %= P2 ; break;
		case P3 : idx %= P3 ; break;
		case P4 : idx %= P4 ; break;
		case P5 : idx %= P5 ; break;
		case P6 : idx %= P6 ; break;
		case P7 : idx %= P7 ; break;
		case P8 : idx %= P8 ; break;
		case P9 : idx %= P9 ; break;
		case P10: idx %= P10; break;
		case P11: idx %= P11; break;
		case P12: idx %= P12; break;
		case P13: idx %= P13; break;
		case P14: idx %= P14; break;
		case P15: idx %= P15; break;
		case P16: idx %= P16; break;
		case P17: idx %= P17; break;
		case P18: idx %= P18; break;
		case P19: idx %= P19; break;
		case P20: idx %= P20; break;
		case P21: idx %= P21; break;
		case P22: idx %= P22; break;
		case P23: idx %= P23; break;
		case P24: idx %= P24; break;
		case P25: idx %= P25; break;
		case P26: idx %= P26; break;
		case P27: idx %= P27; break;
		case P28: idx %= P28; break;
		case P29: idx %= P29; break;
		case P30: idx %= P30; break;
		case P31: idx %= P31; break;
		default : idx %= hsz; break;
	}
	return idx;
}

#include <stdarg.h>
hfl_nt* hfl_set_cmn(hfl_t* obj, char* k, int tp, ...){
	hfl_nt* p = hfl_find(obj, k);
	int rewflg=0;
	p?rewflg=1 :0;
	if(p==NULL){ p = hfl_insnode(obj, k, NULL); }
	if(p==NULL){ return NULL; }
	
	char* yp;
	int yu;
	
	p->tp = tp;
	va_list va;
	va_start(va, tp);
	switch(tp){
		case 'i': p->i = va_arg(va,int); break;
		case 'd': p->d = va_arg(va,double); break;
		case 'p': p->p = va_arg(va,void*); break;
		case 's':
			rewflg? free(p->s),1  :0;
			yp = va_arg(va,char*);
			yu = va_arg(va,int);
			if(yu<0){ p->s = strdup(yp); break; }
			p->s = malloc(yu);
			STOP(p->s == NULL, "mem err");
			memcpy(p->s, yp, yu);
			p->ssz = yu;
			break;
		default: STOP(1, "invalid typechar");
	}
	va_end(va);
	return p;
}

hfl_nt* hfl_unset(hfl_t* obj, char* k){
	hfl_nt*  p = hfl_find(obj, k);
	if(p==NULL){ return NULL; }
	hfl_delnode(obj, p);
	return p;
}


//strcmp()はmemcmp()にくらべて遅い。memはword単位で比較可能
hfl_nt* hfl_find(hfl_t* obj, char* k){
	if(obj==NULL || k==NULL){ return NULL; }
	uint pos = getslot(obj->hashcb, obj->mcnt, k);
	hfl_nt* p = obj->arr[pos];
	if(p==NULL){ return NULL; }
	for(;p;){
		if(p->slot != pos || p==obj->enode){ p=NULL; break; }
		if(strcmp(p->key, k)==0){ break;}
		//if(memcmp(k, np->key, sz)==0){ break;}
		p = p->pre;
	}
	return p;
}


hfl_t* hfl_clear(hfl_t* hdb){
	hfl_foreach(node, hdb){
		hfl_unset(hdb, node->key);
	}
	return hdb;
}










/*SH_SMP
#include <stdio.h>
#include "hfl.h"

int main(int argc, char** argv){
	hfl_t* obj = hfl_new();
	hfl_nt* rp = hfl_set_i(obj, "k1", 12);
	puts(rp->key);	// "k1"
	int tp = rp->tp;	// int == 'i'
	hfl_unset(obj, "k1");	//del

	rp = hfl_set_s(obj, "k2", "hw");
	puts(rp->s);	// "hw"
	rp = hfl_find(obj, "k2");
	puts(rp->s);	// "hw"
	
	float n = 12.3;
	rp = hfl_set_s(obj, "k3", &n, sizeof(float) );
	// ag4 isnt >> strdup()
	// ag4 exist >> malloc()
	n = *(float*)(rp->s);
	
	hfl_foreach(p, obj){
		puts(p->key);
		hfl_set_i(obj, p->key, 2);	//ok, replace/update
		hfl_unset(obj, p->key);	//ok
		// hfl_set_i(obj, "new", 2);	//NG, add newkey
		p->i = 10;	//ok
	}
	
	rp = hfl_unset(obj, "k2");
	rp = hfl_find(obj, "k2");	//rp == NULL;
	int cnt = obj->cnt;	//holding keys
	
	hfl_free(obj);
	return 0;
}
//~$ gcc src.c libhfl.a
//SH_SMPE*/


/*
 change log
 --
2022-07-11 Momi-g	<dmy@dmy.dmy>

	* hfl.sh.c(set_cmn() ): fix memleak, add rewflg for del premem

2022-07-10 Momi-g	<dmy@dmy.dmy>

	* hfl.sh.c(hfl.h): add hpp impl code

2022-07-02	Momi-g	<dmy@dmy.dmy>

	* hfl.sh.c(hfl.h): change type: node->s (void*) >> (char*)

2022-06-28	Momi-g	<dmy@dmy.dmy>

	* hfl.sh.c: fix hfl_set_s() ag3 macro, brp macro

2022-06-17	Momi-g	<dmy@dmy.dmy>

	* hfl.sh.c: add hfl_clear(), fix doc. v1.0.2

2022-06-08	Momi-g	<dmy@dmy.dmy>

	* hfl.sh.c: update, fix doc

2022-05-29	Momi-g	<dmy@dmy.dmy>

	* hfl.sh.c: init.

*/

/*--copyend hfl.c*/
#endif

#endif /* inc_guard */
/*--copyend hfl.h*/

#define TWL_VER "v1.0.0"
#define TWL_MGCMT "#_twl:"
enum {
	  ERR_TKN = -1
	, ERR_STX = -2
	, ERR_INN = -3
};

typedef struct twl_tag{
	int rc;
	char* emsg;
	double cnt;
	hfl_t* hdb;
	void* pvm;

//sv_state
	FILE* svpf;
	int igtab;
	int line;	//hdoc memo
	int clm;
	int tdx;
	int tdxsz;
//
	char* lb;
	int svcnt;
	int ldcnt;
//lexbuff
	void* sepp;
	char* lex0;
	char* lex1;
//yaccbuff
	void* sehd;
	void* setmp;
	void* seyc;
} twl_t;

#define twl_new(...)	twl_new_impl(__VA_ARGS__ +0)
twl_t* twl_new_impl(const char* flname);

#define twl_pushfile(obj, name) twl_pushfile_impl(obj, name)
int twl_pushfile_impl(twl_t* obj, const char* s);

#define twl_pushstr(obj, ...)	twl_pushstr_impl(obj,  __VA_ARGS__, -1)
int twl_pushstr_impl(twl_t* obj, const char* s, int ssz, ...);

twl_t* twl_clear(twl_t* obj);
void* twl_free(twl_t* obj);
int twl_dbgout(twl_t* obj, FILE* fp);

//getter
typedef struct twlval_tag {
	const char* k;
	const char* t;
	double n;
	const char* s;
	double ssz;
	const char* date;	//date rfc3339 format str
	double y,m,d,hh,mm,ss,z,zh,zm; //date: z==zone 0/1
} twl_vt;

twl_vt twl_get(twl_t* obj, const char* key);

#undef twl_foreach
#define twl_foreach(k, v, obj)	for(hfl_nt* twl_nd=obj->hdb->enode->pre, *hfl_p_; \
	 hfl_p_=twl_nd->pre, twl_nd != obj->hdb->enode; twl_nd=hfl_p_ ) \
	for(twl_vt v={0}; strncmp(twl_nd->key, "t=",2)==0 && v.t==NULL && (v=twl_get(obj, twl_nd->key+2), 1);) \
	for(const char* k = twl_nd->key+2; k; k=NULL)
/*
#define hfl_foreach(node, obj)	for(hfl_nt* (node)=obj->enode->pre, *hfl_p_; \
	hfl_p_=(node)->pre, (node) != obj->enode; (node)=hfl_p_ )
*/

#define twl_set_i(obj,k,v)		hfl_set_i(obj->hdb, k, v)
#define twl_set_d(obj,k,v)		hfl_set_d(obj->hdb, k, v)
#define twl_set_p(obj,k,v)		hfl_set_p(obj->hdb, k, v)
#define twl_set_s(obj,k,v)		hfl_set_s(obj->hdb, k, v)
#define twl_unset(obj, k)		hfl_unset(obj->hdb, k)

#endif /* inc_guard: twl */
