/*
 * uClinux for H8/300 gzipped kernel inflate loader
 * 
 * Copyright (C) 2003 Yoshinori Sato
 * $Id: inflate_main.c,v 1.5 2003/07/17 13:30:50 ysato Exp $
 *
 */

#if defined(DEBUG)
#include <stdarg.h>
#include <stdio.h>
#include <zlib.h>
#else
#include <cyg/infra/diag.h>
#include <cyg/compress/zlib.h>
#endif

#if !defined(DEBUG)
extern unsigned char deflate_kernel[];
extern unsigned char kernel_entry[];
extern int           _deflate_kernel_size;
extern int           _inflate_area_size;
extern unsigned char command[];
extern unsigned char linux_command[];
#define DEFLATE_KERNEL_SIZE ((int)(&_deflate_kernel_size))
#define INFLATE_AREA_SIZE ((int)(&_inflate_area__size))
#else
unsigned char *deflate_kernel;
unsigned char *kernel_entry;
#define DEFLATE_KERNEL_SIZE (0x80000)
#define INFLATE_AREA_SIZE (0x180000)
#define DEFLATE_FILENAME "linux.bin.gz"
#define INFLATE_FILENAME "linux.bin"
#endif

#define HEAD_CRC     0x02 /* bit 1 set: header CRC present */
#define EXTRA_FIELD  0x04 /* bit 2 set: extra field present */
#define ORIG_NAME    0x08 /* bit 3 set: original file name present */
#define COMMENT      0x10 /* bit 4 set: file comment present */

#define BLOCK 0x10000

unsigned char *check_deflate_data(unsigned char *data)
{
    const unsigned char gz_magic[]={0x1f,0x8b};
    unsigned char flags;
    int len;
    if((data[0] != gz_magic[0]) || (data[1] != gz_magic[1]))
	return NULL;
    data += 2;
    data++;
    flags = *data++;
    data += 6; /* skip time/xflags/OS code */
    if (flags & EXTRA_FIELD) {
	len = *data++;
	len |= (*data++) << 8;
	data += len;
    }
    if (flags & ORIG_NAME) {
#if defined(VERBOSE)
	diag_printf("filename: %s\n",data);
#endif
	while(*data++); /* skip filename */
    }
    if (flags & COMMENT) {
#if defined(VERBOSE)
	diag_printf("comment: %s\n",data);
#endif
	while(*data++); /* skip comment */
	data++;
    }
    if (flags & HEAD_CRC) {
	data += 2; /* skip crc */
    }
    return data;
}

int inflate_kernel(unsigned char *data)
{
    z_stream z;
    int status;
    int length;

    z.zalloc = Z_NULL;
    z.zfree  = Z_NULL;
    z.opaque = Z_NULL;

    if (inflateInit2(&z, -MAX_WBITS) != Z_OK)
	return 0;

    z.next_in   = data;
    z.avail_in  = DEFLATE_KERNEL_SIZE;
    z.next_out  = kernel_entry;
    z.avail_out = BLOCK;

    length = 0;
    while((status = inflate(&z, Z_NO_FLUSH)) != Z_STREAM_END) {
	if (status != Z_OK) {
	    inflateEnd(&z);  /* infrate error abort */
	    return 0;
	}
#if defined(VERBOSE)
	diag_printf(".");
#endif
	z.avail_out = BLOCK;
	length += BLOCK;
    }
    length += BLOCK - z.avail_out;
    if (inflateEnd(&z) == Z_OK) {
#if defined(VERBOSE)
	diag_printf("\ninflate done %08x - %08x",
                    (unsigned long)kernel_entry,(unsigned long)z.next_out-1);
#endif
	return length;
    } else
	return 0;
}

void start_kernel(int length)
{
#if !defined(DEBUG)
    unsigned char *s,*d;
    s = command;
    d = linux_command;
    while(*d++ = *s++);
#if defined(VERBOSE)
    diag_printf("kernel command: %s\n",linux_command);
#endif
    asm("orc #0x80,ccr\n\t"
	"jmp @_kernel_entry");
#else
    FILE *fp;
    fp = fopen(INFLATE_FILENAME,"wb");
    fwrite(kernel_entry,sizeof(char),length,fp);
    fclose(fp);
#endif
}

#if defined(DEBUG)
void init_work(void)
{
    FILE *fp;
    deflate_kernel = (unsigned char *)malloc(DEFLATE_KERNEL_SIZE);
    kernel_entry   = (unsigned char *)malloc(INFLATE_AREA_SIZE);
    fp = fopen(DEFLATE_FILENAME,"rb");
    fread(deflate_kernel,sizeof(char),DEFLATE_KERNEL_SIZE,fp);
    fclose(fp);
}

void diag_printf(char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);
    vprintf(fmt,ap);
    va_end(ap);
}
#endif

void err(char *msg)
{
#if defined(VERBOSE)
    diag_printf("\ninflate_boot: %s\n",msg);
#endif
#if defined(DEBUG)
    exit(1);
#else
    for(;;);
#endif
}

int main(void)
{
    char *gz_ptr;
    int length;
#if defined(DEBUG)
    init_work();
#endif
#if defined(VERBOSE)
    diag_printf("inflating binary\n");
#endif
    gz_ptr = check_deflate_data(deflate_kernel);
    if(gz_ptr == NULL)
	err("inflate data error");
    length = inflate_kernel(gz_ptr);
    if (length == 0)
	err("inflate error");
#if defined(VERBOSE)
    diag_printf("\ninflate size = %d\n",length);
#endif
    start_kernel(length);
}
