/*
 * uClinux for H8/300 gzipped kernel inflate loader
 * 
 * Copyright (C) 2004 Yoshinori Sato
 * $Id: main.c,v 1.1.1.1 2004/11/21 10:10:09 ysato Exp $
 *
 */

#if defined(DEBUG)
#include <stdio.h>
#else
typedef unsigned int size_t;
#define NULL (0)
#endif

#define HEAP_SIZE 0x10000
#define MAXSIZE 0x180000

typedef unsigned char  uch;
typedef unsigned short ush;
typedef unsigned long  ulg;

#define COMMAND_LEN 512

static unsigned long free_mem_ptr;
static unsigned long free_mem_end_ptr;

static unsigned char *inbuf;

static int  fill_inbuf(void);
static void flush_window(void);
static void error(char *m);
static void gzip_mark(void **);
static void gzip_release(void **);
static void puts(const char *);
  
char *input_data;
int input_len;

static long bytes_out = 0;
static uch *output_data;
static unsigned long output_ptr = 0;

extern int _text;		/* Defined in vmlinux.lds.S */
extern int _end;
static unsigned long free_mem_ptr;
static unsigned long free_mem_end_ptr;
 
#define HEAP_SIZE             0x10000

#if defined(DEBUG)
unsigned char *deflate_kernel;
unsigned char *kernel_entry;
int verbose = 1;
#define DEFLATE_KERNEL_SIZE (0x80000)
#define INFLATE_AREA_SIZE (0x180000)
#define DEFLATE_FILENAME "linux.bin.gz"
#define INFLATE_FILENAME "linux.bin"
#define malloc(x) my_malloc(x)
#undef putc
#define putc(x) my_putc(x)
#define printk(x...) printf(x)
char command[512];
char linux_command[512];
char _heap_top[HEAP_SIZE];
#else
extern char command[];
extern char linux_command[];
extern char _heap_top[];
extern char _erom[];
extern char kernel_entry[];
extern volatile char SCI_TDR;
extern volatile char SCI_SSR;
#endif

void start_kernel(int length)
{
#if !defined(DEBUG)
	asm("orc #0x80,ccr\n\t"
	    "jmp @_kernel_entry");
#else
	FILE *fp;
	fp = fopen(INFLATE_FILENAME,"wb");
	fwrite(kernel_entry,sizeof(char),length,fp);
	fclose(fp);
#endif
}

#if defined(DEBUG)
void init_work(void)
{
	FILE *fp;
	deflate_kernel = (unsigned char *)realloc(NULL,DEFLATE_KERNEL_SIZE);
	kernel_entry   = (unsigned char *)realloc(NULL,INFLATE_AREA_SIZE);
	fp = fopen(DEFLATE_FILENAME,"rb");
	fread(deflate_kernel,sizeof(char),DEFLATE_KERNEL_SIZE,fp);
	fclose(fp);
}
#endif

void *malloc(int size)
{
	void *p;

	if (size <0) error("Malloc error\n");
	if (free_mem_ptr == 0) error("Memory error\n");

	free_mem_ptr = (free_mem_ptr + 3) & ~3;	/* Align */

	p = (void *)free_mem_ptr;
	free_mem_ptr += size;

	if (free_mem_ptr >= free_mem_end_ptr)
		error("\nOut of memory\n");

	return p;
}

void free(void *where)
{	/* Don't care */
}

unsigned char get_byte(void)
{
	return *inbuf++;
}

void gzip_mark(void **ptr)
{
	*ptr = (void *) free_mem_ptr;
}

void gzip_release(void **ptr)
{
	free_mem_ptr = (long) *ptr;
}

void* memset(void* s, int c, size_t n)
{
	int i;
	char *ss = (char*)s;

	for (i=0;i<n;i++) ss[i] = c;
	return s;
}

void* memcpy(void* __dest, __const void* __src,
			    size_t __n)
{
	int i;
	char *d = (char *)__dest, *s = (char *)__src;

	for (i=0;i<__n;i++) d[i] = s[i];
	return __dest;
}

#define OF(args)  args
#define STATIC static

#undef memset
#undef memcpy
#define memzero(s, n)     memset ((s), 0, (n))

#define WSIZE 0x8000		/* Window size must be at least 32k, */
				/* and a power of two */

static uch *inbuf;	     /* input buffer */
static uch window[WSIZE];    /* Sliding window buffer */

static unsigned insize = 0;  /* valid bytes in inbuf */
static unsigned inptr = 0;   /* index of next byte to be processed in inbuf */
static unsigned outcnt = 0;  /* bytes in output buffer */

/* gzip flag byte */
#define ASCII_FLAG   0x01 /* bit 0 set: file probably ASCII text */
#define CONTINUATION 0x02 /* bit 1 set: continuation of multi-part gzip file */
#define EXTRA_FIELD  0x04 /* bit 2 set: extra field present */
#define ORIG_NAME    0x08 /* bit 3 set: original file name present */
#define COMMENT      0x10 /* bit 4 set: file comment present */
#define ENCRYPTED    0x20 /* bit 5 set: file is encrypted */
#define RESERVED     0xC0 /* bit 6,7:   reserved */

#define get_byte()  (inptr < insize ? inbuf[inptr++] : fill_inbuf())
		
/* Diagnostic functions */
#ifdef DEBUG
#  define Assert(cond,msg) {if(!(cond)) error(msg);}
#  define Trace(x) fprintf x
#  define Tracev(x) {if (verbose) fprintf x ;}
#  define Tracevv(x) {if (verbose>1) fprintf x ;}
#  define Tracec(c,x) {if (verbose && (c)) fprintf x ;}
#  define Tracecv(c,x) {if (verbose>1 && (c)) fprintf x ;}
#else
#  define Assert(cond,msg)
#  define Trace(x)
#  define Tracev(x)
#  define Tracevv(x)
#  define Tracec(c,x)
#  define Tracecv(c,x)
#endif

/* ===========================================================================
 * Fill the input buffer. This is called only when the buffer is empty
 * and at least one byte is really needed.
 */
static int fill_inbuf(void)
{
	if (insize != 0) {
		error("ran out of input data\n");
	}

	inbuf = input_data;
	insize = input_len;
	inptr = 1;
	return inbuf[0];
}

#include "inflate.c"

/* ===========================================================================
 * Write the output window window[0..outcnt-1] and update crc and bytes_out.
 * (Used for the decompressed data only.)
 */
static void flush_window(void)
{
    ulg c = crc;         /* temporary variable */
    unsigned n;
    uch *in, *out, ch;
    
    if (output_ptr >= MAXSIZE)
	    return ;
    in = window;
    out = &output_data[output_ptr]; 
    for (n = 0; n < outcnt; n++) {
	    ch = *out++ = *in++;
	    c = crc_32_tab[((int)c ^ ch) & 0xff] ^ (c >> 8);
    }
    crc = c;
    bytes_out += (ulg)outcnt;
    output_ptr += (ulg)outcnt;
    outcnt = 0;
}

static void error(char *x)
{
	puts("\n\n");
	puts(x);
	puts("\n\n -- System halted");

	while(1);	/* Halt */
}

static void putc(char c)
{
#if !defined(DEBUG)
	while((SCI_SSR & 0x80) == 0);
	SCI_TDR = c;
	SCI_SSR &= 0x7f;
#else
	putchar(c);
#endif
}

static void puts(const char *s)
{
	while (*s)
		putc(*s++);
}

static void search_gzhead(void)
{
#if !defined(DEBUG)
	char *p = command;
	while (*p != 0x1f) p++;
	input_data = p;
	input_len = (unsigned long)&_erom - (unsigned long)p;
#else
	input_data = deflate_kernel;
	input_len = 0x80000;
#endif
	output_data = kernel_entry;
}

static void set_command(void)
{
	char *s = command;
	char *d = linux_command;
	int len = COMMAND_LEN;
	while (*s >= ' ') {
		*d++ = *s++;
		--len;
	}
	for(; len > 0; --len)
		*d++ = '\0';
}

int main(void)
{
	char *gz_ptr;
	int length;
#if defined(DEBUG)
	init_work();
#endif
	free_mem_ptr = (unsigned long)&_heap_top;
	free_mem_end_ptr = free_mem_ptr + HEAP_SIZE;
	search_gzhead();
	makecrc();
	puts("Uncompressing Linux... ");
	gunzip();
	puts("Ok, booting the kernel.\r\n");
	
	set_command();
	start_kernel(length);
}
