﻿/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
  <file> Program.cs </file>
  <brief>
    Tokenize and dump the css3 file.
  </brief>
  <author>
    Stanley Hong <link2next@gmail.com>, Dec. 2013.
  </author>
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using uls.polaris;
using uls.collection;

namespace uls
{
    namespace tests
    {
        namespace Css3Toks
        {
            public class Util
            {
                // <brief>
                // Get the basename of file-path!
                // Returns 'basename.c when the 'filepath' is '\a\b\basename.c'.
                // </brief>
                // <parm name="filepath">input path string</parm>
                public static String uls_filename(String filepath)
                {
                    String fname;
                    int i;

                    if ((i = filepath.LastIndexOf('\\')) < 0)
                    {
                        fname = filepath;
                    }
                    else
                    {
                        fname = filepath.Substring(i + 1);
                    }

                    return fname;
                }

                // <brief>
                // This appends the line-feed character '\n' at the message formatted by 'fmtstr'.
                // </brief>
                public static void err_log(String fmtstr, params object[] objs)
                {
                    String dotstr;
                    dotstr = String.Format(fmtstr, objs);
                    Console.WriteLine(dotstr);
                }

                // <brief>
                // This calls err_log() above and terminates the program to abnormal state.
                // </brief>
                public static void err_panic(String fmtstr, params object[] objs)
                {
                    err_log(fmtstr, objs);
                    Environment.Exit(1);
                }
            }

            class Program
            {
                static String config_name = "css3.ulc";
                static String PACKAGE_NAME = "Css3Toks";

                static void usage()
                {
                    Util.err_log("Usage: {0} <css3-file>", PACKAGE_NAME);
                    Util.err_log("   Dumping the tokens in css3 file.");
                }

                static int options(int opt, String optarg)
                {
                    int stat = 0;

                    switch (opt)
                    {
                        case 'v':
                            break;
                        case 'h':
                            usage();
                            stat = -1;
                            break;
                        case 'V':
                            stat = -1;
                            break;
                        default:
                            Util.err_log("undefined option -{0}", opt);
                            stat = -1;
                            break;
                    }

                    return stat;
                }

                static void dumpToken(Css3Lex css3lex)
                {
                    int t = css3lex.TokNum;
                    String tstr = css3lex.TokStr;

                    if (t == Css3Lex.CSS_PATH)
                    {
                        css3lex.dumpTok("\t", "PATH", tstr, "\n");
                    }
                    else if (t == Css3Lex.CSS_NUM)
                    {
                        String str2;

                        if (css3lex.isLexemeReal)
                        {
                            double ff = css3lex.lexemeAsDouble;
                            str2 = ff + " '" + tstr + "'";
                        }
                        else
                        {
                            int ii = css3lex.lexemeAsInt32;
                            str2 = ii + " '" + tstr + "'";
                        }

                        css3lex.dumpTok("\t", "NUM", str2, "\n");
                    }
                    else
                    {
                        css3lex.dumpTok("\t", "\n");
                    }
                }

                static void dumpTokens(Css3Lex css3lex)
                {
                    for ( ; ; )
                    {
                        if (css3lex.getTok() == Css3Lex.CSS_EOI) break;
                        dumpToken(css3lex);
                    }
                }

                static void Main(String[] args)
                {
                    int i0;
                    String input_file;

                    config_name = "css3.ulc";

                    using (UlsContext run_context = new UlsContext("vhV"))
                    {

                        UlsContext.DelegateOptProc opt_proc = new UlsContext.DelegateOptProc(options);
                        if ((i0 = run_context.parseOpts(args, opt_proc)) != 0 || i0 >= args.Length)
                        {
                            usage();
                            return;
                        }

                        if (i0 < args.Length)
                        {
                            input_file = args[i0];
                        }
                        else
                        {
                            usage();
                            return;
                        }

                        try
                        {
                            Css3Lex css3lex = new Css3Lex(config_name);

                            css3lex.setFile(input_file);
                            dumpTokens(css3lex);
                        }
                        catch (Exception e)
                        {
                            Console.WriteLine("An exception caught: {0}", e);
                        }
                    }
                }
            }
        }
    }
}
