/*
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

/*
  <file> UlsLog.cpp </file>
  <brief>
   This implements logging subsystem for UlsFactory Class.
  </brief>
  <author>
    Stanley Hong <link2next@gmail.com>, April 2017.
  </author>
*/
#include "Stdafx.h"
#include "UlsLog.h"
#include <ctype.h>

using namespace System;

#define _uls_log_static(proc) UlsLog::proc##_static

namespace uls {
	namespace polaris {
#define ULS_CLASS_NAME uls_xport_t
		ULS_CLASS_NAME::ULS_CLASS_NAME(delegate_uls_putstr_t ^puts)
		{
			putstr = puts;
		}

		ULS_CLASS_NAME::~ULS_CLASS_NAME()
		{
		}

		void ULS_CLASS_NAME::setXdat(Object ^udat)
		{
			xdat = udat;
		}

		Object ^ULS_CLASS_NAME::getXdat()
		{
			return xdat;
		}

		void ULS_CLASS_NAME::putStr(const char *line, int len)
		{
			putstr(xdat, line, len);
		}
#undef ULS_CLASS_NAME

#define ULS_CLASS_NAME UlsLog
		void ULS_CLASS_NAME::ulslog_lock_null(void)
		{
		}

		void ULS_CLASS_NAME::ulslog_unlock_null(void)
		{
		}

		void ULS_CLASS_NAME::__put_str_fillchars(uls_xport_t ^xport, int n, char ch)
		{
			int i, buf_len;
			char buf[16];

			buf_len = sizeof(buf) - 1;
			for (i=0; i<buf_len; i++) buf[i] = ch;
			buf[buf_len] = '\0';

			for (i=0; i < n/buf_len; i++) {
				xport->putStr(buf, buf_len);
			}

			if ((i=n % buf_len) > 0) {
				xport->putStr(buf, i);
			}
		}

		const char* ULS_CLASS_NAME::__check_line_perc(uls_xport_t ^xport, const char *line, int *ptr_len)
		{
			int len;
			const char *lptr;
			char ch;

			for (lptr = line; (ch=*lptr) != '\0' && ch != '%'; lptr++) {
				/* NOTHING */
			}

			if ((len = (int) (lptr - line)) > 0) {
				xport->putStr(line, len);
				*ptr_len = len;
			} else {
				*ptr_len = 0;
			}

			if (*lptr != '\0') ++lptr;
			return lptr;
		}

		void ULS_CLASS_NAME::err_exit(void)
		{
			Environment::Exit(1);
		}

#define ULSDOT_MAXSZ_NUMSTR 64
		int ULS_CLASS_NAME::uls_num2str(char *numstr, unsigned int num, int base_shift)
		{
			unsigned int  base_mask;
			register int   i;

			i = ULSDOT_MAXSZ_NUMSTR;
			numstr[ULSDOT_MAXSZ_NUMSTR] = '\0';

			if (num == 0) {
				numstr[--i] = '0';
				return i;
			}

			if (base_shift > 0) {
				base_mask = (1 << base_shift) - 1;
				do {
					numstr[--i] = "0123456789ABCDEF"[ (int) (num & base_mask) ];
					num >>= base_shift;
				} while (num!=0);

			} else { // decimal
				do {
					numstr[--i] = "0123456789"[(int) (num%10)];
					num /= 10;
				} while (num!=0);
			}

			return i;
		}

		void ULS_CLASS_NAME::__put_str_width(
			uls_xport_t ^xport, const char *line, int len, int width, char padding_ch)
		{
			int left_just;
			int n_margins;

			if (width < 0) {
				left_just = 1;
				width = -width;
			} else {
				left_just = 0;
			}

			if (width <= len) {
				xport->putStr(line, len);

			} else { // len < width
				n_margins = width - len;
				if (left_just) {
					xport->putStr(line, len);
					__put_str_fillchars(xport, n_margins, padding_ch);
				} else {
					__put_str_fillchars(xport, n_margins, padding_ch);
					xport->putStr(line, len);
				}
			}
		}

		int ULS_CLASS_NAME::__format_strseg(uls_xport_t ^xport, const char **ptr_fmt, const void *udat)
		{
			int len1, len, k, minus, width=0;
			const char *lptr, *str, *fmt = *ptr_fmt;
			char numbuf[ULSDOT_MAXSZ_NUMSTR+1], ch, padding_ch=' ';
			unsigned int num;

			for (len1=0,lptr=fmt; ; lptr++) {
				lptr = __check_line_perc(xport, lptr, &k);
				len1 += k;

				if ((ch = *lptr) == '\0') {
					return len1;
				}

				if (ch != '%') {
					break;
				}

				xport->putStr(&ch, 1);
				++len1;
			}

			if (isdigit(ch) || ch == '-') {
				if (ch == '-') {
					minus = 1;
					++lptr;
				} else {
					minus = 0;
				}

				if ((ch=*lptr) == '0') {
					padding_ch = '0';
					++lptr;
				}

				for ( ; isdigit(ch=*lptr); lptr++) {
					width = width*10 + (ch - '0');
				}
				if (minus) width = -width;
			}

			switch (ch) {
			case 's':
				if ((str = (const char *) udat) == NULL) {
					str = "<<null>>";
				}
				len = (int) strlen(str);
				__put_str_width(xport, str, len, width, ' ');
				break;

			case 'd':
				if ((k = *((int *) udat)) < 0) {
					minus = 1;
					num = (unsigned int) -k;
				} else {
					minus = 0;
					num = (unsigned int) k;
				}

				k = uls_num2str(numbuf, num, 0);
				if (minus) numbuf[--k] = '-';

				len = ULSDOT_MAXSZ_NUMSTR - k;
				__put_str_width(xport, numbuf + k, len, width, padding_ch);
				break;

			case 'x':
			case 'X':
				num = *((unsigned int *) udat);
				k = uls_num2str(numbuf, num, 4);
				len = ULSDOT_MAXSZ_NUMSTR - k;
				__put_str_width(xport, numbuf + k, len, width, padding_ch);
				break;

			case 'c':
				ch = *((char *) udat);
				xport->putStr(&ch, 1);
				len = 1;
				break;

			default:
				xport->putStr(&ch, 1);
				len = 1;
				break;
			}

			*ptr_fmt = ++lptr;
			return len1 + len;
		}

		void ULS_CLASS_NAME::__put_str_strbuf(Object ^xdat, const char *line, int len)
		{
			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xdat;
			char *bufptr;
			int i, m;

			if ((m = xstr->bufsiz - xstr->buflen) <= 0) return;
			if (len > m) len = m;

			bufptr = xstr->buf + xstr->buflen;
			for (i=0; i<len; i++) {
				*bufptr++ = line[i];
			}

			xstr->buflen += len;
		}

		void ULS_CLASS_NAME::__put_str_file(Object ^xdat, const char *line, int len)
		{
			uls_xport_fp_t ^xstr = (uls_xport_fp_t ^) xdat;
			FILE *fp = xstr->fp;
			int i;

			for (i=0; i<len; i++) {
				fputc(line[i], fp);
			}
		}

		void ULS_CLASS_NAME::ulslog_reset_lock_syserr(ulslog_lock_proc_t lock, ulslog_lock_proc_t unlock)
		{
			if (lock == nullptr) {
				lock = gcnew _delegate_ulslog_lock(this, &ULS_CLASS_NAME::ulslog_lock_null);
			}

			if (unlock == nullptr) {
				unlock = gcnew _delegate_ulslog_lock(this, &ULS_CLASS_NAME::ulslog_unlock_null);
			}

			ulslog_lock_syserr = lock;
			ulslog_unlock_syserr = unlock;

			ulslog_lock_sysprn = lock;
			ulslog_unlock_sysprn = unlock;
		}

		void ULS_CLASS_NAME::ulslog_reset_lock_sysprn(ulslog_lock_proc_t lock, ulslog_lock_proc_t unlock)
		{
			ulslog_lock_sysprn = lock;
			ulslog_unlock_sysprn = unlock;
		}

		//
		// sysprn
		//
		void ULS_CLASS_NAME::uls_initialize_sysprn(void)
		{
			sysprn_opened = false;
			fp_sysprn = stdout;
		}

		void ULS_CLASS_NAME::uls_finalize_sysprn(void)
		{
		}

		//
		// syslog
		//
		void ULS_CLASS_NAME::uls_initialize_syserr(FILE *fp)
		{
			fp_syslog = fp;
		}

		void ULS_CLASS_NAME::uls_finalize_syserr(void)
		{
			fp_syslog = stderr;
		}

		//
		// ULS_CLASS_NAME
		//
		ULS_CLASS_NAME::ULS_CLASS_NAME(ulslog_lock_proc_t lock, ulslog_lock_proc_t unlock)
		{
			ulslog_reset_lock_syserr(lock, unlock);

			uls_initialize_syserr(stderr);
			uls_initialize_sysprn();

			isDisposed = isFinalized = false;
#ifdef _MANAGE_ULS_OBJECTS
			addUlsObject2List(objs_list);
#endif
		}

		//
		// ~ULS_CLASS_NAME
		//
		void ULS_CLASS_NAME::finalizer()
		{
			if (isFinalized) return;
			uls_finalize_sysprn();
			uls_finalize_syserr();
			ulslog_lock_syserr = ulslog_unlock_syserr = nullptr;
			isFinalized = true;
		}

		ULS_CLASS_NAME::~ULS_CLASS_NAME()
		{
			if (isDisposed) return;
			finalizer();
#ifdef _MANAGE_ULS_OBJECTS
			delUlsObject2List(objs_list);
#endif
			isDisposed = true;
		}

		ULS_CLASS_NAME::!ULS_CLASS_NAME()
		{
			finalizer();
		}

		//
		// snprintf
		//
		uls_xport_t^ ULS_CLASS_NAME::get_xport_strbuf(char *buf, int bufsiz)
		{
			uls_xport_str_t ^xstr;

			xstr = gcnew uls_xport_str_t();

			xstr->buf = buf;
			xstr->bufsiz = bufsiz - 1;
			xstr->buflen = 0;

			uls_xport_t::delegate_uls_putstr_t ^puts =
				gcnew uls_xport_t::delegate_uls_putstr_t(this, &ULS_CLASS_NAME::__put_str_strbuf);

			uls_xport_t  ^xport = gcnew uls_xport_t(puts);
			xport->setXdat(xstr);

			return xport;

		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, const char *s1)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, int d1)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, const char *s1, const char *s2)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len =__format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, int d1, int d2)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, &d2);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, const char *s1, int d2)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, &d2);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, int d1, const char *s2)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, const char *s1, const char *s2, const char *s3)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, s3);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, const char *s1, int d2, int d3)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, &d2);
			len += __format_strseg(xport, &fmt, &d3);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		int ULS_CLASS_NAME::uls_snprintf(char *buf, int bufsiz, const char *fmt, const char *s1, int d2, const char *s3)
		{
			uls_xport_t  ^xport = get_xport_strbuf(buf, bufsiz);
			int len;

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, &d2);
			len += __format_strseg(xport, &fmt, s3);
			len += __format_strseg(xport, &fmt, NULL);

			uls_xport_str_t ^xstr = (uls_xport_str_t ^) xport->getXdat();
			len = xstr->buflen;
			xstr->buf[xstr->buflen] = '\0';

			return len;
		}

		//
		// uls_fprintf
		//
		uls_xport_t^ ULS_CLASS_NAME::get_xport_filebuf(FILE *fp)
		{
			uls_xport_fp_t ^xfp;

			xfp = gcnew uls_xport_fp_t();
			xfp->fp = fp;

			uls_xport_t::delegate_uls_putstr_t ^puts =
				gcnew uls_xport_t::delegate_uls_putstr_t(this, &ULS_CLASS_NAME::__put_str_file);

			uls_xport_t  ^xport = gcnew uls_xport_t(puts);
			xport->setXdat(xfp);

			return xport;

		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, const char *s1)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, int d1)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, const char *s1, const char *s2)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, int d1, int d2)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, &d2);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, const char *s1, int d2)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, &d2);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, int d1, const char *s2)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, const char *s1, const char *s2, const char *s3)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, s3);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, int d1, int d2, int d3)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, &d2);
			len += __format_strseg(xport, &fmt, &d3);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, int d1, const char *s2, const char *s3)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, &d1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, s3);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		int ULS_CLASS_NAME::uls_fprintf(FILE *fp, const char *fmt, const char *s1, const char *s2, int d3)
		{
			int len;

			uls_xport_t  ^xport = get_xport_filebuf(fp);

			len = __format_strseg(xport, &fmt, s1);
			len += __format_strseg(xport, &fmt, s2);
			len += __format_strseg(xport, &fmt, &d3);
			len += __format_strseg(xport, &fmt, NULL);

			return len;
		}

		//
		// printf
		//
		void ULS_CLASS_NAME::uls_printf(const char *fmt)
		{
			uls_fprintf(stdout, fmt);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, const char *s1)
		{
			uls_fprintf(stdout, fmt, s1);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, int d1)
		{
			uls_fprintf(stdout, fmt, d1);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, const char *s1, const char *s2)
		{
			uls_fprintf(stdout, fmt, s1, s2);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, int d1, int d2)
		{
			uls_fprintf(stdout, fmt, d1, d2);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, const char *s1, int d2)
		{
			uls_fprintf(stdout, fmt, s1, d2);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, int d1, const char *s2)
		{
			uls_fprintf(stdout, fmt, d1, s2);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, const char *s1, const char *s2, const char *s3)
		{
			uls_fprintf(stdout, fmt, s1, s2, s3);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, int d1, int d2, int d3)
		{
			uls_fprintf(stdout, fmt, d1, d2, d3);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, int d1, const char *s2, const char *s3)
		{
			uls_fprintf(stdout, fmt, d1, s2, s3);
		}

		void ULS_CLASS_NAME::uls_printf(const char *fmt, const char *s1, const char *s2, int d3)
		{
			uls_fprintf(stdout, fmt, s1, s2, d3);
		}

		//
		// sysprn
		//
		int ULS_CLASS_NAME::uls_sysprn_open(FILE* fp, void *unused)
		{
			// assert: fp != NULL

			if (sysprn_opened) {
				return -1;
			}

			ulslog_lock_sysprn();
			fp_sysprn = fp;
			sysprn_opened = true;

			return 0;
		}

		void ULS_CLASS_NAME::uls_sysprn_close(void)
		{
			if (sysprn_opened) {
				fp_sysprn = stdout;
				sysprn_opened = false;
				ulslog_unlock_sysprn();
  			}
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt)
		{
			return uls_fprintf(fp_sysprn, fmt);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, const char *s1)
		{
			return uls_fprintf(fp_sysprn, fmt, s1);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, int d1)
		{
			return uls_fprintf(fp_sysprn, fmt, d1);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, const char *s1, const char *s2)
		{
			return uls_fprintf(fp_sysprn, fmt, s1, s2);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, int d1, int d2)
		{
			return uls_fprintf(fp_sysprn, fmt, d1, d2);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, const char *s1, int d2)
		{
			return uls_fprintf(fp_sysprn, fmt, s1, d2);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, int d1, const char *s2)
		{
			return uls_fprintf(fp_sysprn, fmt, d1, s2);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, const char *s1, const char *s2, const char *s3)
		{
			return uls_fprintf(fp_sysprn, fmt, s1, s2, s3);
		}

		int ULS_CLASS_NAME::uls_sysprn(const char *fmt, const char *s1, const char *s2, int d3)
		{
			return uls_fprintf(fp_sysprn, fmt, s1, s2, d3);
		}

		void ULS_CLASS_NAME::uls_sysprn_puttabs(int n)
		{
			uls_xport_t  ^xport = get_xport_filebuf(fp_sysprn);
			__put_str_fillchars(xport, n, '\t');
		}

		//
		// syslog
		//
		void ULS_CLASS_NAME::err_log_static(const char *fmt)
		{
			Console::WriteLine("{0}", gcnew String(fmt));
		}

		void ULS_CLASS_NAME::err_log(const char *fmt)
		{
			ulslog_lock_syserr();
			uls_fprintf(fp_syslog, fmt);
			uls_fprintf(fp_syslog, "\n");
			ulslog_unlock_syserr();
		}

		void ULS_CLASS_NAME::err_log(const char *fmt, const char *s1)
		{
			ulslog_lock_syserr();
			uls_fprintf(fp_syslog, fmt, s1);
			uls_fprintf(fp_syslog, "\n");
			ulslog_unlock_syserr();
		}

		void ULS_CLASS_NAME::err_log(const char *fmt, int d1)
		{
			ulslog_lock_syserr();
			uls_fprintf(fp_syslog, fmt, d1);
			uls_fprintf(fp_syslog, "\n");
			ulslog_unlock_syserr();
		}

		void ULS_CLASS_NAME::err_log(const char *fmt, const char *s1, const char *s2)
		{
			ulslog_lock_syserr();
			uls_fprintf(fp_syslog, fmt, s1, s2);
			uls_fprintf(fp_syslog, "\n");
			ulslog_unlock_syserr();
		}

		void ULS_CLASS_NAME::err_log(const char *fmt, int d1, int d2)
		{
			ulslog_lock_syserr();
			uls_fprintf(fp_syslog, fmt, d1, d2);
			uls_fprintf(fp_syslog, "\n");
			ulslog_unlock_syserr();
		}

		void ULS_CLASS_NAME::err_log(const char *fmt, const char *s1, int d2)
		{
			ulslog_lock_syserr();
			uls_fprintf(fp_syslog, fmt, s1, d2);
			uls_fprintf(fp_syslog, "\n");
			ulslog_unlock_syserr();
		}

		void ULS_CLASS_NAME::err_log(const char *fmt, int d1, const char *s2)
		{
			ulslog_lock_syserr();
			uls_fprintf(fp_syslog, fmt, d1, s2);
			uls_fprintf(fp_syslog, "\n");
			ulslog_unlock_syserr();
		}

		//
		// syslog : panic
		//
		void ULS_CLASS_NAME::err_panic_static(const char *fmt)
		{
			err_log_static(fmt);
			err_exit();
		}

		void ULS_CLASS_NAME::err_panic(const char *fmt)
		{
			err_log(fmt);
			err_exit();
		}

		void ULS_CLASS_NAME::err_panic(const char *fmt, const char *s1)
		{
			err_log(fmt, s1);
			err_exit();
		}

		void ULS_CLASS_NAME::err_panic(const char *fmt, int d1)
		{
			err_log(fmt, d1);
			err_exit();
		}

		void ULS_CLASS_NAME::err_panic(const char *fmt, const char *s1, const char *s2)
		{
			err_log(fmt, s1, s2);
			err_exit();
		}

		void ULS_CLASS_NAME::err_panic(const char *fmt, int d1, int d2)
		{
			err_log(fmt, d1, d2);
			err_exit();
		}

		void ULS_CLASS_NAME::err_panic(const char *fmt, const char *s1, int d2)
		{
			err_log(fmt, s1, d2);
			err_exit();
		}

		void ULS_CLASS_NAME::err_panic(const char *fmt, int d1, const char *s2)
		{
			err_log(fmt, d1, s2);
			err_exit();
		}
#undef ULS_CLASS_NAME
	} //  End of polaris
} //  End of uls
