/*
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

/*
  <file> UlsToolbase.cpp </file>
  <brief>
   This implements utility subsystem for UlsFactory Class.
  </brief>
  <author>
    Stanley Hong <link2next@gmail.com>, April 2017.
  </author>
*/
#pragma once

#include "UlsStream.h"
#include "UlsLex.h"
#include "UlsIStream.h"

using namespace System;

namespace uls {
	namespace polaris {

		public ref class UlsOStream : UlsStream
		{
			bool do_numbering;
			bool isFinalized, isDisposed;

			UlsLex ^uls_lex;
			UlsFactory::uls_ostream_t ^ohdr;

			bool makeOStream(String ^filepath, String ^subtag, bool numbering);

			static UlsObjectListExp ^objs_list;

			static UlsOStream()
			{
				objs_list = gcnew UlsObjectListExp();
			}

		public:
			// <brief>
			// This makes an object that supports write-only sequential IO.
			// Its purpose is to save the token sequence form input source. 
			// </brief>
			// <parm name="filepath">
			// It's a file path for a new uls-stream file.
			// If the file already exists it'll be overwritten.
			// </parm>
			// <parm name="lex">The lexical analyzer associated with uls-stream file</parm>
			// <parm name="subtag">
			// A user provided tag to the file 'filepath'.
			// More often than not, it tends to be the file path.
			// This string will be written to the output uls file.
			// </parm>
			// <parm name="numbering">
			// Specify whether the line number information is inserted or not.
			// The information is automatically inserted whenever the line of source code is changed.
			// </parm>
			UlsOStream(String ^filepath, UlsLex ^lex, String ^subtag, bool numbering);
			UlsOStream(String ^filepath, UlsLex ^lex, String ^subtag);
			UlsOStream(String ^filepath, UlsLex ^lex);
			virtual ~UlsOStream();
			virtual void finalizer() override;
			!UlsOStream();

			// <brief>
			// This method prints a record of <tokid, tokstr> pair to the output stream.
			// </brief>
			// <parm name="tokid">the token number to be printed</parm>
			// <parm name="tokstr">the lexeme associated with the 'tokid'</parm>
			// <return>none</return>
			void printTok(int tokid, String ^tokstr);

			// <brief>
			// This method prints an annotation <linenum, tag> pair to the output file.
			// </brief>
			// <parm name="lno">the line number of the source file</parm>
			// <parm name="tagstr">the tag of the source file</parm>
			// <return>none</return>
			void printTokLineNum(int lno, String ^tagstr);

			// <brief>
			// Start writing the lexical streaming with input-stream 'ifile'.
			// </brief>
			// <parm name="ifile">
			// This input uls-stream will be written to the 'UlsOStream' object.
			// </parm>
			bool startStream(UlsIStream ^ifile);

			// <brief>
			// Close the stream (file).
			// </brief>
			void close();

			// <brief>
			// Returns just the stored 'UlsLex' object. 
			// </brief>
			UlsFactory::uls_ostream_t ^getCoreHdr();

			static void releaseUlsObjectList()
			{
				UlsObject::finalizeUlsObjectList(objs_list);
				objs_list = nullptr;
			}
		};
	}
}
