/*
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

/*
  <file> UlsToolbase.cpp </file>
  <brief>
   This implements utility subsystem for UlsFactory Class.
  </brief>
  <author>
    Stanley Hong <link2next@gmail.com>, April 2017.
  </author>
*/
#include "Stdafx.h"

#include <direct.h>
#include <io.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "UlsToolbase.h"

using namespace System;

namespace uls {
	namespace polaris {
		//
		// Class: UlsToolbase
		//
#define ULS_CLASS_NAME UlsToolbase

#define _uls_log_primitive(proc) _log_->proc
#define _uls_log_static(proc) UlsLog::proc##_static

#define _uls_log(proc) _log_->proc
#define _uls_log_(proc) _log_->uls_##proc

#define _uls_tool(proc) proc
#define _uls_tool_(proc) uls_##proc
#define __uls_tool_(proc) _uls_##proc

#define _uls_sysinfo_(attr) uls_sysinfo->attr

#define ULS_EXCLUDE_HFILES
#include "uls_prim.c"
#include "uls_version.c"
#include "csz_stream.c"
#include "uls_sysprops.c"
#include "uls_ieee754.c"
#include "uls_num.c"
#include "uls_auw.c"
#include "uls_fileio.c"
#include "utf8_enc.c"
#include "uls_misc.c"
#undef ULS_EXCLUDE_HFILES

	void ULS_CLASS_NAME::uls_toolbase_lock_syserr(void)
	{
		uls_mutex_t mtx = uls_ptr(dfl_lck_syserr);
		uls_lock_mutex(mtx);
	}

	void ULS_CLASS_NAME::uls_toolbase_unlock_syserr(void)
	{
		uls_mutex_t mtx = uls_ptr(dfl_lck_syserr);
		uls_unlock_mutex(mtx);
	}

	void ULS_CLASS_NAME::uls_toolbase_lock_sysprn(void)
	{
		uls_mutex_t mtx = uls_ptr(dfl_lck_sysprn);
		uls_lock_mutex(mtx);
	}

	void ULS_CLASS_NAME::uls_toolbase_unlock_sysprn(void)
	{
		uls_mutex_t mtx = uls_ptr(dfl_lck_sysprn);
		uls_unlock_mutex(mtx);
	}

	ULS_CLASS_NAME::ULS_CLASS_NAME()
	{
		UlsLog::ulslog_lock_proc_t lock, unlock;

		_log_ = gcnew UlsLog(nullptr, nullptr);

		// delegate
		_uls_alloc_callback(uls_fgetc_ascii_str, fgetch);
		_uls_alloc_callback(consume_ms_mbcs_onechar, fgetch);


		_uls_alloc_callback(fill_utf8_buf, fill_utf_rawbuf);
		_uls_alloc_callback(fill_utf16_buf, fill_utf_rawbuf);
		_uls_alloc_callback(fill_utf32_buf,fill_utf_rawbuf);

		_uls_alloc_callback(dec_utf8_buf, dec_utf_rawbuf);
		_uls_alloc_callback(dec_utf16_buf, dec_utf_rawbuf);
		_uls_alloc_callback(dec_utf16_buf, dec_utf_rawbuf);

		if (__initialize_uls_misc() < 0) {
			Console::WriteLine("Failed to initialize at UlsToolbase!");
		}

		uls_init_mutex(%dfl_lck_syserr);
		uls_init_mutex(%dfl_lck_sysprn);

		lock = gcnew UlsLog::_delegate_ulslog_lock(this, &ULS_CLASS_NAME::uls_toolbase_lock_syserr);
		unlock = gcnew UlsLog::_delegate_ulslog_lock(this, &ULS_CLASS_NAME::uls_toolbase_unlock_syserr);
		_uls_log(ulslog_reset_lock_syserr)(lock, unlock);

		lock = gcnew UlsLog::_delegate_ulslog_lock(this, &ULS_CLASS_NAME::uls_toolbase_lock_sysprn);
		unlock = gcnew UlsLog::_delegate_ulslog_lock(this, &ULS_CLASS_NAME::uls_toolbase_unlock_sysprn);
		_uls_log(ulslog_reset_lock_sysprn)(lock, unlock);

		isDisposed = isFinalized = false;
#ifdef _MANAGE_ULS_OBJECTS
		addUlsObject2List(objs_list);
#endif
	}

	void ULS_CLASS_NAME::finalizer()
	{
		if (isFinalized) return;
		__finalize_uls_misc();
		isFinalized = true;
	}

	ULS_CLASS_NAME::~ULS_CLASS_NAME()
	{
		if (isDisposed) return;
		finalizer();
#ifdef _MANAGE_ULS_OBJECTS
		delUlsObject2List(objs_list);
#endif
		// delegate
		_uls_dealloc_callback(uls_fgetc_ascii_str);
		_uls_dealloc_callback(consume_ms_mbcs_onechar);

		_uls_dealloc_callback(fill_utf8_buf);
		_uls_dealloc_callback(fill_utf16_buf);
		_uls_dealloc_callback(fill_utf32_buf);

		_uls_dealloc_callback(dec_utf8_buf);
		_uls_dealloc_callback(dec_utf16_buf);
		_uls_dealloc_callback(dec_utf32_buf);

		delete _log_;
		isDisposed = true;
	}

	ULS_CLASS_NAME::!ULS_CLASS_NAME()
	{
		finalizer();
	}

#undef ULS_CLASS_NAME

	} //  End of polaris
} //  End of uls
