#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>

#include "l7vsagent.h"
#include "l7vsClusterInfo.h"
#include "l7vsServiceTable.h"
#include "l7vsRealTable.h"

int raise_trap(char *);
static int trap_loop_enter;
int msgid;
key_t key;

static int keep_running;
static char *l7vs_version_string = L7VS_VERSION_STRING;


/* This method sets the keep_runing to 0. This in turn will terminate the l7vsagent */
void stop_server(int sig) {
    keep_running = 0;
}


/* This method checks the trap and is raised by config.c whenever there is a change
*  in the L7VS Kernel Table */
void check_l7vs_trap(int sig) {

	char trap_message[BUFSIZ];
	char *wordToken;

	/* With combination of WHILE Loop and trap_loop_enter setting, we could allow
	*  the Queuing up of SIGUSR1 requests */
	while (keep_running)
	{
		if (trap_loop_enter == 0)
		{
			/* Added the enter flag just to allow the queuing up */
			trap_loop_enter = 1;

			/* Read the messages from the queue.
			*  Start a LOOP and break if there is no message in the queue */
			while (keep_running)
			{
				/* Create or Get the message ID for the message queue */
				msgid = msgget(key, 0666 | IPC_CREAT);
				if (msgid == -1)
				{
					VANESSA_LOGGER_ERR("Error in getting the message ID for the message queue");
					break;
				}

				if (msgrcv(msgid, (void *) &trap_message, BUFSIZ, 0, IPC_NOWAIT) ==  -1)
					break;
				else
				{
					/* The actual trap is raised here */
					raise_trap(trap_message);
				}
			}

			/* Set trap_loop_enter = 0. Now other SIGUSR1 requests can enter the funtion loop */
			trap_loop_enter = 0;
			return;
		}
	}
}


/* This method obtains the number of Virtual services in the L7VS Kernel Table
*  Most of the code has been taken from the method of l7vsadm.c */
int get_int_numServices(int * virtualServiceCount)
{
	struct l7vs_config_req_list_vs req;
	struct l7vs_config_rsp_list_vs rsp;
	struct iovec iov[2];
	void *buf;
	int bufsize;
	int ret;

	bufsize = L7VSAGENT_LIST_VS_BUFSIZE;

	retry:
	buf = malloc(bufsize);
	if (buf == NULL)
	{
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return;
	}

	req.cmd = L7VS_CONFIG_LIST_VS;
	ret = send_request(socket_num, &req, sizeof(req));
	if (ret < 0)
	{
		free(buf);
		return;
	}

	iov[0].iov_base = &rsp;
	iov[0].iov_len = sizeof(rsp);
	iov[1].iov_base = buf;
	iov[1].iov_len = bufsize;

	ret = recv_response(socket_num, iov, sizeof(iov) / sizeof(iov[0]));
	if (ret < 0)
	{
		free(buf);
		if (ret == L7VSAGENT_RECV_TOO_LARGE)
		{
			bufsize += L7VSAGENT_LIST_VS_BUFSIZE;
			goto retry;
		}
		return;
	}

	if (rsp.code != 0)
	{
		VANESSA_LOGGER_ERR(config_strerror(rsp.code));
		free(buf);
		return;
	}

	/* rsp.num indicates the total number of Virtual services currently in the L7VS Table */
	*virtualServiceCount = rsp.num;
	free(buf);

	return L7_OK;
}


/* This method gets the L7VS version */
int get_str_version(char* * l7vsVersion)
{
	*l7vsVersion = l7vs_version_string;
    return L7_OK;
}


/* This method creates the socket for communicating with L7VS.
*  This is taken from the method of l7vsadm.c */
static int create_socket(void)
{
	struct sockaddr_un addr;
	char *sockname;
	int opt;
	int ret;
	int socket_num;

	ret = asprintf(&sockname, "%s/l7vsagent-%d",
	L7VS_CONFIG_SOCK_PATH, getpid());
	if (ret < 0) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return ret;
	}

	unlink(sockname);
	if (ret >= sizeof(addr.sun_path)) {
		VANESSA_LOGGER_ERR("Internal error. socket name too long.");
		free(sockname);
		return -1;
	}

	memset(&addr, 0, sizeof(addr));
	memcpy(addr.sun_path, sockname, ret);
	addr.sun_family = AF_LOCAL;

	socket_num = socket(PF_LOCAL, SOCK_DGRAM, 0);
	if (socket_num < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("socket: %s", strerror(errno));
		free(sockname);
		return socket_num;
	}

	opt = 1;
	ret = setsockopt(socket_num, SOL_SOCKET, SO_PASSCRED, &opt, sizeof(opt));
	if (ret < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("setsockopt SO_PASSCRED: %s",
			strerror(errno));
		free(sockname);
		close(socket_num);
		return ret;
	}

	ret = bind(socket_num, (struct sockaddr *)&addr, SUN_LEN(&addr));
	if (ret < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("bind on %s: %s",
			sockname, strerror(errno));
		close(socket_num);
		return ret;
	}

	local_sockname = sockname;

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_LOCAL;
	strcpy(addr.sun_path, l7vs_config_sockname);

	ret = connect(socket_num, (struct sockaddr *)&addr, SUN_LEN(&addr));
	if (ret < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("connect to daemon: %s",
			strerror(errno));
		close(socket_num);
		return ret;
	}

	return socket_num;
}


/* This method destroys the socket earlier created.
*  This is taken from the method of l7vsadm.c */
static void destroy_socket(void)
{
	if (local_sockname == NULL) {
		return;
	}

	unlink(local_sockname);
}


/* This method sends the request to obtain data from the L7VS Kernel Table.
*  This is for Virtual Services only.
*  This is taken from the method of l7vsadm.c */
static int send_request(int socket_num, void *req, size_t len)
{
	int ret;

	ret = send(socket_num, req, len, 0);
	if (ret < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("send to daemon: %s",
			strerror(errno));
	}

	return ret;
}


/* This method receives the response from the L7VS Kernel Table.
*  This is taken from the method of l7vsadm.c */
static int recv_response(int socket_num, struct iovec *iov, size_t iovlen)
{
	struct msghdr msg;
	struct ucred *cred;
	struct cmsghdr *cmsg;
	unsigned char cbuf[CMSG_LEN(sizeof(struct ucred))];
	int ret;

	memset(&msg, 0, sizeof(msg));
	msg.msg_control = cbuf;
	msg.msg_controllen = sizeof(cbuf);
	msg.msg_iov = iov;
	msg.msg_iovlen = iovlen;
	cmsg = (struct cmsghdr *)cbuf;

	ret = recvmsg(socket_num, &msg, 0);
	if (ret < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("recvmsg: %s", strerror(errno));
		return L7VSAGENT_RECV_FAILED;
	}

	if (msg.msg_flags & MSG_CTRUNC) {
		VANESSA_LOGGER_ERR("Invalid response from l7vsd");
		return L7VSAGENT_RECV_FAILED;
	}

	if (msg.msg_flags & MSG_TRUNC) {
		return L7VSAGENT_RECV_TOO_LARGE;
	}

	if (! (cmsg->cmsg_level == SOL_SOCKET &&
		cmsg->cmsg_type == SCM_CREDENTIALS)) {
		VANESSA_LOGGER_ERR("Could not receive a remote credential");
		return L7VSAGENT_RECV_FAILED;
	}

	cred = (struct ucred *)CMSG_DATA(cmsg);
	if (cred->uid != 0) {
		VANESSA_LOGGER_ERR("Response from unprivileged user");
		return L7VSAGENT_RECV_FAILED;
	}

	return ret;
}


/* This method is for retrieving the error message for Vanessa logger.
*  This is taken from the method of l7vsadm.c */
static const char * config_strerror(int err)
{
	switch (err) {
	case 0:
			return "No error";
	case L7VS_CONFIG_ERR_INVALID_COMMAND:
			return "Invalid command";
	case L7VS_CONFIG_ERR_NOMEM:
			return "Could not allocate memory";
	case L7VS_CONFIG_ERR_NOVS:
			return "No such virtual service";
	case L7VS_CONFIG_ERR_NOSOCK:
			return "Could not create a service socket";
	default:
			return "Unknown error";
	}
}


/* This method gets the service name (IP Address and Port) in character format
*  This is taken from the method of l7vsadm.c */
static char * get_servicename(struct sockaddr_in *addr, int isnumeric)
{
	char *sname;
	char hostname[40], portname[20];
	int flags;
	int ret;

	flags = 0;
	if (isnumeric) {
		flags = NI_NUMERICHOST | NI_NUMERICSERV;
	}

	ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
	hostname, sizeof(hostname),
	portname, sizeof(portname), flags);
	if (ret != 0) {
		VANESSA_LOGGER_ERR_UNSAFE("getnameinfo: %s",
			gai_strerror(ret));
		return NULL;
	}

	ret = asprintf(&sname, "%s:%s", hostname, portname);
	if (ret < 0) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return NULL;
	}

	return sname;
}


/* This method sends the request to obtain data from the L7VS Kernel Table.
*  This is for Real servers only.
*  This is taken from the method of l7vsadm.c */
static int sendv_request(int socket_num, struct iovec *iov, int iovlen)
{
	int ret;

	ret = writev(socket_num, iov, iovlen);
	if (ret < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("send to daemon: %s",
			strerror(errno));
	}

	return ret;
}


/* This method clears off the Linked list memory created for Virtual services.
*  The linked list would be freed */
void freeVirtualServiceTableMemory()
{
	l7vsServiceTable *tempVirtualServiceNode;

	if(l7vsServiceTable_head != NULL)
	{
		while( l7vsServiceTable_head != NULL ) {
		   tempVirtualServiceNode = l7vsServiceTable_head->next;
		   SNMP_FREE(l7vsServiceTable_head);
		   l7vsServiceTable_head = tempVirtualServiceNode;
		}
	}
	l7vsServiceTable_head = NULL;

}


/* This method clears off the Linked list memory created for Real servers.
*  The linked list would be freed */
void freeRealServerTableMemory()
{
	l7vsRealTable *tempRealServerNode;

	if(l7vsRealTable_head != NULL)
	{
		while( l7vsRealTable_head ) {
		   tempRealServerNode = l7vsRealTable_head->next;
		   SNMP_FREE(l7vsRealTable_head);
		   l7vsRealTable_head = tempRealServerNode;
		}
	}
	l7vsRealTable_head = NULL;
}


/* This method populates the Linked list memory for Virtual services.
*  It in turn calls populateReal() which populates the Linked list for Real servers.
*  The data retrieval from L7VS Kernel Table is similar to that done in l7vsadm.c.
*  Taken from the method list_vs() of l7vsadm.c */
void populateServiceTable()
{
	struct l7vs_config_req_list_vs req;
	struct l7vs_config_rsp_list_vs rsp;
	struct l7vs_service_arg *sarg;
	struct iovec iov[2];
	void *buf;
	char *srvname;
	int bufsize;
	int i, off;
	int ret;
	char * wordToken;

	l7vsServiceTable* pServiceEntry;

	bufsize = L7VSAGENT_LIST_VS_BUFSIZE;

	retry:
	buf = malloc(bufsize);
	if (buf == NULL)
	{
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return;
	}

	req.cmd = L7VS_CONFIG_LIST_VS;
	ret = send_request(socket_num, &req, sizeof(req));
	if (ret < 0)
	{
		free(buf);
		return;
	}

	iov[0].iov_base = &rsp;
	iov[0].iov_len = sizeof(rsp);
	iov[1].iov_base = buf;
	iov[1].iov_len = bufsize;

	ret = recv_response(socket_num, iov, sizeof(iov) / sizeof(iov[0]));
	if (ret < 0)
	{
		free(buf);
		if (ret == L7VSAGENT_RECV_TOO_LARGE)
		{
			bufsize += L7VSAGENT_LIST_VS_BUFSIZE;
			goto retry;
		}
		return;
	}

	if (rsp.code != 0)
	{
		VANESSA_LOGGER_ERR(config_strerror(rsp.code));
		free(buf);
		return;
	}

	/* Put it in the SNMP structure */
	sarg = (struct l7vs_service_arg *)buf;
	off = 0;
	for (i = 0; i < rsp.num; i++)
	{
		srvname = get_servicename(&sarg->addr, 1);

		if (srvname == NULL) {
				break;
		}

		/* Create a row for the Virtual service Linked list */
		pServiceEntry = (l7vsServiceTable *) l7vsServiceTable_createEntry(i+1);

		if(pServiceEntry != NULL)
		{
			/* Set all the values of the Linked list*/

			strcpy(pServiceEntry->l7vsServiceSchedType, sarg->schedmod);

			/* Currently only TCP protocol is used. Thus it is set to 6 */
			pServiceEntry->l7vsServiceProto = 6;

			/* Get IP Address by parsing the string for the COLON character */
			wordToken = strtok(srvname, ":");
			strcpy(pServiceEntry->l7vsServiceAddr, wordToken);

			/* Just take the remainder of the string as the Port */
			wordToken = strtok(NULL, " ");
			pServiceEntry->l7vsServicePort = atol(wordToken);

			pServiceEntry->l7vsServicePersist = sarg->persist;
			pServiceEntry->l7vsServiceBackLog = sarg->backlog;

			/* Protocol module related entries */
			strcpy(pServiceEntry->l7vsProtocolModuleName, sarg->protomod);
			strcpy(pServiceEntry->l7vsProtocolModuleKey, sarg->protomod_key_string);
			strcpy(pServiceEntry->l7vsProtocolModuleString, sarg->protomod_opt_string);

			pServiceEntry->l7vsServiceReschedule = sarg->reschedule;

			free(srvname);

			/* Call the real service with the virtual service number
			  * Initialize to ZERO and then call the function */
			pServiceEntry->l7vsServiceNumDests = 0;
			pServiceEntry->l7vsServiceNumDests = populateReal(socket_num, sarg, i+1);

			off += sarg->len;
			if (off >= bufsize)
			{
				VANESSA_LOGGER_INFO("Message size did not match");
				VANESSA_LOGGER_INFO("version mismatch between"
				" l7vsd and l7vsagent?");
				break;
			}
		}
		sarg = (struct l7vs_service_arg *)((char *)buf + off);
	}

	free(buf);

}


/* This method populates the Linked list memory for real servers.
*  This is called by populateServiceTable.
*  The data retrieval from L7VS Kernel Table is similar to that done in l7vsadm.c.
*  Taken from the method list_rs() of l7vsadm.c */
int populateReal(int socket_num, struct l7vs_service_arg *sarg, int virt_service_num)
{
	struct l7vs_config_req_list_rs req;
	struct l7vs_config_rsp_list_vs rsp;
	struct l7vs_dest_arg *darg;
	struct iovec iov[2];
	char *srvname;
	int ret;
	int i, num;
	l7vsRealTable* pRealEntry;
	char * wordToken;

	num = L7VSAGENT_LIST_RS_RSNUM;
	retry:
	darg = (struct l7vs_dest_arg *)malloc(num * sizeof(*darg));

	if (darg == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return 0;
	}

	req.cmd = L7VS_CONFIG_LIST_RS;
	iov[0].iov_base = &req;
	iov[0].iov_len = sizeof(req);
	iov[1].iov_base = sarg;
	iov[1].iov_len = sarg->len;
	ret = sendv_request(socket_num, iov, sizeof(iov) / sizeof(iov[0]));

	if (ret < 0) {
		free(darg);
		return 0;
	}

	iov[0].iov_base = &rsp;
	iov[0].iov_len = sizeof(rsp);
	iov[1].iov_base = darg;
	iov[1].iov_len = num * sizeof(*darg);

	ret = recv_response(socket_num, iov, sizeof(iov) / sizeof(iov[0]));
	if (ret < 0) {
		free(darg);
		if (ret == L7VSAGENT_RECV_TOO_LARGE) {
			num = rsp.num;
			goto retry;
		}
		return 0;
	}

	if (rsp.code != 0) {
		VANESSA_LOGGER_ERR(config_strerror(rsp.code));
		return 0;
	}

	/* Put it in the SNMP structure */
	num = rsp.num;
	for (i = 0; i < num; i++)
	{
		srvname = get_servicename(&darg[i].addr, 1);
		if (srvname == NULL) {
			break;
		}

		/* Create a row for the Real server Linked list */
		pRealEntry = (l7vsRealTable*)l7vsRealTable_createEntry(virt_service_num, i+1);

		if(pRealEntry!=NULL)
		{
			/* Set all the values of the Linked list*/

			/* Get IP Address by parsing the string for the COLON character */
			wordToken = strtok(srvname, ":");
			strcpy(pRealEntry->l7vsRealServerAddr, wordToken);

			/* Just take the remainder of the string as the Port */
			wordToken = strtok(NULL, " ");
			pRealEntry->l7vsRealServerPort = atol(wordToken);

			/* Currently only MASQ is supported. Thus it is set to 0 */
			pRealEntry->l7vsRealServerFlag = 0;

			pRealEntry->l7vsRealServerWeight = darg[i].weight;
			pRealEntry->l7vsRealServerActiveConn = darg[i].nactive;
			pRealEntry->l7vsRealServerInactiveConn = darg[i].ninact;
		}
			free(srvname);
	}

	free(darg);
	return num;
}


/* Raise the Actual trap here */
int raise_trap(char * trap_message)
{
	char * wordToken;
	char change_type[30];
	char change_string[BUFSIZ];

    oid objid_snmptrap[] = { 1, 3, 6, 1, 6, 3, 1, 1, 4, 1, 0 };
    size_t objid_snmptrap_len = OID_LENGTH(objid_snmptrap);

	oid change_type_oid[] = { 1, 3, 6, 1, 4, 1, 9999, 1, 900, 1 };
   	size_t change_type_oid_len = OID_LENGTH(change_type_oid);

	oid change_str_oid[] = { 1, 3, 6, 1, 4, 1, 9999, 1, 900, 2 };
   	size_t change_str_oid_len = OID_LENGTH(change_str_oid);

    netsnmp_variable_list *notification_vars = NULL;

	/* For TRAP OID, we just initialise the first 9 members of the array.
	*  The tenth member trap_oid[9] would be initialised based on the Trap OID type */
	oid trap_oid[10] = { 1, 3, 6, 1, 4, 1, 9999, 1, 900 };
	size_t trap_oid_len = OID_LENGTH(trap_oid);

	/* Split into tokens. For "l7vsadm -C" since there is no ":" token,
	*  The entire string would be returned as it is */
	wordToken = strtok(trap_message, ":");
	strcpy(change_type, wordToken);

	/*  Populate the tenth member trap_oid[9] */
	if (strstr(change_type, "Virtual") != NULL)
		trap_oid[9] = 3;
	else if (strstr(change_type, "Real") != NULL)
		trap_oid[9] = 4;
	else if (strcmp(change_type, "Clear L7VS Table") == 0)
		trap_oid[9] = 5;
	else
		return 0; /* Something wrong here in the string */

	snmp_varlist_add_variable(&notification_vars,
		objid_snmptrap, objid_snmptrap_len,
		ASN_OBJECT_ID,
		(u_char *) trap_oid,
		trap_oid_len * sizeof(oid));

	snmp_varlist_add_variable(&notification_vars,
		change_type_oid,
		change_type_oid_len,
		ASN_OCTET_STR,
		(const u_char *) change_type,
		strlen(change_type));

	/* If it is for "l7vsadm -C" then just copy the text string, else, parse it into tokens
	 * Other than for l7vsadm -C, send the change string too */
	if (strcmp(change_type, "Clear L7VS Table") != 0)
	{
		/* Put some dummy token string. We just need the rest of the string */
		wordToken = strtok(NULL, "???");
		strcpy(change_string, wordToken);

		snmp_varlist_add_variable(&notification_vars,
			change_str_oid,
			change_str_oid_len,
			ASN_OCTET_STR,
			(const u_char *) change_string,
			strlen(change_string));
	}

    send_v2trap(notification_vars);
    snmp_free_varbind(notification_vars);

    return L7_OK;
}


/* Main function that is called when l7vsagent program is executed */
int main(int argc, char ** argv)
{

	int agentx_subagent=1; /* change this if you want to be a SNMP master agent */
	int background = 0; /* change this if you want to run in the background */
	int syslog = 0; /* change this if you want to use syslog */

	vanessa_logger_t *vl;

	/* PID and file handling related variables */
	FILE *fp, *fpTemp;
	char l7vsagent_pid_folder[30];
	char l7vsagent_string[100];
	char l7vsagent_pid[10];

	/* For signal handling of SIGINT, SIGTERM and SIGUSR1 */
	struct sigaction actINT_TERM, actUSR1;

	/* Open Vanessa logger for logging messages */
	vl = vanessa_logger_openlog_syslog(LOG_LOCAL0, "l7vsagent", LOG_INFO, 0);
	vanessa_logger_set(vl);

	/* Open the PID file to see whether the snmp agent process is already running */
	fp = fopen("/var/run/l7vsagent.pid", "r");
	if (fp != NULL)
	{
		fscanf(fp, "%s", l7vsagent_pid);
		fclose(fp);

		sprintf(l7vsagent_pid_folder, "/proc/%s/cmdline", l7vsagent_pid);

		/* Confirm whether the existing PID file is not stale */
		fpTemp = fopen(l7vsagent_pid_folder, "r");
		if (fpTemp != NULL)
		{
			fscanf(fpTemp, "%s", l7vsagent_string);
			fclose(fpTemp);

			/* Using strstr, confirm that the PID file is not stale or does not
			belong to some other process */
			if (strstr(l7vsagent_string, "l7vsagent") != NULL)
			{
				VANESSA_LOGGER_ERR("l7vsagent is already running. Check the system file under /proc folder");
				exit(1);
			}
		}
	}

	/* set msgid = 0 */
	msgid = 0;

	/* print log errors to syslog or stderr */
	if (syslog)
		snmp_enable_calllog();
	else
		snmp_enable_stderrlog();

	/* we're an agentx subagent? */
	if (agentx_subagent) {
		/* make us a agentx client. */
		netsnmp_ds_set_boolean(NETSNMP_DS_APPLICATION_ID, NETSNMP_DS_AGENT_ROLE, 1);
	}

	/* run in background, if requested */
	if (background && netsnmp_daemonize(1, !syslog))
		exit(1);

	/* initialize tcpip, if necessary */
	SOCK_STARTUP;

	/* initialize the agent library */
	init_agent("l7vsagent");

	/* Open a file for PID writing purposes and write PID value. This is read
	  * by config.c for raising SIGUSR1 */
	fp = fopen("/var/run/l7vsagent.pid", "w");
	if (fp == NULL) {
		VANESSA_LOGGER_ERR("Error in opening PID file for l7vsagent at /var/run folder");
		exit(1);
	} else {
		fprintf(fp, "%d", (int) getpid());
		fclose(fp);
	}

	/* Create key for the message queue */
	key = ftok("/var/run/l7vsagent.pid", 'b');
	if ((int) key == -1)
	{
		VANESSA_LOGGER_ERR("Error in creating Key for the message queue");
		exit(1);
	}

	/* Initialises the L7VS module */
	l7vs_module_init(NULL);

	/* Create the socket for communicating with L7VS */
	socket_num = create_socket();
	if (socket_num < 0) {
		exit(1);
	}

	/* mib code: init all the objects */
	init_l7vsClusterInfo();
	init_l7vsServiceTable();
	init_l7vsRealTable();

	init_snmp("l7vsagent");

	/* If we're going to be a snmp master agent, initial the ports */
	if (!agentx_subagent)
		init_master_agent();  /* open the port to listen on (defaults to udp:161) */

	keep_running = 1;

	/* Handle TERM or INT signal interrupts */
	actINT_TERM.sa_handler = stop_server;
	sigemptyset(&actINT_TERM.sa_mask);
	actINT_TERM.sa_flags = 0;

	sigaction(SIGINT, &actINT_TERM, 0);
	sigaction(SIGTERM, &actINT_TERM, 0);

	/* The method check_l7vs_trap is called whenever there is a SIGUSR1 event.
	*  SIGUSR1 is called from config.c whenever there is a change in the L7VS Kernel Table */
	actUSR1.sa_handler = check_l7vs_trap;
	sigemptyset(&actUSR1.sa_mask);
	actUSR1.sa_flags = 0;
	sigaction(SIGUSR1, &actUSR1, 0);

	/* Log message indicating that the agent is UP */
	VANESSA_LOGGER_INFO("l7vsagent is up and running");

	/* your main loop here... */
	while(keep_running) {
		/* if you use select(), see snmp_select_info() in snmp_api(3) */
		/*     --- OR ---  */
		agent_check_and_process(1); /* 0 == don't block */
	}

	/* Destroy the socket that was earlier created */
	destroy_socket();

	/* Clear the linked list just in case */
	freeVirtualServiceTableMemory();
	freeRealServerTableMemory();

	/* at shutdown time */
	snmp_shutdown("l7vsagent");
	SOCK_CLEANUP;

	/* Remove the message queue */
	if (msgid != 0)
	{
		if (msgctl(msgid, IPC_RMID, 0) ==  -1)
			VANESSA_LOGGER_ERR_UNSAFE("Error in deleting the message queue %d. Clear up manually using ipcrm", msgid);
	}

	/* Remove the created PID file */
	system("rm -f /var/run/l7vsagent.pid");

	/* Log message indicating that the agent is UP */
	VANESSA_LOGGER_INFO("Exiting l7vsagent");

	return 0;
}
