#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>
#include <signal.h>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <vanessa_logger.h>

static int keep_running;

int raise_trap(char*);


void stop_server(int a)
{
    keep_running = 0;
}


/* Main program that monitors Ultra Monkey processes and raises traps
 * Processes l7vsd, l7directord, ldirectord and heartbeat as seen in ps -ax would be monitored.
 * Modules tcps and ip_vs would be monitored */
int main(int argc, char **argv)
{
	int agentx_subagent=1; /* change this if you want to be a SNMP master agent */
	int background = 0; /* change this if you want to run in the background */
	int syslog = 0; /* change this if you want to use syslog */

	vanessa_logger_t *vl;
	FILE *fp;
	char buffer[BUFSIZ + 1];
	int chars_read;

	/* For signal handling */
	struct sigaction act;

	/* Flags for the processes */
	int l7vsd_flag = 0;
	int l7directord_flag = 0;
	int ldirectord_flag = 0;
	int heartbeat_flag = 0;
	int tcps_flag = 0;
	int ip_vs_flag = 0;

	/* Open Vanessa logger for logging messages */
	vl = vanessa_logger_openlog_syslog(LOG_LOCAL0, "umagent", LOG_INFO, 0);
	vanessa_logger_set(vl);

	/* print log errors to syslog or stderr */
	if (syslog)
		snmp_enable_calllog();
	else
		snmp_enable_stderrlog();

	/* we're an agentx subagent? */
	if (agentx_subagent)
	{
	    /* make us a agentx client. */
	    netsnmp_ds_set_boolean(NETSNMP_DS_APPLICATION_ID, NETSNMP_DS_AGENT_ROLE, 1);
	}

	/* run in background, if requested */
	if (background && netsnmp_daemonize(1, !syslog))
		exit(1);

	/* initialize tcpip, if necessary */
	SOCK_STARTUP;

	/* initialize the agent library */
	init_agent("umagent");

	init_snmp("umagent");

	/* If we're going to be a snmp master agent, initial the ports */
	if (!agentx_subagent)
		init_master_agent();  /* open the port to listen on (defaults to udp:161) */

	/* Signal handling of SIGINT and SIGTERM */
	keep_running = 1;

	act.sa_handler = stop_server;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;

	sigaction(SIGINT, &act, 0);
	sigaction(SIGTERM, &act, 0);

	VANESSA_LOGGER_INFO("umagent is up and running");

	/* Main loop here... */
	while(keep_running)
	{
		int l7vsd_running = 0;
		int l7directord_running = 0;
		int ldirectord_running = 0;
		int heartbeat_running = 0;
		int tcps_running = 0;
		int ip_vs_running = 0;

		/* Initialize buffer to NULL */
		memset(buffer, '\0', sizeof(buffer));

		/* Check the agent whether it is running
		*  This is necessary in cases where the SNMPD master daemon has stopped and restarted */
		agent_check_and_process(1);

		/* Get status for l7directord, ldirectord, heartbeat and l7vsd
		 * These could be obtained by "ps -ax" command */
		fp = popen("ps -ax -w --no-headers", "r");

		/* Reading the PIPE */
		if (fp != NULL)
		{
			chars_read = fread(buffer, sizeof(char), BUFSIZ, fp);
			buffer[chars_read - 1] = '\0';

 			while (chars_read > 0)
			{
				/* NULL Terminate the string */
				buffer[chars_read - 1] = '\0';

				if (strstr(buffer, "l7vsd") != NULL)
					l7vsd_running = 1;

				if (strstr(buffer, "l7directord") != NULL)
					l7directord_running = 1;

				if (strstr(buffer, "ldirectord") != NULL)
					ldirectord_running = 1;

				if (strstr(buffer, "heartbeat") != NULL)
					heartbeat_running = 1;

				chars_read = fread(buffer, sizeof(char), BUFSIZ, fp);

			} /* while (chars_read > 0) */

			/* Close the PIPE */
			pclose(fp);
		} /* if (fp != NULL) */

		/* Check if l7vsd is running and raise trap accordingly
		   l7vsd_flag == 0 is used for checking the very first time */
		if (l7vsd_running
			&& (l7vsd_flag == 0 || l7vsd_flag == 1))
		{
			/* Raise l7vsd started trap */
			raise_trap("l7vsd is running");

			/* Set flag so as to not raise the trap repeatedly */
			l7vsd_flag = 2;

		}
		else if ((!l7vsd_running)
			&& (l7vsd_flag == 0 || l7vsd_flag == 2))
		{
			/* raise l7vsd not running trap */
			raise_trap("l7vsd is not running");

			/* Set flag so as to not raise the trap repeatedly */
			l7vsd_flag = 1;
		}

		/* Check if l7directord is running and raise trap accordingly */
		if (l7directord_running
			&& (l7directord_flag == 0 || l7directord_flag == 1))
		{
			/* Raise l7directord started trap */
			raise_trap("l7directord is running");

			/* Set flag so as to not raise the trap repeatedly */
			l7directord_flag = 2;
		}
		else if ((!l7directord_running)
			&& (l7directord_flag == 0 || l7directord_flag == 2))
		{
			/*  Raise l7directord not running trap */
			raise_trap("l7directord is not running");

			/* Set flag so as to not raise the trap repeatedly */
			l7directord_flag = 1;
		}

		/* Check if ldirectord is running and raise trap accordingly */
		if (ldirectord_running
			&& (ldirectord_flag == 0 || ldirectord_flag == 1))
		{
			/* Raise ldirectord started trap */
			raise_trap("ldirectord is running");

			/* Set flag so as to not raise the trap repeatedly */
			ldirectord_flag = 2;
		}
		else if ((!ldirectord_running)
			&& (ldirectord_flag == 0 || ldirectord_flag == 2))
		{
			/* Raise ldirectord not running trap */
			raise_trap("ldirectord is not running");

			/* Set flag so as to not raise the trap repeatedly */
			ldirectord_flag = 1;
		}

		/* Check if heartbeat is running and raise trap accordingly */
		if (heartbeat_running
			&& (heartbeat_flag == 0 || heartbeat_flag == 1))
		{
			/* Raise heartbeat started trap */
			raise_trap("heartbeat is running");

			/* Set flag so as to not raise the trap repeatedly */
			heartbeat_flag = 2;
		}
		else if ((!heartbeat_running)
			&& (heartbeat_flag == 0 || heartbeat_flag == 2))
		{
			/*  Raise heartbeat not running trap */
			raise_trap("heartbeat is not running");

			/* Set flag so as to not raise the trap repeatedly */
			heartbeat_flag = 1;
		}


		/* Get status for tcps and ip_vs.
		 * These are Kernel level modules and can be obtained by "lsmod" command */
		fp = popen("lsmod", "r");

		/* Reading the PIPE */
		if (fp != NULL)
		{
			chars_read = fread(buffer, sizeof(char), BUFSIZ, fp);
			buffer[chars_read - 1] = '\0';

			while (chars_read > 0)
			{
				/* NULL Terminate the string */
				buffer[chars_read - 1] = '\0';

				if (strstr(buffer, "tcps") != NULL)
					tcps_running = 1;

				if (strstr(buffer, "ip_vs") != NULL)
					ip_vs_running = 1;

				chars_read = fread(buffer, sizeof(char), BUFSIZ, fp);
			} /* while (chars_read > 0) */

			/* Close the PIPE */
			pclose(fp);
		} /* if (fp != NULL) */

		/* Check if tcps is running and raise trap accordingly */
		if (tcps_running
			&& (tcps_flag == 0 || tcps_flag == 1))
		{
			/* Raise tcps started trap */
			raise_trap("tcps is running");

			/* Set flag so as to not raise the trap repeatedly */
			tcps_flag = 2;
		}
		else if ((!tcps_running)
				&& (tcps_flag == 0 || tcps_flag == 2))
		{
			/* Raise tcps not running trap */
			raise_trap("tcps is not running");

			/* Set flag so as to not raise the trap repeatedly */
			tcps_flag = 1;
		}

		/* Check if ip_vs is running and raise trap accordingly */
		if (ip_vs_running
			&& (ip_vs_flag == 0 || ip_vs_flag == 1))
		{
			/* Raise ip_vs started trap */
			raise_trap("ip_vs is running");

			/* Set flag so as to not raise the trap repeatedly */
			ip_vs_flag = 2;
		}
		else if ((!ip_vs_running)
				&& (ip_vs_flag == 0 || ip_vs_flag == 2))
		{
			/*  Raise ip_vs not running trap */
			raise_trap("ip_vs is not running");

			/* Set flag so as to not raise the trap repeatedly */
			ip_vs_flag = 1;
		}

	} /* while(keep_running) */

	/* at shutdown time */
	snmp_shutdown("umagent");
	SOCK_CLEANUP;

	VANESSA_LOGGER_INFO("Exiting umagent");
	return 0;
}


/* Method for raising the Actual trap */
int raise_trap(char* trap_message)
{
	char change_string[BUFSIZ];

	oid objid_snmptrap[] = { 1, 3, 6, 1, 6, 3, 1, 1, 4, 1, 0 };
	size_t objid_snmptrap_len = OID_LENGTH(objid_snmptrap);

	oid change_str_oid[] = { 1, 3, 6, 1, 4, 1, 9999, 2, 900, 1 };
   	size_t change_str_oid_len = OID_LENGTH(change_str_oid);

	oid trap_oid[] = { 1, 3, 6, 1, 4, 1, 9999, 2, 900, 2 };
	size_t trap_oid_len = OID_LENGTH(trap_oid);

	netsnmp_variable_list *notification_vars = NULL;

	snmp_varlist_add_variable(&notification_vars,
		objid_snmptrap, objid_snmptrap_len,
		ASN_OBJECT_ID,
		(u_char *) trap_oid,
		trap_oid_len * sizeof(oid));

	strcpy(change_string, trap_message);

	snmp_varlist_add_variable(&notification_vars,
		change_str_oid,
		change_str_oid_len,
		ASN_OCTET_STR,
		(const u_char *) change_string,
		strlen(change_string));

	send_v2trap(notification_vars);
	snmp_free_varbind(notification_vars);

	return 1;
}
