#include "Mix/Private/Graphics/DX11/TextureDynamic.h"
#include "Mix/Private/Graphics/DX11/Device.h"

namespace Mix{ namespace Graphics{ namespace DX11{

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics::DX11::TextureDynamic
////////////////////////////////////////////////////////////////////////////////////////////////////

TextureDynamic* TextureDynamic::CreateInstance(	Mix::Graphics::DX11::Device* pDevice,
												UInt32 width, UInt32 height, Mix::Graphics::FORMAT format,
												ID3D11Texture2D* pTexture, ID3D11ShaderResourceView* pSRView )
{
	return MIX_LIB_NEW_T( Mix::Memory::SECTION_GRAPHICS, TextureDynamic, pDevice, width, height, format, pTexture, pSRView );
}

TextureDynamic::TextureDynamic(	Mix::Graphics::DX11::Device* pDevice,
								UInt32 width, UInt32 height, Mix::Graphics::FORMAT format,
								ID3D11Texture2D* pTexture, ID3D11ShaderResourceView* pSRView ) :
DeviceObject( pDevice ),
m_pDevice( pDevice ),
m_Width( width ),
m_Height( height ),
m_Format( format ),
m_pTexture( pTexture ),
m_pSRView( pSRView ),
m_bLock( False )
{
	MIX_ASSERT( pDevice != NULL );

	MIX_ADD_REF( pDevice );
	m_pDevice = pDevice;
}

TextureDynamic::~TextureDynamic( void )
{
	MIX_RELEASE( m_pSRView );
	MIX_RELEASE( m_pTexture );
	MIX_RELEASE( m_pDevice );
}

void TextureDynamic::Attach( ID3D11Texture2D* pD3DTextureDynamic, ID3D11ShaderResourceView* pShaderResourceView )
{
	D3D11_TEXTURE2D_DESC desc;

	m_pTexture = pD3DTextureDynamic;
	m_pSRView = pShaderResourceView;

	m_pTexture->GetDesc( &desc );

	m_Width = desc.Width;
	m_Height = desc.Height;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics::DX11::Texture
////////////////////////////////////////////////////////////////////////////////////////////////////

ID3D11Resource* TextureDynamic::GetInterface( void )
{
	return m_pTexture;
}

ID3D11ShaderResourceView* TextureDynamic::GetSRView( void )
{
	return m_pSRView;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics::ITexture
////////////////////////////////////////////////////////////////////////////////////////////////////

Mix::Graphics::ITexture::DIMENSION_TYPE TextureDynamic::GetDimensionType( void ) const
{
	return ITexture::DIMENSION_PLANE;
}

Mix::Graphics::ITexture::BIND_TYPE TextureDynamic::GetBindType( void ) const
{
	return Mix::Graphics::ITexture::BIND_DEFAULT;
}

UInt32 TextureDynamic::GetAccessFlags( void ) const
{
	return ITexture::ACCESS_CPU_WRITE | ITexture::ACCESS_GPU_READ;
}

UInt32 TextureDynamic::GetWidth( void ) const
{
	return m_Width;
}

UInt32 TextureDynamic::GetHeight( void ) const
{
	return m_Height;
}

Mix::Graphics::FORMAT TextureDynamic::GetFormat( void ) const
{
	return m_Format;
}

const wchar_t* TextureDynamic::GetFilePath( void ) const
{
	return Mix::STR_NULL;
}

Boolean TextureDynamic::Lock( Mix::Graphics::ITexture::LOCK& lock )
{
	if( ( m_pTexture == NULL ) || ( m_bLock == True ) )
	{
		return False;
	}

	D3D11_MAPPED_SUBRESOURCE mapped;

	if( m_pDevice->Map( m_pTexture, True, mapped ) != S_OK )
	{
		return False;
	}

	lock.bits = mapped.pData;
	lock.pitch = mapped.RowPitch;

	m_bLock = True;

	return True;
}

void TextureDynamic::Unlock( void )
{
	if( ( m_pTexture == NULL ) || ( m_bLock == False ) )
	{
		return;
	}

	m_pDevice->Unmap( m_pTexture );
	m_bLock = False;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics::IDeviceObject
////////////////////////////////////////////////////////////////////////////////////////////////////

Mix::Graphics::IDeviceObject::TYPE TextureDynamic::GetType( void ) const
{
	return Mix::Graphics::IDeviceObject::TEXTURE;
}

}}}
